/*
 *			GPAC - MPEG-4 Systems C Development Kit
 *
 *			Copyright (c) Jean Le Feuvre 2000-2003 
 *					All rights reserved
 *
 *  This file is part of GPAC / MP4 reader plugin
 *
 *  GPAC is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *   
 *  GPAC is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *   
 *  You should have received a copy of the GNU General Public License
 *  along with GNU Make; see the file COPYING.  If not, write to
 *  the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA. 
 *		
 */

#include "mp3_in.h"
#include <gpac/m4_author.h>


static Bool MP3_CanHandleURL(NetClientPlugin *plug, const char *url)
{
	char *sExt, szExt[30];
	const char *szExtList;
	if (!strnicmp(url, "rtsp://", 7)) return 0;
	sExt = strrchr(url, '.');
	if (!sExt) return 0;
	strcpy(szExt, &sExt[1]);
	strlwr(szExt);

	/*check supported extension list*/
	szExtList = PMI_GetOpt(plug, "FileAssociations", "GPAC MP3 Reader");
	if (!szExtList) {
		szExtList = "mp3";
		PMI_SetOpt(plug, "FileAssociations", "GPAC MP3 Reader", szExtList);
	}
	if (strstr(szExtList, szExt)) return 1;
	return 0;
}

static Bool mp3_is_local(const char *url)
{
	if (!strnicmp(url, "file://", 7)) return 1;
	if (strstr(url, "://")) return 0;
	return 1;
}

static Bool MP3_Configure(MP3Reader *read)
{
	u32 hdr, size, pos;
	if (!read->stream) return 0;

	hdr = MP3_GetNextHeader(read->stream);
	if (!hdr) return 0;
	read->sample_rate = MP3_GetSamplingRate(hdr);
	read->oti = MP3_GetObjectTypeIndication(hdr);
	fseek(read->stream, 0, SEEK_SET);
	if (!read->oti) return 0;

	/*we don't have the full file...*/
	if (read->is_remote) return	1;

	read->duration = MP3_GetSamplesPerFrame(hdr);
	size = MP3_GetFrameSize(hdr);
	pos = ftell(read->stream);
	fseek(read->stream, pos + size - 4, SEEK_SET);
	while (1) {
		hdr = MP3_GetNextHeader(read->stream);
		if (!hdr) break;
		read->duration += MP3_GetSamplesPerFrame(hdr);
		size = MP3_GetFrameSize(hdr);
		pos = ftell(read->stream);
		fseek(read->stream, pos + size - 4, SEEK_SET);
	}
	fseek(read->stream, 0, SEEK_SET);
	return 1;
}


void MP3_OnData(NetDownloader *dnload, char *data, u32 data_size)
{
	M4Err e;
	char sMsg[1024];
	Float perc;
	const char *szCache;
	MP3Reader *read = (MP3Reader *) dnload->user_cbck;

	e = M4OK;
	if (read->dnload != dnload) e = M4ServiceError;

	if (!dnload->bytes_done) {
		switch (dnload->net_status) {
		case NM_WaitingForAck:
			NM_OnMessage(read->service, M4OK, "Connecting...");
			return;
		case NM_Connected:
			NM_OnMessage(read->service, M4OK, "Connected");
			return;
		}
	}

	else if ((dnload->error == M4EOF) && read->stream) {
		read->is_remote = 0;
		return;
	}
	else if (dnload->error >= M4OK) {
		/*notify some connection / ...*/
		if (dnload->total_size) {
			perc = (Float) (100 * dnload->bytes_done) / (Float) dnload->total_size;
			sprintf(sMsg, "Download %.2f %% (%.2f kBps)", perc, dnload->bytes_per_sec/1024);
			NM_OnMessage(read->service, M4OK, sMsg);
		}
		if (read->stream) return;

		/*open service*/
		szCache = dnload->GetCacheFileName(dnload);
		if (!szCache) e = M4InvalidPlugin;
		else {
			read->stream = fopen((char *) szCache, "rb");
			if (!read->stream) e = M4ServiceError;
			else {
				/*if full file at once (in cache) parse duration*/
				if (dnload->bytes_done==dnload->total_size) read->is_remote = 0;
				/*not enough data*/
				if (!MP3_Configure(read)) {
					/*bad data - there's likely some ID3 around...*/
					if (dnload->bytes_done>10*1024) {
						e = M4CorruptedData;
					} else {
						fclose(read->stream);
						read->stream = NULL;
						return;
					}
				}
			}
		}
	}
	/*error in service*/
	else e = dnload->error;

	/*OK confirm*/
	if (read->status == NM_Setup) {
		NM_OnConnect(read->service, NULL, e);
		read->status = e ? NM_Unavailable : NM_Connected;
	}
}

void mp3_download_file(NetClientPlugin *plug, char *url)
{
	M4Err e;
	u32 count, i;
	MP3Reader *read = (MP3Reader*) plug->priv;

	read->status = NM_Setup;

	count = PMI_GetPluginsCount((BaseInterface *) plug);
	for (i=0; i<count; i++) {
		if (PMI_LoadInterface((BaseInterface *) plug, i, M4NETDOWNLOADER, (void **) &read->dnload)) {
			if (read->dnload->CanHandleURL(url)) break;
			PM_ShutdownInterface(read->dnload);
		}
		read->dnload = NULL;
	}
	if (!read->dnload) {
		NM_OnConnect(read->service, NULL, M4UnsupportedURL);
		return;
	}
	read->dnload->user_cbck = read;
	read->dnload->OnData = MP3_OnData;
	/*download*/
	e = read->dnload->Connect(read->dnload, url, 0);
	if (e) {
		NM_OnConnect(read->service, NULL, e);
		read->status = NM_Unavailable;
	}
	/*service confirm is done once fetched*/
}

static M4Err MP3_ConnectService(NetClientPlugin *plug, LPNETSERVICE serv, const char *url)
{
	MP3Reader *read = plug->priv;
	read->service = serv;

	if (read->dnload) read->dnload->Close(read->dnload);

	/*remote fetch*/
	read->is_remote = !mp3_is_local(url);
	if (read->is_remote) {
		mp3_download_file(plug, (char *) url);
		return M4OK;
	}

	read->stream = fopen(url, "rb");
	if (read->stream && !MP3_Configure(read)) {
		fclose(read->stream);
		read->stream = 0;
	}
	
	read->status = read->stream ? NM_Connected : NM_Unavailable;

	NM_OnConnect(serv, NULL, read->stream ? M4OK : M4URLNotFound);
	return M4OK;
}

static M4Err MP3_CloseService(NetClientPlugin *plug, Bool immediate_shutdown)
{
	MP3Reader *read = plug->priv;
	if (read->stream) fclose(read->stream);
	read->stream = NULL;
	read->status = NM_Setup;

	NM_OnDisconnect(read->service, NULL, M4OK);
	return M4OK;
}

static M4Err MP3_Get_MPEG4_IOD(NetClientPlugin *plug, u32 expect_type, char **raw_iod, u32 *raw_iod_size)
{
	ESDescriptor *esd;
	M4Err e;
	BitStream *bs;
	MP3Reader *read = plug->priv;
	ObjectDescriptor *od = (ObjectDescriptor *) OD_NewDescriptor(ObjectDescriptor_Tag);

	od->objectDescriptorID = 1;
	/*audio object*/
	if (expect_type==NM_OD_AUDIO) {
		esd = OD_NewESDescriptor(0);
		esd->slConfig->timestampResolution = read->sample_rate;
		esd->decoderConfig->streamType = M4ST_AUDIO;
		esd->decoderConfig->objectTypeIndication = read->oti;
		esd->ESID = 3;
		ChainAddEntry(od->ESDescriptors, esd);
		e = OD_EncDesc((Descriptor *) od, raw_iod, raw_iod_size);
		OD_DeleteDescriptor((Descriptor **)&od);
		return e;
	}
	/*inline scene*/
	/*OD ESD*/
	esd = OD_NewESDescriptor(0);
	esd->slConfig->timestampResolution = 1000;
	esd->decoderConfig->streamType = M4ST_OD;
	esd->decoderConfig->objectTypeIndication = 0x01;
	esd->ESID = 1;
	ChainAddEntry(od->ESDescriptors, esd);
	/*BIFS ESD*/
	esd = OD_NewESDescriptor(0);
	esd->slConfig->timestampResolution = 1000;
	esd->decoderConfig->streamType = M4ST_BIFS;
	esd->decoderConfig->objectTypeIndication = 0x01;
	esd->ESID = 2;
	esd->OCRESID = 1;
	bs = NewBitStream(NULL, 0, BS_WRITE);
	BS_WriteInt(bs, 0, 10);
	BS_WriteInt(bs, 1, 1);
	BS_WriteInt(bs, 1, 1);	/*pixel metrics*/
	BS_WriteInt(bs, 0, 1);	/*no size info*/
	BS_GetContent(bs, (unsigned char **) &esd->decoderConfig->decoderSpecificInfo->data, &esd->decoderConfig->decoderSpecificInfo->dataLength);
	DeleteBitStream(bs);
	ChainAddEntry(od->ESDescriptors, esd);
	e = OD_EncDesc((Descriptor *) od, raw_iod, raw_iod_size);
	OD_DeleteDescriptor((Descriptor **)&od);
	return e;
}

static M4Err MP3_ConnectChannel(NetClientPlugin *plug, LPNETCHANNEL channel, const char *url, Bool upstream)
{
	u32 ES_ID;
	M4Err e;
	MP3Reader *read = plug->priv;

	e = M4ServiceError;
	if ((read->es_ch==channel) || (read->bifs_ch==channel) || (read->od_ch==channel)) goto exit;

	e = M4OK;
	if (strstr(url, "ES_ID")) {
		sscanf(url, "ES_ID=%d", &ES_ID);
	}
	/*URL setup*/
	else if (!read->es_ch && MP3_CanHandleURL(plug, url)) ES_ID = 3;

	switch (ES_ID) {
	case 2:
		read->bifs_status = NM_Connected;
		read->bifs_ch = channel;
		break;
	case 1:
		read->od_status = NM_Connected;
		read->od_ch = channel;
		break;
	case 3:
		read->es_status = NM_Connected;
		read->es_ch = channel;
		break;
	}

exit:
	NM_OnConnect(read->service, channel, e);
	return e;
}

static M4Err MP3_DisconnectChannel(NetClientPlugin *plug, LPNETCHANNEL channel)
{
	MP3Reader *read = plug->priv;

	if (read->es_ch == channel) {
		read->es_status = NM_Disconnected;
		read->es_ch = NULL;
	} else if (read->od_ch == channel) {
		read->od_status = NM_Disconnected;
		read->od_ch = NULL;
	} else if (read->bifs_ch == channel) {
		read->bifs_status = NM_Disconnected;
		read->bifs_ch = NULL;
	}
	NM_OnDisconnect(read->service, channel, M4OK);
	return M4OK;
}

static M4Err MP3_GetStatus(NetClientPlugin *plug, LPNETCHANNEL channel, u32 *status)
{
	MP3Reader *read = plug->priv;
	if (!channel) *status = read->status;
	else if (read->es_ch == channel) *status = read->es_status;
	else if (read->od_ch == channel) *status = read->od_status;
	else if (read->bifs_ch == channel) *status = read->bifs_status;
	return M4OK;
}

static M4Err MP3_ServiceCommand(NetClientPlugin *plug, NetworkCommand *com)
{
	MP3Reader *read = plug->priv;

	if (!com->on_channel) return M4NotSupported;
	switch (com->command_type) {
	case CHAN_SET_PADDING:
		read->pad_bytes = com->padding_bytes;
		return M4OK;
	case CHAN_DURATION:
		com->duration = read->duration;
		com->duration /= read->sample_rate;
		return M4OK;
	case CHAN_PLAY:
		read->start_range = com->start_range;
		read->end_range = com->end_range;
		read->current_time = 0;
		if (read->stream) fseek(read->stream, 0, SEEK_SET);

		if (read->es_ch == com->on_channel) { 
			read->es_status = NM_Running; 
			read->es_done = 0; 
			/*PLAY after complete download, estimate duration*/
			if (!read->is_remote && !read->duration) {
				MP3_Configure(read);
				if (read->duration) {
					NetworkCommand rcfg;
					rcfg.on_channel = read->es_ch;
					rcfg.command_type = CHAN_DURATION;
					rcfg.duration = read->duration;
					rcfg.duration /= read->sample_rate;
					NM_OnCommand(read->service, &rcfg);
				}
			}
		}
		else if (read->bifs_ch == com->on_channel) { read->bifs_status = NM_Running; read->bifs_done = 0; }
		else if (read->od_ch == com->on_channel) { read->od_status = NM_Running; read->od_done = 0; }
		return M4OK;
	case CHAN_STOP:
		if (read->es_ch == com->on_channel) read->es_status = NM_Connected;
		else if (read->bifs_ch == com->on_channel) read->bifs_status = NM_Connected;
		else if (read->od_ch == com->on_channel) read->od_status = NM_Connected;
		return M4OK;
	default:
		return M4OK;
	}
}


static M4Err MP3_ChannelGetSLP(NetClientPlugin *plug, LPNETCHANNEL channel, char **out_data_ptr, u32 *out_data_size, struct tagSLHeader *out_sl_hdr, Bool *sl_compressed, M4Err *out_reception_status, Bool *is_new_data)
{
	u32 pos, hdr, start_from;
	MP3Reader *read = plug->priv;

	*out_reception_status = M4OK;
	*sl_compressed = 0;
	*is_new_data = 0;

	memset(&read->sl_hdr, 0, sizeof(SLHeader));
	read->sl_hdr.randomAccessPointFlag = 1;
	read->sl_hdr.compositionTimeStampFlag = 1;

	if (read->bifs_ch == channel) {
		if (read->bifs_done) {
			*out_reception_status = M4EOF;
			return M4OK;
		}
		read->sl_hdr.compositionTimeStamp = (u64) (read->start_range * 1000);
		*out_sl_hdr = read->sl_hdr;
		*out_data_ptr = (char *) ISMA_BIFS_AUDIO;
		*out_data_size = 8;
		return M4OK;
	}
	if (read->od_ch == channel) {
		LPODCODEC codec;
		ObjectDescriptor *od;
		ObjectDescriptorUpdate *odU;
		ESDescriptor *esd;
		if (read->od_done) {
			*out_reception_status = M4EOF;
			return M4OK;
		}
		read->sl_hdr.compositionTimeStamp = (u64) (read->start_range * 1000);
		*out_sl_hdr = read->sl_hdr;
		if (!read->od_data) {
			*is_new_data = 1;
			odU = (ObjectDescriptorUpdate *) OD_NewCommand(ODUpdate_Tag);
			od = (ObjectDescriptor *) OD_NewDescriptor(ObjectDescriptor_Tag);
			od->objectDescriptorID = ISMA_AUDIO_OD_ID;
			esd = OD_NewESDescriptor(0);
			esd->slConfig->timestampResolution = read->sample_rate;
			esd->ESID = 3;
			esd->OCRESID = 1;
			esd->decoderConfig->streamType = M4ST_AUDIO;
			esd->decoderConfig->objectTypeIndication = read->oti;
			ChainAddEntry(od->ESDescriptors, esd);
			ChainAddEntry(odU->objectDescriptors, od);
			codec = OD_NewCodec(OD_WRITE);
			OD_AddCommand(codec, (ODCommand *)odU);
			OD_EncodeAU(codec);
			OD_GetEncodedAU(codec, &read->od_data, &read->od_data_size);
			OD_DeleteCodec(codec);
		}
		*out_data_ptr = read->od_data;
		*out_data_size = read->od_data_size;
		return M4OK;
	}

	if (read->es_ch != channel) return M4ChannelNotFound;

	/*fetching es data*/
	if (read->es_done) {
		*out_reception_status = M4EOF;
		return M4OK;
	}
	if (!read->es_data) {
		if (!read->stream) {
			*out_data_ptr = NULL;
			*out_data_size = 0;
			return M4OK;
		}
		*is_new_data = 1;

fetch_next:
		pos = ftell(read->stream);
		hdr = MP3_GetNextHeader(read->stream);
		if (!hdr) {
			if (!read->dnload) {
				*out_reception_status = M4EOF;
				read->es_done = 1;
			} else {
				fseek(read->stream, pos, SEEK_SET);
				*out_reception_status = M4OK;
			}
			return M4OK;
		}
		read->es_data_size = MP3_GetFrameSize(hdr);
		if (!read->es_data_size) {
			*out_reception_status = M4EOF;
			read->es_done = 1;
			return M4OK;
		}
		read->nb_samp = MP3_GetSamplesPerFrame(hdr);
		/*we're seeking*/
		if (read->start_range && read->duration) {
			start_from = (u32) (read->start_range * read->sample_rate);
			if (read->current_time + read->nb_samp < start_from) {
				read->current_time += read->nb_samp;
				goto fetch_next;
			} else {
				read->start_range = 0;
			}
		}
		
		read->sl_hdr.compositionTimeStamp = read->current_time;

		read->es_data = malloc(sizeof(char) * (read->es_data_size+read->pad_bytes));
		read->es_data[0] = (hdr >> 24) & 0xFF;
		read->es_data[1] = (hdr >> 16) & 0xFF;
		read->es_data[2] = (hdr >> 8) & 0xFF;
		read->es_data[3] = (hdr ) & 0xFF;
		/*end of file*/
		if (fread(&read->es_data[4], 1, read->es_data_size - 4, read->stream) != read->es_data_size-4) {
			free(read->es_data);
			read->es_data = NULL;
			if (read->is_remote) {
				fseek(read->stream, pos, SEEK_SET);
				*out_reception_status = M4OK;
			} else {
				*out_reception_status = M4EOF;
			}
			return M4OK;
		}
		if (read->pad_bytes) memset(read->es_data + read->es_data_size, 0, sizeof(char) * read->pad_bytes);
	}
	*out_sl_hdr = read->sl_hdr;
	*out_data_ptr = read->es_data;
	*out_data_size = read->es_data_size;
	return M4OK;
}

static M4Err MP3_ChannelReleaseSLP(NetClientPlugin *plug, LPNETCHANNEL channel)
{
	MP3Reader *read = plug->priv;

	if (read->es_ch == channel) {
		if (!read->es_data) return M4BadParam;
		free(read->es_data);
		read->es_data = NULL;
		read->current_time += read->nb_samp;
		return M4OK;
	}
	if (read->bifs_ch == channel) {
		read->bifs_done = 1;
		return M4OK;
	}
	if (read->od_ch == channel) {
		if (!read->od_data) return M4BadParam;
		free(read->od_data);
		read->od_data = NULL;
		read->od_done = 1;
		return M4OK;
	}
	return M4OK;
}

NetClientPlugin *MP3_LoadPlugin()
{
	MP3Reader *reader;
	NetClientPlugin *plug = malloc(sizeof(NetClientPlugin));
	memset(plug, 0, sizeof(NetClientPlugin));
	M4_REG_PLUG(plug, M4STREAMINGCLIENT, "GPAC MP3 Reader", "gpac distribution", 0)

	plug->CanHandleURL = MP3_CanHandleURL;
	plug->ConnectService = MP3_ConnectService;
	plug->CloseService = MP3_CloseService;
	plug->Get_MPEG4_IOD = MP3_Get_MPEG4_IOD;
	plug->ConnectChannel = MP3_ConnectChannel;
	plug->DisconnectChannel = MP3_DisconnectChannel;
	plug->GetStatus = MP3_GetStatus;
	plug->ServiceCommand = MP3_ServiceCommand;
	/*we do support pull mode*/
	plug->ChannelGetSLP = MP3_ChannelGetSLP;
	plug->ChannelReleaseSLP = MP3_ChannelReleaseSLP;

	reader = malloc(sizeof(MP3Reader));
	memset(reader, 0, sizeof(MP3Reader));
	reader->status = NM_Setup;

	plug->priv = reader;
	return plug;
}

void MP3_Delete(void *ifce)
{
	NetClientPlugin *plug = (NetClientPlugin *) ifce;
	MP3Reader *read = plug->priv;

	if (read->dnload) {
		read->dnload->Close(read->dnload);
		PM_ShutdownInterface(read->dnload);
	}

	free(read);
	free(plug);
}

Bool QueryInterface(u32 InterfaceType) 
{
	if (InterfaceType == M4STREAMINGCLIENT) return 1;
	return 0;
}

void *LoadInterface(u32 InterfaceType) 
{
	if (InterfaceType == M4STREAMINGCLIENT) return MP3_LoadPlugin();
	return NULL;
}

void ShutdownInterface(void *ifce)
{
	BaseInterface *ptr = (BaseInterface *)ifce;
	switch (ptr->InterfaceType) {
	case M4STREAMINGCLIENT:
		MP3_Delete(ptr);
		break;
	}
}
