/*
 *			GPAC - MPEG-4 Systems C Development Kit
 *
 *			Copyright (c) Jean Le Feuvre 2000-2003 
 *					All rights reserved
 *
 *  This file is part of GPAC / MP4 reader plugin
 *
 *  GPAC is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *   
 *  GPAC is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *   
 *  You should have received a copy of the GNU General Public License
 *  along with GNU Make; see the file COPYING.  If not, write to
 *  the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA. 
 *		
 */

#include "mp4_io.h"
#include <gpac/m4_author.h>
#include <gpac/m4_decoder.h>

/*loads a default IOD*/
M4Err M4R_LoadIOD(M4Reader *read, InitialObjectDescriptor *iod, char **raw_iod, u32 *raw_iod_size)
{
	LPODCODEC codec;
	ObjectDescriptorUpdate *odU;
	M4Err e;
	BitStream *bs;
	u32 w, h;
	ShortTextualDescriptor *std;
	InitialObjectDescriptor *fake_iod;
	ESDescriptor *esd, *st_esd;

	st_esd = ChainGetEntry(iod->ESDescriptors, 0);
	
	read->av_mode = (st_esd->decoderConfig->streamType==M4ST_VISUAL) ? 1 : 0;
	fake_iod = (InitialObjectDescriptor *) OD_NewDescriptor(InitialObjectDescriptor_Tag);
	
	esd = OD_NewESDescriptor(0);
	esd->slConfig->timestampResolution = 1000;
	esd->slConfig->useRandomAccessPointFlag = 1;
	esd->slConfig->useTimestampsFlag = 1;
	
	w = 200;
	h = 20;

	if (st_esd->decoderConfig->objectTypeIndication == 0x20) {
		M4VDecoderSpecificInfo dsi;
		M4V_GetConfig(st_esd->decoderConfig->decoderSpecificInfo->data, st_esd->decoderConfig->decoderSpecificInfo->dataLength, &dsi);
		w = dsi.width;
		h = dsi.height;
	}
	/*make BIFS cfg v1*/
	bs = NewBitStream(NULL, 0, BS_WRITE);
	BS_WriteInt(bs, 0, 10);
	BS_WriteInt(bs, 1, 1);
	BS_WriteInt(bs, 1, 1);
	BS_WriteInt(bs, 1, 1);
	BS_WriteInt(bs, w, 16);
	BS_WriteInt(bs, h, 16);
	esd->decoderConfig->streamType = M4ST_BIFS;
	esd->decoderConfig->objectTypeIndication = 0x01;
	BS_Align(bs);
	BS_GetContent(bs, (unsigned char **) &esd->decoderConfig->decoderSpecificInfo->data, &esd->decoderConfig->decoderSpecificInfo->dataLength);
	DeleteBitStream(bs);
	if (st_esd->ESID>=10) {
		read->BIFS_ESID = esd->ESID = 1;
	} else {
		read->BIFS_ESID = esd->ESID = 10;
	}
	ChainAddEntry(fake_iod->ESDescriptors, esd);

	/*make OD*/	
	esd = OD_NewESDescriptor(0);
	esd->slConfig->timestampResolution = 1000;
	esd->slConfig->useRandomAccessPointFlag = 1;
	esd->slConfig->useTimestampsFlag = 1;
	read->OD_ESID = esd->ESID = read->BIFS_ESID + 1;
	esd->OCRESID = read->BIFS_ESID;
	esd->decoderConfig->streamType = M4ST_OD;
	esd->decoderConfig->objectTypeIndication = 0x01;
	ChainAddEntry(fake_iod->ESDescriptors, esd);
	fake_iod->graphics_profileAndLevel = 1;
	fake_iod->OD_profileAndLevel = 1;
	fake_iod->scene_profileAndLevel = 1;
	
	fake_iod->audio_profileAndLevel = read->av_mode ? 0xFF : 1;
	fake_iod->visual_profileAndLevel = read->av_mode ? 1 : 0xFF;
	/*if the IOD is properly formatted keep its PL info*/
	if (iod->tag==InitialObjectDescriptor_Tag) {
		if (!read->av_mode && (iod->audio_profileAndLevel != 0xFF)) 
			fake_iod->audio_profileAndLevel = iod->audio_profileAndLevel;

		if (read->av_mode && (iod->visual_profileAndLevel != 0xFF)) 
			fake_iod->visual_profileAndLevel = iod->visual_profileAndLevel;
	}

	/*make OD AU*/
	codec = OD_NewCodec(OD_WRITE);	
	odU = (ObjectDescriptorUpdate *) OD_NewCommand(ODUpdate_Tag);
	iod->objectDescriptorID = read->av_mode ? ISMA_VIDEO_OD_ID : ISMA_AUDIO_OD_ID;
	st_esd->OCRESID = read->BIFS_ESID;
	ChainAddEntry(odU->objectDescriptors, iod);
	OD_AddCommand(codec, (ODCommand *)odU);
	OD_EncodeAU(codec);
	OD_GetEncodedAU(codec, &read->od_au, &read->od_au_size);
	OD_DeleteCodec(codec);


	std = (ShortTextualDescriptor *) OD_NewDescriptor(ShortTextualDescriptor_Tag);
	std->eventName = strdup("MP4 info");
	std->eventText = strdup("Automatically generated IOD");
	std->langCode = 0;
	ChainAddEntry(fake_iod->OCIDescriptors, std);
	fake_iod->objectDescriptorID = 1;
	e = OD_EncDesc((Descriptor *)fake_iod, raw_iod, raw_iod_size);
	OD_DeleteDescriptor((Descriptor **)&fake_iod);
	return e;
}



/*emulate a default IOD for all other files (3GP, weird MP4, QT )*/
M4Err M4R_EmulateIOD(M4Reader *read, char **raw_iod, u32 *raw_iod_size)
{
	LPODCODEC codec;
	ObjectDescriptorUpdate *odU;
	M4Err e;
	BitStream *bs;
	u32 w, h, i;
	u32 a_track, v_track, startID;
	ShortTextualDescriptor *std;
	InitialObjectDescriptor *fake_iod;
	ObjectDescriptor *od;
	ESDescriptor *esd, *tmp_esd;

	a_track = v_track = 0;

	for (i=0; i<M4_GetTrackCount(read->mov); i++) {
		switch (M4_GetMediaType(read->mov, i+1)) {
		case M4_AudioMediaType:
			if (!a_track) a_track = i+1;
			break;
		case M4_VisualMediaType:
			if (!v_track || (M4_GetSampleCount(read->mov, i+1) > M4_GetSampleCount(read->mov, v_track) ) ) v_track = i+1;
			break;
		}
	}
	if (!a_track && !v_track) return M4InvalidMP4File;

	read->av_mode = 0;
	if (v_track) {
		read->av_mode = 1;
		if (a_track) read->av_mode = 2;
	}
	startID = 0;
	if (a_track) startID = M4_GetTrackID(read->mov, a_track);
	if (v_track && (startID < M4_GetTrackID(read->mov, v_track))) startID = M4_GetTrackID(read->mov, v_track);
	
	startID++;

	fake_iod = (InitialObjectDescriptor *) OD_NewDescriptor(InitialObjectDescriptor_Tag);
	
	/*load a BIFS stream description*/
	esd = OD_NewESDescriptor(0);
	esd->slConfig->timestampResolution = 1000;
	esd->slConfig->useRandomAccessPointFlag = 1;
	esd->slConfig->useTimestampsFlag = 1;

	w = 200;
	h = 20;
	if (v_track) {
		M4_GetVisualEntrySize(read->mov, v_track, 1, &w, &h);
		if (M4_GetMediaSubType(read->mov, v_track, 1) == M4_MPEG4_SubType) {
			tmp_esd = M4_GetStreamDescriptor(read->mov, v_track, 1);
			if ((tmp_esd->decoderConfig->objectTypeIndication == 0x20) 
			&& tmp_esd->decoderConfig->decoderSpecificInfo
			&& tmp_esd->decoderConfig->decoderSpecificInfo->data) {
				M4VDecoderSpecificInfo dsi;
				M4V_GetConfig(tmp_esd->decoderConfig->decoderSpecificInfo->data, tmp_esd->decoderConfig->decoderSpecificInfo->dataLength, &dsi);
				w = dsi.width;
				h = dsi.height;
			}
			OD_DeleteDescriptor((Descriptor **) &tmp_esd);
		}
	}

	/*make BIFS cfg v1*/
	bs = NewBitStream(NULL, 0, BS_WRITE);
	BS_WriteInt(bs, 0, 10);
	BS_WriteInt(bs, 1, 1);
	BS_WriteInt(bs, 1, 1);
	BS_WriteInt(bs, 1, 1);
	BS_WriteInt(bs, w, 16);
	BS_WriteInt(bs, h, 16);
	esd->decoderConfig->streamType = M4ST_BIFS;
	esd->decoderConfig->objectTypeIndication = 0x01;
	BS_Align(bs);
	BS_GetContent(bs, (unsigned char **) &esd->decoderConfig->decoderSpecificInfo->data, &esd->decoderConfig->decoderSpecificInfo->dataLength);
	DeleteBitStream(bs);
	read->BIFS_ESID = esd->OCRESID = esd->ESID = startID;
	ChainAddEntry(fake_iod->ESDescriptors, esd);

	/*load an OD stream description*/	
	esd = OD_NewESDescriptor(0);
	esd->slConfig->timestampResolution = 1000;
	esd->slConfig->useRandomAccessPointFlag = 1;
	esd->slConfig->useTimestampsFlag = 1;
	read->OD_ESID = esd->ESID = read->BIFS_ESID + 1;
	esd->OCRESID = read->BIFS_ESID;
	esd->decoderConfig->streamType = M4ST_OD;
	esd->decoderConfig->objectTypeIndication = 0x01;
	ChainAddEntry(fake_iod->ESDescriptors, esd);
	fake_iod->graphics_profileAndLevel = 1;
	fake_iod->OD_profileAndLevel = 1;
	fake_iod->scene_profileAndLevel = 1;
	
	fake_iod->audio_profileAndLevel = a_track ? 0xFE : 0;
	fake_iod->visual_profileAndLevel = v_track ? 0xFE : 0;

	/*make OD AU*/
	codec = OD_NewCodec(OD_WRITE);	
	odU = (ObjectDescriptorUpdate *) OD_NewCommand(ODUpdate_Tag);
	if (a_track) {
		esd = MP4T_MapESDescriptor(read->mov, a_track);
		if (esd) {
			od = (ObjectDescriptor *) OD_NewDescriptor(ObjectDescriptor_Tag);
			od->objectDescriptorID = ISMA_AUDIO_OD_ID;
			esd->OCRESID = read->BIFS_ESID;
			ChainAddEntry(od->ESDescriptors, esd);
			ChainAddEntry(odU->objectDescriptors, od);
		}
	}

	if (v_track) {
		esd = MP4T_MapESDescriptor(read->mov, v_track);
		if (esd) {
			od = (ObjectDescriptor *) OD_NewDescriptor(ObjectDescriptor_Tag);
			od->objectDescriptorID = ISMA_VIDEO_OD_ID;
			esd = M4_GetStreamDescriptor(read->mov, v_track, 1);
			esd->OCRESID = read->BIFS_ESID;
			ChainAddEntry(od->ESDescriptors, esd);
			ChainAddEntry(odU->objectDescriptors, od);
		}
	}

	OD_AddCommand(codec, (ODCommand *)odU);
	OD_EncodeAU(codec);
	OD_GetEncodedAU(codec, &read->od_au, &read->od_au_size);
	OD_DeleteCodec(codec);

	std = (ShortTextualDescriptor *) OD_NewDescriptor(ShortTextualDescriptor_Tag);
	std->eventName = strdup("MP4 info");
	std->eventText = strdup("Automatically generated IOD");
	std->langCode = 0;
	ChainAddEntry(fake_iod->OCIDescriptors, std);
	fake_iod->objectDescriptorID = 1;
	e = OD_EncDesc((Descriptor *)fake_iod, raw_iod, raw_iod_size);
	OD_DeleteDescriptor((Descriptor **)&fake_iod);
	return e;
}

Bool QueryInterface(u32 InterfaceType) 
{
	if (InterfaceType == M4STREAMINGCLIENT) return 1;
	return 0;
}

void *LoadInterface(u32 InterfaceType) 
{
	if (InterfaceType == M4STREAMINGCLIENT) return M4R_LoadPlugin();
	return NULL;
}

void ShutdownInterface(void *ifce)
{
	BaseInterface *ptr = (BaseInterface *)ifce;
	switch (ptr->InterfaceType) {
	case M4STREAMINGCLIENT:
		M4Read_Delete(ptr);
		break;
	}
}
