/*
 *			GPAC - MPEG-4 Systems C Development Kit
 *
 *			Copyright (c) Jean Le Feuvre 2000-2004 
 *					All rights reserved
 *
 *  This file is part of GPAC / Scene Rendering sub-project
 *
 *  GPAC is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *   
 *  GPAC is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *   
 *  You should have received a copy of the GNU General Public License
 *  along with GNU Make; see the file COPYING.  If not, write to
 *  the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA. 
 *
 */



#include "stacks3d.h"


#ifdef M4_DEF_Background2D

static void DestroyBackground2D(SFNode *node)
{
	Background2DStack *ptr;
	ptr = (Background2DStack *) Node_GetPrivate(node);
	PreDestroyBindable(node, ptr->reg_stacks);
	DeleteChain(ptr->reg_stacks);
	texture_destroy(&ptr->txh);
	mesh_free(ptr->mesh);
	free(ptr);
}

static Bool back_use_texture(B_Background2D *bck)
{
	if (!bck->url.count) return 0;
	if (bck->url.vals[0].OD_ID > 0) return 1;
	if (bck->url.vals[0].url && strlen(bck->url.vals[0].url)) return 1;
	return 0;
}

static void RenderBackground2D(SFNode *node, void *rs)
{
	B_Background2D *bck;
	Bool use_texture;
	Background2DStack *st;
	RenderEffect *eff = (RenderEffect *)rs;
	Render3D *sr;

	if (eff->traversing_mode==TRAVERSE_PICK) return;
	
	bck = (B_Background2D *)node;
	st = (Background2DStack *) Node_GetPrivate(node);
	sr = (Render3D*)st->compositor->visual_renderer->user_priv;


	assert(eff->backgrounds);

	/*first traverse, bound if needed*/
	if (ChainFindEntry(eff->backgrounds, node) < 0) {
		ChainAddEntry(eff->backgrounds, node);
		assert(ChainFindEntry(st->reg_stacks, eff->backgrounds)==-1);
		ChainAddEntry(st->reg_stacks, eff->backgrounds);
		/*only bound if we're on top*/
		if (ChainGetEntry(eff->backgrounds, 0) == bck) {
			if (!bck->isBound) Bindable_SetIsBound(node, 1);
		}
		/*open the stream if any*/
		if (back_use_texture(bck) && !st->txh.is_open) {
			st->needs_texture = 1;
			texture_play(&st->txh, &bck->url);
		}
		/*in any case don't draw the first time (since the background could have been declared last)*/
		SR_Invalidate(st->compositor, NULL);
		return;
	}
	if (!bck->isBound) return;
	if (eff->traversing_mode != TRAVERSE_RENDER_BINDABLE) return;

	if (!bbox_equal(&st->prev_bounds, &eff->bbox)) {
		SFVec2f s;
		st->prev_bounds = eff->bbox;
		s.x = eff->bbox.max_edge.x - eff->bbox.min_edge.x;
		s.y = eff->bbox.max_edge.y - eff->bbox.min_edge.y;
		mesh_new_rectangle(st->mesh, s);
	}

	use_texture = back_use_texture(bck);
	if (use_texture) {
		/*texture not ready*/
		if (!st->txh.hwtx) {
			use_texture = 0;
			SR_Invalidate(st->compositor, NULL);
		}
	}

	VS3D_PushState(eff->surface);
	/*no lights on background*/
	VS3D_SetState(eff->surface, F3D_LIGHT, 0);

	if (use_texture) use_texture = tx_enable(&st->txh, NULL);

	/*little opt: if we clear the main surface clear it entirely - ONLY IF NOT IN LAYER*/
	if ((eff->surface == sr->surface) && (eff->surface->back_stack == eff->backgrounds)) {
		VS3D_ClearSurface(eff->surface, bck->backColor, 1.0f);
		if (!use_texture) {
			VS3D_PopState(eff->surface);
			return;
		}
	}
	if (!use_texture) VS3D_SetMaterial2D(eff->surface, bck->backColor, 1.0);
	VS3D_DrawMesh(eff->surface, st->mesh);
	VS3D_PopState(eff->surface);
}


static void b2D_set_bind(SFNode *node)
{
	Background2DStack *st = (Background2DStack *)Node_GetPrivate(node);
	Bindable_OnSetBind(node, st->reg_stacks);
	/*and redraw scene*/
	SR_Invalidate(st->compositor, NULL);
}

static void UpdateBackgroundTexture(TextureHandler *txh)
{
	texture_update_frame(txh);
	/*restart texture if needed (movie background controled by MediaControl)*/
	if (txh->stream_finished && MO_GetLoop(txh->stream, 0)) texture_restart(txh);
}

void R3D_InitBackground2D(Render3D *sr, SFNode *node)
{
	Background2DStack *ptr = malloc(sizeof(Background2DStack));
	memset(ptr, 0, sizeof(Background2DStack));

	traversable_setup(ptr, node, sr->compositor);
	ptr->reg_stacks = NewChain();
	((B_Background2D *)node)->on_set_bind = b2D_set_bind;
	texture_setup(&ptr->txh, sr->compositor, node);
	ptr->txh.update_texture_fcnt = UpdateBackgroundTexture;
	Node_SetPrivate(node, ptr);
	Node_SetPreDestroyFunction(node, DestroyBackground2D);
	Node_SetRenderFunction(node, RenderBackground2D);
	ptr->mesh = new_mesh();
}

void R3D_Background2DModified(SFNode *node)
{
	B_Background2D *bck = (B_Background2D *)node;
	Background2DStack *st = (Background2DStack *) Node_GetPrivate(node);
	if (!st) return;

	/*if open and changed, stop and play*/
	if (st->txh.is_open) {
		if (! texture_check_url_change(&st->txh, &bck->url)) return;
		texture_stop(&st->txh);
		texture_play(&st->txh, &bck->url);
		return;
	}
	/*if not open and changed play*/
	if (bck->url.count) 
		texture_play(&st->txh, &bck->url);
	SR_Invalidate(st->txh.compositor, NULL);
}

#endif

