/*
 *			GPAC - MPEG-4 Systems C Development Kit
 *
 *			Copyright (c) Jean Le Feuvre 2000-2004 
 *					All rights reserved
 *
 *  This file is part of GPAC / Scene Rendering sub-project
 *
 *  GPAC is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *   
 *  GPAC is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *   
 *  You should have received a copy of the GNU General Public License
 *  along with GNU Make; see the file COPYING.  If not, write to
 *  the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA. 
 *
 */


#include "visual_surface.h"


/*
		Shape 
*/
static void RenderShape(SFNode *node, void *rs)
{
#ifdef M4_DEF_Shape
	RenderEffect *eff = (RenderEffect *) rs;
	B_Shape *shape = (B_Shape *) node;
	if (!shape->geometry) return;
	eff->appear = (SFNode *) shape->appearance;

	if ((eff->trav_flags & TF_SWITCHED_OFF) || eff->traversing_mode) {
		Node_Render((SFNode *) shape->geometry, eff);
	} else {
		DrawableStack *st = Node_GetPrivate(shape->geometry);
		u32 cull_bckup = eff->cull_flag;
		if (st && node_cull_or_render(eff, &st->mesh->bounds)) {
			VS3D_PushState(eff->surface);
			Node_Render((SFNode *) shape->geometry, eff);
			VS3D_PopState(eff->surface);
		}
		eff->cull_flag = cull_bckup;
	}

	/*reset all appearance children dirty flag (this is needed for texture & bitmap handling)*/
	//Node_ClearDirtyChildren(node, 1);
	eff->appear = NULL;
#endif
}

void R3D_InitShape(Render3D *sr, SFNode *node)
{
	Node_SetRenderFunction(node, RenderShape);
}


#ifdef M4_DEF_Box

static void RenderBox(SFNode *n, void *rs)
{
	RenderEffect *eff = (RenderEffect *)rs;
	DrawableStack *st = Node_GetPrivate(n);
	B_Box *box = (B_Box *)n;

	if (Node_GetDirty(n)) {
		mesh_new_box(st->mesh, box->size);
		Node_ClearDirty(n);
	}
	/*not drawing*/
	if (eff->trav_flags & TF_SWITCHED_OFF)  return;

	if (!eff->traversing_mode) {
		VS_setup_appearance(eff);
		VS3D_DrawMesh(eff->surface, st->mesh);
	}
	else if (eff->traversing_mode==TRAVERSE_GET_BOUNDS) {
		eff->bbox = st->mesh->bounds;
	}
}

void R3D_InitBox(Render3D *sr, SFNode *node)
{
	BaseDrawableStack(sr->compositor, node);
	Node_SetRenderFunction(node, RenderBox);
}


#endif

#ifdef M4_DEF_Cone

static void RenderCone(SFNode *n, void *rs)
{
	RenderEffect *eff = (RenderEffect *)rs;
	DrawableStack *st = Node_GetPrivate(n);
	B_Cone *co = (B_Cone *)n;

	if (Node_GetDirty(n)) {
		mesh_new_cone(st->mesh, co->height, co->bottomRadius, co->bottom, co->side, eff->surface->render->compositor->high_speed);
		Node_ClearDirty(n);
	}

	/*not drawing*/
	if (eff->trav_flags & TF_SWITCHED_OFF)  return;

	if (!eff->traversing_mode) {
		VS_setup_appearance(eff);
		VS3D_DrawMesh(eff->surface, st->mesh);
	}
	else if (eff->traversing_mode==TRAVERSE_GET_BOUNDS) {
		eff->bbox = st->mesh->bounds;
	}
}

void R3D_InitCone(Render3D *sr, SFNode *node)
{
	BaseDrawableStack(sr->compositor, node);
	Node_SetRenderFunction(node, RenderCone);
}

#endif

#ifdef M4_DEF_Cylinder

static void RenderCylinder(SFNode *n, void *rs)
{
	RenderEffect *eff = (RenderEffect *)rs;
	DrawableStack *st = Node_GetPrivate(n);
	B_Cylinder *cy = (B_Cylinder *)n;

	if (Node_GetDirty(n) ) {
		mesh_new_cylinder(st->mesh, cy->height, cy->radius, cy->bottom, cy->side, cy->top, eff->surface->render->compositor->high_speed);
		Node_ClearDirty(n);
	}
	/*not drawing*/
	if (eff->trav_flags & TF_SWITCHED_OFF)  return;

	if (!eff->traversing_mode) {
		VS_setup_appearance(eff);
		VS3D_DrawMesh(eff->surface, st->mesh);
	}
	else if (eff->traversing_mode==TRAVERSE_GET_BOUNDS) {
		eff->bbox = st->mesh->bounds;
	}
}

void R3D_InitCylinder(Render3D *sr, SFNode *node)
{
	BaseDrawableStack(sr->compositor, node);
	Node_SetRenderFunction(node, RenderCylinder);
}

#endif


#ifdef M4_DEF_Cone

static void RenderSphere(SFNode *n, void *rs)
{
	RenderEffect *eff = (RenderEffect *)rs;
	DrawableStack *st = Node_GetPrivate(n);
	B_Sphere *sp = (B_Sphere *)n;

	if (Node_GetDirty(n)) {
		mesh_new_sphere(st->mesh, sp->radius, eff->surface->render->compositor->high_speed);
		Node_ClearDirty(n);
	}
	/*not drawing*/
	if (eff->trav_flags & TF_SWITCHED_OFF)  return;

	if (!eff->traversing_mode) {
		VS_setup_appearance(eff);
		VS3D_DrawMesh(eff->surface, st->mesh);
	}
	else if (eff->traversing_mode==TRAVERSE_GET_BOUNDS) {
		eff->bbox = st->mesh->bounds;
	}
}

void R3D_InitSphere(Render3D *sr, SFNode *node)
{
	BaseDrawableStack(sr->compositor, node);
	Node_SetRenderFunction(node, RenderSphere);
}

#endif


#ifdef M4_DEF_Circle

/*
		Circle 
*/
static void RenderCircle(SFNode *node, void *rs)
{
	stack2D *st = Node_GetPrivate(node);
	RenderEffect *eff = rs;

	if (Node_GetDirty(node)) {
		Float a = ((B_Circle *) node)->radius * 2;
		stack2D_reset(st);
		m4_path_add_ellipse(st->path, a, a);
		mesh_new_ellipse(st->mesh, a, a, eff->surface->render->compositor->high_speed);
		Node_ClearDirty(node);
	}
	if (eff->trav_flags & TF_SWITCHED_OFF)  return;

	if (!eff->traversing_mode) 
		stack2D_draw(st, eff);
	else if (eff->traversing_mode==TRAVERSE_GET_BOUNDS) 
		eff->bbox = st->mesh->bounds;
}

void R3D_InitCircle(Render3D *sr, SFNode *node)
{
	BaseStack2D(sr->compositor, node);
	Node_SetRenderFunction(node, RenderCircle);
}
#endif

#ifdef M4_DEF_Ellipse
/*
		Ellipse
*/
static void RenderEllipse(SFNode *node, void *rs)
{
	stack2D *st = Node_GetPrivate(node);
	RenderEffect *eff = rs;

	if (Node_GetDirty(node)) {
		Float a = ((B_Ellipse *) node)->radius.x;
		Float b = ((B_Ellipse *) node)->radius.y;
		stack2D_reset(st);
		m4_path_add_ellipse(st->path, a, b);
		mesh_new_ellipse(st->mesh, a, b, eff->surface->render->compositor->high_speed);
		Node_ClearDirty(node);
	}
	if (eff->trav_flags & TF_SWITCHED_OFF)  return;

	if (!eff->traversing_mode) 
		stack2D_draw(st, eff);
	else if (eff->traversing_mode==TRAVERSE_GET_BOUNDS) 
		eff->bbox = st->mesh->bounds;
}

void R3D_InitEllipse(Render3D *sr, SFNode *node)
{
	BaseStack2D(sr->compositor, node);
	Node_SetRenderFunction(node, RenderEllipse);
}
#endif

#ifdef M4_DEF_Rectangle
/*
		Rectangle 
*/
static void RenderRectangle(SFNode *node, void *rs)
{
	stack2D *st = (stack2D *) Node_GetPrivate(node);
	RenderEffect *eff = rs;

	/*build vec path*/
	if (Node_GetDirty(node)) {
		SFVec2f s = ((B_Rectangle *) node)->size;
		stack2D_reset(st);
		m4_path_add_rectangle(st->path, 0, 0, s.x, s.y);
		mesh_new_rectangle(st->mesh, s);
		Node_ClearDirty(node);
	}
	if (eff->trav_flags & TF_SWITCHED_OFF)  return;

	if (!eff->traversing_mode) 
		stack2D_draw(st, eff);
	else if (eff->traversing_mode==TRAVERSE_GET_BOUNDS) 
		eff->bbox = st->mesh->bounds;
}

void R3D_InitRectangle(Render3D *sr, SFNode *node)
{
	BaseStack2D(sr->compositor, node);
	Node_SetRenderFunction(node, RenderRectangle);
}
#endif

#ifdef M4_DEF_Curve2D

#ifndef M4_DEF_Coordinate2D
#error "Coordinate2D node MUST be defined when enabling Curve2D"
#endif

//#define CHECK_VALID_C2D(nbPts) if (cur_index+nbPts>=pt_count) { m4_path_reset(st->path); return; }
#define CHECK_VALID_C2D(nbPts)

static void build_curve2D(stack2D *st, B_Curve2D *c2D)
{
	SFVec2f orig, ct_orig, ct_end, end;
	u32 cur_index, i, remain, type_count, pt_count;
	SFVec2f *pts;
	B_Coordinate2D *coord = (B_Coordinate2D *)c2D->point;

	pts = coord->point.vals;
	if (!pts) 
		return;

	cur_index = c2D->type.count ? 1 : 0;
	/*if the first type is a moveTo skip initial moveTo*/
	i=0;
	if (cur_index && (c2D->type.vals[0]==0)) i += 1;
	ct_orig = orig = pts[0];

	m4_path_add_move_to(st->path, orig.x, orig.y);

	pt_count = coord->point.count;
	type_count = c2D->type.count;
	for (; i<type_count; i++) {

		switch (c2D->type.vals[i]) {
		//moveTo, 1 point
		case 0:
			CHECK_VALID_C2D(0);
			orig = pts[cur_index];
			if (i) m4_path_add_move_to(st->path, orig.x, orig.y);
			cur_index += 1;
			break;
		//lineTo, 1 point
		case 1:
			CHECK_VALID_C2D(0);
			end = pts[cur_index];
			m4_path_add_line_to(st->path, end.x, end.y);
			orig = end;
			cur_index += 1;
			break;
		
		//curveTo, 3 points
		case 2:
			CHECK_VALID_C2D(2);
			ct_orig = pts[cur_index];
			ct_end = pts[cur_index+1];
			end = pts[cur_index+2];
			m4_path_add_cubic_to(st->path, ct_orig.x, ct_orig.y, ct_end.x, ct_end.y, end.x, end.y);
			cur_index += 3;
			ct_orig = ct_end;
			orig = end;
			break;
		//nextCurveTo, 2 points (cf spec)
		case 3:
			CHECK_VALID_C2D(1);
			ct_orig.x = 2*orig.x - ct_orig.x;
			ct_orig.y = 2*orig.y - ct_orig.y;
			ct_end = pts[cur_index];
			end = pts[cur_index+1];
			m4_path_add_cubic_to(st->path, ct_orig.x, ct_orig.y, ct_end.x, ct_end.y, end.x, end.y);
			cur_index += 2;
			ct_orig = ct_end;
			orig = end;
			break;
		/*the rest is XCurve2D specific*/

		/*quadratic CurveTo, 2 points*/
		case 7:
			CHECK_VALID_C2D(1);
			ct_end = pts[cur_index];
			end = pts[cur_index+1];
			m4_path_add_quadratic_to(st->path, ct_end.x, ct_end.y, end.x, end.y);
			cur_index += 2;
			ct_orig = ct_end;
			orig = end;
			break;
		/*CW and CCW ArcTo - TO DO*/
		case 4:
		case 5:
			break;
		/*ClosePath*/
		case 6:
			m4_path_close(st->path);
			break;
		}
		/*bad content*/
		if (cur_index>=pt_count) return;
	}

	/*what's left is an N-bezier spline*/
	if (pt_count > cur_index) {
		/*first moveto*/
		if (!cur_index) cur_index++;

		remain = pt_count - cur_index;

		if (remain>1)
			m4_path_add_bezier(st->path, (M4Point2D *) &pts[cur_index], remain);
	}
}

static void RenderCurve2D(SFNode *node, void *rs)
{
	B_Curve2D *c2D = (B_Curve2D *)node;
	stack2D *st = Node_GetPrivate(node);
	RenderEffect *eff = rs;

	if (!c2D->point) return;

	if (Node_GetDirty(node)) {
		stack2D_reset(st);
		st->path->resolution = st->compositor->base_curve_resolution;
		if (st->compositor->high_speed) st->path->resolution /= HIGH_SPEED_RATIO;
		st->path->fineness = c2D->fineness;
		build_curve2D(st, c2D);
		mesh_from_path(st->mesh, st->path);
		Node_ClearDirty(node);
	}
	
	if (eff->trav_flags & TF_SWITCHED_OFF)  return;
	
	if (!eff->traversing_mode) 
		stack2D_draw(st, eff);
	else if (eff->traversing_mode==TRAVERSE_GET_BOUNDS) 
		eff->bbox = st->mesh->bounds;
}

void R3D_InitCurve2D(Render3D *sr, SFNode *node)
{
	BaseStack2D(sr->compositor, node);
	Node_SetRenderFunction(node, RenderCurve2D);
}

#endif


#ifdef M4_DEF_IndexedLineSet2D

#ifndef M4_DEF_Coordinate2D
#error "Coordinate2D node MUST be defined when enabling IndexedLineSet2D"
#endif

static void build_ils2d(stack2D *st, B_IndexedLineSet2D *ils2D)
{
	u32 i;
	Bool started;
	SFVec2f *pts;
	B_Coordinate2D *coord = (B_Coordinate2D *)ils2D->coord;

	pts = coord->point.vals;
	if (ils2D->coordIndex.count > 0) {
		started = 0;
		for (i=0; i < ils2D->coordIndex.count; i++) {
			/*NO close on ILS2D*/
			if (ils2D->coordIndex.vals[i] == -1) {
				started = 0;
			} else if (!started) {
				started = 1;
				m4_path_add_move_to(st->path, pts[ils2D->coordIndex.vals[i]].x, pts[ils2D->coordIndex.vals[i]].y);
			} else {
				m4_path_add_line_to(st->path, pts[ils2D->coordIndex.vals[i]].x, pts[ils2D->coordIndex.vals[i]].y);
			}
		}
	} else if (coord->point.count) {
		m4_path_add_move_to(st->path, pts[0].x, pts[0].y);
		for (i=1; i < coord->point.count; i++) {
			m4_path_add_line_to(st->path, pts[i].x, pts[i].y);
		}
	}
}

static void RenderILS2D(SFNode *node, void *rs)
{
	Aspect2D asp;
	StrikeInfo *si;
	B_IndexedLineSet2D *ils2D = (B_IndexedLineSet2D *)node;
	stack2D *st = Node_GetPrivate(node);
	RenderEffect *eff = rs;

	if (!ils2D->coord) return;

	if (Node_GetDirty(node)) {
		stack2D_reset(st);
		build_ils2d(st, ils2D);
		mesh_new_ils(st->mesh, ils2D->coord, &ils2D->coordIndex, ils2D->color, &ils2D->colorIndex, ils2D->colorPerVertex, 0);
		Node_ClearDirty(node);
	}

	VS_GetAspect2D(eff, &asp);

	if (eff->trav_flags & TF_SWITCHED_OFF)  return;
	
	if (!eff->traversing_mode) {
		if (ils2D->color) {
			VS3D_StrikeMesh(eff->surface, st->mesh, Aspect_GetLineWidth(&asp), asp.pen_props.dash);
		} else {
			si = VS_GetStrikeInfo(st, &asp, eff);
			if (si) {
				VS_Set2DStrikeAspect(eff->surface, &asp);
				if (!si->is_vectorial) {
					VS3D_StrikeMesh(eff->surface, si->outline, Aspect_GetLineWidth(&asp), asp.pen_props.dash);
				} else {
					VS3D_DrawMesh(eff->surface, si->outline);
				}
			}
		}
	} else if (eff->traversing_mode==TRAVERSE_GET_BOUNDS) {
		eff->bbox = st->mesh->bounds;
	}
	
}

static void ILS2D_SetColorIndex(SFNode *node)
{
	B_IndexedLineSet2D *ils2D = (B_IndexedLineSet2D *)node;
	SG_CopyField(&ils2D->colorIndex, &ils2D->set_colorIndex, FT_MFInt32);
}

static void ILS2D_SetCoordIndex(SFNode *node)
{
	B_IndexedLineSet2D *ils2D = (B_IndexedLineSet2D *)node;
	SG_CopyField(&ils2D->coordIndex, &ils2D->set_coordIndex, FT_MFInt32);
}

void R3D_InitILS2D(Render3D *sr, SFNode *node)
{
	B_IndexedLineSet2D *ils2D = (B_IndexedLineSet2D *)node;
	BaseStack2D(sr->compositor, node);
	Node_SetRenderFunction(node, RenderILS2D);
	ils2D->on_set_colorIndex = ILS2D_SetColorIndex;
	ils2D->on_set_coordIndex = ILS2D_SetCoordIndex;
}

#endif

#ifdef M4_DEF_PointSet2D

#ifndef M4_DEF_Coordinate2D
#error "Coordinate2D node MUST be defined when enabling PointSet2D"
#endif

static void RenderPointSet2D(SFNode *node, void *rs)
{
	B_PointSet2D *ps2D = (B_PointSet2D *)node;
	DrawableStack *st = Node_GetPrivate(node);
	RenderEffect *eff = rs;
	if (!ps2D->coord) return;

	if (Node_GetDirty(node)) {
		mesh_new_ps(st->mesh, ps2D->coord, ps2D->color);
		Node_ClearDirty(node);
	}

	if (eff->trav_flags & TF_SWITCHED_OFF)  return;
	
	if (!eff->traversing_mode) {
		Aspect2D asp;
		VS_GetAspect2D(eff, &asp);
		VS3D_SetMaterial2D(eff->surface, asp.fill_color, asp.alpha);
		VS3D_DrawMesh(eff->surface, st->mesh);
	}
	else if (eff->traversing_mode==TRAVERSE_GET_BOUNDS) {
		eff->bbox = st->mesh->bounds;
	}
}


void R3D_InitPointSet2D(Render3D *sr, SFNode *node)
{
	BaseDrawableStack(sr->compositor, node);
	Node_SetRenderFunction(node, RenderPointSet2D);
}

#endif

#ifdef M4_DEF_IndexedFaceSet2D

#ifndef M4_DEF_Coordinate2D
#error "Coordinate2D node MUST be defined when enabling IndexedFaceSet2D"
#endif

static void build_ifs2d(stack2D *st, B_IndexedFaceSet2D *ifs2D)
{
	u32 i;
	SFVec2f *pts;
	u32 ci_count, c_count;
	Bool started;
	B_Coordinate2D *coord = (B_Coordinate2D *)ifs2D->coord;

	c_count = coord->point.count;
	ci_count = ifs2D->coordIndex.count;
	pts = coord->point.vals;

	if (ci_count > 0) {
		started = 0;
		for (i=0; i < ci_count; i++) {
			if (ifs2D->coordIndex.vals[i] == -1) {
				m4_path_close(st->path);
				started = 0;
			} else if (!started) {
				started = 1;
				m4_path_add_move_to(st->path, pts[ifs2D->coordIndex.vals[i]].x, pts[ifs2D->coordIndex.vals[i]].y);
			} else {
				m4_path_add_line_to(st->path, pts[ifs2D->coordIndex.vals[i]].x, pts[ifs2D->coordIndex.vals[i]].y);
			}
		}
		if (started) m4_path_close(st->path);
	} else if (c_count) {
		m4_path_add_move_to(st->path, pts[0].x, pts[0].y);
		for (i=1; i < c_count; i++) {
			m4_path_add_line_to(st->path, pts[i].x, pts[i].y);
		}
		m4_path_close(st->path);
	}
}

static void RenderIFS2D(SFNode *node, void *rs)
{
	Aspect2D asp;
	StrikeInfo *si;
	B_IndexedFaceSet2D *ifs2D = (B_IndexedFaceSet2D *)node;
	stack2D *st = Node_GetPrivate(node);
	RenderEffect *eff = rs;

	if (!ifs2D->coord) return;

	if (Node_GetDirty(node)) {
		stack2D_reset(st);
		build_ifs2d(st, ifs2D);
		mesh_new_ifs2d(st->mesh, node);
		Node_ClearDirty(node);
	}
	if (eff->trav_flags & TF_SWITCHED_OFF)  return;

	VS_GetAspect2D(eff, &asp);

	if (!eff->traversing_mode) {
		if (ifs2D->color && !asp.filled) {
			/*use special func to disable outline recompute and handle recompute ourselves*/
			si = VS_GetStrikeInfoIFS(st, &asp, eff);
			if (!si->outline) {
				si->outline = new_mesh();
				mesh_new_ils(si->outline, ifs2D->coord, &ifs2D->coordIndex, ifs2D->color, &ifs2D->colorIndex, ifs2D->colorPerVertex, 1);
			}
			VS3D_StrikeMesh(eff->surface, si->outline, Aspect_GetLineWidth(&asp), asp.pen_props.dash);
		} else {
			stack2D_draw(st, eff);
		}
	} 
	else if (eff->traversing_mode==TRAVERSE_GET_BOUNDS) {
		eff->bbox = st->mesh->bounds;
	}
}

static void IFS2D_SetColorIndex(SFNode *node)
{
	B_IndexedFaceSet2D *ifs2D = (B_IndexedFaceSet2D *)node;
	SG_CopyField(&ifs2D->colorIndex, &ifs2D->set_colorIndex, FT_MFInt32);
}

static void IFS2D_SetCoordIndex(SFNode *node)
{
	B_IndexedFaceSet2D *ifs2D = (B_IndexedFaceSet2D *)node;
	SG_CopyField(&ifs2D->coordIndex, &ifs2D->set_coordIndex, FT_MFInt32);
}

static void IFS2D_SetTexCoordIndex(SFNode *node)
{
	B_IndexedFaceSet2D *ifs2D = (B_IndexedFaceSet2D *)node;
	SG_CopyField(&ifs2D->texCoordIndex, &ifs2D->set_texCoordIndex, FT_MFInt32);
}

void R3D_InitIFS2D(Render3D *sr, SFNode *node)
{
	B_IndexedFaceSet2D *ifs2D = (B_IndexedFaceSet2D *)node;
	BaseStack2D(sr->compositor, node);
	Node_SetRenderFunction(node, RenderIFS2D);
	ifs2D->on_set_colorIndex = IFS2D_SetColorIndex;
	ifs2D->on_set_coordIndex = IFS2D_SetCoordIndex;
	ifs2D->on_set_texCoordIndex = IFS2D_SetTexCoordIndex;
}

#endif



#ifdef M4_DEF_IndexedFaceSet

#ifndef M4_DEF_Coordinate
#error "Coordinate node MUST be defined when enabling IndexedFaceSet"
#endif

static void RenderIFS(SFNode *node, void *rs)
{
	B_IndexedFaceSet *ifs = (B_IndexedFaceSet *)node;
	DrawableStack *st = Node_GetPrivate(node);
	RenderEffect *eff = rs;

	if (!ifs->coord) return;

	if (Node_GetDirty(node)) {
		mesh_new_ifs(st->mesh, node);
		Node_ClearDirty(node);
	}
	/*not drawing*/
	if (eff->trav_flags & TF_SWITCHED_OFF)  return;

	if (!eff->traversing_mode) {
		VS_setup_appearance(eff);
		VS3D_DrawMesh(eff->surface, st->mesh);
	} else if (eff->traversing_mode==TRAVERSE_GET_BOUNDS) {
		eff->bbox = st->mesh->bounds;
	}
}

static void IFS_SetColorIndex(SFNode *node)
{
	B_IndexedFaceSet *ifs = (B_IndexedFaceSet *)node;
	SG_CopyField(&ifs->colorIndex, &ifs->set_colorIndex, FT_MFInt32);
}

static void IFS_SetCoordIndex(SFNode *node)
{
	B_IndexedFaceSet *ifs = (B_IndexedFaceSet *)node;
	SG_CopyField(&ifs->coordIndex, &ifs->set_coordIndex, FT_MFInt32);
}

static void IFS_SetNormalIndex(SFNode *node)
{
	B_IndexedFaceSet *ifs = (B_IndexedFaceSet *)node;
	SG_CopyField(&ifs->normalIndex, &ifs->set_normalIndex, FT_MFInt32);
}

static void IFS_SetTexCoordIndex(SFNode *node)
{
	B_IndexedFaceSet *ifs = (B_IndexedFaceSet *)node;
	SG_CopyField(&ifs->texCoordIndex, &ifs->set_texCoordIndex, FT_MFInt32);
}

void R3D_InitIFS(Render3D *sr, SFNode *node)
{
	B_IndexedFaceSet *ifs = (B_IndexedFaceSet *)node;
	BaseDrawableStack(sr->compositor, node);
	Node_SetRenderFunction(node, RenderIFS);
	ifs->on_set_colorIndex = IFS_SetColorIndex;
	ifs->on_set_coordIndex = IFS_SetCoordIndex;
	ifs->on_set_normalIndex = IFS_SetNormalIndex;
	ifs->on_set_texCoordIndex = IFS_SetTexCoordIndex;
}

#endif


#ifdef M4_DEF_IndexedLineSet

#ifndef M4_DEF_Coordinate
#error "Coordinate node MUST be defined when enabling IndexedLineSet"
#endif

static void RenderILS(SFNode *node, void *rs)
{
	B_IndexedLineSet *ils = (B_IndexedLineSet *)node;
	DrawableStack *st = Node_GetPrivate(node);
	RenderEffect *eff = rs;

	if (!ils->coord) return;

	if (Node_GetDirty(node)) {
		mesh_new_ils(st->mesh, ils->coord, &ils->coordIndex, ils->color, &ils->colorIndex, ils->colorPerVertex, 0);
		Node_ClearDirty(node);
	}

	/*not drawing*/
	if (eff->trav_flags & TF_SWITCHED_OFF)  return;

	if (!eff->traversing_mode) {
		VS_setup_appearance(eff);
		VS3D_DrawMesh(eff->surface, st->mesh);
	} else if (eff->traversing_mode==TRAVERSE_GET_BOUNDS) {
		eff->bbox = st->mesh->bounds;
	}
}

static void ILS_SetColorIndex(SFNode *node)
{
	B_IndexedLineSet *ils = (B_IndexedLineSet *)node;
	SG_CopyField(&ils->colorIndex, &ils->set_colorIndex, FT_MFInt32);
}

static void ILS_SetCoordIndex(SFNode *node)
{
	B_IndexedLineSet *ils = (B_IndexedLineSet *)node;
	SG_CopyField(&ils->coordIndex, &ils->set_coordIndex, FT_MFInt32);
}

void R3D_InitILS(Render3D *sr, SFNode *node)
{
	B_IndexedLineSet *ils = (B_IndexedLineSet *)node;
	BaseDrawableStack(sr->compositor, node);
	Node_SetRenderFunction(node, RenderILS);
	ils->on_set_colorIndex = ILS_SetColorIndex;
	ils->on_set_coordIndex = ILS_SetCoordIndex;
}

#endif

