/*
 *			GPAC - MPEG-4 Systems C Development Kit
 *
 *			Copyright (c) Jean Le Feuvre 2000-2004 
 *					All rights reserved
 *
 *  This file is part of GPAC / Scene Rendering sub-project
 *
 *  GPAC is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *   
 *  GPAC is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *   
 *  You should have received a copy of the GNU General Public License
 *  along with GNU Make; see the file COPYING.  If not, write to
 *  the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA. 
 *
 */

#ifndef GROUPING_H
#define GROUPING_H

#include "visual_surface.h"

/*
		NOTES about grouping nodes

***when a grouping node impacts placement of its children, rendering is done in 2 step: first traverse
the children and collect each direct child bounding rect (empty if 3D), then computes local transform 
per child according to the display rules of the node (form, layout).  

*/

typedef struct 
{
	/*the one and only child*/
	SFNode *child;
	/*set to true if the bounding rect of this group is not the sum of bounding rects of its children
	(typically all grouping node with post-layout/clipping)*/
	Bool bounds_forced;
	/*bounds before (projection of bound_vol) and after placement*/
	M4Rect original, final;

#ifdef M4_DEF_Text
	/*set if this group contains only text graphic nodes*/
	Bool is_text_group;
	/*in which case ascent and descent of the text is stored for baseline alignment*/
	Float ascent, descent;
	u32 split_text_idx;
#endif
} ChildGroup;


/*
	@children: pointer to children field
	@groups: list of ChildGroup visually render - used for post placement
	@bbox: bounding box of sub-tree
*/

#define GROUPINGNODESTACK	\
	Chain *children;			\
	Chain *groups;			\
	M4BBox bbox;			\
	Bool dont_cull;			\

typedef struct _parent_group
{
	SFNode *owner;
	struct scene_renderer *compositor;
	GROUPINGNODESTACK
} GroupingNode;

/*performs stack init/destroy - doesn't free stack*/
void SetupGroupingNode(GroupingNode *ptr, LPSCENERENDER sr, SFNode *node, Chain *children);
void DeleteGroupingNode(GroupingNode *gr);
/*destroy base stack*/
void DestroyBaseGrouping(SFNode *node);
/*creates grouping stack and register callbacks*/
void NewGroupingNodeStack(LPSCENERENDER sr, SFNode *node, Chain *children);


/*traverse all children of the node*/
void grouping_traverse(GroupingNode *group, RenderEffect *effects);

/*starts new child group. If n is NULL, uses previous child group node for node (this is for text only)*/
void group_start_child(GroupingNode *group, SFNode *n);
void group_end_child(GroupingNode *group, M4BBox *bounds);
#ifdef M4_DEF_Text
void group_end_text_child(GroupingNode *group, M4Rect *bounds, Float ascent, Float descent, u32 split_text_idx);
#endif
/*reset ChildGroup info - does NOT reset real children*/
void group_reset_children(GroupingNode *group);


/*update clipers and matrices in regular mode (translations only)*/
void child_render_done(ChildGroup *cg, RenderEffect *eff);
/*update clipers and matrices in complex mode (free-form transforms) - if @mat is NULL group isn't rendered*/
void child_render_done_complex(ChildGroup *cg, RenderEffect *eff, M4Matrix2D *mat);

/*activates all sensors in a group*/
void grouping_activate_sensors(SFNode *group, Bool mouse_over, Bool is_active, SFVec3f *hit_point);

#endif

