/*
 *			GPAC - MPEG-4 Systems C Development Kit
 *
 *			Copyright (c) Jean Le Feuvre 2000-2004 
 *					All rights reserved
 *
 *  This file is part of GPAC / Scene Rendering sub-project
 *
 *  GPAC is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *   
 *  GPAC is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *   
 *  You should have received a copy of the GNU General Public License
 *  along with GNU Make; see the file COPYING.  If not, write to
 *  the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA. 
 *
 */

#include "stacks3d.h"

#ifdef M4_DEF_PathLayout

static void DestroyPathLayout(SFNode *node)
{
	GroupingNode *st = (GroupingNode *)Node_GetPrivate(node);
	DeleteGroupingNode((GroupingNode *)st);
	free(st);
}

static void RenderPathLayout(SFNode *node, void *rs)
{
	u32 i, count, minor, major, int_bck;
	Float length, offset, length_after_point;
	Bool res;
	u32 mode_bckup;
	ChildGroup *cg;
	M4Matrix mat;
	M4Matrix2D mx2d;
	GroupingNode *parent_bck;
	stack2D *path;
	GroupingNode *gr = (GroupingNode *) Node_GetPrivate(node);
	B_PathLayout *pl = (B_PathLayout *)node;
	RenderEffect *eff = (RenderEffect *) rs;
	
	if (!pl->geometry) return;
	
	/*only low-level primitives allowed*/
	switch (Node_GetTag((SFNode *) pl->geometry)) {
#ifdef M4_DEF_Rectangle
	case TAG_Rectangle: return;
#endif
#ifdef M4_DEF_Circle
	case TAG_Circle: return;
#endif
#ifdef M4_DEF_Ellipse
	case TAG_Ellipse: return;
#endif
	}

	/*store effect*/
	mx_copy(mat, eff->model_view);
	parent_bck = eff->parent;

	mx_init(eff->model_view);
	eff->parent = NULL;

	/*render geometry without drawing*/
	int_bck = eff->trav_flags;
	eff->trav_flags |= TF_SWITCHED_OFF;
	Node_Render((SFNode *)pl->geometry, eff);
	eff->trav_flags = int_bck;

	eff->parent = gr;
	int_bck = eff->text_split_mode;
	eff->text_split_mode = 2;
	mode_bckup = eff->traversing_mode;
	eff->traversing_mode = TRAVERSE_GET_BOUNDS;
	grouping_traverse(gr, eff);
	eff->text_split_mode = int_bck;
	eff->traversing_mode = mode_bckup;

	/*restore effect*/
	mx_copy(eff->model_view, mat);
	eff->parent = parent_bck;

	count = ChainGetCount(gr->groups);
	
	/*get the drawable*/
	path = (stack2D *) Node_GetPrivate( (SFNode *) pl->geometry);
	/*init iteration*/
	if (m4_path_init_iteration(path->path) != M4OK ) return;
	/*get length*/
	if (m4_path_get_length(path->path, &length) != M4OK ) return;

	/*place all children*/
	offset = length * pl->pathOffset;

	major = pl->alignment.count ? pl->alignment.vals[0] : 0;
	minor = (pl->alignment.count==2) ? pl->alignment.vals[1] : 0;

	if (pl->wrapMode==1) {
		while (offset<0) offset += length;
	}

	for (i=0; i<count; i++) {
		cg = ChainGetEntry(gr->groups, i);
		if (cg->original.width>length) break;

		/*first set our center and baseline*/
		mx2d_init(mx2d);

		/*major align*/
		switch (major) {
		case 2:
			if (cg->is_text_group) mx2d_add_translation(&mx2d, -1*cg->original.x - cg->original.width, 0);
			else mx2d_add_translation(&mx2d, -1 * cg->original.width/2, 0);
			length_after_point = 0;
			break;
		case 1:
			length_after_point = cg->original.width/2;
			if (cg->is_text_group) mx2d_add_translation(&mx2d, -1*cg->original.x - cg->original.width / 2, 0);
			break;
		default:
		case 0:
			if (cg->is_text_group) mx2d_add_translation(&mx2d, cg->original.x, 0);
			else mx2d_add_translation(&mx2d, cg->original.width/2, 0);
			length_after_point = cg->original.width;
			break;
		}

		/*if wrapping and out of path, restart*/
		if ((pl->wrapMode==1) && (offset+length_after_point>=length)) {
			offset += length_after_point;
			offset -= length;
			i--;
			continue;
		}
		/*if not wrapping and not yet in path skip */
		if (!pl->wrapMode && (offset+length_after_point < 0)) {
			child_render_done_complex(cg, (RenderEffect *)rs, NULL);
			goto next;
		}

		/*minor justify*/
		switch (minor) {
		/*top alignment*/
		case 3:
#ifdef M4_DEF_Text
			if (cg->is_text_group) 
				mx2d_add_translation(&mx2d, 0, -1 * cg->ascent);
			else 
#endif
				mx2d_add_translation(&mx2d, 0, -1 * cg->original.height / 2);
			
			break;
		/*baseline*/
		case 1:
#ifdef M4_DEF_Text
			/*move to bottom align if not text*/
			if (!cg->is_text_group) 
#endif
				mx2d_add_translation(&mx2d, 0, cg->original.height / 2);
			break;
		/*middle*/
		case 2:
#ifdef M4_DEF_Text
			/*if text use (asc+desc) /2 as line height since glyph height differ*/
			if (cg->is_text_group) 
				mx2d_add_translation(&mx2d, 0, cg->descent - (cg->ascent + cg->descent) / 2);
#endif
			break;
		/*bottomline alignment*/
		case 0:
		default:
#ifdef M4_DEF_Text
			if (cg->is_text_group)
				mx2d_add_translation(&mx2d, 0, cg->descent);
			else
#endif
				mx2d_add_translation(&mx2d, 0, cg->original.height / 2);
			
			break;
		}
		res = m4_path_get_transform_at_offset(path->path, offset, (Bool) (pl->wrapMode==2), &mx2d, 1, length_after_point);
		if (!res) break;

		child_render_done_complex(cg, (RenderEffect *)rs, &mx2d);

next:
		if (i+1<count) {
			ChildGroup *cg_next = ChainGetEntry(gr->groups, i+1);

			/*update offset according to major alignment */
			switch (major) {
			case 2:
				if (cg_next->is_text_group) offset += pl->spacing * cg_next->original.x;
				offset += pl->spacing * cg_next->original.width;
				break;
			case 1:
				if (cg->is_text_group) offset += pl->spacing * cg->original.x / 2;
				offset += pl->spacing * cg->original.width / 2;
				offset += cg_next->original.width / 2;
				break;
			default:
			case 0:
				if (cg->is_text_group) offset += pl->spacing * cg->original.x;
				offset += pl->spacing * cg->original.width;
				break;
			}
		}
		/*wrap*/
		if ((pl->wrapMode==1) && (offset>=length)) offset-=length;
	}

	/*undrawn nodes*/
	for (;i<count; i++) {
		cg = ChainGetEntry(gr->groups, i);
		child_render_done_complex(cg, (RenderEffect *)rs, NULL);
	}

	group_reset_children(gr);
}

void R3D_InitPathLayout(Render3D *sr, SFNode *node)
{
	GroupingNode *stack = malloc(sizeof(GroupingNode));
	memset(stack, 0, sizeof(GroupingNode));
	SetupGroupingNode((GroupingNode*)stack, sr->compositor, node, ((B_PathLayout *)node)->children);
	Node_SetPrivate(node, stack);
	Node_SetPreDestroyFunction(node, DestroyPathLayout);
	Node_SetRenderFunction(node, RenderPathLayout);
}


#endif

