/*
 *			GPAC - MPEG-4 Systems C Development Kit
 *
 *			Copyright (c) Jean Le Feuvre 2000-2004 
 *					All rights reserved
 *
 *  This file is part of GPAC / Scene Rendering sub-project
 *
 *  GPAC is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *   
 *  GPAC is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *   
 *  You should have received a copy of the GNU General Public License
 *  along with GNU Make; see the file COPYING.  If not, write to
 *  the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA. 
 *
 */


#include "render3d.h"
#include "visual_surface.h"
#include "stacks3d.h"


void R3D_MapCoordsToAR(VisualRenderer *vr, s32 X, s32 Y, Float *bifsX, Float *bifsY)
{
	*bifsX = (Float) X;
	*bifsY = (Float) Y;
}


void effect_reset(RenderEffect *eff)
{
	memset(eff, 0, sizeof(RenderEffect));
	mx_init(eff->model_view);
	cmat_init(&eff->color_mat);
}

void effect_delete(RenderEffect *eff)
{
	free(eff);
}

SFNode *R3D_PickNode(VisualRenderer *vr, s32 X, s32 Y)
{
	return NULL;
}

static void DestroyLineProps(SFNode *n)
{
	u32 i;
	LinePropStack *st = Node_GetPrivate(n);
	Render3D *sr = (Render3D *)st->sr;
	
	for (i=0; i<ChainGetCount(sr->strike_bank); i++) {
		StrikeInfo *si = ChainGetEntry(sr->strike_bank, i);
		if (si->lineProps == n) {
			/*remove from node*/
			if (si->node2D) {
				stack2D *st = (stack2D *) Node_GetPrivate(si->node2D);
				ChainDeleteItem(st->strike_list, si);
			}
			ChainDeleteEntry(sr->strike_bank, i);
			delete_strikeinfo(si);
			i--;
		}
	}
	free(st);
}

void R3D_InitLineProps(Render3D *sr, SFNode *node)
{
	LinePropStack *st = malloc(sizeof(LinePropStack));
	st->sr = sr;
	st->last_mod_time = 1;
	Node_SetPrivate(node, st);
	Node_SetPreDestroyFunction(node, DestroyLineProps);
}

u32 R3D_LP_GetLastUpdateTime(SFNode *node)
{
	LinePropStack *st = Node_GetPrivate(node);
	if (!st) return 0;
	if (Node_GetDirty(node) & SG_NODE_DIRTY) {
		st->last_mod_time ++;
		Node_ClearDirty(node);
	}
	return st->last_mod_time;
}


void R3D_DrawScene(VisualRenderer *vr, SFNode *top_node)
{
	RenderEffect static_eff;
	Render3D *sr = (Render3D *)vr->user_priv;

	if (top_node) {
		memcpy(&static_eff, sr->top_effect, sizeof(RenderEffect));

		/*note we wait for top node before setup to know if we need to create specific 3D stacks*/
		if (!sr->surface) {
			u32 tag = Node_GetTag(top_node);
			sr->surface = VS_New(((tag==TAG_Layer2D) || (tag==TAG_OrderedGroup)) ? 0 : 1);
			sr->surface->render = sr;
			sr->surface->width = sr->compositor->scene_width;
			sr->surface->height = sr->compositor->scene_height;
			assert(sr->surface);
		}
		VS_InitRender(sr->surface, sr->top_effect);
		mx_init(sr->top_effect->model_view);
		Node_Render(top_node, sr->top_effect);
		sr->top_effect->surface = NULL;
		memcpy(sr->top_effect, &static_eff, sizeof(RenderEffect));
	} else {
		VS3D_ClearSurface(sr->surface, sr->compositor->clear_color, 1.0f);
	}

	sr->compositor->video_out->FlushVideo(sr->compositor->video_out, NULL);
}

static void R3D_ExecuteEvent(VisualRenderer *vr, u32 event_type, Float x, Float y)
{
}

static void R3D_SetScaling(Render3D *sr, Float sx, Float sy)
{
	sr->scale_x = sx;
	sr->scale_y = sy;
}

M4Err R3D_RecomputeAR(VisualRenderer *vr)
{
	Double ratio;
	Float scaleX, scaleY;
	Render3D *sr = (Render3D *)vr->user_priv;
	if (!sr->compositor->scene_height || !sr->compositor->scene_width) return M4OK;
	if (!sr->compositor->height || !sr->compositor->width) return M4OK;

	sr->out_width = sr->compositor->width;
	sr->out_height = sr->compositor->height;
	sr->out_x = 0;
	sr->out_y = 0;

	if (!sr->compositor->has_size_info) {
		R3D_SetScaling(sr, 1, 1);
		if (sr->surface) {
			sr->surface->width = sr->compositor->scene_width;
			sr->surface->height = sr->compositor->scene_height;
		}
		return M4OK;
	}

	switch (sr->compositor->aspect_ratio) {
	case M4_AR_None:
		break;
	case M4_AR_16_9:
		sr->out_width = sr->compositor->width;
		sr->out_height = 9 * sr->compositor->width / 16;
		break;
	case M4_AR_4_3:
		sr->out_width = sr->compositor->width;
		sr->out_height = 3 * sr->compositor->width / 4;
		break;
	default:
		ratio = sr->compositor->scene_height;
		ratio /= sr->compositor->scene_width;
		if (sr->out_width * ratio > sr->out_height) {
			sr->out_width = sr->out_height * sr->compositor->scene_width;
			sr->out_width /= sr->compositor->scene_height;
		}
		else {
			sr->out_height = sr->out_width * sr->compositor->scene_height;
			sr->out_height /= sr->compositor->scene_width;
		}
		break;
	}
	sr->out_x = (sr->compositor->width - sr->out_width) / 2;
	sr->out_y = (sr->compositor->height - sr->out_height) / 2;
	/*update size info for main surface*/
	if (sr->surface) {
		sr->surface->width = sr->compositor->scene_width;
		sr->surface->height = sr->compositor->scene_height;
	}
	/*scaling is still needed for bitmap*/
	scaleX = (Float)sr->out_width / sr->compositor->scene_width;
	scaleY = (Float)sr->out_height / sr->compositor->scene_height;
	R3D_SetScaling(sr, scaleX, scaleY);
	return M4OK;
}


void R3D_SceneReset(VisualRenderer *vr)
{
	Render3D *sr = (Render3D*)vr->user_priv;
	/*destroy main surface*/
	if (sr->surface) VS_Delete(sr->surface);
	sr->surface = NULL;
}

void R3D_ReloadConfig(VisualRenderer *vr)
{
	const char *sOpt;
	Render3D *sr = (Render3D *)vr->user_priv;

	SR_Lock(sr->compositor, 1);

	sOpt = PMI_GetOpt(vr, "Render3D", "RasterOutlines");
	sr->raster_outlines = (sOpt && !stricmp(sOpt, "yes") ) ? 1 : 0;
	sOpt = PMI_GetOpt(vr, "Render3D", "EmulatePOW2");
	sr->emul_pow2 = (sOpt && !stricmp(sOpt, "yes") ) ? 1 : 0;
	sOpt = PMI_GetOpt(vr, "Render3D", "PolygonAA");
	sr->poly_aa = (sOpt && !stricmp(sOpt, "yes") ) ? 1 : 0;
	sOpt = PMI_GetOpt(vr, "Render3D", "Wireframe");
	if (sOpt && !stricmp(sOpt, "WireOnly")) sr->wiremode = M4_WireOnly;
	else if (sOpt && !stricmp(sOpt, "WireOnSolid")) sr->wiremode = M4_WireOnSolid;
	else sr->wiremode = M4_WireNone;

	sOpt = PMI_GetOpt(vr, "Render3D", "BitmapCopyPixels");
	sr->bitmap_use_pixels = (sOpt && !stricmp(sOpt, "yes") ) ? 1 : 0;

	sOpt = PMI_GetOpt(vr, "Render3D", "DisableRectExt");
	sr->disable_rect_ext = (sOpt && !stricmp(sOpt, "yes") ) ? 1 : 0;
	/*RECT texture support - we must reload HW*/
	SR_ResetGraphics(sr->compositor);

	SR_Lock(sr->compositor, 0);
}

M4Err R3D_LoadRenderer(VisualRenderer *vr, SceneRenderer *compositor)
{
	Render3D *sr;
	if (vr->user_priv) return M4BadParam;

	sr = malloc(sizeof(Render3D));
	if (!sr) return M4OutOfMem;
	memset(sr, 0, sizeof(Render3D));

	sr->compositor = compositor;
	sr->strike_bank = NewChain();
	/*create default unit sphere and box for bounds*/
	sr->unit_bbox = new_mesh();
	mesh_new_unit_bbox(sr->unit_bbox);
	sr->unit_bsphere = new_mesh();
	mesh_new_sphere(sr->unit_bsphere, 1.0f, 1);


	sr->top_effect = malloc(sizeof(RenderEffect));
	memset(sr->top_effect, 0, sizeof(RenderEffect));
	/*note we create main surface only when attached to scene*/

	vr->user_priv = sr;

	R3D_ReloadConfig(vr);
	return M4OK;
}

void R3D_UnloadRenderer(VisualRenderer *vr)
{
	Render3D *sr = (Render3D *)vr->user_priv;
	if (!sr) return;

	assert(!ChainGetCount(sr->strike_bank));
	DeleteChain(sr->strike_bank);

	if (sr->unit_bbox) mesh_free(sr->unit_bbox);
	if (sr->unit_bsphere) mesh_free(sr->unit_bsphere);
	effect_delete(sr->top_effect);
	free(sr);
	vr->user_priv = NULL;
}

void R3D_GraphicsReset(VisualRenderer *vr)
{
	Render3D *sr = (Render3D *)vr->user_priv;
	if (!sr) return;
	R3D_LoadExtensions(sr);
}

M4Err R3D_SetOption(VisualRenderer *vr, u32 option, u32 value)
{
	Render3D *sr = (Render3D *)vr->user_priv;
	switch (option) {
	case M4O_RasterOutlines: sr->raster_outlines = value; return M4OK;
	case M4O_EmulatePOW2: sr->emul_pow2 = value; return M4OK;
	case M4O_PolygonAA: sr->poly_aa = value; return M4OK;
	case M4O_Wireframe: sr->wiremode = value; return M4OK;
	case M4O_ReloadConfig:
		R3D_ReloadConfig(vr);
		return M4OK;
	case M4O_NoRectExt:
		if (value != sr->disable_rect_ext) {
			sr->disable_rect_ext = value;
			/*RECT texture support - we must reload HW*/
			SR_ResetGraphics(sr->compositor);
		}
		return M4OK;
	case M4O_BitmapCopyPixels: sr->bitmap_use_pixels = value; return M4OK;

	default:
		return M4BadParam;
	}
}

u32 R3D_GetOption(VisualRenderer *vr, u32 option)
{
	Render3D *sr = (Render3D *)vr->user_priv;
	switch (option) {
	case M4O_RasterOutlines: return sr->raster_outlines;
	case M4O_EmulatePOW2: return sr->emul_pow2;
	case M4O_PolygonAA: return sr->poly_aa;
	case M4O_Wireframe: return sr->wiremode;
	case M4O_NoRectExt: return sr->disable_rect_ext;
	case M4O_BitmapCopyPixels: return sr->bitmap_use_pixels;
	default: return 0;
	}
}

M4Err R3D_GetScreenBuffer(VisualRenderer *vr, M4VideoSurface *framebuffer);
M4Err R3D_ReleaseScreenBuffer(VisualRenderer *vr, M4VideoSurface *framebuffer);

/*interface create*/
void *LoadInterface(u32 InterfaceType)
{
	VisualRenderer *sr;
	if (InterfaceType != M4_RENDERER_INTERFACE) return NULL;
	
	sr = malloc(sizeof(VisualRenderer));
	if (!sr) return NULL;
	memset(sr, 0, sizeof(VisualRenderer));
	M4_REG_PLUG(sr, M4_RENDERER_INTERFACE, "GPAC 3D Renderer", "gpac distribution", 0);

	sr->LoadRenderer = R3D_LoadRenderer;
	sr->UnloadRenderer = R3D_UnloadRenderer;
	sr->GraphicsReset = R3D_GraphicsReset;
	sr->NodeChanged = R3D_NodeChanged;
	sr->NodeInit = R3D_NodeInit;
	sr->MapCoordsToAR = R3D_MapCoordsToAR;
	sr->DrawScene = R3D_DrawScene;
	sr->ExecuteEvent = R3D_ExecuteEvent;
	sr->RecomputeAR = R3D_RecomputeAR;
	sr->SceneReset = R3D_SceneReset;
	sr->AllocTexture = tx_allocate;
	sr->ReleaseTexture = tx_delete;
	sr->SetTextureData = R3D_SetTextureData;
	sr->TextureHWReset = R3D_TextureHWReset;

	sr->SetOption = R3D_SetOption;
	sr->GetOption = R3D_GetOption;
	sr->GetScreenBuffer = R3D_GetScreenBuffer;
	sr->ReleaseScreenBuffer = R3D_ReleaseScreenBuffer;

	/*signal we need openGL*/
	sr->bNeedsGL = 1;
	sr->user_priv = NULL;
	return sr;
}


/*interface destroy*/
void ShutdownInterface(void *ifce)
{
	VisualRenderer *rend = (VisualRenderer *)ifce;
	if (rend->InterfaceType != M4_RENDERER_INTERFACE) return;
	assert(rend->user_priv==NULL);
	free(rend);
}

/*interface query*/
Bool QueryInterface(u32 InterfaceType)
{
	if (InterfaceType == M4_RENDERER_INTERFACE) return 1;
	return 0;
}


