/*
 *			GPAC - MPEG-4 Systems C Development Kit
 *
 *			Copyright (c) Jean Le Feuvre 2000-2004 
 *					All rights reserved
 *
 *  This file is part of GPAC / Scene Rendering sub-project
 *
 *  GPAC is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *   
 *  GPAC is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *   
 *  You should have received a copy of the GNU General Public License
 *  along with GNU Make; see the file COPYING.  If not, write to
 *  the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA. 
 *
 */

#ifndef RENDER3D_H
#define RENDER3D_H

#include <gpac/intern/m4_render_dev.h>

#include "m4_mesh.h"


typedef struct _render3d
{
	/*remember main renderer*/
	SceneRenderer *compositor;
	/*all outlines cached*/
	Chain *strike_bank;
	/*main 3D surface*/
	struct _visual_surface *surface;

	/*options*/
	/*emulate power-of-2 for video texturing by using a po2 texture and texture scaling. If any textureTransform
	is applied to this texture, black stripes will appear on the borders.
	If not set video is written through glDrawPixels with bitmap (slow scaling), or converted to
	po2 texture*/
	Bool emul_pow2;
	/*use openGL for outline rather than vectorial ones*/
	Bool raster_outlines;
	/*disable RECT extensions (except for Bitmap...)*/
	Bool disable_rect_ext;
	/*disable RECT extensions (except for Bitmap...)*/
	Bool bitmap_use_pixels;

	/*top level effect for zoom/pan*/
	struct _render3d_effect *top_effect;

	/*current output info: screen size and top-left point of video surface, and current scaled scene size*/
	u32 out_width, out_height, out_x, out_y, cur_width, cur_height;
	/*needed for bitmap scaling when using DrawPixels rather than texture*/
	Float scale_x, scale_y;

	Bool poly_aa;
	u32 wiremode;
	/*unit box (1.0 size) and unit sphere (1.0 radius)*/
	M4Mesh *unit_bbox, *unit_bsphere;
} Render3D;

/*user interaction event*/
typedef struct
{
	u32 event_type;
	Float x, y;
} UserEvent3D;

/*sensor node handler - this is not defined as a stack because Anchor is both a grouping node and a 
sensor node, and we DO need the groupingnode stack...*/
typedef struct _sensorhandler
{
	/*sensor enabled or not ?*/
	Bool (*IsEnabled)(SFNode *node);
	/*user input on sensor:
	is_over: pointing device is over a shape the sensor is attached to
	is_down: pointing device down (mouse down)
	local_pt: pointing device coord (local system)
	world_pt: pointing device coord (world coord system)
	mx: inverse matrix of world transform at local coord system
	*/
	void (*OnUserEvent)(struct _sensorhandler *sh, Bool is_over, Bool is_down, SFVec3f *local_pt, SFVec3f *world_pt, M4Matrix *mx);
	/*set the node pointer here*/
	SFNode *owner;
} SensorHandler;


/*rendering modes*/
enum
{
	/*when set objects are drawn as soon as traversed, at each frame - THIS MODE IS AND WILL ALWAYS BE 0*/
	TRAVERSE_RENDER = 0,
	/*set to signal bindable render - only called on bindable stack top if present.
	for background: drawing request
	for viewports/viewpoints: viewpoint setup
	for fog: fog setup
	not used for navigation info
	*/
	TRAVERSE_RENDER_BINDABLE,
	/*recomputes bounds - result shall be stored in effect bbox field*/
	TRAVERSE_GET_BOUNDS,
	/*picking mode*/
	TRAVERSE_PICK,
};

enum
{
	/*only valid at root node*/
	CULL_NOT_SET = 0,
	/*subtree completely outside view vol*/
	CULL_OUTSIDE,
	/*subtree completely inside view vol*/
	CULL_INSIDE,
	/*subtree overlaps view vol - FIXME: would be nice to keep track of intersecting planes*/
	CULL_INTERSECTS
};

/*the traversing context: set_up at top-level and passed through SFNode_Render*/
typedef struct _render3d_effect
{
	AUDIO_EFFECT_CLASS

	
	u32 traversing_mode;

	/*the one and only surface currently traversed*/
	struct _visual_surface *surface;

	/*current transformation from top-level: cumulated modelview transforms from root
	*/
	M4Matrix model_view;
	/*current color transformation from top-level*/
	M4ColorMatrix color_mat;
	/*set if cmat is used*/
	Bool has_cmat;

	/*current appearance when traversing geometry nodes*/
	SFNode *appear;
	/*parent group for composition: can be Form, Layout or Layer2D*/
	struct _parent_group *parent;

	/*current bindable stacks - READ ONLY*/
	Chain *backgrounds;	/*holds 2D and 3D backgrounds*/
	Chain *viewpoints; /*holds viewpoints (3D) and viewports (2D)*/
	Chain *fogs; /*holds fogs info*/
	Chain *navigations; /*holds navigation info*/

	/*text splitting*/
	u32 text_split_mode;
	/*1-based idx of text element drawn*/
	u32 split_text_idx;
	/*bounds for TRAVERSE_GET_BOUNDS*/
	M4BBox bbox;
	/*cull flag (used to skip culling of children when parent bbox is completely inside/outside frustum*/
	u32 cull_flag;

	/*ray used for TRAVERSE_PICK*/
	M4Ray ray;
} RenderEffect;

void effect_reset(RenderEffect *eff);
void effect_delete(RenderEffect *eff);

Bool is_sensor_node(SFNode *node);


void R3D_LoadExtensions(Render3D *sr);

void R3D_LinePropsRemoved(Render3D *sr, SFNode *n);
Bool R3D_NodeChanged(VisualRenderer *vr, SFNode *byObj);
void R3D_NodeInit(VisualRenderer *vr, SFNode *node);
TextureHandler *R3D_GetTextureHandler(SFNode *n);

M4Err tx_allocate(TextureHandler *txh);
void tx_delete(TextureHandler *txh);
M4Err R3D_SetTextureData(TextureHandler *hdl);
void R3D_TextureHWReset(TextureHandler *hdl);
Bool tx_enable(TextureHandler *txh, SFNode *tx_transform);
void tx_disable(TextureHandler *txh);
char *tx_get_data(TextureHandler *txh, u32 *pix_format);
Bool tx_needs_reload(TextureHandler *hdl);
Bool tx_check_bitmap_load(TextureHandler *hdl);
void tx_copy_to_texture(TextureHandler *txh);
Bool tx_setup_format(TextureHandler *txh);
Bool tx_set_image(TextureHandler *txh, Bool generate_mipmaps);

#endif

