/*
 *			GPAC - MPEG-4 Systems C Development Kit
 *
 *			Copyright (c) Jean Le Feuvre 2000-2004 
 *					All rights reserved
 *
 *  This file is part of GPAC / Scene Rendering sub-project
 *
 *  GPAC is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *   
 *  GPAC is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *   
 *  You should have received a copy of the GNU General Public License
 *  along with GNU Make; see the file COPYING.  If not, write to
 *  the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA. 
 *
 */



#include "stacks3d.h"

static void DestroyViewStack(SFNode *node)
{
	ViewStack *st = (ViewStack *) Node_GetPrivate(node);
	PreDestroyBindable(node, st->reg_stacks);
	DeleteChain(st->reg_stacks);
	free(st);
}

#ifdef M4_DEF_Viewport

static void viewport_set_bind(SFNode *node)
{
	ViewStack *st = (ViewStack *) Node_GetPrivate(node);
	Bindable_OnSetBind(node, st->reg_stacks);
	SR_Invalidate(st->compositor, NULL);
}

static void RenderViewport(SFNode *node, void *rs)
{
	Float ar, sx, sy, w, h, tx, ty;
	M4Matrix2D mat;
	M4Matrix mx;
	M4Rect rc, rc_bckup;
	RenderEffect *eff = (RenderEffect *)rs;
	B_Viewport *vp = (B_Viewport*) node;
	ViewStack *st = (ViewStack *) Node_GetPrivate(node);

	if (eff->traversing_mode==TRAVERSE_PICK) return;

	assert(eff->viewpoints);

	/*first traverse, bound if needed*/
	if (ChainFindEntry(eff->viewpoints, node) < 0) {
		ChainAddEntry(eff->viewpoints, node);
		if (ChainGetEntry(eff->viewpoints, 0) == vp) {
			if (!vp->isBound) Bindable_SetIsBound(node, 1);
		}
		/*in any case don't draw the first time (since the viewport could have been declared last)*/
		SR_Invalidate(st->compositor, NULL);
		return;
	}
	/*not bound or in 3D surface*/
	if (!vp->isBound || eff->navigations) return;
	if (eff->traversing_mode != TRAVERSE_RENDER_BINDABLE) return;


	mx2d_init(mat);
	mx2d_add_translation(&mat, vp->position.x, vp->position.y);
	mx2d_add_rotation(&mat, 0, 0, vp->orientation);

	//compute scaling ratio
	m4_rect_center(&rc, vp->size.x, vp->size.y);
	mx2d_apply_rect(&mat, &rc);

	w = eff->bbox.max_edge.x - eff->bbox.min_edge.x;
	h = eff->bbox.max_edge.y - eff->bbox.min_edge.y;
	ar = h / w;

	rc_bckup = rc;

	switch (vp->fit) {
	/*covers all area and respect aspect ratio*/
	case 2:
		if (rc.width/w > rc.height/h) {
			rc.width *= h/rc.height;
			rc.height = h;
		} else {
			rc.height *= w/rc.width;
			rc.width = w;
		}
		break;
	/*fits inside the area and respect AR*/
	case 1:
		if (rc.width/w> rc.height/h) {
			rc.height *= w/rc.width;
			rc.width = w;
		} else {
			rc.width *= h/rc.height;
			rc.height = h;
		}
		break;
	/*fit entirely: nothing to change*/
	case 0:
		rc.width = w;
		rc.height = h;
		break;
	default:
		return;
	}
	sx = rc_bckup.width / rc.width;
	sy = rc_bckup.height / rc.height;

	rc.x = - rc.width/2;
	rc.y = rc.height/2;

	tx = ty = 0;
	if (vp->fit) {
		/*left alignment*/
		if (vp->alignment.vals[0] == -1) tx = rc.width/2 - w/2;
		else if (vp->alignment.vals[0] == 1) tx = w/2 - rc.width/2;

		/*top-alignment*/
		if (vp->alignment.vals[1]==-1) ty = rc.height/2 - h/2;
		else if (vp->alignment.vals[1]==1) ty = h/2 - rc.height/2;
	}

	mx_from_mx2d(&mx, &mat);
	mx_add_scale(&mx, sx, sy, 1.0);
	mx_add_translation(&mx, -tx, -ty, 0);
	mx_inverse(&mx);
	mx_add_matrix(&eff->surface->modelview, &mx);
}

void R3D_InitViewport(Render3D *sr, SFNode *node)
{
	ViewStack *st = malloc(sizeof(ViewStack));
	memset(st, 0, sizeof(ViewStack));
	st->reg_stacks = NewChain();
	traversable_setup(st, node, sr->compositor);
	Node_SetPrivate(node, st);
	Node_SetRenderFunction(node, RenderViewport);
	Node_SetPreDestroyFunction(node, DestroyViewStack);
	((B_Viewport*)node)->on_set_bind = viewport_set_bind;
}
#endif


#ifdef M4_DEF_Viewpoint

static void viewpoint_set_bind(SFNode *node)
{
	ViewStack *st = (ViewStack *) Node_GetPrivate(node);
	Bindable_OnSetBind(node, st->reg_stacks);
	SR_Invalidate(st->compositor, NULL);
}

static void RenderViewpoint(SFNode *node, void *rs)
{
	Float d, sin_a, cos_a, icos_a;

	RenderEffect *eff = (RenderEffect *)rs;
	B_Viewpoint *vp = (B_Viewpoint*) node;
	ViewStack *st = (ViewStack *) Node_GetPrivate(node);

	assert(eff->viewpoints);

	/*first traverse, bound if needed*/
	if (ChainFindEntry(eff->viewpoints, node) < 0) {
		ChainAddEntry(eff->viewpoints, node);
		if (ChainGetEntry(eff->viewpoints, 0) == vp) {
			if (!vp->isBound) Bindable_SetIsBound(node, 1);
		}
		/*in any case don't draw the first time (since the viewport could have been declared last)*/
		SR_Invalidate(st->compositor, NULL);
		return;
	}
	/*not bound or in 2D surface*/
	if (!vp->isBound || !eff->navigations) return;
	/*not rendering vp, return*/
	if (eff->traversing_mode != TRAVERSE_RENDER_BINDABLE) {
		/*store world matrix - NOTE: we always have a 1-frame delay between VP used and real world...
		we could remove this by pre-rendering the scene before applying vp, but that would mean 2 scene 
		traversals*/
		mx_copy(st->world_view_mx, eff->model_view);
		return;
	}

	/*set frustrum param*/
	eff->surface->frustum.position = vp->position;
	mx_apply_vec(&st->world_view_mx, &eff->surface->frustum.position);
	eff->surface->frustum.fieldOfView = vp->fieldOfView; 
    eff->surface->frustum.z_near = 0.5f*eff->surface->avatar_size.x; 
	if (eff->surface->frustum.z_near<=0) eff->surface->frustum.z_near = 0.01f;
	eff->surface->frustum.z_far = eff->surface->visibility; 
	if (eff->surface->frustum.z_far<=0) eff->surface->frustum.z_far = 1000.0f;

	/*compute up & target vectors*/
    d = 10.0f * eff->surface->avatar_size.x;
    if ((d<eff->surface->frustum.z_near) || (d>eff->surface->frustum.z_far)) d = 0.2f * (eff->surface->avatar_size.x + eff->surface->frustum.z_far);

	sin_a = (Float) sin(vp->orientation.angle);
	cos_a = (Float) cos(vp->orientation.angle);
	icos_a = 1.0f - cos_a;

	eff->surface->frustum.target.x = icos_a * vp->orientation.xAxis * vp->orientation.zAxis + sin_a * vp->orientation.yAxis;
	eff->surface->frustum.target.y = icos_a * vp->orientation.yAxis * vp->orientation.zAxis - sin_a * vp->orientation.xAxis;
	eff->surface->frustum.target.z = icos_a * vp->orientation.zAxis * vp->orientation.zAxis + cos_a;
	eff->surface->frustum.target = vec_scale(&eff->surface->frustum.target, -d);
	eff->surface->frustum.target = vec_add(&eff->surface->frustum.target, &vp->position);

	eff->surface->frustum.up.x = icos_a * vp->orientation.xAxis * vp->orientation.yAxis - sin_a * vp->orientation.zAxis;
	eff->surface->frustum.up.y = icos_a * vp->orientation.yAxis * vp->orientation.yAxis + cos_a;
	eff->surface->frustum.up.z = icos_a * vp->orientation.yAxis * vp->orientation.zAxis + sin_a * vp->orientation.xAxis;
	eff->surface->frustum.is_ortho = 0;
}

void R3D_InitViewpoint(Render3D *sr, SFNode *node)
{
	ViewStack *st = malloc(sizeof(ViewStack));
	memset(st, 0, sizeof(ViewStack));
	st->reg_stacks = NewChain();
	traversable_setup(st, node, sr->compositor);
	Node_SetPrivate(node, st);
	Node_SetRenderFunction(node, RenderViewpoint);
	Node_SetPreDestroyFunction(node, DestroyViewStack);
	((B_Viewpoint*)node)->on_set_bind = viewpoint_set_bind;
}

#endif


#ifdef M4_DEF_NavigationInfo

static void navinfo_set_bind(SFNode *node)
{
	ViewStack *st = (ViewStack *) Node_GetPrivate(node);
	Bindable_OnSetBind(node, st->reg_stacks);
	SR_Invalidate(st->compositor, NULL);
}

static void RenderNavigationInfo(SFNode *node, void *rs)
{
	u32 i;

	RenderEffect *eff = (RenderEffect *)rs;
	B_NavigationInfo *ni = (B_NavigationInfo *) node;
	ViewStack *st = (ViewStack *) Node_GetPrivate(node);

	if (!eff->navigations) return;

	/*first traverse, bound if needed*/
	if (ChainFindEntry(eff->viewpoints, node) < 0) {
		ChainAddEntry(eff->viewpoints, node);
		if (ChainGetEntry(eff->viewpoints, 0) == ni) {
			if (!ni->isBound) Bindable_SetIsBound(node, 1);
		}
		/*in any case don't draw the first time (since the viewport could have been declared last)*/
		SR_Invalidate(st->compositor, NULL);
		return;
	}
	/*not bound*/
	if (!ni->isBound) return;
	/*not rendering, return*/
	if (eff->traversing_mode != TRAVERSE_RENDER_BINDABLE) return;

	eff->surface->navigation_flags = 0;
	for (i=0; i<ni->type.count; i++) {
		if (ni->type.vals[i] && stricmp(ni->type.vals[i], "ANY")) eff->surface->navigation_flags |= NAV_ANY;
		else if (ni->type.vals[i] && stricmp(ni->type.vals[i], "WALK")) eff->surface->navigation_flags |= NAV_WALK;
		else if (ni->type.vals[i] && stricmp(ni->type.vals[i], "EXAMINE")) eff->surface->navigation_flags |= NAV_EXAMINE;
		else if (ni->type.vals[i] && stricmp(ni->type.vals[i], "FLY")) eff->surface->navigation_flags |= NAV_FLY;
		else if (ni->type.vals[i] && stricmp(ni->type.vals[i], "NONE")) eff->surface->navigation_flags |= NAV_NONE;
	}
	if (ni->headlight) eff->surface->navigation_flags |= NAV_HEADLIGHT;

	eff->surface->speed = ni->speed;
    eff->surface->visibility = ni->visibilityLimit;
    
	if (ni->avatarSize.count) eff->surface->avatar_size.x = ni->avatarSize.vals[0];
	if (ni->avatarSize.count>1) eff->surface->avatar_size.y = ni->avatarSize.vals[1];
	if (ni->avatarSize.count>2) eff->surface->avatar_size.z = ni->avatarSize.vals[2];
}

void R3D_InitNavigationInfo(Render3D *sr, SFNode *node)
{
	ViewStack *st = malloc(sizeof(ViewStack));
	memset(st, 0, sizeof(ViewStack));
	st->reg_stacks = NewChain();
	traversable_setup(st, node, sr->compositor);
	Node_SetPrivate(node, st);
	Node_SetRenderFunction(node, RenderNavigationInfo);
	Node_SetPreDestroyFunction(node, DestroyViewStack);
	((B_NavigationInfo*)node)->on_set_bind = navinfo_set_bind;
}

#endif


