/*
 *			GPAC - MPEG-4 Systems C Development Kit
 *
 *			Copyright (c) Jean Le Feuvre 2000-2003 
 *					All rights reserved
 *
 *  This file is part of GPAC / plugins interfaces
 *
 *  GPAC is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *   
 *  GPAC is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *   
 *  You should have received a copy of the GNU General Public License
 *  along with GNU Make; see the file COPYING.  If not, write to
 *  the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA. 
 *
 */



#ifndef __M4_DECODER_H_
#define __M4_DECODER_H_


#include <gpac/m4_tools.h>

#ifdef __cplusplus
extern "C" {
#endif

/*multimedia processing levels*/
enum
{
	/*normal, full processing*/
	MM_LEVEL_NORMAL,
	/*codec is late, should scale down processing*/
	MM_LEVEL_LATE,
	/*codec is very late, should turn off post-processing, even drop*/
	MM_LEVEL_VERY_LATE,
	/*input frames are already late before decoding*/
	MM_LEVEL_DROP,
	/*this is a special level indicating that a seek is happening (decode but no dispatch)
	it is set dynamically*/
	MM_LEVEL_SEEK
};

/*the structure for capabilities*/
typedef struct 
{
	/*cap code cf below*/
	u16 CapCode;
	union {
		u32 valueINT;
		Float valueFT;
	} cap;
} CapObject;

/*media codec interface name*/
#define M4MEDIADECODERINTERFACE		FOUR_CHAR_INT('M', 'D', 'E', 'C')

/*the media plugin interface. A media plugin MUST be implemented in synchronous mode as time 
and resources management is done by the terminal
NOTE: all functions pass the interface object so that app functions can be used
*/

typedef struct _decoderinterface
{
	/* interface declaration*/
	M4_DECL_PLUGIN_INTERFACE

	/* Add a Stream to the codec. If DependsOnESID is NULL, the stream is a base layer
	UpStream means that the decoder should send feedback on this channel. 
	WARNING: Feedback format is not standardized by MPEG
	the same API is used for both encoder and decoder (decSpecInfo is ignored
	for an encoder) */
	M4Err	(*Codec_AttachStream)(struct _decoderinterface *, u16 ES_ID, unsigned char *decSpecInfo, u32 decSpecInfoSize, u16 DependsOnES_ID, u32 objectTypeIndication, Bool UpStream);
	/*Remove stream*/
	M4Err	(*Codec_DetachStream)(struct _decoderinterface *, u16 ES_ID);

	
	/*Get the desired capability given its code*/
	M4Err	(*Codec_GetCapabilities)(struct _decoderinterface *, CapObject *capability);
	
	/*Set the desired capability given its code if possible
	if the codec does not support the request capability, return M4NotSupported*/
	M4Err	(*Codec_SetCapabilities)(struct _decoderinterface *, CapObject capability);
	
	/*Process the media data in inAU. 
	@inBuffer, inBufferLength: encoded input AU
	@ES_ID: stream this data belongs too (scalable object)
	@outBuffer, outBufferLength: allocated data for decoding - if outBufferLength is not enough
		you should set the size in outBufferLength and M4BufferTooSmall 

	@PaddingBits is the padding at the end of the buffer
	@mmlevel is managed by the main scheduler and indicates how late the stream is
		use this var to drop / adjust filtering / ... when possible*/
	M4Err (*Codec_Process)(struct _decoderinterface *, 
			unsigned char *inBuffer, u32 inBufferLength,
			u16 ES_ID,
			unsigned char *outBuffer, u32 *outBufferLength,
			Bool isRAP, u8 PaddingBits, u32 mmlevel);

	/*	SHAPE CODED VIDEO ONLY: gets X and Y offset pixels (from top-left) in main frame
		the function is called after each frame for shaped video
				can be set to NULL if not supported
	*/
	M4Err (*Codec_GetFramePosition)(struct _decoderinterface *, u32 *offset_x, u32 *offset_y);

	/*Can plugin handle this codec? Return 0 if No and !0 otherwise
	decoderSpecificInfo is provided for MPEG4 audio/visual where a bunch of codecs are defined 
	with same objectType
		NOTE: until the plugin has return TRUE no other function (except ShutdownInterface) are called, 
		it is thus allowed to dynamically change the function pointers of the interface in the IsYourCodec
		function (except of course IsYourCodec !!)
		NOTE2: when passed with only streamType specified, the plugin should return whether it can handle this stream 
		type or not
	*/
	Bool (*CanHandleStream)(struct _decoderinterface *, u32 StreamType, u32 ObjectType, unsigned char *decSpecInfo, u32 decSpecInfoSize, u32 PL);

	/*returns codec name - only called once the stream is successfully attached*/
	const char *(*GetCodecName)(struct _decoderinterface *);

	/*interface private data*/
	void *privateStack;

} DecoderInterface;


/*
			all codecs capabilities
*/

enum
{
	/*size of a single composition unit */
	CAP_OUTPUTSIZE			=	0x01,
	/*average and max bitrate */
	CAP_AVGBITRATE			=	0x02,
	CAP_MAXBITRATE			=	0x03,
	/*profile and level */
	CAP_PL					=	0x04,
	/*resilency: if packets are lost within an AU, resilience means the AU won't be discarded and the codec
	will try to decode */
	CAP_HASRESILIENCE		=	0x05,
	/*critical level of composition memory - if below, media management for the object */
	CAP_BUFFER_MIN			=	0x06,
	/*maximum size in CU of composition memory */
	CAP_BUFFER_MAX			=	0x07,
	/*flags that all AUs should be discarded till next RAP (needed in case RAPs are not carried by the transport
	protocol */
	CAP_WAIT_RAP			=	0x08,
	/*number of padding bytes needed - if the decoder needs padding input cannot be pulled and data is duplicated*/
	CAP_PADDING_BYTES		=	0x09,
	/*codecs can be threaded at will - by default a single thread is used for all decoders and priority is handled
	by the app, but a codec can configure itself to run in a dedicated thread*/
	CAP_CODEC_WANTSTHREAD	=	0x10,

	/*video width and height and horizontal pitch (in YV12 we assume half Y pitch for U and V planes) */
	CAP_WIDTH				=	0x11,
	CAP_HEIGHT				=	0x12,
	CAP_PITCH				=	0x13,
	/*video color mode*/
	CAP_COLORMODE			=	0x14,
	/*post-processing options*/
	CAP_VID_POSTPROC		=	0x15,
	/*query-only, indicates whether the video is shape-coded or not
	if yes, shape offset in main frame has to be retrieved for each frame*/
	CAP_VID_SHAPE		=	0x16,
	/*isgnal decoder performs frame re-ordering in temporal scalability*/
	CAP_VID_REORDER		=	0x17,
	
	/*Audio sample rate*/
	CAP_SAMPLERATE			=	0x30,
	/*Audio num channels*/
	CAP_NBCHANNELS			=	0x31,
	/*Audio bps*/
	CAP_BITSPERSAMPLE		=	0x32,

	/*this is only used for audio in case transport mapping relies on sampleRate (RTP)
	gets the CU duration in samplerate unit (type: int) */
	CAP_CU_DURATION			=	0x33
};


/*the video post-processing types (flags) */
enum
{
	CAP_VID_NONE = 0,
	CAP_VID_DEBLOCK = (1),
	CAP_VID_DERING = (1<<2),

	/*for scalable*/
	CAP_VID_BASEONLY = (1<<5),
};


/*color modes are in m4_tools.h*/



/*extensions for undefined codecs - this allows demuxers and codecs to talk the same language*/

/*this is the OTI (user-priv) used for all undefined codec using MP4/QT 4CC codes*/
#define GPAC_QT_CODECS_OTI				0x80

/*The decoder specific info for all unknown decoders - it is always carried encoded

	u32 codec_four_cc: the codec 4CC reg code
	- for audio - 
	u32 sample_rate: sampling rate or 0 if unknown
	u32 nb_channels: num channels or 0 if unknown
	u32 nb_bits_per_sample: nb bits or 0 if unknown
	u32 num_samples: num audio samples per frame or 0 if unknown

  	- for video - 
	u32 width: video width or 0 if unknown;
	u32 height: video height or 0 if unknown;

	- till end of DSI bitstream-
	char *data: per-codec extensions 
*/

#ifdef __cplusplus
}
#endif

#endif	/*__M4_DECODER_H_*/

