/*
 *			GPAC - MPEG-4 Systems C Development Kit
 *
 *			Copyright (c) Jean Le Feuvre 2000-2003 
 *					All rights reserved
 *
 *  This file is part of GPAC / Scene Graph sub-project
 *
 *  GPAC is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *   
 *  GPAC is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *   
 *  You should have received a copy of the GNU General Public License
 *  along with GNU Make; see the file COPYING.  If not, write to
 *  the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA. 
 *
 */


#ifndef M4_SCENEGRAPH_H_
#define M4_SCENEGRAPH_H_

#ifdef __cplusplus
extern "C" {
#endif


#include <gpac/m4_tools.h>



/*
	VRML / BIFS TYPES DEFINITION
*/

/*
				event types, as defined in the specs 
	this should not be needed by non binary codecs
*/
enum
{
	ET_Field		=	0,
	ET_ExposedField	=	1,
	ET_EventIn		=	2,
	ET_EventOut		=	3,
	ET_Unknown	=	4
};
const char *GetEventTypeName(u32 EventType);

/*
				field coding mode

	BIFS defines the bitstream syntax contextually, and therefore sometimes refer to fields as indexed
  in the node ("all" mode) or just as a sub-set (in, out, def, dyn modes) of similar types
*/
enum
{
	/*all fields and events*/
	FCM_ALL		=	0,
	/*defined fields (exposedField and Field)*/
	FCM_DEF		=	1,
	/*input field (exposedField and eventIn)*/
	FCM_IN		=	2,
	/*output field (exposedField and eventOut)*/
	FCM_OUT		=	3,
	/*field that can be animated (subset of inFields) used in BIFS_Anim only*/
	FCM_DYN		=	4
};



/*		SF Types	*/
typedef Bool SFBool;
typedef s32 SFInt32;
typedef s32 SFInt;
typedef Float SFFloat;

typedef struct
{
	unsigned char* buffer;
} SFString;

typedef Double SFTime;

typedef struct {
	Float	red;
	Float	green;
	Float	blue;
} SFColor;
typedef struct {
	Float	xAxis;
	Float	yAxis;
	Float	zAxis;
	Float	angle;
} SFRotation;
typedef struct {
	u32 OD_ID;
	char *url;
} SFURL;
typedef struct {
	Float	x;
	Float	y;
} SFVec2f;
typedef struct {
	Float	x;
	Float	y;
	Float	z;
} SFVec3f;

typedef struct {
	Float	x;
	Float	y;
	Float	z;
	Float	q;
} SFVec4f;
typedef struct {
	u32 width;
	u32 height;
	u8 numComponents;
	unsigned char* pixels;
} SFImage;
typedef struct {
	u32 bufferSize;
	unsigned char* buffer;
	/*uncompressed command list*/
	Chain *commandList;
} SFCommandBuffer;

/*Note on SFScript: the javascript or vrml script is handled in its decompressed (textual) format
since most JS interpreter work with text*/
typedef struct {
	unsigned char* script_text;
} SFScript;


/*		MF Types	*/

/*generic MF field: all MF fields use the same syntax except MFNode which uses Chain. You  can thus use
this structure to safely typecast MF field pointers*/
typedef struct {
	u32 count;
	char *array;
} GenMFField;

typedef struct {
	u32 count;
	s32* vals;
} MFInt32;
typedef struct {
	u32 count;
	s32* vals;
} MFInt;
typedef struct {
	u32 count;
	Float* vals;
} MFFloat;
typedef struct {
	u32 count;
	u32* vals;
} MFBool;
typedef struct {
	u32 count;
	SFColor* vals;
} MFColor;
typedef struct {
	u32 count;
	SFRotation*	vals;
} MFRotation;
typedef struct {
	u32 count;
	Double* vals;
} MFTime;
typedef struct {
	u32 count;
	SFVec2f* vals;
} MFVec2f;
typedef struct {
	u32 count;
	SFVec3f* vals;
} MFVec3f;
typedef struct {
	u32 count;
	SFVec4f* vals;
} MFVec4f;

typedef struct {
	u32 count;
	SFURL* vals;
} MFURL;
typedef struct {
	u32 count;
	char** vals;
} MFString;
typedef struct {
	u32 count;
	SFScript *vals;
} MFScript;




/*field types, as defined in BIFS encoding (used for scripts and proto coding)*/
enum
{
	FT_SFBool		=	0,
	FT_SFFloat		=	1,
	FT_SFTime		=	2,
	FT_SFInt32		=	3,
	FT_SFString		=	4,
	FT_SFVec3f		=	5,
	FT_SFVec2f		=	6,
	FT_SFColor		=	7,
	FT_SFRotation	=	8,
	FT_SFImage		=	9,
	FT_SFNode		=	10,
	/*TO CHECK*/
	FT_SFVec4f		=	11,

	FT_MFBool		=	32,
	FT_MFFloat		=	33,
	FT_MFTime		=	34,
	FT_MFInt32		=	35,
	FT_MFString		=	36,
	FT_MFVec3f		=	37,
	FT_MFVec2f		=	38,
	FT_MFColor		=	39,
	FT_MFRotation	=	40,
	FT_MFImage		=	41,
	FT_MFNode		=	42,
	/*TO CHECK*/
	FT_MFVec4f		=	43,

	/*these types are not defined in the Specs*/
	FT_SFURL		=	50,
	FT_MFURL		=	51,
	FT_SFCommandBuffer		=	52,
	FT_SFScript		=	53,
	FT_MFScript		=	54,

	FT_Unknown		=	55
};
const char *GetFieldTypeName(u32 FieldType);


/*
allocates a new field and gets it back. 
	NOTE:
			FT_MFNode will return a pointer to a Chain structure (eg Chain *)
			FT_SFNode will return NULL
*/
void *SG_NewFieldPointer(u32 FieldType);
/*deletes a field pointer (including SF an,d MF nodes)*/
void SG_DeleteFieldPointer(void *field, u32 FieldType);

Bool SG_IsSFField(u32 FieldType);

/*translates MF/SF to SF type*/
u32 SG_GetSFType(u32 FieldType);


/*
	MFField manipulation  - MFNode cannot use these, use the Chain functions instead
	or the Node_* insertion functions
	FieldType shall always be given when manipulating MFFields
*/
/*Insert (+alloc) a slot in the MFField with a specified position for insertion and sets the ptr
to the newly created slot
@InsertAt is the 0-based index for the new slot
*/
M4Err MFField_Insert(void *mf, u32 FieldType, void **new_ptr, u32 InsertAt);
/*adds at the end and gets the ptr*/
M4Err MFField_Append(void *mf, u32 FieldType, void **new_ptr);
/*remove the desired item*/
M4Err MFField_Remove(void *mf, u32 FieldType, u32 RemoveFrom);
/*alloc a fixed array*/
M4Err MFField_Alloc(void *mf, u32 FieldType, u32 NbItems);
/*get the item in the array*/
M4Err MFField_GetItem(void *mf, u32 FieldType, void **new_ptr, u32 ItemPos);
/*remove all items of the MFField*/
M4Err MFField_Reset(void *mf, u32 FieldType);

/*clones a field content EXCEPT SF/MFNode. Pointers to field shall be used
@dest, @orig: pointers to field
@FieldType: type of the field
*/
void SG_CopyField(void *dest, void *orig, u32 FieldType);

/*indicates whether 2 fields of same type EXCEPT SF/MFNode are equal
@dest, @orig: pointers to field
@FieldType: type of the field
*/
Bool SG_FieldsEqual(void *dest, void *orig, u32 FieldType);


/*private handler for this library on all nodes*/
#define BASE_NODE	struct _nodepriv *sgprivate;

/*base node type*/
typedef struct _sfNode
{
	BASE_NODE
} SFNode;




/*
	grouping nodes macro

All grouping nodes (with "children" field) implement the following: 

addChildren: chain containing nodes to add passed as eventIn - handled internally through ROUTE

void (*on_addChildren)(SFNode *pNode): add eventIn signaler - this is handled internally by the scene_graph and SHALL 
NOT BE OVERRIDEN since it takes care of node(s) routing

removeChildren: chain containing nodes to remove passed as eventIn - handled internally through ROUTE

void (*on_removeChildren)(SFNode *pNode): remove eventIn signaler - this is handled internally by the scene_graph and SHALL 
NOT BE OVERRIDEN since it takes care of node(s) routing
  

children: list of children SFNodes
*/

#define CHILDREN									\
	Chain *addChildren;								\
	void (*on_addChildren)(SFNode *pNode);			\
	Chain *removeChildren;							\
	void (*on_removeChildren)(SFNode *pNode);		\
	Chain *children;


/*tag is set upon creation and cannot be modified*/
u32 Node_GetTag(SFNode*);
/*set node def
ID: if !0 set def node - if a different node with the same ID exists, returns error. If ID=0, 
node is undef'ed (use with care..) but the refcount is untounched. You may change the node ID by recalling the function 
with a different ID value
defName: optional readable name (script, MPEGJ). To change the name, recall the function with a different name and the same ID
*/
M4Err Node_SetDEF(SFNode*n, u32 nodeID, const char *nodeDEFName);
/*get def name of the node , NULL if not set*/
const char *Node_GetDefName(SFNode*);
/*get def ID of the node, 0 if node not def*/
u32 Node_GetID(SFNode*);

/*get/set user private stack*/
void *Node_GetPrivate(SFNode*);
void Node_SetPrivate(SFNode*, void *);

/*set rendering function. When rendering a scene graph, the render stack is passed
through the graph without being touched. Instanciated Protos are handled internally
as well as interpolators, valuators and scripts
NOTE: if a node with sub-nodes (MF or SF node fields) has no associated RenderNode()
the traversing of the tree won't propagate below it. It is the app responsability to setup traversing 
functions as needed*/
M4Err Node_SetRenderFunction(SFNode *, void (*RenderNode)(SFNode *node, void *render_stack) );
/*set pre-destroy function in order to delete any private data*/
M4Err Node_SetPreDestroyFunction(SFNode *, void (*PreDestroyNode)(struct _sfNode *node) );


/*register a node with parent (node may or not be DEF'ed). Parent may be NULL (DEF root node, commands). 
This MUST be called whenever a node is instanciated (child of a parent node)*/
M4Err Node_Register(SFNode *node, SFNode *parent_node);

/*unregister a node from parent (node may or not be DEF'ed). Parent may be NULL (DEF root node, commands).
This MUST be called whenever a node is destroyed (removed from a parent node)
If this is the last instance of the node, the node is destroyed*/
M4Err Node_Unregister(SFNode *node, SFNode *parent_node);

/*get all parents of the node and replace the instances of the node by the new node and finally destroy the node
Note: if the new node is not DEFed, only the first instance of "old_node" will be replaced, the other ones deleted*/
M4Err Node_ReplaceAllInstances(SFNode *old_node, SFNode *new_node, Bool updateOrderedGroup);

/*deletes all node instances in the given list (can be the "children" field or any MFNode field)*/
void Node_ResetChildren(SFNode *node, Chain *childrenlist);

/*adds a new node to the "children" field
position is the 0-BASED index in the list of children, -1 means end of list (append)*/
M4Err Node_InsertChild(SFNode *parent, SFNode *new_child, s32 Position);

/*removes an existing node from the "children" field*/
M4Err Node_RemoveChild(SFNode *parent, SFNode *toremove_child);

/*remove and replace given child by specified node. If node is NULL, only delete target node
position is the 0-BASED index in the list of children, -1 means end of list (append)*/
M4Err Node_ReplaceChild(SFNode *node, Chain *container, s32 pos, SFNode *newNode);

/*signals eventOut has been set. FieldIndex/eventName identify the eventOut field. Routes are automatically triggered
when the event is signaled*/
void Node_OnEventOut(SFNode *node, u32 FieldIndex);
void Node_OnEventOutSTR(SFNode *node, const char *eventName);


/*calls RenderNode on this node*/
void Node_Render(SFNode *node, void *renderStack);
/*blindly calls RenderNode on all nodes in the "children" list*/
void Node_RenderChildren(SFNode *node, void *renderStack);

/*returns 0 if field is not default value - NOT IMPLEMENTED YET*/
u32 Node_FieldIsDefault(SFNode *Node, u32 Index);
/*returns 1 if node is DEFed, 0 otherwise*/
Bool Node_IsDEF(SFNode *node);

/*returns number of parent for this node (parent are kept regardless of DEF state)*/
u32 Node_GetParentCount(SFNode *node);
/*returns desired parent for this node (parent are kept regardless of DEF state)
idx is 0-based parent index*/
SFNode *Node_GetParent(SFNode *node, u32 idx);

/* gets node built-in name (eg 'Appearance', ..) */
const char *Node_GetName(SFNode *Node);

enum
{
	/*flag set whenever a field of the node has been modified*/
	SG_NODE_DIRTY = 1,
	/*flag set whenever a child node of this node has been modified*/
	SG_CHILD_DIRTY = 1<<1,
};

/*set dirty flag on (eg SG_NODE_DIRTY).
If invalidate_parents is set, all parent subtrees nodes are marked as dirty (eg SG_CHILD_DIRTY)
(marking aborts if a node in the subtree is already marked)*/
void Node_SetDirty(SFNode *node, Bool invalidate_parents);
/*set dirty flag off. It is the user responsability to clear dirty flags.
Note that a node doesn't have to be cleared to have parent signaled (eg SG_CHILD_DIRTY
is set regardless of SG_NODE_DIRTY state). this alows skipping dirty detecion on fields of type 
SFNode (for ex, lineProps in material)*/
void Node_ClearDirty(SFNode *node);
/*same as ClearDirty, but also clears all subtrees
if skip_if_clean is set and node is not dirty, children are not cleared*/
void Node_ClearDirtyChildren(SFNode *node, Bool skip_if_clean);
/*get dirty flag value*/
u16 Node_GetDirty(SFNode *node);
/*marks node and all its subtrees nodes as dirty (SG_NODE_DIRTY)*/
void Node_SetDirtyChildren(SFNode *node);


/*all eventIn handlers are of this types. If you wish to handle a given eventIn, overwrite the 
handler of the field by your specific call*/
typedef void (*OnEventIn)(SFNode *pNode);

typedef struct
{	
	/*field type, SF/MFXXX*/
	u32 fieldType;
	/*far ptr to the field (eg SFNode **, Chain**, MFInt32 *, ...)*/
	void *far_ptr;
	/*NDT type in case of SF/MFNode field - cf BIFS specific tools*/
	u32 NDTtype;
	/*event type*/
	u32 eventType;
	/*eventin handler if any*/
	OnEventIn on_event_in;
	/*field name*/
	const char *name;
	/*0-based index of the field in the node*/
	u32 allIndex;
} FieldInfo;

/*fill the field info structure for the given field*/
M4Err Node_GetField(SFNode *node, u32 FieldIndex, FieldInfo *info);

/*get the field by its name*/
M4Err Node_GetFieldByName(SFNode *node, char *name, FieldInfo *field);

/*returns node tag by name*/
u32 Node_GetTagByName(const char *node_name);


/*opaque handler for the scene graph object*/
typedef struct _tagSceneGraph *LPSCENEGRAPH;

/*scene graph constructor: a set of callback function for the user. 
All these functions and callbacks are optional, you can pass NULL if you don't need them*/
LPSCENEGRAPH NewSceneGraph(
		/*function called upon node creation. Application should instanciate the node rendering stack and 
		any desired callback*/
		void (*UserNodeInit)(void *NodeInitCallback, SFNode *newNode), void *NodeInitCallback,
		/*the codec will notify any modification on nodes so that an application can decide
		whether the scene must be redrawned or not. You typucally will set the dirty flags here*/
		void (*NodeModified)(void *ModifCallback, SFNode *node), void *ModifCallback			
	);

/*creates a sub scene graph (typically used with Inline node): independent graph with same private stack, 
and user callbacks as parent. All routes triggered in this subgraph are executed in the parent graph (this 
means you only have to activate routes on the main graph)
NOTE: the resulting graph is not destroyed when the parent graph is 
*/
LPSCENEGRAPH SG_NewSubScene(LPSCENEGRAPH scene);

/*set the scene timer - callback is the same as simulation time callback*/
void SG_SetSceneTime(LPSCENEGRAPH scene, Double (*GetSceneTime)(void *SceneCallback), void *SceneCallback);

/*set proto loader - callback is the same as simulation time callback
	GetExternProtoLib is a pointer to the proto lib loader - this callback shall return the LPSCENEGRAPH
of the extern proto lib if found and loaded, NULL if not found and SG_INTERNAL_PROTO for internal
hardcoded protos (extensions of MPEG-4 scene graph used for module deveopment)
*/
#define SG_INTERNAL_PROTO	(LPSCENEGRAPH) 0xFFFFFFFF
void SG_SetProtoLoader(LPSCENEGRAPH scene, LPSCENEGRAPH (*GetExternProtoLib)(void *SceneCallback, MFURL *lib_url));

/*destructor - all nodes, routes and protos are destroyed*/
void SG_Delete(LPSCENEGRAPH sg);
/*set/get user private data*/
void SG_SetPrivate(LPSCENEGRAPH sg, void *);
void *SG_GetPrivate(LPSCENEGRAPH sg);

/*BIFS allows working in pixelMetrics and can specify the rendering size
by default graphs have no size and are in meter metrics
if any of width or height is 0, the graph has no size info*/
void SG_SetSizeInfo(LPSCENEGRAPH sg, u32 width, u32 Height, Bool usePixelMetrics);
/*returns 1 if pixelMetrics*/
Bool SG_UsePixelMetrics(LPSCENEGRAPH sg);
/*returns 0 if no size info, otherwise 1 and set width/height*/
Bool SG_GetSizeInfo(LPSCENEGRAPH sg, u32 *width, u32 *Height);

/*reset the full graph (including protos) needed for graph random access points such as BIFS ReplaceScene*/
void SG_Reset(LPSCENEGRAPH sg);

/*get/set the root node of the graph*/
SFNode *SG_GetRootNode(LPSCENEGRAPH sg);
void SG_SetRootNode(LPSCENEGRAPH sg, SFNode *node);

/*finds a registered node by ID*/
SFNode *SG_FindNode(LPSCENEGRAPH sg, u32 nodeID);
/*finds a registered node by DEF name*/
SFNode *SG_FindNodeByName(LPSCENEGRAPH sg, char *name);

/*used to signal modification of a node, indicating which field is modified - exposed for BIFS codec, 
should not be needed by other apps*/
void SG_NodeChanged(SFNode *node, FieldInfo *fieldChanged);

/*creates a node of the given tag. sg is the parent scenegraph of the node,
eg the root one for scene nodes or the proto one for proto code (cf proto)
this doesn't perform application setup for the node, you shall call Node_Init in order to init
your specific extensions*/
SFNode *SG_NewNode(LPSCENEGRAPH sg, u32 tag);

/*clones a node in a(nother) scene graph
WARNING: currently all DEF/USE are removed (eg, a hard copy of each child node instance is created) which may be problematic
with protos/scripts.
The clone returned is NOT registered with any parent, this is the caller responsability (as with SG_NewNode)
*/
SFNode *Node_Clone(LPSCENEGRAPH inScene, SFNode *orig);

/*returns the graph this node belongs to*/
LPSCENEGRAPH Node_GetParentGraph(SFNode *node);

/*inits node (either internal stack or user-defined) - typically called once the node has been loaded */
void Node_Init(SFNode *node);

/*gets scene time for scene this node belongs too, 0 if timeline not specified*/
Double Node_GetSceneTime(SFNode *node);
/*gets scene scene*/
Double SG_GetSceneTime(LPSCENEGRAPH sg);


/*Route manip: routes are used to pass events between nodes. Event handling is managed by the scene graph
however only the nodes overloading the EventIn handler associated with the event will process the eventIn*/
typedef struct _route *LPROUTE;

/*creates a new route:
	@fromNode: @fromField: address of the eventOut field triggering the route
	@toNode: @toField: address of the destination eventIn field
NOTE: routes are automatically destroyed if either the target or origin node of the route is destroyed
*/
LPROUTE SG_NewRoute(LPSCENEGRAPH sg, SFNode *fromNode, u32 fromField, SFNode *toNode, u32 toField);
/*creates a new route using field names instead of indexes*/
LPROUTE SG_NewRouteByName(LPSCENEGRAPH sg, SFNode *fromNode, char *fromField, SFNode *toNode, char *toField);

/*delete route*/
void SG_DeleteRoute(LPROUTE route);
M4Err SG_DeleteRouteByID(LPSCENEGRAPH sg,u32 routeID);

/*locate route by ID/name*/
LPROUTE SG_FindRoute(LPSCENEGRAPH sg, u32 RouteID);
LPROUTE SG_FindRouteByName(LPSCENEGRAPH sg, char *name);
/*assign route ID - fails if a route with same ID already exist*/
M4Err Route_SetID(LPROUTE route, u32 ID);
u32 Route_GetID(LPROUTE route);
/*assign route name if desired*/
M4Err Route_SetName(LPROUTE route, char *name);
char *Route_GetName(LPROUTE route);

/*activates all routes currently triggered - this follows the event cascade model of VRML/MPEG4:
	- routes are collected during eventOut generation
	- routes are activated. If eventOuts are generated during activation the cycle goes on.

  A route cannot be activated twice in the same simulation tick, hence this function shall be called 
  ONCE AND ONLY ONCE per simulation tick

Note that children scene graphs register their routes with the top-level graph, so only the main 
scene graph needs to be activated*/
void SG_ActivateRoutes(LPSCENEGRAPH sg);


/*
				proto handling

	The lib allows you to construct prototype nodes as defined in VRML/MPEG4 by constructing 
	proto interfaces and instanciating them. An instanciated proto is handled as a single node for
	rendering, thus an application will never handle proto instances for rendering
*/

/*opaque handler for a proto object (declaration)*/
typedef struct _proto *LPPROTO;
/*opaque handler for a proto field object (declaration)*/
typedef struct _protofield *LPPROTOFIELD;

/*proto constructor identified by ID/name in the given scene
2 protos in the same scene may not have the same ID/name

@unregistered: used for memory handling of scene graph only, the proto is not stored
in the graph main proto list but in an alternate list. Several protos with the same ID/Name can be stored unregistered
*/
LPPROTO SG_NewProto(LPSCENEGRAPH inScene, u32 ProtoID, char *name, Bool unregistered);

/*used for memory handling of scene graph only. move proto from off-graph to in-graph or reverse*/
M4Err Proto_SetInGraph(LPPROTO proto, LPSCENEGRAPH inScene, Bool set_in);

/*destroy proto interface - can be used even if instances of the proto are still present*/
M4Err SG_DeleteProto(LPPROTO proto);

/*returns graph associated with this proto. Such a graph cannot be used for rendering but is needed during
construction of proto dictionaries in case of nested protos*/
LPSCENEGRAPH Proto_GetSceneGraph(LPPROTO proto);

/*get/set private data*/
void Proto_SetPrivate(LPPROTO proto, void *ptr, void (*OnDelete)(void *ptr) );
void *Proto_GetPrivate(LPPROTO proto);

/*get a pointer to the MF URL field for externProto info - DO NOT TOUCH THIS FIELD*/
MFURL *Proto_GetExternURLFieldPointer(LPPROTO proto);
/*add node code - a proto is build of several nodes, the first node is used for rendering
and the others are kept private. This set of nodes is refered to as the proto "node code"*/
M4Err Proto_AddNodeCode(LPPROTO proto, SFNode *pNode);

/*gets number of field in the proto interface*/
u32 Proto_GetFieldCount(LPPROTO proto);
/*locates field declaration by name*/
LPPROTOFIELD Proto_FindFieldByName(LPPROTO proto, char *fieldName);
/*locates field declaration by index (0-based)*/
LPPROTOFIELD Proto_FindField(LPPROTO proto, u32 fieldIndex);

/*creates a new field declaration in the proto. of given fieldtype and eventType
fieldName can be NULL, if so the name will be fieldN, N being the index of the created field*/
LPPROTOFIELD Proto_NewField(LPPROTO proto, u32 fieldType, u32 eventType, char *fieldName);

/*assign the node field to a field of the proto (the node field IS the proto field)
the node shall be a node of the proto scenegraph, and the fieldtype/eventType of both fields shall match
(except SF/MFString and MF/SFURL which are allowed) due to BIFS semantics*/
M4Err Proto_SetISField(LPPROTO proto, u32 protoFieldIndex, SFNode *node, u32 nodeFieldIndex);
/*set/get user private data for the proto field declaration*/
void ProtoField_SetPrivate(LPPROTOFIELD field, void *ptr, void (*OnDelete)(void *ptr) );
void *ProtoField_GetPrivate(LPPROTOFIELD field);
/*returns field info of the field - this is typically used to setup the default value of the field*/
M4Err ProtoField_GetField(LPPROTOFIELD field, FieldInfo *info);

/*
	NOTE on proto instances:
		The proto instance is handled as an SFNode outside the scenegraph lib, and is manipulated with the same functions 
		as an SFNode 
		The proto instance may or may not be loaded. 
		An unloaded instance only contains the proto instance fields 
		A loaded instance contains the proto instance fields plus all the proto code (Nodes, routes) and 
		will load any scripts present in it. This allows keeping the memory usage of proto very low, especially
		when nested protos (protos used as building blocks of their parent proto) are used.
*/

/*creates the proto interface without the proto code.*/
SFNode *Proto_CreateInstance(LPSCENEGRAPH sg, LPPROTO proto);

/*lodes code in this instance - all subprotos are automatically created, thus you must only instanciate
top-level protos. VRML/BIFS doesn't allow for non top-level proto instanciation in the main graph
All nodes created in this proto will be forwarded to the app for initialization*/
M4Err Proto_LoadCode(SFNode *proto_inst);

/*locate a prototype definition by ID or by name*/
LPPROTO SG_FindProto(LPSCENEGRAPH sg, u32 ProtoID, char *name);

/*deletes all protos in given scene - does NOT delete instances of protos, only the proto object is destroyed */
M4Err SG_DeleteAllProtos(LPSCENEGRAPH scene);


/*tools for hardcoded proto*/
/*gets proto of this node - if the ndoe is not a prototype instan,ce, returns NULL*/
LPPROTO Node_GetProto(SFNode *node);
/*returns the ID of the proto*/
u32 Proto_GetID(LPPROTO proto);
/*returns proto name*/
const char *Proto_GetName(LPPROTO proto);

/*Returns 1 if the given field is ISed to a startTime/stopTime field (MPEG-4 specific for updates)*/
Bool Proto_FieldIsSFTimeOffset(SFNode *node, FieldInfo *field);

/*set an ISed field in a proto instance (not a proto) - this is needed with dynamic node creation inside a proto
instance (conditionals)*/
M4Err ProtoInstance_SetISField(SFNode *protoinst, u32 protoFieldIndex, SFNode *node, u32 nodeFieldIndex);

/*
			JavaScript tools
*/

/*script fields type don't have the same value as the bifs ones...*/
enum
{
	SFET_Field = 0,
	SFET_EventIn,
	SFET_EventOut,
};

typedef struct _scriptfield *LPSCRIPTFIELD;
/*creates new sript field - script fields are dynamically added to the node, and thus can be accessed through the
same functions as other SFNode fields*/
LPSCRIPTFIELD SG_NewScriptField(SFNode *script, u32 eventType, u32 fieldType, const char *name);
/*retrieves field info, usefull to get the field index*/
M4Err ScriptField_GetInfo(LPSCRIPTFIELD field, FieldInfo *info);


/*JavaScript interface with user*/
typedef struct
{
	/*user defined callback*/
	void *callback;
	/*returns app name*/
	const char *(*GetName)(void *opaque);
	/*returns app version*/
	const char *(*GetVersion)(void *opaque);
	/*returns world URL name*/
	const char *(*GetWorldURL)(void *opaque);
	/*signals error*/
	void (*Error)(void *opaque, const char *msg);
	/*signals message*/
	void (*Print)(void *opaque, const char *msg);
	/*ask the app to load a URL*/
	Bool (*LoadURL)(void *opaque, const char *url);
} JSInterface;

/*assign API to scene graph - by default, sub-graphs inherits the API if set*/
void SG_SetJavaScriptAPI(LPSCENEGRAPH scene, JSInterface *ifce);

/*load script into engine - this should be called only for script in main scene, loading of scripts
in protos is done internally when instanciating the proto*/
void Script_Load(SFNode *script);

/*activate eventIn for script node - needed for BIFS field replace*/
void Script_EventIn(SFNode *node, FieldInfo *in_field);

/*returns true if current lib has javascript support*/
Bool SG_HasScripting();

/*
			BIFS specific tools	- cf BIFS codec for their usage
*/

/* QUANTIZATION AND BIFS_Anim Info */
/*set QP and anim info for a proto field (BIFS allows for that in proto coding)*/
M4Err ProtoField_SetQuantizationInfo(LPPROTOFIELD field, u32 QP_Type, u32 hasMinMax, u32 QPSFType, void *qp_min_value, void *qp_max_value, u32 QP13_NumBits);
/*get field QP and anim info*/
Bool Node_GetAQInfo(SFNode *Node, u32 FieldIndex, u8 *QType, u8 *AType, Float *b_min, Float *b_max, u32 *QT13_bits);

/*get the absolute field 0_based index (or ALL mode) given the field index in IndexMode*/
M4Err Node_GetFieldIndex(SFNode *Node, u32 inField, u8 IndexMode, u32 *allField);
/*get the number of field in the given mode*/
u32 Node_GetNumFields(SFNode *Node, u8 IndexMode);

/*return number of bits needed to code all nodes present in the specified NDT*/
u32 NDT_GetNumBits(u32 NDT_Tag, u32 Version);
/*return absolute node tag given its type in the NDT and the NDT version number*/
u32 NDT_GetNodeTag(u32 NDT_Tag, u32 NodeType, u32 Version);

/*returns the opaque NodeDataType of the node "children" field if any, or 0*/
u32 Node_GetChildTable(SFNode *Node);

/*converts field index from all_mode to given mode*/
M4Err Node_ModeFieldIndex(SFNode *node, u32 all_ind, u8 indexMode, u32 *outField);

/*returns binary type of node in the given version of the desired NDT*/
u32 NDT_GetNodeType(u32 NDT_Tag, u32 NodeTag, u32 Version);

/*
			BIFS command tools
		These are used to store updates in memory without applying changes to the graph, 
	for dumpers, encoders ... Field Indexes are in "ALL" mode
		The commands can then be applied through this lib
*/

/*
		Currently defined possible modifications
*/
enum
{
	SG_SceneReplace = 0,
	SG_NodeReplace,
	SG_FieldReplace, 
	SG_IndexedReplace,
	SG_RouteReplace,
	SG_NodeDelete,
	SG_IndexedDelete,
	SG_RouteDelete,
	SG_NodeInsert,
	SG_IndexedInsert,
	SG_RouteInsert,
	/*extended updates*/
	SG_ProtoInsert,
	SG_ProtoDelete,
	SG_ProtoDeleteAll,
	SG_MultipleReplace,
	SG_MultipleIndexedReplace,
	SG_GlobalQuantizer,
	/*same as NodeDelete, and also updates OrderedGroup.order when deleting a child*/
	SG_NodeDeleteEx,

	SG_UNDEFINED
};


/*
				single command wrapper

  NOTE: In order to maintain node registry, the nodes replaced/inserted MUST be registered with 
  their parents even when the command is never applied. Registering shall be performed 
  with Node_Register (see below).
  If you fail to do so, a node may be destroyed when destroying a command while still used
  in another command or in the graph - this will just crash.
*/

/*structure used to store field info, pos and static pointers to SFNode/MFNode in commands*/
typedef struct
{
	u32 fieldIndex;
	/*field type*/
	u32 fieldType;
	/*field pointer for multiple replace/ multiple indexed replace - if multiple indexed replace, must be the SF field being changed*/
	void *field_ptr;
	/*replace/insert/delete pos - -1 is append except in multiple indexed replace*/
	s32 pos;

	/*Whenever field pointer is of type SFNode, store the node here and set the far pointer to this address.*/
	SFNode *new_node;
	/*Whenever field pointer is of type MFNode, store the node list here and set the far pointer to this address.*/
	Chain *node_list;
} CommandFieldInfo;

typedef struct
{
	u32 tag;
	/*scene replace command only - must be allocated by user, is destroyed when applied*/
	LPSCENEGRAPH graph;

	/*node the command applies to - may be NULL*/
	SFNode *node;

	/*list of CommandFieldInfo for all field commands replace/ index insert / index replace / index delete / MultipleReplace / MultipleIndexedreplace 
	the content is destroyed when deleting the command*/
	Chain *command_fields;
	
	/*route insert, replace and delete*/
	u32 RouteID;
	char *def_name;
	u32 fromNodeID, fromFieldIndex;
	u32 toNodeID, toFieldIndex;

	/*for scene replace*/
	Bool use_names;

	/*proto list to insert*/
	Chain *new_proto_list;
	/*proto ID list to delete*/
	u32 *del_proto_list;
	u32 del_proto_list_size;

	/*for authoring purposes - must be cleaned by user*/
	Bool unresolved;
	char *unres_name;
} SGCommand;


/*creates command - graph only needed for SceneReplace*/
SGCommand *SG_NewCommand(u32 tag);
/*deletes command*/
void SG_DeleteCommand(SGCommand *com);
/*apply command to graph - the command content is kept unchanged for authoring purposes - THIS NEEDS TESTING AND FIXING*/
M4Err SG_ApplyCommand(LPSCENEGRAPH inScene, SGCommand *com);
/*apply list if command to graph - the command content is kept unchanged for authoring purposes - THIS NEEDS TESTING AND FIXING*/
M4Err SG_ApplyCommandList(LPSCENEGRAPH graph, Chain *comList);

/*returns new commandFieldInfo structure and registers it with command*/
CommandFieldInfo *SG_NewFieldCommand(SGCommand *com);

/*same as Node_ResetChildren, excepts that it forces implicit reference count decrementation if the child
is not DEF. This is only used for commands handling*/
void Node_ForceResetChildren(SFNode *node, Chain *list);


/*authoring tools*/

/*retuns next available NodeID*/
u32 SG_GetNextNodeID(LPSCENEGRAPH sg);


#ifdef __cplusplus
}
#endif



#endif /*M4_SCENEGRAPH_H_*/


