#! /usr/bin/perl -w
######################################################################
##  This program is copyright (c) 1998, 1999 Bruce Ravel
##  <ravel@phys.washington.edu>
##  http://feff.phys.washington.edu/~ravel/software/atoms/
##
## -------------------------------------------------------------------
##     All rights reserved. This program is free software; you can
##     redistribute it and/or modify it under the same terms as Perl
##     itself.
##
##     This program is distributed in the hope that it will be useful,
##     but WITHOUT ANY WARRANTY; without even the implied warranty of
##     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
##     Artistic License for more details.
## -------------------------------------------------------------------
######################################################################
## Time-stamp: <2004/04/26 14:56:29 bruce>
######################################################################
##
##  This program write the space groups data to a database file using
##  either the MLDBM module with Storable serializing, DB_File format,
##  and portable byte ordering.  The database filename is
##  space_groups.db.
##
##  This is perhaps the world's most boring program.  It has all of
##  three lines.  It imports the MLDBM module, sets a variable, then
##  writes the contents of that variable to disk.  Ho hum.
##
##  See the documentation of the Xtal module for details about using
##  the contents of this database.
##
##  I intend that this program be run once at install time to create
##  the database.  After that it should only be necessary to rerun
##  this program if the database in this file is updated.
##
##  Each entry of %space_groups looks rather like this:
## "" => {
## 	 number      => ,             number in table
## 	 schoenflies => "_^",         Schoenflies notation
## 	 full        => "",           full Hermann-Maguin notation
##	 settings    => [ ba-c, cab, -cba, bca, a-cb], orthorhombic permutations
## 	 shiftvec    => [  ],         vector for groups with two origins
## 	 new_symbol  => "",           double glide plane symbol, cf. sec 1.3
##       thirtyfive  => "",           symbols from the 1935 edition
## 	 positions   => [             internal symmetries
## 		         [$x, $y, $z],
## 		        ],
##       rhombohedral=> [             internal symmetries, rhombohedral
## 		         [$x, $y, $z],
## 		        ],
## 		       },
######################################################################
## Code:

use File::Spec;
##Storable:
use Storable qw/nstore/;
##end Storable:
##MLDBM:
## use MLDBM qw(DB_File Storable);
## use Fcntl;
##end MLDBM:

my $cvs_info = '$Id: space_groups.PL,v 1.5 1999/07/28 22:32:50 bruce Exp $ ';
my $version = 1.6; #(split(' ', $cvs_info))[2] || "pre_release";


$| = 1;
print "Space group database generation tool $version Xray::Xtal 0.30$/";
print "  Munching on space group database.\n";
my %space_groups =
    (
     'version' => (split(' ', $cvs_info))[2],
     ## triclinic groups
     'p 1' => {
	 number => 1,
	 schoenflies => 'c_1^1',
	 positions => [['$x', '$y', '$z']]
     },
     'p -1' => {
	 number => 2,
	 schoenflies => 'c_i^1',
	 positions => [['$x', '$y', '$z'],
		       ['-$x', '-$y', '-$z']]
     },

     ## monoclinic groups
     'p 2' => {
	 number => 3,
	 schoenflies => 'c_2^1',
	 settings => ['p 1 2 1', 'p 1 1 2', 'p 2 1 1'],
	 b_unique => [['$x', '$y', '$z'],
		      ['-$x', '$y', '-$z']],
	 c_unique => [['$x', '$y', '$z'],
		      ['-$x', '-$y', '$z']],
	 a_unique => [['$x', '$y', '$z'],
		      ['$x', '-$y', '-$z']]
     },
     'p 21' => {
	 number => 4,
	 schoenflies => 'c_2^2',
	 settings => ['p 1 21 1', 'p 1 1 21', 'p 21 1 1'],
	 b_unique => [['$x', '$y', '$z'],
		      ['-$x', '$y+1/2', '-$z']],
	 c_unique => [['$x', '$y', '$z'],
		      ['-$x', '-$y', '$z+1/2']],
	 a_unique => [['$x', '$y', '$z'],
		      ['$x+1/2', '-$y', '-$z']]
	       },
     'c 2' => {
	 number => 5,
	 schoenflies => 'c_2^3',
	 short => ['c 2', 'a 2', 'i 2', 'a 2', 'b 2', 'i 2', 'b 2', 'c 2', 'i 2'],
	 settings => ['c 1 2 1', 'a 1 2 1', 'i 1 2 1',
		      'a 1 1 2', 'b 1 1 2', 'i 1 1 2',
		      'b 2 1 1', 'b 2 1 1', 'i 2 1 1'],
	 b_unique => [['$x', '$y', '$z'],
		      ['-$x', '$y', '-$z']],
	 c_unique => [['$x', '$y', '$z'],
		      ['-$x', '-$y', '$z']],
	 a_unique => [['$x', '$y', '$z'],
		      ['$x', '-$y', '-$z']]
     },
     'p m' => {
	 number => 6,
	 schoenflies => 'c_s^1',
	 settings => ['p 1 m 1', 'p 1 1 m', 'p m 1 1'],
	 b_unique => [['$x', '$y', '$z'],
		      ['$x', '-$y', '$z']],
	 c_unique => [['$x', '$y', '$z'],
		      ['$x', '$y', '-$z']],
	 a_unique => [['$x', '$y', '$z'],
		      ['-$x', '$y', '$z']]
     },
     'p c' => {
	 number => 7,
	 schoenflies => 'c_s^2', # note that the short notation is ambiguous
	 short => ['p c', 'p n', 'p a', 'p a', 'p n', 'p b', 'p b', 'p n', 'p c'],
	 settings => ['p 1 c 1', 'p 1 n 1', 'p 1 a 1',
		      'p 1 1 a', 'p 1 1 n', 'p 1 1 b',
		      'p b 1 1', 'p n 1 1', 'p c 1 1'],
	 b_unique_1 => [['$x', '$y', '$z'],
			['$x', '-$y', '$z+1/2']],
	 b_unique_2 => [['$x', '$y', '$z'],
			['$x+1/2', '-$y', '$z+1/2']],
	 b_unique_3 => [['$x', '$y', '$z'],
			['$x+1/2', '-$y', '$z']],
	 c_unique_1 => [['$x', '$y', '$z'],
			['$x+1/2', '$y', '-$z']],
	 c_unique_2 => [['$x', '$y', '$z'],
			['$x+1/2', '$y+1/2', '-$z']],
	 c_unique_3 => [['$x', '$y', '$z'],
			['$x', '$y+1/2', '-$z']],
	 a_unique_1 => [['$x', '$y', '$z'],
			['-$x', '$y+1/2', '$z']],
	 a_unique_2 => [['$x', '$y', '$z'],
			['-$x', '$y+1/2', '$z+1/2']],
	 a_unique_3 => [['$x', '$y', '$z'],
			['-$x', '$y', '$z+1/2']]
     },
     'c m' => {
	 number => 8,
	 schoenflies => 'c_s^3', # note that the short notation is ambiguous
	 short => ['c m', 'a m', 'i m', 'a m', 'b m', 'i m', 'b m', 'c m', 'i m'],
	 settings => ['c 1 m 1', 'a 1 m 1', 'i 1 m 1',
		      'a 1 1 m', 'b 1 1 m', 'i 1 1 m',
		      'b m 1 1', 'c m 1 1', 'i m 1 1'],
	 b_unique => [['$x', '$y', '$z'],
		      ['$x', '-$y', '$z']],
	 c_unique => [['$x', '$y', '$z'],
		      ['$x', '$y', '-$z']],
	 a_unique => [['$x', '$y', '$z'],
		      ['-$x', '$y', '$z']]
     },
     'c c' => {
	 number => 9,
	 schoenflies => 'c_s^4',
	 short => ['c c', 'a n', 'i a', 'a a', 'b n', 'i b', 'b b', 'c n', 'i c'],
	 settings => ['c 1 c 1', 'a 1 n 1', 'i 1 a 1',
		      'a 1 1 a', 'b 1 1 n', 'i 1 1 b',
		      'b b 1 1', 'c n 1 1', 'i c 1 1'],
	 b_unique_1 => [['$x', '$y', '$z'],
			['$x', '-$y', '$z+1/2']],
	 b_unique_2 => [['$x', '$y', '$z'],
			['$x+1/2', '-$y', '$z+1/2']],
	 b_unique_3 => [['$x', '$y', '$z'],
			['$x+1/2', '-$y', '$z']],
	 c_unique_1 => [['$x', '$y', '$z'],
			['$x+1/2', '$y', '-$z']],
	 c_unique_2 => [['$x', '$y', '$z'],
			['$x+1/2', '$y+1/2', '-$z']],
	 c_unique_3 => [['$x', '$y', '$z'],
			['$x', '$y+1/2', '-$z']],
	 a_unique_1 => [['$x', '$y', '$z'],
			['-$x', '$y+1/2', '$z']],
	 a_unique_2 => [['$x', '$y', '$z'],
			['-$x', '$y+1/2', '$z+1/2']],
	 a_unique_3 => [['$x', '$y', '$z'],
			['-$x', '$y', '$z+1/2']]
     },
     'p 2/m' => {
	 number => 10,
	 schoenflies => 'c_2h^1',
	 settings => ['p 1 2/m 1', 'p 1 1 2/m', 'p 2/m 1 1'],
	 b_unique => [['$x', '$y', '$z'],
		      ['-$x', '$y', '-$z'],
		      ['-$x', '-$y', '-$z'],
		      ['$x', '-$y', '$z']],
	 c_unique => [['$x', '$y', '$z'],
		      ['-$x', '-$y', '$z'],
		      ['-$x', '-$y', '-$z'],
		      ['$x', '$y', '-$z']],
	 a_unique => [['$x', '$y', '$z'],
		      ['$x', '-$y', '-$z'],
		      ['-$x', '-$y', '-$z'],
		      ['-$x', '$y', '$z']]
     },
     'p 21/m' => {
	 number => 11,
	 schoenflies => 'c_2h^2',
	 settings => ['p 1 21/m 1', 'p 1 1 21/m', 'p 21/m 1 1'],
	 b_unique => [['$x', '$y', '$z'],
		      ['-$x', '$y+1/2', '-$z'],
		      ['-$x', '-$y', '-$z'],
		      ['$x', '-$y+1/2', '$z']],
	 c_unique => [['$x', '$y', '$z'],
		      ['-$x', '-$y', '$z+1/2'],
		      ['-$x', '-$y', '-$z'],
		      ['$x', '$y', '-$z+1/2']],
	 a_unique => [['$x', '$y', '$z'],
		      ['$x+1/2', '-$y', '-$z'],
		      ['-$x', '-$y', '-$z'],
		      ['-$x+1/2', '$y', '$z']]
     },
     'c 2/m' => {
	 number => 12,
	 schoenflies => 'c_2h^3', # note that the short notation is ambiguous
	 short => ['c 2/m', 'a 2/m', 'i 2/m',
		   'a 2/m', 'b 2/m', 'i 2/m',
		   'b 2/m', 'c 2/m', 'i 2/m'],
	 settings => ['c 1 2/m 1', 'a 1 2/m 1', 'i 1 2/m 1',
		      'a 1 1 2/m', 'b 1 1 2/m', 'i 1 1 2/m',
		      'b 2/m 1 1', 'c 2/m 1 1', 'i 2/m 1 1'],
	 b_unique => [['$x', '$y', '$z'],
		      ['-$x', '$y', '-$z'],
		      ['-$x', '-$y', '-$z'],
		      ['$x', '-$y', '$z']],
	 c_unique => [['$x', '$y', '$z'],
		      ['-$x', '-$y', '$z'],
		      ['-$x', '-$y', '-$z'],
		      ['$x', '$y', '-$z']],
	 a_unique => [['$x', '$y', '$z'],
		      ['$x', '-$y', '-$z'],
		      ['-$x', '-$y', '-$z'],
		      ['-$x', '$y', '$z']]
     },
     'p 2/c' => {
	 number => 13,
	 schoenflies => 'c_2h^4', # note that the short notation is ambiguous
	 short => ['p 2/c', 'p 2/n', 'p 2/a',
		   'p 2/a', 'p 2/b', 'p 2/b',
		   'p 2/b', 'p 2/n', 'p 2/c'],
	 settings => ['p 1 2/c 1', 'p 1 2/n 1', 'p 1 2/a 1',
		      'p 1 1 2/a', 'p 1 1 2/n', 'p 1 1 2/b',
		      'p 2/a 1 1', 'p 2/n 1 1', 'p 2/c 1 1'],
	 b_unique_1 => [['$x', '$y', '$z'],
			['-$x', '$y', '-$z+1/2'],
			['-$x', '-$y', '-$z'],
			['$x', '-$y', '$z+1/2']],
	 b_unique_2 => [['$x', '$y', '$z'],
			['-$x+1/2', '$y', '-$z+1/2'],
			['-$x', '-$y', '-$z'],
			['$x+1/2', '-$y', '$z+1/2']],
	 b_unique_3 => [['$x', '$y', '$z'],
			['-$x+1/2', '$y', '-$z'],
			['-$x', '-$y', '-$z'],
			['$x+1/2', '-$y', '$z']],

	 c_unique_1 => [['$x', '$y', '$z'],
			['-$x+1/2', '-$y', '$z'],
			['-$x', '-$y', '-$z'],
			['$x+1/2', '$y', '-$z']],
	 c_unique_2 => [['$x', '$y', '$z'],
			['-$x+1/2', '-$y+1/2', '$z'],
			['-$x', '-$y', '-$z'],
			['$x+1/2', '$y+1/2', '-$z']],
	 c_unique_3 => [['$x', '$y', '$z'],
			['-$x', '-$y+1/2', '$z'],
			['-$x', '-$y', '-$z'],
			['$x', '$y+1/2', '-$z']],

	 a_unique_1 => [['$x', '$y', '$z'],
			['$x', '-$y+1/2', '-$z'],
			['-$x', '-$y', '-$z'],
			['-$x', '$y+1/2', '$z']],
	 a_unique_2 => [['$x', '$y', '$z'],
			['$x', '-$y+1/2', '-$z+1/2'],
			['-$x', '-$y', '-$z'],
			['-$x', '$y+1/2', '$z+1/2']],
	 a_unique_3 => [['$x', '$y', '$z'],
			['$x', '-$y', '-$z+1/2'],
			['-$x', '-$y', '-$z'],
			['-$x', '$y', '$z+1/2']]
		},
     'p 21/c' => {
	 number => 14,
	 schoenflies => 'c_2h^5', # note that the short notation is ambiguous
	 short => ['p 21/c', 'p 21/n', 'p 21/a',
		   'p 21/a', 'p 21/b', 'p 21/b',
		   'p 21/b', 'p 21/n', 'p 21/c'],
	 settings => ['p 1 21/c 1', 'p 1 21/n 1', 'p 1 21/a 1',
		      'p 1 1 21/a', 'p 1 1 21/n', 'p 1 1 21/b',
		      'p 21/a 1 1', 'p 21/n 1 1', 'p 21/c 1 1'],
	 b_unique_1 => [['$x', '$y', '$z'],
			['-$x', '$y+1/2', '-$z+1/2'],
			['-$x', '-$y', '-$z'],
			['$x', '-$y+1/2', '$z+1/2']],
	 b_unique_2 => [['$x', '$y', '$z'],
			['-$x+1/2', '$y+1/2', '-$z+1/2'],
			['-$x', '-$y', '-$z'],
			['$x+1/2', '-$y+1/2', '$z+1/2']],
	 b_unique_3 => [['$x', '$y', '$z'],
			['-$x+1/2', '$y+1/2', '-$z'],
			['-$x', '-$y', '-$z'],
			['$x+1/2', '-$y+1/2', '$z']],

	 c_unique_1 => [['$x', '$y', '$z'],
			['-$x+1/2', '-$y', '$z+1/2'],
			['-$x', '-$y', '-$z'],
			['$x+1/2', '$y', '-$z+1/2']],
	 c_unique_2 => [['$x', '$y', '$z'],
			['-$x+1/2', '-$y+1/2', '$z+1/2'],
			['-$x', '-$y', '-$z'],
			['$x+1/2', '$y+1/2', '-$z+1/2']],
	 c_unique_3 => [['$x', '$y', '$z'],
			['-$x', '-$y+1/2', '$z+1/2'],
			['-$x', '-$y', '-$z'],
			['$x', '$y+1/2', '-$z+1/2']],

	 a_unique_1 => [['$x', '$y', '$z'],
			['$x+1/2', '-$y+1/2', '-$z'],
			['-$x', '-$y', '-$z'],
			['-$x+1/2', '$y+1/2', '$z']],
	 a_unique_2 => [['$x', '$y', '$z'],
			['$x+1/2', '-$y+1/2', '-$z+1/2'],
			['-$x', '-$y', '-$z'],
			['-$x+1/2', '$y+1/2', '$z+1/2']],
	 a_unique_3 => [['$x', '$y', '$z'],
			['$x+1/2', '-$y', '-$z+1/2'],
			['-$x', '-$y', '-$z'],
			['-$x+1/2', '$y', '$z+1/2']]
      },
     'c 2/c' => {
	 number => 15,
	 schoenflies => 'c_2h^4', # note that the short notation is ambiguous
	 short => ['c 2/c', 'a 2/n', 'i 2/a',
		   'a 2/a', 'b 2/b', 'i 2/b',
		   'b 2/b', 'c 2/n', 'i 2/c'],
	 settings => ['c 1 2/c 1', 'a 1 2/n 1', 'i 1 2/a 1',
		      'a 1 1 2/a', 'b 1 1 2/n', 'i 1 1 2/b',
		      'b 2/a 1 1', 'c 2/n 1 1', 'i 2/c 1 1'],
	 b_unique_1 => [['$x', '$y', '$z'],
			['-$x', '$y', '-$z+1/2'],
			['-$x', '-$y', '-$z'],
			['$x', '-$y', '$z+1/2']],
	 b_unique_2 => [['$x', '$y', '$z'],
			['-$x+1/2', '$y', '-$z+1/2'],
			['-$x', '-$y', '-$z'],
			['$x+1/2', '-$y', '$z+1/2']],
	 b_unique_3 => [['$x', '$y', '$z'],
			['-$x+1/2', '$y', '-$z'],
			['-$x', '-$y', '-$z'],
			['$x+1/2', '-$y', '$z']],

	 c_unique_1 => [['$x', '$y', '$z'],
			['-$x+1/2', '-$y', '$z'],
			['-$x', '-$y', '-$z'],
			['$x+1/2', '$y', '-$z']],
	 c_unique_2 => [['$x', '$y', '$z'],
			['-$x+1/2', '-$y+1/2', '$z'],
			['-$x', '-$y', '-$z'],
			['$x+1/2', '$y+1/2', '-$z']],
	 c_unique_3 => [['$x', '$y', '$z'],
			['-$x', '-$y+1/2', '$z'],
			['-$x', '-$y', '-$z'],
			['$x', '$y+1/2', '-$z']],

	 a_unique_1 => [['$x', '$y', '$z'],
			['$x', '-$y+1/2', '-$z'],
			['-$x', '-$y', '-$z'],
			['-$x', '$y+1/2', '$z']],
	 a_unique_2 => [['$x', '$y', '$z'],
			['$x', '-$y+1/2', '-$z+1/2'],
			['-$x', '-$y', '-$z'],
			['-$x', '$y+1/2', '$z+1/2']],
	 a_unique_3 => [['$x', '$y', '$z'],
			['$x', '-$y', '-$z+1/2'],
			['-$x', '-$y', '-$z'],
			['-$x', '$y', '$z+1/2']],
      },

     ## orthorhombic groups
     'p 2 2 2' => {
	 number => 16,
	 schoenflies => 'd_2^1',
	 positions => [['$x', '$y', '$z'],
		       ['-$x', '-$y', '$z'],
		       ['-$x', '$y', '-$z'],
		       ['$x', '-$y', '-$z']],
     },
     'p 2 2 21' => {
	 number => 17,
	 schoenflies => 'd_2^2',
	 settings => [ 'p 2 2 21', 'p 21 2 2', 'p 21 2 2', 'p 2 21 2', 'p 2 21 2'],
	 positions => [['$x', '$y', '$z'],
		       ['-$x', '-$y', '$z+1/2'],
		       ['-$x', '$y', '-$z+1/2'],
		       ['$x', '-$y', '-$z']],
     },
     'p 21 21 2' => {
	 number => 18,
	 schoenflies => 'd_2^3',
	 settings => [ 'p 21 21 2', 'p 2 21 21', 'p 2 21 21', 'p 21 2 21', 'p 21 2 21'],
	 positions => [['$x', '$y', '$z'],
		       ['-$x', '-$y', '$z'],
		       ['-$x+1/2', '$y+1/2', '-$z'],
		       ['$x+1/2', '-$y+1/2', '-$z']],
     },
     'p 21 21 21' => {
	 number => 19,
	 schoenflies => 'd_2^4',
	 positions => [['$x', '$y', '$z'],
		       ['-$x+1/2', '-$y', '$z+1/2'],
		       ['-$x', '$y+1/2', '-$z+1/2'],
		       ['$x+1/2', '-$y+1/2', '-$z']],
     },
     'c 2 2 21' => {
	 number => 20,
	 schoenflies => 'd_2^5',
	 settings => [ 'c 2 2 21', 'a 21 2 2', 'a 21 2 2', 'b 2 21 2', 'b 2 21 2'],
	 positions => [['$x', '$y', '$z'],
		       ['-$x', '-$y', '$z+1/2'],
		       ['-$x', '$y', '-$z+1/2'],
		       ['$x', '-$y', '-$z']],
     },
     'c 2 2 2' => {
	 number => 21,
	 schoenflies => 'd_2^6',
	 settings => ['c 2 2 2', 'a 2 2 2', 'a 2 2 2', 'b 2 2 2', 'b 2 2 2'],
	 positions => [['$x', '$y', '$z'],
		       ['-$x', '-$y', '$z'],
		       ['-$x', '$y', '-$z'],
		       ['$x', '-$y', '-$z']],
     },
     'f 2 2 2' => {
	 number => 22,
	 schoenflies => 'd_2^7',
	 positions => [['$x', '$y', '$z'],
		       ['-$x', '-$y', '$z'],
		       ['-$x', '$y', '-$z'],
		       ['$x', '-$y', '-$z']],
     },
     'i 2 2 2' => {
	 number => 23,
	 schoenflies => 'd_2^8',
	 positions => [['$x', '$y', '$z'],
		       ['-$x', '-$y', '$z'],
		       ['-$x', '$y', '-$z'],
		       ['$x', '-$y', '-$z']],
     },
     'i 21 21 21' => {
	 number => 24,
	 schoenflies => 'd_2^9',
	 positions => [['$x', '$y', '$z'],
		       ['-$x+1/2', '-$y', '$z+1/2'],
		       ['-$x', '$y+1/2', '-$z+1/2'],
		       ['$x+1/2', '-$y+1/2', '-$z']],
     },
     'p m m 2' => {
	 number => 25,
	 schoenflies => 'c_2v^1',
	 settings => ['p m m 2', 'p 2 m m', 'p 2 m m', 'p m 2 m', 'p m 2 m'],
	 positions => [['$x', '$y', '$z'],
		       ['-$x', '-$y', '$z'],
		       ['$x', '-$y', '$z'],
		       ['-$x', '$y', '$z']],
     },
     'p m c 21' => {
	 number => 26,
	 schoenflies => 'c_2v^2',
	 settings => ['p c m 21', 'p 21 m a', 'p 21 a m', 'p b 21 m', 'p m 21 b'],
	 positions => [['$x', '$y', '$z'],
		       ['-$x', '-$y', '$z+1/2'],
		       ['$x', '-$y', '$z+1/2'],
		       ['-$x', '$y', '$z']],
     },
     'p c c 2' => {
	 number => 27,
	 schoenflies => 'c_2v^3',
	 settings => ['p c c 2', 'p 2 a a', 'p 2 a a', 'p b 2 b', 'p b 2 b' ],
	 positions => [['$x', '$y', '$z'],
		       ['-$x', '-$y', '$z'],
		       ['$x', '-$y', '$z+1/2'],
		       ['-$x', '$y', '$z+1/2']],
     },
     'p m a 2' => {
	 number => 28,
	 schoenflies => 'c_2v^4',
	 settings => ['p m a 2', 'p b m 2', 'p 2 m b', 'p 2 c m', 'p m 2 a' ],
	 positions => [['$x', '$y', '$z'],
		       ['-$x', '-$y', '$z'],
		       ['$x+1/2', '-$y', '$z'],
		       ['-$x+1/2', '$y', '$z']],
     },
     'p c a 21' => {
	 number => 29,
	 schoenflies => 'c_2v^5',
	 settings => ['p b c 21', 'p 21 a b', 'p 21 c a', 'p c 21 b', 'p b 21 a' ],
	 positions => [['$x', '$y', '$z'],
		       ['-$x', '-$y', '$z+1/2'],
		       ['$x+1/2', '-$y', '$z'],
		       ['-$x+1/2', '$y', '$z+1/2']],
     },
     'p n c 2' => {
	 number => 30,
	 schoenflies => 'c_2v^6',
	 settings => ['p c n 2', 'p 2 n a', 'p 2 a n', 'p b 2 n', 'p n 2 b' ],
	 positions => [['$x', '$y', '$z'],
		       ['-$x', '-$y', '$z'],
		       ['$x', '-$y+1/2', '$z+1/2'],
		       ['-$x', '$y+1/2', '$z+1/2']],
     },
     'p m n 21' => {
	 number => 31,
	 schoenflies => 'c_2v^7',
	 settings => ['p n m 21', 'p 21 m n', 'p 21 n m', 'p n 21 m', 'p m 21 n' ],
	 positions => [['$x', '$y', '$z'],
		       ['-$x+1/2', '-$y', '$z+1/2'],
		       ['$x+1/2', '-$y', '$z+1/2'],
		       ['-$x', '$y', '$z']],
     },
     'p b a 2' => {
	 number => 32,
	 schoenflies => 'c_2v^8',
	 settings => ['p b a 2', 'p 2 c b', 'p 2 c b', 'p c 2 a', 'p c 2 a' ],
	 positions => [['$x', '$y', '$z'],
		       ['-$x', '-$y', '$z'],
		       ['$x+1/2', '-$y+1/2', '$z'],
		       ['-$x+1/2', '$y+1/2', '$z']],
     },
     'p n a 21' => {
	 number => 33,
	 schoenflies => 'c_2v^9',
	 settings => ['p b n 21', 'p 21 n b', 'p 21 c n', 'p c 21 n', 'p n 21 a' ],
	 positions => [['$x', '$y', '$z'],
		       ['-$x', '-$y', '$z+1/2'],
		       ['$x+1/2', '-$y+1/2', '$z'],
		       ['-$x+1/2', '$y+1/2', '$z+1/2']],
     },
     'p n n 2' => {
	 number => 34,
	 schoenflies => 'c_2v^10',
	 settings => ['p n n 2', 'p 2 n n', 'p 2 n n', 'p n 2 n', 'p n 2 n' ],
	 positions => [['$x', '$y', '$z'],
		       ['-$x', '-$y', '$z'],
		       ['$x+1/2', '-$y+1/2', '$z+1/2'],
		       ['-$x+1/2', '$y+1/2', '$z+1/2']],
     },
     'c m m 2' => {
	 number => 35,
	 schoenflies => 'c_2v^11',
	 settings => ['c m m 2', 'a 2 m m', 'a 2 m m', 'b m 2 m', 'b m 2 m' ],
	 positions => [['$x', '$y', '$z'],
		       ['-$x', '-$y', '$z'],
		       ['$x', '-$y', '$z'],
		       ['-$x', '$y', '$z']],
     },
     'c m c 21' => {
	 number => 36,
	 schoenflies => 'c_2v^12',
	 settings => ['c c m 21', 'a 21 m a', 'a 21 a m', 'b b 21 m', 'b m 21 b' ],
	 positions => [['$x', '$y', '$z'],
		       ['-$x', '-$y', '$z+1/2'],
		       ['$x', '-$y', '$z+1/2'],
		       ['-$x', '$y', '$z']],
     },
     'c c c 2' => {
	 number => 37,
	 schoenflies => 'c_2v^13',
	 settings => ['c c c 2', 'a 2 a a', 'a 2 a a', 'b b 2 b', 'b b 2 b' ],
	 positions => [['$x', '$y', '$z'],
		       ['-$x', '-$y', '$z'],
		       ['$x', '-$y', '$z+1/2'],
		       ['-$x', '$y', '$z+1/2']],
     },
     'a m m 2' => {
	 number => 38,
	 schoenflies => 'c_2v^14',
	 settings => ['b m m 2', 'b 2 m m', 'c 2 m m', 'c m 2 m', 'a m 2 m' ],
	 positions => [['$x', '$y', '$z'],
		       ['-$x', '-$y', '$z'],
		       ['$x', '-$y', '$z'],
		       ['-$x', '$y', '$z']],
     },
     'a b m 2' => {
	 number => 39,
	 schoenflies => 'c_2v^15',
	 settings => ['b m a 2', 'b 2 c m', 'c 2 m b', 'c m 2 a', 'a c 2 m' ],
	 new_symbol => 'a e m 2',
	 positions => [['$x', '$y', '$z'],
		       ['-$x', '-$y', '$z'],
		       ['$x', '-$y+1/2', '$z'],
		       ['-$x', '$y+1/2', '$z']],
     },
     'a m a 2' => {
	 number => 40,
	 schoenflies => 'c_2v^16',
	 settings => ['b b m 2', 'b 2 m b', 'c 2 c m', 'c c m 2', 'a m 2 a' ],
	 positions => [['$x', '$y', '$z'],
		       ['-$x', '-$y', '$z'],
		       ['$x+1/2', '-$y', '$z'],
		       ['-$x+1/2', '$y', '$z']],
     },
     'a b a 2' => {
	 number => 41,
	 schoenflies => 'c_2v^17',
	 settings => ['b b a 2', 'b 2 c b', 'c 2 c b', 'c c 2 a', 'a c 2 a' ],
	 new_symbol => 'a e a 2',
	 positions => [['$x', '$y', '$z'],
		       ['-$x', '-$y', '$z'],
		       ['$x+1/2', '-$y+1/2', '$z'],
		       ['-$x+1/2', '$y+1/2', '$z']],
     },
     'f m m 2' => {
	 number => 42,
	 schoenflies => 'c_2v^18',
	 settings => ['f m m 2', 'f 2 m m', 'f 2 m m', 'f m 2 m', 'f m 2 m' ],
	 positions => [['$x', '$y', '$z'],
		       ['-$x', '-$y', '$z'],
		       ['$x', '-$y', '$z'],
		       ['-$x', '$y', '$z']],
     },
     'f d d 2' => {
	 number => 43,
	 schoenflies => 'c_2v^19',
	 settings => ['f d d 2', 'f 2 d d', 'f 2 d d', 'f d 2 d', 'f d 2 d' ],
	 positions => [['$x', '$y', '$z'],
		       ['-$x', '-$y', '$z'],
		       ['$x+1/4', '-$y+1/4', '$z+1/4'],
		       ['-$x+1/4', '$y+1/4', '$z+1/4']],
     },
     'i m m 2' => {
	 number => 44,
	 schoenflies => 'c_2v^20',
	 settings => ['i m m 2', 'i 2 m m', 'i 2 m m', 'i m 2 m', 'i m 2 m' ],
	 positions => [['$x', '$y', '$z'],
		       ['-$x', '-$y', '$z'],
		       ['$x', '-$y', '$z'],
		       ['-$x', '$y', '$z']],
     },
     'i b a 2' => {
	 number => 45,
	 schoenflies => 'c_2v^21',
	 settings => ['i b a 2', 'i 2 c b', 'i 2 c b', 'i c 2 a', 'i c 2 a' ],
	 positions => [['$x', '$y', '$z'],
		       ['-$x', '-$y', '$z'],
		       ['$x+1/2', '-$y+1/2', '$z'],
		       ['-$x+1/2', '$y+1/2', '$z']],
     },
     'i m a 2' => {
	 number => 46,
	 schoenflies => 'c_2v^22',
	 settings => ['i b m 2', 'i 2 m b', 'i 2 c m', 'i c 2 m', 'i m 2 a' ],
	 positions => [['$x', '$y', '$z'],
		       ['-$x', '-$y', '$z'],
		       ['$x+1/2', '-$y', '$z'],
		       ['-$x+1/2', '$y', '$z']],
     },
     'p m m m' => {
	 number => 47,
	 schoenflies => 'd_2h^1',
	 full => 'p 2/m 2/m 2/m',
	 positions => [['$x', '$y', '$z'],
		       ['-$x', '-$y', '$z'],
		       ['-$x', '$y', '-$z'],
		       ['$x', '-$y', '-$z'],
		       ['-$x', '-$y', '-$z'],
		       ['$x', '$y', '-$z'],
		       ['$x', '-$y', '$z'],
		       ['-$x', '$y', '$z']],
     },
     'p n n n' => {
	 number => 48,
	 schoenflies => 'd_2h^2',
	 full => 'p 2/n 2/n 2/n',
	 shiftvec => [1/4, 1/4, 1/4],
	 positions => [['$x', '$y', '$z'],
		       ['-$x+1/2', '-$y+1/2', '$z'],
		       ['-$x+1/2', '$y', '-$z+1/2'],
		       ['$x', '-$y+1/2', '-$z+1/2'],
		       ['-$x', '-$y', '-$z'],
		       ['$x+1/2', '$y+1/2', '-$z'],
		       ['$x+1/2', '-$y', '$z+1/2'],
		       ['-$x', '$y+1/2', '$z+1/2']],
     },
     'p c c m' => {
	 number => 49,
	 schoenflies => 'd_2h^3',
	 full => 'p 2/c 2/c 2/m',
	 settings => ['p c c m', 'p m a a', 'p m a a', 'p b m b', 'p b m b'],
	 positions => [['$x', '$y', '$z'],
		       ['-$x', '-$y', '$z'],
		       ['-$x', '$y', '-$z+1/2'],
		       ['$x', '-$y', '-$z+1/2'],
		       ['-$x', '-$y', '-$z'],
		       ['$x', '$y', '-$z'],
		       ['$x', '-$y', '$z+1/2'],
		       ['-$x', '$y', '$z+1/2']],
     },
     'p b a n' => {
	 number => 50,
	 schoenflies => 'd_2h^4',
	 full => 'p 2/b 2/a 2/n',
	 settings => ['p b a n', 'p n c b', 'p n c b', 'p c n a', 'p c n a'],
	 shiftvec => [1/4, 1/4, 0],
	 positions => [['$x', '$y', '$z'],
		       ['-$x+1/2', '-$y+1/2', '$z'],
		       ['-$x+1/2', '$y', '-$z'],
		       ['$x', '-$y+1/2', '-$z'],
		       ['-$x', '-$y', '-$z'],
		       ['$x+1/2', '$y+1/2', '-$z'],
		       ['$x+1/2', '-$y', '$z'],
		       ['-$x', '$y+1/2', '$z']],
     },
     'p m m a' => {
	 number => 51,
	 schoenflies => 'd_2h^5',
	 full => 'p 21/m 2/m 2/a',
	 settings => ['p m m b', 'p b m m', 'p c m m', 'p m c m', 'p m a m'],
	 positions => [['$x', '$y', '$z'],
		       ['-$x+1/2', '-$y', '$z'],
		       ['-$x', '$y', '-$z'],
		       ['$x+1/2', '-$y', '-$z'],
		       ['-$x', '-$y', '-$z'],
		       ['$x+1/2', '$y', '-$z'],
		       ['$x', '-$y', '$z'],
		       ['-$x+1/2', '$y', '$z']],
     },
     'p n n a' => {
	 number => 52,
	 schoenflies => 'd_2h^6',
	 full => 'p 2/n 21/n 2/a',
	 settings => ['p n n b', 'p b n n', 'p c n n', 'p n c n', 'p n a n'],
	 positions => [['$x', '$y', '$z'],
		       ['-$x+1/2', '-$y', '$z'],
		       ['-$x+1/2', '$y+1/2', '-$z+1/2'],
		       ['$x', '-$y+1/2', '-$z+1/2'],
		       ['-$x', '-$y', '-$z'],
		       ['$x+1/2', '$y', '-$z'],
		       ['$x+1/2', '-$y+1/2', '$z+1/2'],
		       ['-$x', '$y+1/2', '$z+1/2']],
     },
     'p m n a' => {
	 number => 53,
	 schoenflies => 'd_2h^7',
	 full => 'p 2/m 2/n 21/a',
	 settings => ['p n m b', 'p b m n', 'p c n m', 'p n c m', 'p m a n'],
	 positions => [['$x', '$y', '$z'],
		       ['-$x+1/2', '-$y', '$z+1/2'],
		       ['-$x+1/2', '$y', '-$z+1/2'],
		       ['$x', '-$y', '-$z'],
		       ['-$x', '-$y', '-$z'],
		       ['$x+1/2', '$y', '-$z+1/2'],
		       ['$x+1/2', '-$y', '$z+1/2'],
		       ['-$x', '$y', '$z']],
     },
     'p c c a' => {
	 number => 54,
	 schoenflies => 'd_2h^8',
	 full => 'p 21/c 2/c 2/a',
	 settings => ['p c c b', 'p b a a', 'p c a a', 'p b c b', 'p b a b'],
	 positions => [['$x', '$y', '$z'],
		       ['-$x+1/2', '-$y', '$z'],
		       ['-$x', '$y', '-$z+1/2'],
		       ['$x', '-$y', '-$z+1/2'],
		       ['-$x', '-$y', '-$z'],
		       ['$x+1/2', '$y', '-$z'],
		       ['$x', '-$y', '$z+1/2'],
		       ['-$x', '$y', '$z+1/2']],
     },
     'p b a m' => {
	 number => 55,
	 schoenflies => 'd_2h^9',
	 full => 'p 21/b 21/a 2/m',
	 settings => ['p b a m', 'p m c b', 'p m c b', 'p c m a', 'p c m a'],
	 positions => [['$x', '$y', '$z'],
		       ['-$x', '-$y', '$z'],
		       ['-$x+1/2', '$y+1/2', '-$z'],
		       ['$x+1/2', '-$y+1/2', '-$z'],
		       ['-$x', '-$y', '-$z'],
		       ['$x', '$y', '-$z'],
		       ['$x+1/2', '-$y+1/2', '$z'],
		       ['-$x+1/2', '$y+1/2', '$z']],
     },
     'p c c n' => {
	 number => 56,
	 schoenflies => 'd_2h^10',
	 full => 'p 21/c 21/c 2/n',
	 settings => ['p c c n', 'p n a a', 'p n a a', 'p b n b', 'p b n a'],
	 positions => [['$x', '$y', '$z'],
		       ['-$x+1/2', '-$y+1/2', '$z'],
		       ['-$x', '$y+1/2', '-$z+1/2'],
		       ['$x+1/2', '-$y', '-$z+1/2'],
		       ['-$x', '-$y', '-$z'],
		       ['$x+1/2', '$y+1/2', '-$z'],
		       ['$x', '-$y+1/2', '$z+1/2'],
		       ['-$x+1/2', '$y', '$z+1/2']],
     },
     'p b c m' => {
	 number => 57,
	 schoenflies => 'd_2h^11',
	 full => 'p 2/b 21/c 21/m',
	 settings => ['p c a m', 'p m c a', 'p m a b', 'p b m a', 'p c m b'],
	 positions => [['$x', '$y', '$z'],
		       ['-$x', '-$y', '$z+1/2'],
		       ['-$x', '$y+1/2', '-$z+1/2'],
		       ['$x', '-$y+1/2', '-$z'],
		       ['-$x', '-$y', '-$z'],
		       ['$x', '$y', '-$z+1/2'],
		       ['$x', '-$y+1/2', '$z+1/2'],
		       ['-$x', '$y+1/2', '$z']],
     },
     'p n n m' => {
	 number => 58,
	 schoenflies => 'd_2h^12',
	 full => 'p 21/n 21/n 2/m',
	 settings => ['p n n m', 'p m n n', 'p m n n', 'p n m n', 'p n m n'],
	 positions => [['$x', '$y', '$z'],
		       ['-$x', '-$y', '$z'],
		       ['-$x+1/2', '$y+1/2', '-$z+1/2'],
		       ['$x+1/2', '-$y+1/2', '-$z+1/2'],
		       ['-$x', '-$y', '-$z'],
		       ['$x', '$y', '-$z'],
		       ['$x+1/2', '-$y+1/2', '$z+1/2'],
		       ['-$x+1/2', '$y+1/2', '$z+1/2']],
     },
     'p m m n' => {
	 number => 59,
	 schoenflies => 'd_2h^13',
	 full => 'p 21/m 21/m 2/n',
	 shiftvec => [1/4, 1/4, 0],
	 settings => ['p m m n', 'p n m m', 'p n m m', 'p m n m', 'p m n m'],
	 positions => [['$x', '$y', '$z'],
		       ['-$x+1/2', '-$y+1/2', '$z'],
		       ['-$x', '$y+1/2', '-$z'],
		       ['$x+1/2', '-$y', '-$z'],
		       ['-$x', '-$y', '-$z'],
		       ['$x+1/2', '$y+1/2', '-$z'],
		       ['$x', '-$y+1/2', '$z'],
		       ['-$x+1/2', '$y', '$z']],
     },
     'p b c n' => {
	 number => 60,
	 schoenflies => 'd_2h^14',
	 full => 'p 21/b 2/c 21/n',
	 settings => ['p c a n', 'p n c a', 'p n a b', 'p b n a', 'p c n b'],
	 positions => [['$x', '$y', '$z'],
		       ['-$x+1/2', '-$y+1/2', '$z+1/2'],
		       ['-$x', '$y', '-$z+1/2'],
		       ['$x+1/2', '-$y+1/2', '-$z'],
		       ['-$x', '-$y', '-$z'],
		       ['$x+1/2', '$y+1/2', '-$z+1/2'],
		       ['$x', '-$y', '$z+1/2'],
		       ['-$x+1/2', '$y+1/2', '$z']],
     },
     'p b c a' => {
	 number => 61,
	 schoenflies => 'd_2h^15',
	 full => 'p 21/b 21/c 21/a',
	 settings => ['p c a b', 'p b c a', 'p c a b', 'p b c a', 'p c a b'],
	 positions => [['$x', '$y', '$z'],
		       ['-$x+1/2', '-$y', '$z+1/2'],
		       ['-$x', '$y+1/2', '-$z+1/2'],
		       ['$x+1/2', '-$y+1/2', '-$z'],
		       ['-$x', '-$y', '-$z'],
		       ['$x+1/2', '$y', '-$z+1/2'],
		       ['$x', '-$y+1/2', '$z+1/2'],
		       ['-$x+1/2', '$y+1/2', '$z']],
     },
     'p n m a' => {
	 number => 62,
	 schoenflies => 'd_2h^16',
	 full => 'p 21/n 21/m 21/a',
	 settings => ['p m n b', 'p b n m', 'p c m n', 'p m c n', 'p n a m'],
	 positions => [['$x', '$y', '$z'],
		       ['-$x+1/2', '-$y', '$z+1/2'],
		       ['-$x', '$y+1/2', '-$z'],
		       ['$x+1/2', '-$y+1/2', '-$z+1/2'],
		       ['-$x', '-$y', '-$z'],
		       ['$x+1/2', '$y', '-$z+1/2'],
		       ['$x', '-$y+1/2', '$z'],
		       ['-$x+1/2', '$y+1/2', '$z+1/2']],
     },
     'c m c m' => {
	 number => 63,
	 schoenflies => 'd_2h^17',
	 full => 'c 2/m 2/c 21/m',
	 settings => ['c c m m', 'a m m a', 'a m a m', 'b b m m', 'b m m b'],
	 positions => [['$x', '$y', '$z'],
		       ['-$x', '-$y', '$z+1/2'],
		       ['-$x', '$y', '-$z+1/2'],
		       ['$x', '-$y', '-$z'],
		       ['-$x', '-$y', '-$z'],
		       ['$x', '$y', '-$z+1/2'],
		       ['$x', '-$y', '$z+1/2'],
		       ['-$x', '$y', '$z']],
     },
     'c m c a' => {
	 number => 64,
	 schoenflies => 'd_2h^18',
	 full => 'c 2/m 2/c 21/a',
	 new_symbol => 'c m c e',
	 settings => ['c c m b', 'a b m a', 'a c a m', 'b b c m', 'b m a b'],
	 positions => [['$x', '$y', '$z'],
		       ['-$x', '-$y+1/2', '$z+1/2'],
		       ['-$x', '$y+1/2', '-$z+1/2'],
		       ['$x', '-$y', '-$z'],
		       ['-$x', '-$y', '-$z'],
		       ['$x', '$y+1/2', '-$z+1/2'],
		       ['$x', '-$y+1/2', '$z+1/2'],
		       ['-$x', '$y', '$z']],
     },
     'c m m m' => {
	 number => 65,
	 schoenflies => 'd_2h^19',
	 full => 'c 2/m 2/m 2/m',
	 settings => ['c m m m', 'a m m m', 'a m m m', 'b m m m', 'b m m m'],
	 positions => [['$x', '$y', '$z'],
		       ['-$x', '-$y', '$z'],
		       ['-$x', '$y', '-$z'],
		       ['$x', '-$y', '-$z'],
		       ['-$x', '-$y', '-$z'],
		       ['$x', '$y', '-$z'],
		       ['$x', '-$y', '$z'],
		       ['-$x', '$y', '$z']],
     },
     'c c c m' => {
	 number => 66,
	 schoenflies => 'd_2h^20',
	 full => 'c 2/c 2/c 2/m',
	 settings => ['c c c m', 'a m a a', 'a m a a', 'b b m b', 'b b m b'],
	 positions => [['$x', '$y', '$z'],
		       ['-$x', '-$y', '$z'],
		       ['-$x', '$y', '-$z+1/2'],
		       ['$x', '-$y', '-$z+1/2'],
		       ['-$x', '-$y', '-$z'],
		       ['$x', '$y', '-$z'],
		       ['$x', '-$y', '$z+1/2'],
		       ['-$x', '$y', '$z+1/2']],
     },
     'c m m a' => {
	 number => 67,
	 schoenflies => 'd_2h^21',
	 full => 'c 2/m 2/m 2/a',
	 new_symbol => 'c m m e',
	 settings => ['c m m b', 'a b m m', 'a c m m', 'b m c m', 'b m a m'],
	 positions => [['$x', '$y', '$z'],
		       ['-$x', '-$y+1/2', '$z'],
		       ['-$x', '$y+1/2', '-$z'],
		       ['$x', '-$y', '-$z'],
		       ['-$x', '-$y', '-$z'],
		       ['$x', '$y+1/2', '-$z'],
		       ['$x', '-$y+1/2', '$z'],
		       ['-$x', '$y', '$z']],
     },
     'c c c a' => {
	 number => 68,
	 schoenflies => 'd_2h^22',
	 full => 'c 2/c 2/c 2/a',
	 new_symbol => 'c c c e',
	 shiftvec => [0, 1/4, 1/4],
	 settings => ['c c c b', 'a b a a', 'a c a a', 'b b c b', 'b b a b'],
	 positions => [['$x', '$y', '$z'],
		       ['-$x+1/2', '-$y', '$z'],
		       ['-$x', '$y', '-$z+1/2'],
		       ['$x+1/2', '-$y', '-$z+1/2'],
		       ['-$x', '-$y', '-$z'],
		       ['$x+1/2', '$y', '-$z'],
		       ['$x', '-$y', '$z+1/2'],
		       ['-$x+1/2', '$y', '$z+1/2']],
     },
     'f m m m' => {
	 number => 69,
	 schoenflies => 'd_2h^23',
	 full => 'f 2/m 2/m 2/m',
	 positions => [['$x', '$y', '$z'],
		       ['-$x', '-$y', '$z'],
		       ['-$x', '$y', '-$z'],
		       ['$x', '-$y', '-$z'],
		       ['-$x', '-$y', '-$z'],
		       ['$x', '$y', '-$z'],
		       ['$x', '-$y', '$z'],
		       ['-$x', '$y', '$z']],
     },
     'f d d d' => {
	 number => 70,
	 schoenflies => 'd_2h^24',
	 full => 'f 2/d 2/d 2/d',
	 shiftvec => [-1/8, -1/8, -1/8],
	 positions => [['$x', '$y', '$z'],
		       ['-$x+3/4', '-$y+3/4', '$z'],
		       ['-$x+3/4', '$y', '-$z+3/4'],
		       ['$x', '-$y+3/4', '-$z+3/4'],
		       ['-$x', '-$y', '-$z'],
		       ['$x+1/4', '$y+1/4', '-$z'],
		       ['$x+1/4', '-$y', '$z+1/4'],
		       ['-$x', '$y+1/4', '$z+1/4']],
     },
     'i m m m' => {
	 number => 71,
	 schoenflies => 'd_2h^25',
	 full => 'i 2/m 2/m 2/m',
	 positions => [['$x', '$y', '$z'],
		       ['-$x', '-$y', '$z'],
		       ['-$x', '$y', '-$z'],
		       ['$x', '-$y', '-$z'],
		       ['-$x', '-$y', '-$z'],
		       ['$x', '$y', '-$z'],
		       ['$x', '-$y', '$z'],
		       ['-$x', '$y', '$z']],
     },
     'i b a m' => {
	 number => 72,
	 schoenflies => 'd_2h^26',
	 full => 'i 2/b 2/a 2/m',
	 settings => ['i b a m', 'i m c b', 'i m c b', 'i c m a', 'i c m a'],
	 positions => [['$x', '$y', '$z'],
		       ['-$x', '-$y', '$z'],
		       ['-$x+1/2', '$y+1/2', '-$z'],
		       ['$x+1/2', '-$y+1/2', '-$z'],
		       ['-$x', '-$y', '-$z'],
		       ['$x', '$y', '-$z'],
		       ['$x+1/2', '-$y+1/2', '$z'],
		       ['-$x+1/2', '$y+1/2', '$z']],
     },
     'i b c a' => {
	 number => 73,
	 schoenflies => 'd_2h^27',
	 full => '1 21/b 21/c 21/a',
	 settings => ['i c a b', 'i b c a', 'i c a b', 'i b c a', 'i c a b'],
	 positions => [['$x', '$y', '$z'],
		       ['-$x+1/2', '-$y', '$z+1/2'],
		       ['-$x', '$y+1/2', '-$z+1/2'],
		       ['$x+1/2', '-$y+1/2', '-$z'],
		       ['-$x', '-$y', '-$z'],
		       ['$x+1/2', '$y', '-$z+1/2'],
		       ['$x', '-$y+1/2', '$z+1/2'],
		       ['-$x+1/2', '$y+1/2', '$z']],
     },
     'i m m a' => {
	 number => 74,
	 schoenflies => 'd_2h^28',
	 full => 'i 21/m 21/m 21/a',
	 settings => ['i m m b', 'i b m m', 'i c m m', 'i m c m', 'i m a m'],
	 positions => [['$x', '$y', '$z'],
		       ['-$x', '-$y+1/2', '$z'],
		       ['-$x', '$y+1/2', '-$z'],
		       ['$x', '-$y', '-$z'],
		       ['-$x', '-$y', '-$z'],
		       ['$x', '$y+1/2', '-$z'],
		       ['$x', '-$y+1/2', '$z'],
		       ['-$x', '$y', '$z']],
     },

     ## tetragonal groups
     'p 4' => {
	 number => 75,
	 schoenflies => 'c_4^1',
         settings => ['c 4'],
	 positions => [['$x', '$y', '$z'],
		       ['-$x', '-$y', '$z'],
		       ['-$y', '$x', '$z'],
		       ['$y', '-$x', '$z']],
     },
     'p 41' => {
	 number => 76,
	 schoenflies => 'c_4^2',
         settings => ['c 41'],
	 positions => [['$x', '$y', '$z'],
		       ['-$x', '-$y', '$z+1/2'],
		       ['-$y', '$x', '$z+1/4'],
		       ['$y', '-$x', '$z+3/4']],
     },
     'p 42' => {
	 number => 77,
	 schoenflies => 'c_4^3',
         settings => ['c 42'],
	 positions => [['$x', '$y', '$z'],
		       ['-$x', '-$y', '$z'],
		       ['-$y', '$x', '$z+1/2'],
		       ['$y', '-$x', '$z+1/2']],
     },
     'p 43' => {
	 number => 78,
	 schoenflies => 'c_4^4',
         settings => ['c 43'],
	 positions => [['$x', '$y', '$z'],
		       ['-$x', '-$y', '$z+1/2'],
		       ['-$y', '$x', '$z+3/4'],
		       ['$y', '-$x', '$z+1/4']],
     },
     'i 4' => {
	 number => 79,
	 schoenflies => 'c_4^5',
         settings => ['f 4'],
	 positions => [['$x', '$y', '$z'],
		       ['-$x', '-$y', '$z'],
		       ['-$y', '$x', '$z'],
		       ['$y', '-$x', '$z']],
     },
     'i 41' => {
	 number => 80,
	 schoenflies => 'c_4^6',
         settings => ['f 41'],
	 positions => [['$x', '$y', '$z'],
		       ['-$x+1/2', '-$y+1/2', '$z+1/2'],
		       ['-$y', '$x+1/2', '$z+1/4'],
		       ['$y+1/2', '-$x', '$z+3/4']],
     },
     'p -4' => {
	 number => 81,
	 schoenflies => 's_4^1',
         settings => ['c -4'],
	 positions => [['$x', '$y', '$z'],
		       ['-$x', '-$y', '$z'],
		       ['$y', '-$x', '-$z'],
		       ['-$y', '$x', '-$z']],
     },
     'i -4' => {
	 number => 82,
	 schoenflies => 's_4^2',
         settings => ['f -4'],
	 positions => [['$x', '$y', '$z'],
		       ['-$x', '-$y', '$z'],
		       ['$y', '-$x', '-$z'],
		       ['-$y', '$x', '-$z']],
     },
     'p 4/m' => {
	 number => 83,
	 schoenflies => 'c_4h^1',
         settings => ['c 4/m'],
	 positions => [['$x', '$y', '$z'],
		       ['-$x', '-$y', '$z'],
		       ['-$y', '$x', '$z'],
		       ['$y', '-$x', '$z'],
		       ['-$x', '-$y', '-$z'],
		       ['$x', '$y', '-$z'],
		       ['$y', '-$x', '-$z'],
		       ['-$y', '$x', '-$z']],
     },
     'p 42/m' => {
	 number => 84,
	 schoenflies => 'c_4h^2',
         settings => ['c 42/m'],
	 positions => [['$x', '$y', '$z'],
		       ['-$x', '-$y', '$z'],
		       ['-$y', '$x', '$z+1/2'],
		       ['$y', '-$x', '$z+1/2'],
		       ['-$x', '-$y', '-$z'],
		       ['$x', '$y', '-$z'],
		       ['$y', '-$x', '-$z+1/2'],
		       ['-$y', '$x', '-$z+1/2']],
     },
     'p 4/n' => {
	 number => 85,
	 schoenflies => 'c_4h^3',
         settings => ['c 4/a'],
	 shiftvec => [-1/4, 1/4, 0],
	 positions => [['$x', '$y', '$z'],
		       ['-$x+1/2', '-$y+1/2', '$z'],
		       ['-$y+1/2', '$x', '$z'],
		       ['$y', '-$x+1/2', '$z'],
		       ['-$x', '-$y', '-$z'],
		       ['$x+1/2', '$y+1/2', '-$z'],
		       ['$y+1/2', '-$x', '-$z'],
		       ['-$y', '$x+1/2', '-$z']],
     },
     'p 42/n' => {
	 number => 86,
	 schoenflies => 'c_4h^4',
	 settings => ['c 42/a'],
	 shiftvec => [-1/4, -1/4, -1/4],
	 positions => [['$x', '$y', '$z'],
		       ['-$x+1/2', '-$y+1/2', '$z'],
		       ['-$y', '$x+1/2', '$z+1/2'],
		       ['$y+1/2', '-$x+1/2', '$z+1/2'],
		       ['-$x', '-$y', '-$z'],
		       ['$x+1/2', '$y+1/2', '-$z'],
		       ['$y', '-$x+1/2', '-$z+1/2'],
		       ['-$y+1/2', '$x', '-$z+1/2']],
     },
     'i 4/m' => {
	 number => 87,
	 schoenflies => 'c_4h^5',
         settings => ['f 4/m'],
	 positions => [['$x', '$y', '$z'],
		       ['-$x', '-$y', '$z'],
		       ['-$y', '$x', '$z'],
		       ['$y', '-$x', '$z'],
		       ['-$x', '-$y', '-$z'],
		       ['$x', '$y', '-$z'],
		       ['$y', '-$x', '-$z'],
		       ['-$y', '$x', '-$z']],
     },
     'i 41/a' => {
	 number => 88,
	 schoenflies => 'c_4h^6',
         settings => ['f 41/d'],
	 shiftvec => [0, -1/4, -1/8],
	 positions => [['$x', '$y', '$z'],
		       ['-$x+1/2', '-$y', '$z+1/2'],
		       ['-$y+3/4', '$x+1/4', '$z+1/4'],
		       ['$y+3/4', '-$x+3/4', '$z+3/4'],
		       ['-$x', '-$y', '-$z'],
		       ['$x+1/2', '$y', '-$z+1/2'],
		       ['$y+1/4', '-$x+3/4', '-$z+3/4'],
		       ['-$y+1/4', '$x+1/4', '-$z+1/4']],
     },
     'p 4 2 2' => {
	 number => 89,
	 schoenflies => 'd_4^1',
	 thirtyfive => 'p 4 2',
         settings => ['c 4 2 2'],
	 positions => [['$x', '$y', '$z'],
		       ['-$x', '-$y', '$z'],
		       ['-$y', '$x', '$z'],
		       ['$y', '-$x', '$z'],
		       ['-$x', '$y', '-$z'],
		       ['$x', '-$y', '-$z'],
		       ['$y', '$x', '-$z'],
		       ['-$y', '-$x', '-$z']],
     },
     'p 4 21 2' => {
	 number => 90,
	 schoenflies => 'd_4^2',
	 thirtyfive => 'p 4 21',
         settings => ['c 4 21 2'],
	 positions => [['$x', '$y', '$z'],
		       ['-$x', '-$y', '$z'],
		       ['-$y+1/2', '$x+1/2', '$z'],
		       ['$y+1/2', '-$x+1/2', '$z'],
		       ['-$x+1/2', '$y+1/2', '-$z'],
		       ['$x+1/2', '-$y+1/2', '-$z'],
		       ['$y', '$x', '-$z'],
		       ['-$y', '-$x', '-$z']],
     },
     'p 41 2 2' => {
	 number => 91,
	 schoenflies => 'd_4^3',
	 thirtyfive => 'p 41 2',
         settings => ['c 41 2 2'],
	 positions => [['$x', '$y', '$z'],
		       ['-$x', '-$y', '$z+1/2'],
		       ['-$y', '$x', '$z+1/4'],
		       ['$y', '-$x', '$z+3/4'],
		       ['-$x', '$y', '-$z'],
		       ['$x', '-$y', '-$z+1/2'],
		       ['$y', '$x', '-$z+3/4'],
		       ['-$y', '-$x', '-$z+1/4']],
     },
     'p 41 21 2' => {
	 number => 92,
	 schoenflies => 'd_4^4',
	 thirtyfive => 'p 41 21',
         settings => ['c 41 2 21'],
	 positions => [['$x', '$y', '$z'],
		       ['-$x', '-$y', '$z+1/2'],
		       ['-$y+1/2', '$x+1/2', '$z+1/4'],
		       ['$y+1/2', '-$x+1/2', '$z+3/4'],
		       ['-$x+1/2', '$y+1/2', '-$z+1/4'],
		       ['$x+1/2', '-$y+1/2', '-$z+3/4'],
		       ['$y', '$x', '-$z'],
		       ['-$y', '-$x', '-$z+1/2']],
     },
     'p 42 2 2' => {
	 number => 93,
	 schoenflies => 'd_4^5',
	 thirtyfive => 'p 42 2',
         settings => ['c 42 2 2'],
	 positions => [['$x', '$y', '$z'],
		       ['-$x', '-$y', '$z'],
		       ['-$y', '$x', '$z+1/2'],
		       ['$y', '-$x', '$z+1/2'],
		       ['-$x', '$y', '-$z'],
		       ['$x', '-$y', '-$z'],
		       ['$y', '$x', '-$z+1/2'],
		       ['-$y', '-$x', '-$z+1/2']],
     },
     'p 42 21 2' => {
	 number => 94,
	 schoenflies => 'd_4^6',
	 thirtyfive => 'p 42 21',
         settings => ['c 42 2 21'],
	 positions => [['$x', '$y', '$z'],
		       ['-$x', '-$y', '$z'],
		       ['-$y+1/2', '$x+1/2', '$z+1/2'],
		       ['$y+1/2', '-$x+1/2', '$z+1/2'],
		       ['-$x+1/2', '$y+1/2', '-$z+1/2'],
		       ['$x+1/2', '-$y+1/2', '-$z+1/2'],
		       ['$y', '$x', '-$z'],
		       ['-$y', '-$x', '-$z']],
     },
     'p 43 2 2' => {
	 number => 95,
	 schoenflies => 'd_4^7',
	 thirtyfive => 'p 43 2',
         settings => ['c 43 2 2'],
	 positions => [['$x', '$y', '$z'],
		       ['-$x', '-$y', '$z+1/2'],
		       ['-$y', '$x', '$z+3/4'],
		       ['$y', '-$x', '$z+1/4'],
		       ['-$x', '$y', '-$z'],
		       ['$x', '-$y', '-$z+1/2'],
		       ['$y', '$x', '-$z+1/4'],
		       ['-$y', '-$x', '-$z+3/4']],
     },
     'p 43 21 2' => {
	 number => 96,
	 schoenflies => 'd_4^8',
	 thirtyfive => 'p 43 21',
         settings => ['c 43 2 21'],
	 positions => [['$x', '$y', '$z'],
		       ['-$x', '-$y', '$z+1/2'],
		       ['-$y+1/2', '$x+1/2', '$z+3/4'],
		       ['$y+1/2', '-$x+1/2', '$z+1/4'],
		       ['-$x+1/2', '$y+1/2', '-$z+3/4'],
		       ['$x+1/2', '-$y+1/2', '-$z+1/4'],
		       ['$y', '$x', '-$z'],
		       ['-$y', '-$x', '-$z+1/2']],
     },
     'i 4 2 2' => {
	 number => 97,
	 schoenflies => 'd_4^9',
	 thirtyfive => 'i 4 2',
         settings => ['f 4 2 2'],
	 positions => [['$x', '$y', '$z'],
		       ['-$x', '-$y', '$z'],
		       ['-$y', '$x', '$z'],
		       ['$y', '-$x', '$z'],
		       ['-$x', '$y', '-$z'],
		       ['$x', '-$y', '-$z'],
		       ['$y', '$x', '-$z'],
		       ['-$y', '-$x', '-$z']],
     },
     'i 41 2 2' => {
	 number => 98,
	 schoenflies => 'd_4^10',
	 thirtyfive => 'p 41 2',
         settings => ['f 41 2 2'],
	 positions => [['$x', '$y', '$z'],
		       ['-$x+1/2', '-$y+1/2', '$z+1/2'],
		       ['-$y', '$x+1/2', '$z+1/4'],
		       ['$y+1/2', '-$x', '$z+3/4'],
		       ['-$x+1/2', '$y', '-$z+3/4'],
		       ['$x', '-$y+1/2', '-$z+1/4'],
		       ['$y+1/2', '$x+1/2', '-$z+1/2'],
		       ['-$y', '-$x', '-$z']],
     },
     'p 4 m m' => {
	 number => 99,
	 schoenflies => 'c_4v^1',
         settings => ['c 4 m m'],
	 positions => [['$x', '$y', '$z'],
		       ['-$x', '-$y', '$z'],
		       ['-$y', '$x', '$z'],
		       ['$y', '-$x', '$z'],
		       ['$x', '-$y', '$z'],
		       ['-$x', '$y', '$z'],
		       ['-$y', '-$x', '$z'],
		       ['$y', '$x', '$z']],
     },
     'p 4 b m' => {
	 number => 100,
	 schoenflies => 'c_4v^2',
	 settings => ['c 4 m g1'],
	 positions => [['$x', '$y', '$z'],
		       ['-$x', '-$y', '$z'],
		       ['-$y', '$x', '$z'],
		       ['$y', '-$x', '$z'],
		       ['$x+1/2', '-$y+1/2', '$z'],
		       ['-$x+1/2', '$y+1/2', '$z'],
		       ['-$y+1/2', '-$x+1/2', '$z'],
		       ['$y+1/2', '$x+1/2', '$z']],
     },
     'p 42 c m' => {
	 number => 101,
	 schoenflies => 'd_4v^3',
	 settings => ['c 42 m c'],
	 positions => [['$x', '$y', '$z'],
		       ['-$x', '-$y', '$z'],
		       ['-$y', '$x', '$z+1/2'],
		       ['$y', '-$x', '$z+1/2'],
		       ['$x', '-$y', '$z+1/2'],
		       ['-$x', '$y', '$z+1/2'],
		       ['-$y', '-$x', '$z'],
		       ['$y', '$x', '$z']],
     },
     'p 42 n m' => {
	 number => 102,
	 schoenflies => 'c_4v^4',
	 settings => ['c 42 m g2'],
	 positions => [['$x', '$y', '$z'],
		       ['-$x', '-$y', '$z'],
		       ['-$y+1/2', '$x+1/2', '$z+1/2'],
		       ['$y+1/2', '-$x+1/2', '$z+1/2'],
		       ['$x+1/2', '-$y+1/2', '$z+1/2'],
		       ['-$x+1/2', '$y+1/2', '$z+1/2'],
		       ['-$y', '-$x', '$z'],
		       ['$y', '$x', '$z']],
     },
     'p 4 c c' => {
	 number => 103,
	 schoenflies => 'c_4v^5',
	 settings => ['c 4 c c'],
	 positions => [['$x', '$y', '$z'],
		       ['-$x', '-$y', '$z'],
		       ['-$y', '$x', '$z'],
		       ['$y', '-$x', '$z'],
		       ['$x', '-$y', '$z+1/2'],
		       ['-$x', '$y', '$z+1/2'],
		       ['-$y', '-$x', '$z+1/2'],
		       ['$y', '$x', '$z+1/2']],
     },
     'p 4 n c' => {
	 number => 104,
	 schoenflies => 'p_4v^6',
	 settings => ['c 4 c g2'],  # is that right?
	 positions => [['$x', '$y', '$z'],
		       ['-$x', '-$y', '$z'],
		       ['-$y', '$x', '$z'],
		       ['$y', '-$x', '$z'],
		       ['$x+1/2', '-$y+1/2', '$z+1/2'],
		       ['-$x+1/2', '$y+1/2', '$z+1/2'],
		       ['-$y+1/2', '-$x+1/2', '$z+1/2'],
		       ['$y+1/2', '$x+1/2', '$z+1/2']],
     },
     'p 42 m c' => {
	 number => 105,
	 schoenflies => 'c_4v^7',
	 settings => ['c 42 c m'],
	 positions => [['$x', '$y', '$z'],
		       ['-$x', '-$y', '$z'],
		       ['-$y', '$x', '$z+1/2'],
		       ['$y', '-$x', '$z+1/2'],
		       ['$x', '-$y', '$z'],
		       ['-$x', '$y', '$z'],
		       ['-$y', '-$x', '$z+1/2'],
		       ['$y', '$x', '$z+1/2']],
     },
     'p 42 b c' => {
	 number => 106,
	 schoenflies => 'c_4v^8',
	 settings => ['c 42 c g1'],
	 positions => [['$x', '$y', '$z'],
		       ['-$x', '-$y', '$z'],
		       ['-$y', '$x', '$z+1/2'],
		       ['$y', '-$x', '$z+1/2'],
		       ['$x+1/2', '-$y+1/2', '$z'],
		       ['-$x+1/2', '$y+1/2', '$z'],
		       ['-$y+1/2', '-$x+1/2', '$z+1/2'],
		       ['$y+1/2', '$x+1/2', '$z+1/2']],
     },
     'i 4 m m' => {
	 number => 107,
	 schoenflies => 'c_4v^9',
	 settings => ['f 4 m m'],
	 positions => [['$x', '$y', '$z'],
		       ['-$x', '-$y', '$z'],
		       ['-$y', '$x', '$z'],
		       ['$y', '-$x', '$z'],
		       ['$x', '-$y', '$z'],
		       ['-$x', '$y', '$z'],
		       ['-$y', '-$x', '$z'],
		       ['$y', '$x', '$z']],
     },
     'i 4 c m' => {
	 number => 108,
	 schoenflies => 'c_4v^10',
	 settings => ['f 4 m c'],  # is that right?
	 positions => [['$x', '$y', '$z'],
		       ['-$x', '-$y', '$z'],
		       ['-$y', '$x', '$z'],
		       ['$y', '-$x', '$z'],
		       ['$x', '-$y', '$z+1/2'],
		       ['-$x', '$y', '$z+1/2'],
		       ['-$y', '-$x', '$z+1/2'],
		       ['$y', '$x', '$z+1/2']],
     },
     'i 41 m d' => {
	 number => 109,
	 schoenflies => 'c_4v^11',
	 settings => ['f 41 d m'],  # is that right?
	 positions => [['$x', '$y', '$z'],
		       ['-$x+1/2', '-$y+1/2', '$z+1/2'],
		       ['-$y', '$x+1/2', '$z+1/4'],
		       ['$y+1/2', '-$x', '$z+3/4'],
		       ['$x', '-$y', '$z'],
		       ['-$x+1/2', '$y+1/2', '$z+1/2'],
		       ['-$y', '-$x+1/2', '$z+1/4'],
		       ['$y+1/2', '$x', '$z+3/4']],
     },
     'i 41 c d' => {
	 number => 110,
	 schoenflies => 'c_4v^12',
	 settings => ['f 41 d c'],  # is that right?
	 positions => [['$x', '$y', '$z'],
		       ['-$x+1/2', '-$y+1/2', '$z+1/2'],
		       ['-$y', '$x+1/2', '$z+1/4'],
		       ['$y+1/2', '-$x', '$z+3/4'],
		       ['$x', '-$y', '$z+1/2'],
		       ['-$x+1/2', '$y+1/2', '$z'],
		       ['-$y', '-$x+1/2', '$z+3/4'],
		       ['$y+1/2', '$x', '$z+1/4']],
     },
     'p -4 2 m' => {
	 number => 111,
	 schoenflies => 'd_2d^1',
	 settings => ['c -4 m 2'],  # is that right?
	 positions => [['$x', '$y', '$z'],
		       ['-$x', '-$y', '$z'],
		       ['$y', '-$x', '-$z'],
		       ['-$y', '$x', '-$z'],
		       ['-$x', '$y', '-$z'],
		       ['$x', '-$y', '-$z'],
		       ['-$y', '-$x', '$z'],
		       ['$y', '$x', '$z']],
     },
     'p -4 2 c' => {
	 number => 112,
	 schoenflies => 'd_2d^2',
	 settings => ['c -4 c 2'],  # is that right?
	 positions => [['$x', '$y', '$z'],
		       ['-$x', '-$y', '$z'],
		       ['$y', '-$x', '-$z'],
		       ['-$y', '$x', '-$z'],
		       ['-$x', '$y', '-$z+1/2'],
		       ['$x', '-$y', '-$z+1/2'],
		       ['-$y', '-$x', '$z+1/2'],
		       ['$y', '$x', '$z+1/2']],
     },
     'p -4 21 m' => {
	 number => 113,
	 schoenflies => 'd_2d^3',
	 settings => ['c -4 m 21'],  # is that right?
	 positions => [['$x', '$y', '$z'],
		       ['-$x', '-$y', '$z'],
		       ['$y', '-$x', '-$z'],
		       ['-$y', '$x', '-$z'],
		       ['-$x+1/2', '$y+1/2', '-$z'],
		       ['$x+1/2', '-$y+1/2', '-$z'],
		       ['-$y+1/2', '-$x+1/2', '$z'],
		       ['$y+1/2', '$x+1/2', '$z']],
     },
     'p -4 21 c' => {
	 number => 114,
	 schoenflies => 'd_2d^4',
	 settings => ['c -4 c 21'],  # is that right?
	 positions => [['$x', '$y', '$z'],
		       ['-$x', '-$y', '$z'],
		       ['$y', '-$x', '-$z'],
		       ['-$y', '$x', '-$z'],
		       ['-$x+1/2', '$y+1/2', '-$z+1/2'],
		       ['$x+1/2', '-$y+1/2', '-$z+1/2'],
		       ['-$y+1/2', '-$x+1/2', '$z+1/2'],
		       ['$y+1/2', '$x+1/2', '$z+1/2']],
     },
     'p -4 m 2' => {
	 number => 115,
	 schoenflies => 'd_2d^5',
	 settings => ['c -4 2 m'],  # is that right?
	 positions => [['$x', '$y', '$z'],
		       ['-$x', '-$y', '$z'],
		       ['$y', '-$x', '-$z'],
		       ['-$y', '$x', '-$z'],
		       ['$x', '-$y', '$z'],
		       ['-$x', '$y', '$z'],
		       ['$y', '$x', '-$z'],
		       ['-$y', '-$x', '-$z']],
     },
     'p -4 c 2' => {
	 number => 116,
	 schoenflies => 'd_2d^6',
	 settings => ['c -4 2 c'],  # is that right?
	 positions => [['$x', '$y', '$z'],
		       ['-$x', '-$y', '$z'],
		       ['$y', '-$x', '-$z'],
		       ['-$y', '$x', '-$z'],
		       ['$x', '-$y', '$z+1/2'],
		       ['-$x', '$y', '$z+1/2'],
		       ['$y', '$x', '-$z+1/2'],
		       ['-$y', '-$x', '-$z+1/2']],
     },
     'p -4 b 2' => {
	 number => 117,
	 schoenflies => 'd_2d^7',
	 settings => ['c -4 2 g'],  # is that right?
	 positions => [['$x', '$y', '$z'],
		       ['-$x', '-$y', '$z'],
		       ['$y', '-$x', '-$z'],
		       ['-$y', '$x', '-$z'],
		       ['$x+1/2', '-$y+1/2', '$z'],
		       ['-$x+1/2', '$y+1/2', '$z'],
		       ['$y+1/2', '$x+1/2', '-$z'],
		       ['-$y+1/2', '-$x+1/2', '-$z']],
     },
     'p -4 n 2' => {
	 number => 118,
	 schoenflies => 'd_2d^8',
	 settings => ['c -4 2 g'],
	 positions => [['$x', '$y', '$z'],
		       ['-$x', '-$y', '$z'],
		       ['$y', '-$x', '-$z'],
		       ['-$y', '$x', '-$z'],
		       ['$x+1/2', '-$y+1/2', '$z+1/2'],
		       ['-$x+1/2', '$y+1/2', '$z+1/2'],
		       ['$y+1/2', '$x+1/2', '-$z+1/2'],
		       ['-$y+1/2', '-$x+1/2', '-$z+1/2']],
     },
     'i -4 m 2' => {
	 number => 119,
	 schoenflies => 'd_2d^9',
	 settings => ['f -4 2 m'],
	 positions => [['$x', '$y', '$z'],
		       ['-$x', '-$y', '$z'],
		       ['$y', '-$x', '-$z'],
		       ['-$y', '$x', '-$z'],
		       ['$x', '-$y', '$z'],
		       ['-$x', '$y', '$z'],
		       ['$y', '$x', '-$z'],
		       ['-$y', '-$x', '-$z']],
     },
     'i -4 c 2' => {
	 number => 120,
	 schoenflies => 'd_2d^10',
	 settings => ['f -4 2 c'],
	 positions => [['$x', '$y', '$z'],
		       ['-$x', '-$y', '$z'],
		       ['$y', '-$x', '-$z'],
		       ['-$y', '$x', '-$z'],
		       ['$x', '-$y', '$z+1/2'],
		       ['-$x', '$y', '$z+1/2'],
		       ['$y', '$x', '-$z+1/2'],
		       ['-$y', '-$x', '-$z+1/2']],
     },
     'i -4 2 m' => {
	 number => 121,
	 schoenflies => 'd_2d^11',
	 settings => ['f -4 m 2'],
	 positions => [['$x', '$y', '$z'],
		       ['-$x', '-$y', '$z'],
		       ['$y', '-$x', '-$z'],
		       ['-$y', '$x', '-$z'],
		       ['-$x', '$y', '-$z'],
		       ['$x', '-$y', '-$z'],
		       ['-$y', '-$x', '$z'],
		       ['$y', '$x', '$z']],
     },
     'i -4 2 d' => {
	 number => 122,
	 schoenflies => 'd_2d^12',
	 settings => ['f -4 d 2'],  # is that right?
	 positions => [['$x', '$y', '$z'],
		       ['-$x', '-$y', '$z'],
		       ['$y', '-$x', '-$z'],
		       ['-$y', '$x', '-$z'],
		       ['-$x+1/2', '$y', '-$z+3/4'],
		       ['$x+1/2', '-$y', '-$z+3/4'],
		       ['-$y+1/2', '-$x', '$z+3/4'],
		       ['$y+1/2', '$x', '$z+3/4']],
     },
     'p 4/m m m' => {
	 number => 123,
	 schoenflies => 'd_4h^1',
	 full => 'p 4/m 2/m 2/m',
         settings => ['c 4/m m m'],
	 positions => [['$x', '$y', '$z'],
		       ['-$x', '-$y', '$z'],
		       ['-$y', '$x', '$z'],
		       ['$y', '-$x', '$z'],
		       ['-$x', '$y', '-$z'],
		       ['$x', '-$y', '-$z'],
		       ['$y', '$x', '-$z'],
		       ['-$y', '-$x', '-$z'],
		       ['-$x', '-$y', '-$z'],
		       ['$x', '$y', '-$z'],
		       ['$y', '-$x', '-$z'],
		       ['-$y', '$x', '-$z'],
		       ['$x', '-$y', '$z'],
		       ['-$x', '$y', '$z'],
		       ['-$y', '-$x', '$z'],
		       ['$y', '$x', '$z']],
     },
     'p 4/m c c' => {
	 number => 124,
	 schoenflies => 'd_4h^2',
	 full => 'p 4/m 2/c 2/c',
         settings => ['c 4/m c c'],
	 positions => [['$x', '$y', '$z'],
		       ['-$x', '-$y', '$z'],  		# 2
		       ['-$y', '$x', '$z'],   		# 3
		       ['$y', '-$x', '$z'],   		# 4
		       ['-$x', '$y', '-$z+1/2'],  		# 5
		       ['$x', '-$y', '-$z+1/2'],  		# 6
		       ['$y', '$x', '-$z+1/2'],   		# 7
		       ['-$y', '-$x', '-$z+1/2'], 		# 8
		       ['-$x', '-$y', '-$z'], 		# 9
		       ['$x', '$y', '-$z'],   		# 10
		       ['$y', '-$x', '-$z'],  		# 11
		       ['-$y', '$x', '-$z'],  		# 12
		       ['$x', '-$y', '$z+1/2'],   		# 13
		       ['-$x', '$y', '$z+1/2'],   		# 14
		       ['-$y', '-$x', '$z+1/2'],  		# 15
		       ['$y', '$x', '$z+1/2']],    		# 16
		   },
     'p 4/n b m' => {  # is that right?
	 number => 125,
	 schoenflies => 'd_4h^3',
	 full => 'p 4/n 2/b 2/m',
         settings => ['c 4/a m b'],  # is that right?
	 shiftvec => [-1/4, -1/4, 0],
	 positions => [['$x', '$y', '$z'],
		       ['-$x+1/2', '-$y+1/2', '$z'],  		# 2
		       ['-$y+1/2', '$x', '$z'],   		# 3
		       ['$y', '-$x+1/2', '$z'],   		# 4
		       ['-$x+1/2', '$y', '-$z'],  		# 5
		       ['$x', '-$y+1/2', '-$z'],  		# 6
		       ['$y', '$x', '-$z'],   		# 7
		       ['-$y+1/2', '-$x+1/2', '-$z'], 		# 8
		       ['-$x', '-$y', '-$z'], 		# 9
		       ['$x+1/2', '$y+1/2', '-$z'],   		# 10
		       ['$y+1/2', '-$x', '-$z'],  		# 11
		       ['-$y', '$x+1/2', '-$z'],  		# 12
		       ['$x+1/2', '-$y', '$z'],   		# 13
		       ['-$x', '$y+1/2', '$z'],   		# 14
		       ['-$y', '-$x', '$z'],  		# 15
		       ['$y+1/2', '$x+1/2', '$z']],
     },
     'p 4/n n c' => {
	 number => 126,
	 schoenflies => 'd_4h^4',
	 full => 'p 4/n 2/n 2/c',
         settings => ['c 4/a c n'],
	 shiftvec => [-1/4, -1/4, -1/4],
	 positions => [['$x', '$y', '$z'],
		       ['-$x+1/2', '-$y+1/2', '$z'],  		# 2
		       ['-$y+1/2', '$x', '$z'],   		# 3
		       ['$y', '-$x+1/2', '$z'],   		# 4
		       ['-$x+1/2', '$y', '-$z+1/2'],  		# 5
		       ['$x', '-$y+1/2', '-$z+1/2'],  		# 6
		       ['$y', '$x', '-$z+1/2'],   		# 7
		       ['-$y+1/2', '-$x+1/2', '-$z+1/2'], 		# 8
		       ['-$x', '-$y', '-$z'], 		# 9
		       ['$x+1/2', '$y+1/2', '-$z'],   		# 10
		       ['$y+1/2', '-$x', '-$z'],  		# 11
		       ['-$y', '$x+1/2', '-$z'],  		# 12
		       ['$x+1/2', '-$y', '$z+1/2'],   		# 13
		       ['-$x', '$y+1/2', '$z+1/2'],   		# 14
		       ['-$y', '-$x', '$z+1/2'],  		# 15
		       ['$y+1/2', '$x+1/2', '$z+1/2']],
     },
     'p 4/m b m' => {
	 number => 127,
	 schoenflies => 'd_4h^5',
	 full => 'p 4/m 21/b 2/m',
         settings => ['c 4/m m b'],
	 positions => [['$x', '$y', '$z'],
		       ['-$x', '-$y', '$z'],  		# 2
		       ['-$y', '$x', '$z'],   		# 3
		       ['$y', '-$x', '$z'],   		# 4
		       ['-$x+1/2', '$y+1/2', '-$z'],  		# 5
		       ['$x+1/2', '-$y+1/2', '-$z'],  		# 6
		       ['$y+1/2', '$x+1/2', '-$z'],   		# 7
		       ['-$y+1/2', '-$x+1/2', '-$z'], 		# 8
		       ['-$x', '-$y', '-$z'], 		# 9
		       ['$x', '$y', '-$z'],   		# 10
		       ['$y', '-$x', '-$z'],  		# 11
		       ['-$y', '$x', '-$z'],  		# 12
		       ['$x+1/2', '-$y+1/2', '$z'],   		# 13
		       ['-$x+1/2', '$y+1/2', '$z'],   		# 14
		       ['-$y+1/2', '-$x+1/2', '$z'],  		# 15
		       ['$y+1/2', '$x+1/2', '$z']],
     },
     'p 4/m n c' => {
	 number => 128,
	 schoenflies => 'd_4h^6',
	 full => 'p 4/m 21/n 2/c',
         settings => ['c 4/m c n'],
	 positions => [['$x', '$y', '$z'],
		       ['-$x', '-$y', '$z'],  		# 2
		       ['-$y', '$x', '$z'],   		# 3
		       ['$y', '-$x', '$z'],   		# 4
		       ['-$x+1/2', '$y+1/2', '-$z+1/2'],  		# 5
		       ['$x+1/2', '-$y+1/2', '-$z+1/2'],  		# 6
		       ['$y+1/2', '$x+1/2', '-$z+1/2'],   		# 7
		       ['-$y+1/2', '-$x+1/2', '-$z+1/2'], 		# 8
		       ['-$x', '-$y', '-$z'], 		# 9
		       ['$x', '$y', '-$z'],   		# 10
		       ['$y', '-$x', '-$z'],  		# 11
		       ['-$y', '$x', '-$z'],  		# 12
		       ['$x+1/2', '-$y+1/2', '$z+1/2'],   		# 13
		       ['-$x+1/2', '$y+1/2', '$z+1/2'],   		# 14
		       ['-$y+1/2', '-$x+1/2', '$z+1/2'],  		# 15
		       ['$y+1/2', '$x+1/2', '$z+1/2']],
     },
     'p 4/n m m' => {
	 number => 129,
	 schoenflies => 'd_4h^7',
	 full => 'p 4/n 21/m 2/m',
         settings => ['c 4/a m m'],
	 shiftvec => [1/4, -1/4, 0],
	 positions => [['$x', '$y', '$z'],
		       ['-$x+1/2', '-$y+1/2', '$z'],  		# 2
		       ['-$y+1/2', '$x', '$z'],   		# 3
		       ['$y', '-$x+1/2', '$z'],   		# 4
		       ['-$x', '$y+1/2', '-$z'],  		# 5
		       ['$x+1/2', '-$y', '-$z'],  		# 6
		       ['$y+1/2', '$x+1/2', '-$z'],   		# 7
		       ['-$y', '-$x', '-$z'], 		# 8
		       ['-$x', '-$y', '-$z'], 		# 9
		       ['$x+1/2', '$y+1/2', '-$z'],   		# 10
		       ['$y+1/2', '-$x', '-$z'],  		# 11
		       ['-$y', '$x+1/2', '-$z'],  		# 12
		       ['$x', '-$y+1/2', '$z'],   		# 13
		       ['-$x+1/2', '$y', '$z'],   		# 14
		       ['-$y+1/2', '-$x+1/2', '$z'],  		# 15
		       ['$y', '$x', '$z']],
     },
     'p 4/n c c' => {
	 number => 130,
	 schoenflies => 'd_4h^8',
	 full => 'p 4/n 21/c 2/c',
         settings => ['c 4/a c c'],
	 shiftvec => [-1/4, -1/4, 0],
	 positions => [['$x', '$y', '$z'],
		       ['-$x+1/2', '-$y+1/2', '$z'],  		# 2
		       ['-$y+1/2', '$x', '$z'],   		# 3
		       ['$y', '-$x+1/2', '$z'],   		# 4
		       ['-$x', '$y+1/2', '-$z+1/2'],  		# 5
		       ['$x+1/2', '-$y', '-$z+1/2'],  		# 6
		       ['$y+1/2', '$x+1/2', '-$z+1/2'],   		# 7
		       ['-$y', '-$x', '-$z+1/2'], 		# 8
		       ['-$x', '-$y', '-$z'], 		# 9
		       ['$x+1/2', '$y+1/2', '-$z'],   		# 10
		       ['$y+1/2', '-$x', '-$z'],  		# 11
		       ['-$y', '$x+1/2', '-$z'],  		# 12
		       ['$x', '-$y+1/2', '$z+1/2'],   		# 13
		       ['-$x+1/2', '$y', '$z+1/2'],   		# 14
		       ['-$y+1/2', '-$x+1/2', '$z+1/2'],  		# 15
		       ['$y', '$x', '$z+1/2']],
     },
     'p 42/m m c' => {
	 number => 131,
	 schoenflies => 'd_4h^9',
	 full => 'p 42/m 2/m 2/c',
         settings => ['c 42/m c m'],
	 positions => [['$x', '$y', '$z'],
		       ['-$x', '-$y', '$z'],  		# 2
		       ['-$y', '$x', '$z+1/2'],   		# 3
		       ['$y', '-$x', '$z+1/2'],   		# 4
		       ['-$x', '$y', '-$z'],  		# 5
		       ['$x', '-$y', '-$z'],  		# 6
		       ['$y', '$x', '-$z+1/2'],   		# 7
		       ['-$y', '-$x', '-$z+1/2'], 		# 8
		       ['-$x', '-$y', '-$z'], 		# 9
		       ['$x', '$y', '-$z'],   		# 10
		       ['$y', '-$x', '-$z+1/2'],  		# 11
		       ['-$y', '$x', '-$z+1/2'],  		# 12
		       ['$x', '-$y', '$z'],   		# 13
		       ['-$x', '$y', '$z'],   		# 14
		       ['-$y', '-$x', '$z+1/2'],  		# 15
		       ['$y', '$x', '$z+1/2']],
     },
     'p 42/m c m' => {
	 number => 132,
	 schoenflies => 'd_4h^10',
	 full => 'p 42/m 2/c 2/m',
         settings => ['c 42/m m c'],
	 positions => [['$x', '$y', '$z'],
		       ['-$x', '-$y', '$z'],  		# 2
		       ['-$y', '$x', '$z+1/2'],   		# 3
		       ['$y', '-$x', '$z+1/2'],   		# 4
		       ['-$x', '$y', '-$z+1/2'],  		# 5
		       ['$x', '-$y', '-$z+1/2'],  		# 6
		       ['$y', '$x', '-$z'],   		# 7
		       ['-$y', '-$x', '-$z'], 		# 8
		       ['-$x', '-$y', '-$z'], 		# 9
		       ['$x', '$y', '-$z'],   		# 10
		       ['$y', '-$x', '-$z+1/2'],  		# 11
		       ['-$y', '$x', '-$z+1/2'],  		# 12
		       ['$x', '-$y', '$z+1/2'],   		# 13
		       ['-$x', '$y', '$z+1/2'],   		# 14
		       ['-$y', '-$x', '$z'],  		# 15
		       ['$y', '$x', '$z']],
     },
     'p 42/n b c' => {
	 number => 133,
	 schoenflies => 'd_4h^11',
	 full => 'p 42/n 2/b 2/c',
         settings => ['c 42/a c b'],
	 shiftvec => [-1/4, 1/4, -1/4],
	 positions => [['$x', '$y', '$z'],
		       ['-$x+1/2', '-$y+1/2', '$z'],
		       ['-$y+1/2', '$x', '$z+1/2'],
		       ['$y', '-$x+1/2', '$z+1/2'],
		       ['-$x+1/2', '$y', '-$z'],
		       ['$x', '-$y+1/2', '-$z'],
		       ['$y', '$x', '-$z+1/2'],
		       ['-$y+1/2', '-$x+1/2', '-$z+1/2'],
		       ['-$x', '-$y', '-$z'],
		       ['$x+1/2', '$y+1/2', '-$z'],
		       ['$y+1/2', '-$x', '-$z+1/2'],
		       ['-$y', '$x+1/2', '-$z+1/2'],
		       ['$x+1/2', '-$y', '$z'],
		       ['-$x', '$y+1/2', '$z'],
		       ['-$y', '-$x', '$z+1/2'],
		       ['$y+1/2', '$x+1/2', '$z+1/2']],
     },
     'p 42/n n m' => {
	 number => 134,
	 schoenflies => 'd_4h^12',
	 full => 'p 42/n 2/n 2/m',
	 settings => ['c 42/a m n'],
	 shiftvec => [-1/4, 1/4, -1/4],
	 positions => [['$x', '$y', '$z'],
		       ['-$x+1/2', '-$y+1/2', '$z'],
		       ['-$y+1/2', '$x', '$z+1/2'],
		       ['$y', '-$x+1/2', '$z+1/2'],
		       ['-$x+1/2', '$y', '-$z+1/2'],
		       ['$x', '-$y+1/2', '-$z+1/2'],
		       ['$y', '$x', '-$z'],
		       ['-$y+1/2', '-$x+1/2', '-$z'],
		       ['-$x', '-$y', '-$z'],
		       ['$x+1/2', '$y+1/2', '-$z'],
		       ['$y+1/2', '-$x', '-$z+1/2'],
		       ['-$y', '$x+1/2', '-$z+1/2'],
		       ['$x+1/2', '-$y', '$z+1/2'],
		       ['-$x', '$y+1/2', '$z+1/2'],
		       ['-$y', '-$x', '$z'],
		       ['$y+1/2', '$x+1/2', '$z']],
     },
     'p 42/m b c' => {
	 number => 135,
	 schoenflies => 'd_4h^13',
	 full => 'p 42/m 21/b 2/c',
         settings => ['c 42/m c b'],
	 positions => [['$x', '$y', '$z'],
		       ['-$x', '-$y', '$z'],
		       ['-$y', '$x', '$z+1/2'],
		       ['$y', '-$x', '$z+1/2'],
		       ['-$x+1/2', '$y+1/2', '-$z'],
		       ['$x+1/2', '-$y+1/2', '-$z'],
		       ['$y+1/2', '$x+1/2', '-$z+1/2'],
		       ['-$y+1/2', '-$x+1/2', '-$z+1/2'],
		       ['-$x', '-$y', '-$z'],
		       ['$x', '$y', '-$z'],
		       ['$y', '-$x', '-$z+1/2'],
		       ['-$y', '$x', '-$z+1/2'],
		       ['$x+1/2', '-$y+1/2', '$z'],
		       ['-$x+1/2', '$y+1/2', '$z'],
		       ['-$y+1/2', '-$x+1/2', '$z+1/2'],
		       ['$y+1/2', '$x+1/2', '$z+1/2']],
     },
     'p 42/m n m' => {
	 number => 136,
	 schoenflies => 'd_4h^14',
	 full => 'p 42/m 21/n 2/m',
         settings => ['c 42/m m n'],
	 positions => [['$x', '$y', '$z'],
		       ['-$x', '-$y', '$z'],
		       ['-$y+1/2', '$x+1/2', '$z+1/2'],
		       ['$y+1/2', '-$x+1/2', '$z+1/2'],
		       ['-$x+1/2', '$y+1/2', '-$z+1/2'],
		       ['$x+1/2', '-$y+1/2', '-$z+1/2'],
		       ['$y', '$x', '-$z'],
		       ['-$y', '-$x', '-$z'],
		       ['-$x', '-$y', '-$z'],
		       ['$x', '$y', '-$z'],
		       ['$y+1/2', '-$x+1/2', '-$z+1/2'],
		       ['-$y+1/2', '$x+1/2', '-$z+1/2'],
		       ['$x+1/2', '-$y+1/2', '$z+1/2'],
		       ['-$x+1/2', '$y+1/2', '$z+1/2'],
		       ['-$y', '-$x', '$z'],
		       ['$y', '$x', '$z']],
     },
     'p 42/n m c' => {
	 number => 137,
	 schoenflies => 'd_4h^15',
	 full => 'p 42/n 21/m 2/c',
         settings => ['c 42/a c m'],
	 shiftvec => [-1/4, 1/4, -1/4],
	 positions => [['$x', '$y', '$z'],
		       ['-$x+1/2', '-$y+1/2', '$z'],
		       ['-$y+1/2', '$x', '$z+1/2'],
		       ['$y', '-$x+1/2', '$z+1/2'],
		       ['-$x', '$y+1/2', '-$z'],
		       ['$x+1/2', '-$y', '-$z'],
		       ['$y+1/2', '$x+1/2', '-$z+1/2'],
		       ['-$y', '-$x', '-$z+1/2'],
		       ['-$x', '-$y', '-$z'],
		       ['$x+1/2', '$y+1/2', '-$z'],
		       ['$y+1/2', '-$x', '-$z+1/2'],
		       ['-$y', '$x+1/2', '-$z+1/2'],
		       ['$x', '-$y+1/2', '$z'],
		       ['-$x+1/2', '$y', '$z'],
		       ['-$y+1/2', '-$x+1/2', '$z+1/2'],
		       ['$y', '$x', '$z+1/2']],
     },
     'p 42/n c m' => {
	 number => 138,
	 schoenflies => 'd_4h^16',
	 full => 'p 42/n 21/c 2/m',
         settings => ['c 42/a m c'],
	 shiftvec => [-1/4, 1/4, -1/4],
	 positions => [['$x', '$y', '$z'],
		       ['-$x+1/2', '-$y+1/2', '$z'],
		       ['-$y+1/2', '$x', '$z+1/2'],
		       ['$y', '-$x+1/2', '$z+1/2'],
		       ['-$x', '$y+1/2', '-$z+1/2'],
		       ['$x+1/2', '-$y', '-$z+1/2'],
		       ['$y+1/2', '$x+1/2', '-$z'],
		       ['-$y', '-$x', '-$z'],
		       ['-$x', '-$y', '-$z'],
		       ['$x+1/2', '$y+1/2', '-$z'],
		       ['$y+1/2', '-$x', '-$z+1/2'],
		       ['-$y', '$x+1/2', '-$z+1/2'],
		       ['$x', '-$y+1/2', '$z+1/2'],
		       ['-$x+1/2', '$y', '$z+1/2'],
		       ['-$y+1/2', '-$x+1/2', '$z'],
		       ['$y', '$x', '$z']],
     },
     'i 4/m m m' => {
	 number => 139,
	 schoenflies => 'd_4h^17',
	 full => 'i 4/m 2/m 2/m',
         settings => ['f 4/m m m'],
	 positions => [['$x', '$y', '$z'],
		       ['-$x', '-$y', '$z'],
		       ['-$y', '$x', '$z'],
		       ['$y', '-$x', '$z'],
		       ['-$x', '$y', '-$z'],
		       ['$x', '-$y', '-$z'],
		       ['$y', '$x', '-$z'],
		       ['-$y', '-$x', '-$z'],
		       ['-$x', '-$y', '-$z'],
		       ['$x', '$y', '-$z'],
		       ['$y', '-$x', '-$z'],
		       ['-$y', '$x', '-$z'],
		       ['$x', '-$y', '$z'],
		       ['-$x', '$y', '$z'],
		       ['-$y', '-$x', '$z'],
		       ['$y', '$x', '$z']],
     },
     'i 4/m c m' => {
	 number => 140,
	 schoenflies => 'd_4h^18',
	 full => 'i 4/m 2/c 2/m',
         settings => ['f 4/m m c'],
	 positions => [['$x', '$y', '$z'],
		       ['-$x', '-$y', '$z'],
		       ['-$y', '$x', '$z'],
		       ['$y', '-$x', '$z'],
		       ['-$x', '$y', '-$z+1/2'],
		       ['$x', '-$y', '-$z+1/2'],
		       ['$y', '$x', '-$z+1/2'],
		       ['-$y', '-$x', '-$z+1/2'],
		       ['-$x', '-$y', '-$z'],
		       ['$x', '$y', '-$z'],
		       ['$y', '-$x', '-$z'],
		       ['-$y', '$x', '-$z'],
		       ['$x', '-$y', '$z+1/2'],
		       ['-$x', '$y', '$z+1/2'],
		       ['-$y', '-$x', '$z+1/2'],
		       ['$y', '$x', '$z+1/2']],
     },
     'i 41/a m d' => {
	 number => 141,
	 schoenflies => 'd_4h^19',
	 full => 'i 41/a 2/m 2/d',
         settings => ['f 41/d d m'],
	 shiftvec => [0, 1/4, -1/8],
	 positions => [['$x', '$y', '$z'],
		       ['-$x+1/2', '-$y', '$z+1/2'],
		       ['-$y+1/4', '$x+3/4', '$z+1/4'],
		       ['$y+1/4', '-$x+1/4', '$z+3/4'],
		       ['-$x+1/2', '$y', '-$z+1/2'],
		       ['$x', '-$y', '-$z'],
		       ['$y+1/4', '$x+3/4', '-$z+1/4'],
		       ['-$y+1/4', '-$x+1/4', '-$z+3/4'],
		       ['-$x', '-$y', '-$z'],
		       ['$x+1/2', '$y', '-$z+1/2'],
		       ['$y+3/4', '-$x+1/4', '-$z+3/4'],
		       ['-$y+3/4', '$x+3/4', '-$z+1/4'],
		       ['$x+1/2', '-$y', '$z+1/2'],
		       ['-$x', '$y', '$z'],
		       ['-$y+3/4', '-$x+1/4', '$z+3/4'],
		       ['$y+3/4', '$x+3/4', '$z+1/4']],
     },
     'i 41/a c d' => {
	 number => 142,
	 schoenflies => 'd_4h^20',
	 full => 'i 41/a 2/c 2/d',
         settings => ['f 41/d d c'],
	 shiftvec => [0, 1/4, -1/8],
	 positions => [['$x', '$y', '$z'],
		       ['-$x+1/2', '-$y', '$z+1/2'],
		       ['-$y+1/4', '$x+3/4', '$z+1/4'],
		       ['$y+1/4', '-$x+1/4', '$z+3/4'],
		       ['-$x+1/2', '$y', '-$z'],
		       ['$x', '-$y', '-$z+1/2'],
		       ['$y+1/4', '$x+3/4', '-$z+3/4'],
		       ['-$y+1/4', '-$x+1/4', '-$z+1/4'],
		       ['-$x', '-$y', '-$z'],
		       ['$x+1/2', '$y', '-$z+1/2'],
		       ['$y+3/4', '-$x+1/4', '-$z+3/4'],
		       ['-$y+3/4', '$x+3/4', '-$z+1/4'],
		       ['$x+1/2', '-$y', '$z'],
		       ['-$x', '$y', '$z+1/2'],
		       ['-$y+3/4', '-$x+1/4', '$z+1/4'],
		       ['$y+3/4', '$x+3/4', '$z+3/4']],
     },

     ## trigonal groups
     'p 3' => {
	 number => 143,
	 schoenflies => 'c_3^1',
	 positions => [
		       ['$x', '$y', '$z'],
		       ['-$y', '$x-$y', '$z'],
		       ['-$x+$y', '-$x', '$z']
		       ],
		       },
     'p 31' => {
	 number => 144,
	 schoenflies => 'c_3^2',
	 positions => [
		       ['$x', '$y', '$z'],
		       ['-$y', '$x-$y', '$z+1/3'],
		       ['-$x+$y', '-$x', '$z+2/3']
		       ],
		       },
     'p 32' => {
	 number => 145,
	 schoenflies => 'c_3^3',
	 positions => [
		       ['$x', '$y', '$z'],
		       ['-$y', '$x-$y', '$z+2/3'],
		       ['-$x+$y', '-$x', '$z+1/3'] ],
	 },
     'r 3' => {
	 number => 146,
	 schoenflies => 'c_3^4',
	 positions => [['$x', '$y', '$z'],
		       ['-$y', '$x-$y', '$z'],
		       ['-$x+$y', '-$x', '$z'] ],
	 rhombohedral => [['$x', '$y', '$z'],
			  ['$z', '$x', '$y'],
			  ['$y', '$z', '$x']],

	 },
     'p -3' => {
	 number => 147,
	 schoenflies => 'c_3i^1',
	 positions => [['$x', '$y', '$z'],
		       ['-$y', '$x-$y', '$z'],
		       ['-$x+$y', '-$x', '$z'],
		       ['-$x', '-$y', '-$z'],
		       ['$y', '-$x+$y', '-$z'],
		       ['$x-$y', '$x', '-$z']],
		       },
     'r -3' => {
	 number => 148,
	 schoenflies => 'c_3i^2',
	 positions => [['$x', '$y', '$z'],
		       ['-$y', '$x-$y', '$z'],
		       ['-$x+$y', '-$x', '$z'],
		       ['-$x', '-$y', '-$z'],
		       ['$y', '-$x+$y', '-$z'],
		       ['$x-$y', '$x', '-$z']],
	 rhombohedral => [['$x', '$y', '$z'],
			  ['$z', '$x', '$y'],
			  ['$y', '$z', '$x'],
			  ['-$x', '-$y', '-$z'],
			  ['-$z', '-$x', '-$y'],
			  ['-$y', '-$z', '-$x']],
		       },
     'p 3 1 2' => {
	 number => 149,
	 schoenflies => 'd_3^1',
	 positions => [['$x', '$y', '$z'],
		       ['-$y', '$x-$y', '$z'],
		       ['-$x+$y', '-$x', '$z'],
		       ['-$y', '-$x', '-$z'],
		       ['-$x+$y', '$y', '-$z'],
		       ['$x', '$x-$y', '-$z']],
		       },
     'p 3 2 1' => {
	 number => 150,
	 schoenflies => 'd_3^2',
	 positions => [['$x', '$y', '$z'],
		       ['-$y', '$x-$y', '$z'],
		       ['-$x+$y', '-$x', '$z'],
		       ['$y', '$x', '-$z'],
		       ['$x-$y', '-$y', '-$z'],
		       ['-$x', '-$x+$y', '-$z']],
		       },
     'p 31 1 2' => {
	 number => 151,
	 schoenflies => 'd_3^3',
	 positions => [['$x', '$y', '$z'],
		       ['-$y', '$x-$y', '$z+1/3'],
		       ['-$x+$y', '-$x', '$z+2/3'],
		       ['-$y', '-$x', '-$z+2/3'],
		       ['-$x+$y', '$y', '-$z+1/3'],
		       ['$x', '$x-$y', '-$z']],
		       },
     'p 31 2 1' => {
	 number => 152,
	 schoenflies => 'd_3^4',
	 positions => [['$x', '$y', '$z'],
		       ['-$y', '$x-$y', '$z+1/3'],
		       ['-$x+$y', '-$x', '$z+2/3'],
		       ['$y', '$x', '-$z'],
		       ['$x-$y', '-$y', '-$z+2/3'],
		       ['-$x', '-$x+$y', '-$z+1/3']],
		       },
     'p 32 1 2' => {
	 number => 153,
	 schoenflies => 'd_3^5',
	 positions => [['$x', '$y', '$z'],
		       ['-$y', '$x-$y', '$z+2/3'],
		       ['-$x+$y', '-$x', '$z+1/3'],
		       ['-$y', '-$x', '-$z+1/3'],
		       ['-$x+$y', '$y', '-$z+2/3'],
		       ['$x', '$x-$y', '-$z']],
		       },
     'p 32 2 1' => {
	 number => 154,
	 schoenflies => 'd_3^6',
	 positions => [['$x', '$y', '$z'],
		       ['-$y', '$x-$y', '$z+2/3'],
		       ['-$x+$y', '-$x', '$z+1/3'],
		       ['$y', '$x', '-$z'],
		       ['$x-$y', '-$y', '-$z+1/3'],
		       ['-$x', '-$x+$y', '-$z+2/3']],
		       },
     'r 3 2' => {
	 number => 155,
	 schoenflies => 'd_3^7',
	 positions => [['$x', '$y', '$z'],
		       ['-$y', '$x-$y', '$z'],
		       ['-$x+$y', '-$x', '$z'],
		       ['$y', '$x', '-$z'],
		       ['$x-$y', '-$y', '-$z'],
		       ['-$x', '-$x+$y', '-$z']],
	 rhombohedral => [['$x', '$y', '$z'],
			  ['$z', '$x', '$y'],
			  ['$y', '$z', '$x'],
			  ['-$y', '-$x', '-$z'],
			  ['-$x', '-$z', '-$y'],
			  ['-$z', '-$y', '-$x']],
		       },
     'p 3 m 1' => {
	 number => 156,
	 schoenflies => 'c_3v^1',
	 positions => [['$x', '$y', '$z'],
		       ['-$y', '$x-$y', '$z'],
		       ['-$x+$y', '-$x', '$z'],
		       ['-$y', '-$x', '$z'],
		       ['-$x+$y', '$y', '$z'],
		       ['$x', '$x-$y', '$z']],
		       },
     'p 3 1 m' => {
	 number => 157,
	 schoenflies => 'c_3v^2',
	 positions => [['$x', '$y', '$z'],
		       ['-$y', '$x-$y', '$z'],
		       ['-$x+$y', '-$x', '$z'],
		       ['$y', '$x', '$z'],
		       ['$x-$y', '-$y', '$z'],
		       ['-$x', '-$x+$y', '$z']],
		       },
     'p 3 c 1' => {
	 number => 158,
	 schoenflies => 'c_3v^3',
	 positions => [['$x', '$y', '$z'],
		       ['-$y', '$x-$y', '$z'],
		       ['-$x+$y', '-$x', '$z'],
		       ['-$y', '-$x', '$z+1/2'],
		       ['-$x+$y', '$y', '$z+1/2'],
		       ['$x', '$x-$y', '$z+1/2']],
		       },
     'p 3 1 c' => {
	 number => 159,
	 schoenflies => 'c_3v^4',
	 positions => [['$x', '$y', '$z'],
		       ['-$y', '$x-$y', '$z'],
		       ['-$x+$y', '-$x', '$z'],
		       ['$y', '$x', '$z+1/2'],
		       ['$x-$y', '-$y', '$z+1/2'],
		       ['-$x', '-$x+$y', '$z+1/2']],
		       },
     'r 3 m' => {
	 number => 160,
	 schoenflies => 'c_3v^5',
	 positions => [['$x', '$y', '$z'],
		       ['-$y', '$x-$y', '$z'],
		       ['-$x+$y', '-$x', '$z'],
		       ['-$y', '-$x', '$z'],
		       ['-$x+$y', '$y', '$z'],
		       ['$x', '$x-$y', '$z']],
	 rhombohedral => [['$x', '$y', '$z'],
			  ['$z', '$x', '$y'],
			  ['$y', '$z', '$x'],
			  ['$y', '$x', '$z'],
			  ['$x', '$z', '$y'],
			  ['$z', '$y', '$x']],
     },
     'r 3 c' => {
	 number => 161,
	 schoenflies => 'c_3v^6',
	 positions => [['$x', '$y', '$z'],
		       ['-$y', '$x-$y', '$z'],
		       ['-$x+$y', '-$x', '$z'],
		       ['-$y', '-$x', '$z+1/2'],
		       ['-$x+$y', '$y', '$z+1/2'],
		       ['$x', '$x-$y', '$z+1/2']],
	 rhombohedral => [['$x', '$y', '$z'],
			  ['$z', '$x', '$y'],
			  ['$y', '$z', '$x'],
			  ['$y+1/2', '$x+1/2', '$z+1/2'],
			  ['$x+1/2', '$z+1/2', '$y+1/2'],
			  ['$z+1/2', '$y+1/2', '$x+1/2']],
     },
     'p -3 1 m' => {
	 number => 162,
	 schoenflies => 'd_3d^1',
	 full => 'p -3 1 2/m',
	 positions => [['$x', '$y', '$z'],
		       ['-$y', '$x-$y', '$z'],
		       ['-$x+$y', '-$x', '$z'],
		       ['-$y', '-$x', '-$z'],
		       ['-$x+$y', '$y', '-$z'],
		       ['$x', '$x-$y', '-$z'],
		       ['-$x', '-$y', '-$z'],
		       ['$y', '-$x+$y', '-$z'],
		       ['$x-$y', '$x', '-$z'],
		       ['$y', '$x', '$z'],
		       ['$x-$y', '-$y', '$z'],
		       ['-$x', '-$x+$y', '$z']],
     },
     'p -3 1 c' => {
	 number => 163,
	 schoenflies => 'd_3d^2',
	 full => 'p -3 1 2/c',
	 positions => [['$x', '$y', '$z'],
		       ['-$y', '$x-$y', '$z'],
		       ['-$x+$y', '-$x', '$z'],
		       ['-$y', '-$x', '-$z+1/2'],
		       ['-$x+$y', '$y', '-$z+1/2'],
		       ['$x', '$x-$y', '-$z+1/2'],
		       ['-$x', '-$y', '-$z'],
		       ['$y', '-$x+$y', '-$z'],
		       ['$x-$y', '$x', '-$z'],
		       ['$y', '$x', '$z+1/2'],
		       ['$x-$y', '-$y', '$z+1/2'],
		       ['-$x', '-$x+$y', '$z+1/2']],
     },
     'p -3 m 1' => {
	 number => 164,
	 schoenflies => 'd_3d^3',
	 full => 'p -3 2/m 1',
	 positions => [['$x', '$y', '$z'],
		       ['-$y', '$x-$y', '$z'],
		       ['-$x+$y', '-$x', '$z'],
		       ['$y', '$x', '-$z'],
		       ['$x-$y', '-$y', '-$z'],
		       ['-$x', '-$x+$y', '-$z'],
		       ['-$x', '-$y', '-$z'],
		       ['$y', '-$x+$y', '-$z'],
		       ['$x-$y', '$x', '-$z'],
		       ['-$y', '-$x', '$z'],
		       ['-$x+$y', '$y', '$z'],
		       ['$x', '$x-$y', '$z']],
     },
     'p -3 c 1' => {
	 number => 165,
	 full => 'p -3 2/c 1',
	 schoenflies => 'd_3d^4',
	 positions => [['$x', '$y', '$z'],
		       ['-$y', '$x-$y', '$z'],
		       ['-$x+$y', '-$x', '$z'],
		       ['$y', '$x', '-$z+1/2'],
		       ['$x-$y', '-$y', '-$z+1/2'],
		       ['-$x', '-$x+$y', '-$z+1/2'],
		       ['-$x', '-$y', '-$z'],
		       ['$y', '-$x+$y', '-$z'],
		       ['$x-$y', '$x', '-$z'],
		       ['-$y', '-$x', '$z+1/2'],
		       ['-$x+$y', '$y', '$z+1/2'],
		       ['$x', '$x-$y', '$z+1/2']],
     },
     'r -3 m' => {
	 number => 166,
	 schoenflies => 'd_3d^5',
	 full => 'r -3 2/m',
	 positions => [['$x', '$y', '$z'],
		       ['-$y', '$x-$y', '$z'],
		       ['-$x+$y', '-$x', '$z'],
		       ['$y', '$x', '-$z'],
		       ['$x-$y', '-$y', '-$z'],
		       ['-$x', '-$x+$y', '-$z'],
		       ['-$x', '-$y', '-$z'],
		       ['$y', '-$x+$y', '-$z'],
		       ['$x-$y', '$x', '-$z'],
		       ['-$y', '-$x', '$z'],
		       ['-$x+$y', '$y', '$z'],
		       ['$x', '$x-$y', '$z']],
	 rhombohedral => [['$x', '$y', '$z'],
			  ['$z', '$x', '$y'],
			  ['$y', '$z', '$x'],
			  ['-$y', '-$x', '-$z'],
			  ['-$x', '-$z', '-$y'],
			  ['-$z', '-$y', '-$x'],
			  ['-$x', '-$y', '-$z'],
			  ['-$z', '-$x', '-$y'],
			  ['-$y', '-$z', '-$x'],
			  ['$y', '$x', '$z'],
			  ['$x', '$z', '$y'],
			  ['$z', '$y', '$x']],
     },
     'r -3 c' => {
	 number => 167,
	 schoenflies => 'd_3d^6',
	 full => 'r -3 2/c',
	 positions => [['$x', '$y', '$z'],
		       ['-$y', '$x-$y', '$z'],
		       ['-$x+$y', '-$x', '$z'],
		       ['$y', '$x', '-$z+1/2'],
		       ['$x-$y', '-$y', '-$z+1/2'],
		       ['-$x', '-$x+$y', '-$z+1/2'],
		       ['-$x', '-$y', '-$z'],
		       ['$y', '-$x+$y', '-$z'],
		       ['$x-$y', '$x', '-$z'],
		       ['-$y', '-$x', '$z+1/2'],
		       ['-$x+$y', '$y', '$z+1/2'],
		       ['$x', '$x-$y', '$z+1/2']],
	 rhombohedral => [['$x', '$y', '$z'],
			  ['$z', '$x', '$y'],
			  ['$y', '$z', '$x'],
			  ['-$y+1/2', '-$x+1/2', '-$z+1/2'],
			  ['-$x+1/2', '-$z+1/2', '-$y+1/2'],
			  ['-$z+1/2', '-$y+1/2', '-$x+1/2'],
			  ['-$x', '-$y', '-$z'],
			  ['-$z', '-$x', '-$y'],
			  ['-$y', '-$z', '-$x'],
			  ['$y+1/2', '$x+1/2', '$z+1/2'],
			  ['$x+1/2', '$z+1/2', '$y+1/2'],
			  ['$z+1/2', '$y+1/2', '$x+1/2']],
     },

     ## hexagonal groups
     'p 6' => {
	 number => 168,
	 schoenflies => 'c_6^1',
	 positions => [['$x', '$y', '$z'],
		       ['-$y', '$x-$y', '$z'],
		       ['-$x+$y', '-$x', '$z'],
		       ['-$x', '-$y', '$z'],
		       ['$y', '-$x+$y', '$z'],
		       ['$x-$y', '$x', '$z']],
		       },
     'p 61' => {
	 number => 169,
	 schoenflies => 'c_6^2',
	 positions => [['$x', '$y', '$z'],
		       ['-$y', '$x-$y', '$z+1/3'],
		       ['-$x+$y', '-$x', '$z+2/3'],
		       ['-$x', '-$y', '$z+1/2'],
		       ['$y', '-$x+$y', '$z+5/6'],
		       ['$x-$y', '$x', '$z+1/6']],
     },
     'p 65' => {
	 number => 170,
	 schoenflies => 'c_6^3',
	 positions => [['$x', '$y', '$z'],
		       ['-$y', '$x-$y', '$z+2/3'],
		       ['-$x+$y', '-$x', '$z+1/3'],
		       ['-$x', '-$y', '$z+1/2'],
		       ['$y', '-$x+$y', '$z+1/6'],
		       ['$x-$y', '$x', '$z+5/6']],
     },
     'p 62' => {
	 number => 171,
	 schoenflies => 'c_6^4',
	 positions => [['$x', '$y', '$z'],
		       ['-$y', '$x-$y', '$z+2/3'],
		       ['-$x+$y', '-$x', '$z+1/3'],
		       ['-$x', '-$y', '$z'],
		       ['$y', '-$x+$y', '$z+2/3'],
		       ['$x-$y', '$x', '$z+1/3']],
     },
     'p 64' => {
	 number => 172,
	 schoenflies => 'c_6^5',
	 positions => [['$x', '$y', '$z'],
		       ['-$y', '$x-$y', '$z+1/3'],
		       ['-$x+$y', '-$x', '$z+2/3'],
		       ['-$x', '-$y', '$z'],
		       ['$y', '-$x+$y', '$z+1/3'],
		       ['$x-$y', '$x', '$z+2/3']],
     },
     'p 63' => {
	 number => 173,
	 schoenflies => 'c_6^6',
	 positions => [['$x', '$y', '$z'],
		       ['-$y', '$x-$y', '$z'],
		       ['-$x+$y', '-$x', '$z'],
		       ['-$x', '-$y', '$z+1/2'],
		       ['$y', '-$x+$y', '$z+1/2'],
		       ['$x-$y', '$x', '$z+1/2']],
     },
     'p -6' => {
	 number => 174,
	 schoenflies => 'c_3h^1',
	 positions => [['$x', '$y', '$z'],
		       ['-$y', '$x-$y', '$z'],
		       ['-$x+$y', '-$x', '$z'],
		       ['$x', '$y', '-$z'],
		       ['-$y', '$x-$y', '-$z'],
		       ['-$x+$y', '-$x', '-$z']],
     },
     'p 6/m' => {
	 number => 175,
	 schoenflies => 'c_6h^1',
	 positions => [['$x', '$y', '$z'],
		       ['-$y', '$x-$y', '$z'],
		       ['-$x+$y', '-$x', '$z'],
		       ['-$x', '-$y', '$z'],
		       ['$y', '-$x+$y', '$z'],
		       ['$x-$y', '$x', '$z'],
		       ['-$x', '-$y', '-$z'],
		       ['$y', '-$x+$y', '-$z'],
		       ['$x-$y', '$x', '-$z'],
		       ['$x', '$y', '-$z'],
		       ['-$y', '$x-$y', '-$z'],
		       ['-$x+$y', '-$x', '-$z']],
     },
     'p 63/m' => {
	 number => 176,
	 schoenflies => 'c_6h^2',
	 positions => [['$x', '$y', '$z'],
		       ['-$y', '$x-$y', '$z'],
		       ['-$x+$y', '-$x', '$z'],
		       ['-$x', '-$y', '$z+1/2'],
		       ['$y', '-$x+$y', '$z+1/2'],
		       ['$x-$y', '$x', '$z+1/2'],
		       ['-$x', '-$y', '-$z'],
		       ['$y', '-$x+$y', '-$z'],
		       ['$x-$y', '$x', '-$z'],
		       ['$x', '$y', '-$z+1/2'],
		       ['-$y', '$x-$y', '-$z+1/2'],
		       ['-$x+$y', '-$x', '-$z+1/2']],
     },
     'p 6 2 2' => {
	 number => 177,
	 schoenflies => 'd_6^1',
	 positions => [['$x', '$y', '$z'],
		       ['-$y', '$x-$y', '$z'],
		       ['-$x+$y', '-$x', '$z'],
		       ['-$x', '-$y', '$z'],
		       ['$y', '-$x+$y', '$z'],
		       ['$x-$y', '$x', '$z'],
		       ['$y', '$x', '-$z'],
		       ['$x-$y', '-$y', '-$z'],
		       ['-$x', '-$x+$y', '-$z'],
		       ['-$y', '-$x', '-$z'],
		       ['-$x+$y', '$y', '-$z'],
		       ['$x', '$x-$y', '-$z']],
     },
     'p 61 2 2' => {
	 number => 178,
	 schoenflies => 'd_6^2',
	 positions => [['$x', '$y', '$z'],
		       ['-$y', '$x-$y', '$z+1/3'],
		       ['-$x+$y', '-$x', '$z+2/3'],
		       ['-$x', '-$y', '$z+1/2'],
		       ['$y', '-$x+$y', '$z+5/6'],
		       ['$x-$y', '$x', '$z+1/6'],
		       ['$y', '$x', '-$z+1/3'],
		       ['$x-$y', '-$y', '-$z'],
		       ['-$x', '-$x+$y', '-$z+2/3'],
		       ['-$y', '-$x', '-$z+5/6'],
		       ['-$x+$y', '$y', '-$z+1/2'],
		       ['$x', '$x-$y', '-$z+1/6']],
     },
     'p 65 2 2' => {
	 number => 179,
	 schoenflies => 'd_6^3',
	 positions => [['$x', '$y', '$z'],
		       ['-$y', '$x-$y', '$z+2/3'],
		       ['-$x+$y', '-$x', '$z+1/3'],
		       ['-$x', '-$y', '$z+1/2'],
		       ['$y', '-$x+$y', '$z+1/6'],
		       ['$x-$y', '$x', '$z+5/6'],
		       ['$y', '$x', '-$z+2/3'],
		       ['$x-$y', '-$y', '-$z'],
		       ['-$x', '-$x+$y', '-$z+1/3'],
		       ['-$y', '-$x', '-$z+1/6'],
		       ['-$x+$y', '$y', '-$z+1/2'],
		       ['$x', '$x-$y', '-$z+5/6']],
     },
     'p 62 2 2' => {
	 number => 180,
	 schoenflies => 'd_6^4',
	 positions => [['$x', '$y', '$z'],
		       ['-$y', '$x-$y', '$z+2/3'],
		       ['-$x+$y', '-$x', '$z+1/3'],
		       ['-$x', '-$y', '$z'],
		       ['$y', '-$x+$y', '$z+2/3'],
		       ['$x-$y', '$x', '$z+1/3'],
		       ['$y', '$x', '-$z+2/3'],
		       ['$x-$y', '-$y', '-$z'],
		       ['-$x', '-$x+$y', '-$z+1/3'],
		       ['-$y', '-$x', '-$z+2/3'],
		       ['-$x+$y', '$y', '-$z'],
		       ['$x', '$x-$y', '-$z+1/3']],
     },
     'p 64 2 2' => {
	 number => 181,
	 schoenflies => 'd_6^5',
	 positions => [['$x', '$y', '$z'],
		       ['-$y', '$x-$y', '$z+1/3'],
		       ['-$x+$y', '-$x', '$z+2/3'],
		       ['-$x', '-$y', '$z'],
		       ['$y', '-$x+$y', '$z+1/3'],
		       ['$x-$y', '$x', '$z+2/3'],
		       ['$y', '$x', '-$z+1/3'],
		       ['$x-$y', '-$y', '-$z'],
		       ['-$x', '-$x+$y', '-$z+2/3'],
		       ['-$y', '-$x', '-$z+1/3'],
		       ['-$x+$y', '$y', '-$z'],
		       ['$x', '$x-$y', '-$z+2/3']],
     },
     'p 63 2 2' => {
	 number => 182,
	 schoenflies => 'd_6^6',
	 positions => [['$x', '$y', '$z'],
		       ['-$y', '$x-$y', '$z'],
		       ['-$x+$y', '-$x', '$z'],
		       ['-$x', '-$y', '$z+1/2'],
		       ['$y', '-$x+$y', '$z+1/2'],
		       ['$x-$y', '$x', '$z+1/2'],
		       ['$y', '$x', '-$z'],
		       ['$x-$y', '-$y', '-$z'],
		       ['-$x', '-$x+$y', '-$z'],
		       ['-$y', '-$x', '-$z+1/2'],
		       ['-$x+$y', '$y', '-$z+1/2'],
		       ['$x', '$x-$y', '-$z+1/2']],
     },
     'p 6 m m' => {
	 number => 183,
	 schoenflies => 'c_6v^1',
	 positions => [['$x', '$y', '$z'],
		       ['-$y', '$x-$y', '$z'],
		       ['-$x+$y', '-$x', '$z'],
		       ['-$x', '-$y', '$z'],
		       ['$y', '-$x+$y', '$z'],
		       ['$x-$y', '$x', '$z'],
		       ['-$y', '-$x', '$z'],
		       ['-$x+$y', '$y', '$z'],
		       ['$x', '$x-$y', '$z'],
		       ['$y', '$x', '$z'],
		       ['$x-$y', '-$y', '$z'],
		       ['-$x', '-$x+$y', '$z']],
     },
     'p 6 c c' => {
	 number => 184,
	 schoenflies => 'c_6v^2',
	 positions => [['$x', '$y', '$z'],
		       ['-$y', '$x-$y', '$z'],
		       ['-$x+$y', '-$x', '$z'],
		       ['-$x', '-$y', '$z'],
		       ['$y', '-$x+$y', '$z'],
		       ['$x-$y', '$x', '$z'],
		       ['-$y', '-$x', '$z+1/2'],
		       ['-$x+$y', '$y', '$z+1/2'],
		       ['$x', '$x-$y', '$z+1/2'],
		       ['$y', '$x', '$z+1/2'],
		       ['$x-$y', '-$y', '$z+1/2'],
		       ['-$x', '-$x+$y', '$z+1/2']],
     },
     'p 63 c m' => {
	 number => 185,
	 schoenflies => 'c_6v^3',
	 positions => [['$x', '$y', '$z'],
		       ['-$y', '$x-$y', '$z'],
		       ['-$x+$y', '-$x', '$z'],
		       ['-$x', '-$y', '$z+1/2'],
		       ['$y', '-$x+$y', '$z+1/2'],
		       ['$x-$y', '$x', '$z+1/2'],
		       ['-$y', '-$x', '$z+1/2'],
		       ['-$x+$y', '$y', '$z+1/2'],
		       ['$x', '$x-$y', '$z+1/2'],
		       ['$y', '$x', '$z'],
		       ['$x-$y', '-$y', '$z'],
		       ['-$x', '-$x+$y', '$z']],
     },
     'p 63 m c' => {
	 number => 186,
	 schoenflies => 'c_6v^4',
	 shorthand => ['graphite', 'gra'],
	 positions => [['$x', '$y', '$z'],
		       ['-$y', '$x-$y', '$z'],
		       ['-$x+$y', '-$x', '$z'],
		       ['-$x', '-$y', '$z+1/2'],
		       ['$y', '-$x+$y', '$z+1/2'],
		       ['$x-$y', '$x', '$z+1/2'],
		       ['-$y', '-$x', '$z'],
		       ['-$x+$y', '$y', '$z'],
		       ['$x', '$x-$y', '$z'],
		       ['$y', '$x', '$z+1/2'],
		       ['$x-$y', '-$y', '$z+1/2'],
		       ['-$x', '-$x+$y', '$z+1/2']],
     },
     'p -6 m 2' => {
	 number => 187,
	 schoenflies => 'd_3h^1',
	 positions => [['$x', '$y', '$z'],
		       ['-$y', '$x-$y', '$z'],
		       ['-$x+$y', '-$x', '$z'],
		       ['$x', '$y', '-$z'],
		       ['-$y', '$x-$y', '-$z'],
		       ['-$x+$y', '-$x', '-$z'],
		       ['-$y', '-$x', '$z'],
		       ['-$x+$y', '$y', '$z'],
		       ['$x', '$x-$y', '$z'],
		       ['-$y', '-$x', '-$z'],
		       ['-$x+$y', '$y', '-$z'],
		       ['$x', '$x-$y', '-$z']],
     },
     'p -6 c 2' => {
	 number => 188,
	 schoenflies => 'd_3h^2',
	 positions => [['$x', '$y', '$z'],
		       ['-$y', '$x-$y', '$z'],
		       ['-$x+$y', '-$x', '$z'],
		       ['$x', '$y', '-$z+1/2'],
		       ['-$y', '$x-$y', '-$z+1/2'],
		       ['-$x+$y', '-$x', '-$z+1/2'],
		       ['-$y', '-$x', '$z+1/2'],
		       ['-$x+$y', '$y', '$z+1/2'],
		       ['$x', '$x-$y', '$z+1/2'],
		       ['-$y', '-$x', '-$z'],
		       ['-$x+$y', '$y', '-$z'],
		       ['$x', '$x-$y', '-$z']],
     },
     'p -6 2 m' => {
	 number => 189,
	 schoenflies => 'd_3h^3',
	 positions => [['$x', '$y', '$z'],
		       ['-$y', '$x-$y', '$z'],
		       ['-$x+$y', '-$x', '$z'],
		       ['$x', '$y', '-$z'],
		       ['-$y', '$x-$y', '-$z'],
		       ['-$x+$y', '-$x', '-$z'],
		       ['$y', '$x', '-$z'],
		       ['$x-$y', '-$y', '-$z'],
		       ['-$x', '-$x+$y', '-$z'],
		       ['$y', '$x', '$z'],
		       ['$x-$y', '-$y', '$z'],
		       ['-$x', '-$x+$y', '$z']],
     },
     'p -6 2 c' => {
	 number => 190,
	 schoenflies => 'd_3h^4',
	 positions => [['$x', '$y', '$z'],
		       ['-$y', '$x-$y', '$z'],
		       ['-$x+$y', '-$x', '$z'],
		       ['$x', '$y', '-$z+1/2'],
		       ['-$y', '$x-$y', '-$z+1/2'],
		       ['-$x+$y', '-$x', '-$z+1/2'],
		       ['$y', '$x', '-$z'],
		       ['$x-$y', '-$y', '-$z'],
		       ['-$x', '-$x+$y', '-$z'],
		       ['$y', '$x', '$z+1/2'],
		       ['$x-$y', '-$y', '$z+1/2'],
		       ['-$x', '-$x+$y', '$z+1/2']],
     },
     'p 6/m m m' => {
	 number => 191,
	 schoenflies => 'd_6h^1',
	 full => 'p 6/m 2/m 2/m',
	 positions => [['$x', '$y', '$z'],
		       ['-$y', '$x-$y', '$z'],
		       ['-$x+$y', '-$x', '$z'],
		       ['-$x', '-$y', '$z'],
		       ['$y', '-$x+$y', '$z'],
		       ['$x-$y', '$x', '$z'],
		       ['$y', '$x', '-$z'],
		       ['$x-$y', '-$y', '-$z'],
		       ['-$x', '-$x+$y', '-$z'],
		       ['-$y', '-$x', '-$z'],
		       ['-$x+$y', '$y', '-$z'],
		       ['$x', '$x-$y', '-$z'],
		       ['-$x', '-$y', '-$z'],
		       ['$y', '-$x+$y', '-$z'],
		       ['$x-$y', '$x', '-$z'],
		       ['$x', '$y', '-$z'],
		       ['-$y', '$x-$y', '-$z'],
		       ['-$x+$y', '-$x', '-$z'],
		       ['-$y', '-$x', '$z'],
		       ['-$x+$y', '$y', '$z'],
		       ['$x', '$x-$y', '$z'],
		       ['$y', '$x', '$z'],
		       ['$x-$y', '-$y', '$z'],
		       ['-$x', '-$x+$y', '$z']],
		       },
     'p 6/m c c' => {
	 number => 192,
	 schoenflies => 'd_6h^2',
	 full => 'p 6/m 2/c 2/c',
	 positions => [['$x', '$y', '$z'],
		       ['-$y', '$x-$y', '$z'],
		       ['-$x+$y', '-$x', '$z'],
		       ['-$x', '-$y', '$z'],
		       ['$y', '-$x+$y', '$z'],
		       ['$x-$y', '$x', '$z'],
		       ['$y', '$x', '-$z+1/2'],
		       ['$x-$y', '-$y', '-$z+1/2'],
		       ['-$x', '-$x+$y', '-$z+1/2'],
		       ['-$y', '-$x', '-$z+1/2'],
		       ['-$x+$y', '$y', '-$z+1/2'],
		       ['$x', '$x-$y', '-$z+1/2'],
		       ['-$x', '-$y', '-$z'],
		       ['$y', '-$x+$y', '-$z'],
		       ['$x-$y', '$x', '-$z'],
		       ['$x', '$y', '-$z'],
		       ['-$y', '$x-$y', '-$z'],
		       ['-$x+$y', '-$x', '-$z'],
		       ['-$y', '-$x', '$z+1/2'],
		       ['-$x+$y', '$y', '$z+1/2'],
		       ['$x', '$x-$y', '$z+1/2'],
		       ['$y', '$x', '$z+1/2'],
		       ['$x-$y', '-$y', '$z+1/2'],
		       ['-$x', '-$x+$y', '$z+1/2']],
		       },
     'p 63/m c m' => {
	 number => 193,
	 schoenflies => 'd_6h^3',
	 full => 'p 63/m 2/c 2/m',
	 positions => [['$x', '$y', '$z'],
		       ['-$y', '$x-$y', '$z'],
		       ['-$x+$y', '-$x', '$z'],
		       ['-$x', '-$y', '$z+1/2'],
		       ['$y', '-$x+$y', '$z+1/2'],
		       ['$x-$y', '$x', '$z+1/2'],
		       ['$y', '$x', '-$z+1/2'],
		       ['$x-$y', '-$y', '-$z+1/2'],
		       ['-$x', '-$x+$y', '-$z+1/2'],
		       ['-$y', '-$x', '-$z'],
		       ['-$x+$y', '$y', '-$z'],
		       ['$x', '$x-$y', '-$z'],
		       ['-$x', '-$y', '-$z'],
		       ['$y', '-$x+$y', '-$z'],
		       ['$x-$y', '$x', '-$z'],
		       ['$x', '$y', '-$z+1/2'],
		       ['-$y', '$x-$y', '-$z+1/2'],
		       ['-$x+$y', '-$x', '-$z+1/2'],
		       ['-$y', '-$x', '$z+1/2'],
		       ['-$x+$y', '$y', '$z+1/2'],
		       ['$x', '$x-$y', '$z+1/2'],
		       ['$y', '$x', '$z'],
		       ['$x-$y', '-$y', '$z'],
		       ['-$x', '-$x+$y', '$z']],
		       },
     'p 63/m m c' => {
	 number => 194,
	 schoenflies => 'd_6h^4',
	 full => 'p 63/m 2/m 2/c',
	 shorthand => ['hex', 'hcp'],
	 positions => [['$x', '$y', '$z'],
		       ['-$y', '$x-$y', '$z'],
		       ['-$x+$y', '-$x', '$z'],
		       ['-$x', '-$y', '$z+1/2'],
		       ['$y', '-$x+$y', '$z+1/2'],
		       ['$x-$y', '$x', '$z+1/2'],
		       ['$y', '$x', '-$z'],
		       ['$x-$y', '-$y', '-$z'],
		       ['-$x', '-$x+$y', '-$z'],
		       ['-$y', '-$x', '-$z+1/2'],
		       ['-$x+$y', '$y', '-$z+1/2'],
		       ['$x', '$x-$y', '-$z+1/2'],
		       ['-$x', '-$y', '-$z'],
		       ['$y', '-$x+$y', '-$z'],
		       ['$x-$y', '$x', '-$z'],
		       ['$x', '$y', '-$z+1/2'],
		       ['-$y', '$x-$y', '-$z+1/2'],
		       ['-$x+$y', '-$x', '-$z+1/2'],
		       ['-$y', '-$x', '$z'],
		       ['-$x+$y', '$y', '$z'],
		       ['$x', '$x-$y', '$z'],
		       ['$y', '$x', '$z+1/2'],
		       ['$x-$y', '-$y', '$z+1/2'],
		       ['-$x', '-$x+$y', '$z+1/2']],
		       },

     ## cubic groups
     'p 2 3' => {
	 number => 195,
	 schoenflies => 't^1',
	 positions => [['$x', '$y', '$z'],
		       ['-$x', '-$y', '$z'],
		       ['-$x', '$y', '-$z'],
		       ['$x', '-$y', '-$z'],
		       ['$z', '$x', '$y'],
		       ['$z', '-$x', '-$y'],
		       ['-$z', '-$x', '$y'],
		       ['-$z', '$x', '-$y'],
		       ['$y', '$z', '$x'],
		       ['-$y', '$z', '-$x'],
		       ['$y', '-$z', '-$x'],
		       ['-$y', '-$z', '$x']],
     },
     'f 2 3' => {
	 number => 196,
	 schoenflies => 't^2',
	 positions => [['$x', '$y', '$z'],
		       ['-$x', '-$y', '$z'],
		       ['-$x', '$y', '-$z'],
		       ['$x', '-$y', '-$z'],
		       ['$z', '$x', '$y'],
		       ['$z', '-$x', '-$y'],
		       ['-$z', '-$x', '$y'],
		       ['-$z', '$x', '-$y'],
		       ['$y', '$z', '$x'],
		       ['-$y', '$z', '-$x'],
		       ['$y', '-$z', '-$x'],
		       ['-$y', '-$z', '$x']],
     },
     'i 2 3' => {
	 number => 197,
	 schoenflies => 't^3',
	 positions => [['$x', '$y', '$z'],
		       ['-$x', '-$y', '$z'],
		       ['-$x', '$y', '-$z'],
		       ['$x', '-$y', '-$z'],
		       ['$z', '$x', '$y'],
		       ['$z', '-$x', '-$y'],
		       ['-$z', '-$x', '$y'],
		       ['-$z', '$x', '-$y'],
		       ['$y', '$z', '$x'],
		       ['-$y', '$z', '-$x'],
		       ['$y', '-$z', '-$x'],
		       ['-$y', '-$z', '$x']],
     },
     'p 21 3' => {
	 number => 198,
	 schoenflies => 't^4',
	 positions => [['$x', '$y', '$z'],
		       ['-$x+1/2', '-$y', '$z+1/2'],
		       ['-$x', '$y+1/2', '-$z+1/2'],
		       ['$x+1/2', '-$y+1/2', '-$z'],
		       ['$z', '$x', '$y'],
		       ['$z+1/2', '-$x+1/2', '-$y'],
		       ['-$z+1/2', '-$x', '$y+1/2'],
		       ['-$z', '$x+1/2', '-$y+1/2'],
		       ['$y', '$z', '$x'],
		       ['-$y', '$z+1/2', '-$x+1/2'],
		       ['$y+1/2', '-$z+1/2', '-$x'],
		       ['-$y+1/2', '-$z', '$x+1/2']],
     },
     'i 21 3' => {
	 number => 199,
	 schoenflies => 't^5',
	 positions => [['$x', '$y', '$z'],
		       ['-$x+1/2', '-$y', '$z+1/2'],
		       ['-$x', '$y+1/2', '-$z+1/2'],
		       ['$x+1/2', '-$y+1/2', '-$z'],
		       ['$z', '$x', '$y'],
		       ['$z+1/2', '-$x+1/2', '-$y'],
		       ['-$z+1/2', '-$x', '$y+1/2'],
		       ['-$z', '$x+1/2', '-$y+1/2'],
		       ['$y', '$z', '$x'],
		       ['-$y', '$z+1/2', '-$x+1/2'],
		       ['$y+1/2', '-$z+1/2', '-$x'],
		       ['-$y+1/2', '-$z', '$x+1/2']],
     },
     'p m -3' => {
	 number => 200,
	 schoenflies => 't_h^1',
	 full => 'p 2/m -3',
	 thirtyfive => 'p m 3',
	 positions => [['$x', '$y', '$z'],
		       ['-$x', '-$y', '$z'],
		       ['-$x', '$y', '-$z'],
		       ['$x', '-$y', '-$z'],
		       ['$z', '$x', '$y'],
		       ['$z', '-$x', '-$y'],
		       ['-$z', '-$x', '$y'],
		       ['-$z', '$x', '-$y'],
		       ['$y', '$z', '$x'],
		       ['-$y', '$z', '-$x'],
		       ['$y', '-$z', '-$x'],
		       ['-$y', '-$z', '$x'],
		       ['-$x', '-$y', '-$z'],
		       ['$x', '$y', '-$z'],
		       ['$x', '-$y', '$z'],
		       ['-$x', '$y', '$z'],
		       ['-$z', '-$x', '-$y'],
		       ['-$z', '$x', '$y'],
		       ['$z', '$x', '-$y'],
		       ['$z', '-$x', '$y'],
		       ['-$y', '-$z', '-$x'],
		       ['$y', '-$z', '$x'],
		       ['-$y', '$z', '$x'],
		       ['$y', '$z', '-$x']],
		       },
     'p n -3' => {
	 number => 201,
	 schoenflies => 't_h^3',
	 full => 'p 2/n -3',
	 thirtyfive => 'p n 3',
	 shiftvec => [ -1/4, -1/4, -1/4 ],
	 positions => [['$x', '$y', '$z'],
		       ['-$x+1/2', '-$y+1/2', '$z'],
		       ['-$x+1/2', '$y', '-$z+1/2'],
		       ['$x', '-$y+1/2', '-$z+1/2'],
		       ['$z', '$x', '$y'],
		       ['$z', '-$x+1/2', '-$y+1/2'],
		       ['-$z+1/2', '-$x+1/2', '$y'],
		       ['-$z+1/2', '$x', '-$y+1/2'],
		       ['$y', '$z', '$x'],
		       ['-$y+1/2', '$z', '-$x+1/2'],
		       ['$y', '-$z+1/2', '-$x+1/2'],
		       ['-$y+1/2', '-$z+1/2', '$x'],
		       ['-$x', '-$y', '-$z'],
		       ['$x+1/2', '$y+1/2', '-$z'],
		       ['$x+1/2', '-$y', '$z+1/2'],
		       ['-$x', '$y+1/2', '$z+1/2'],
		       ['-$z', '-$x', '-$y'],
		       ['-$z', '$x+1/2', '$y+1/2'],
		       ['$z+1/2', '$x+1/2', '-$y'],
		       ['$z+1/2', '-$x', '$y+1/2'],
		       ['-$y', '-$z', '-$x'],
		       ['$y+1/2', '-$z', '$x+1/2'],
		       ['-$y', '$z+1/2', '$x+1/2'],
		       ['$y+1/2', '$z+1/2', '-$x']],
		       },
     'f m -3' => {
	 number => 202,
	 schoenflies => 't_h^3',
	 full => 'f 2/m -3',
	 thirtyfive => 'f m 3',
	 positions => [['$x', '$y', '$z'],
		       ['-$x', '-$y', '$z'],
		       ['-$x', '$y', '-$z'],
		       ['$x', '-$y', '-$z'],
		       ['$z', '$x', '$y'],
		       ['$z', '-$x', '-$y'],
		       ['-$z', '-$x', '$y'],
		       ['-$z', '$x', '-$y'],
		       ['$y', '$z', '$x'],
		       ['-$y', '$z', '-$x'],
		       ['$y', '-$z', '-$x'],
		       ['-$y', '-$z', '$x'],
		       ['-$x', '-$y', '-$z'],
		       ['$x', '$y', '-$z'],
		       ['$x', '-$y', '$z'],
		       ['-$x', '$y', '$z'],
		       ['-$z', '-$x', '-$y'],
		       ['-$z', '$x', '$y'],
		       ['$z', '$x', '-$y'],
		       ['$z', '-$x', '$y'],
		       ['-$y', '-$z', '-$x'],
		       ['$y', '-$z', '$x'],
		       ['-$y', '$z', '$x'],
		       ['$y', '$z', '-$x']],
		       },
     'f d -3' => {
	 number => 203,
	 schoenflies => 't_h^4',
	 full => 'p 2/d -3',
	 thirtyfive => 'f d 3',
	 shiftvec => [ -1/8, -1/8, -1/8 ],
	 positions => [['$x', '$y', '$z'],
		       ['-$x+1/4', '-$y+1/4', '$z'],
		       ['-$x+1/4', '$y', '-$z+1/4'],
		       ['$x', '-$y+1/4', '-$z+1/4'],
		       ['$z', '$x', '$y'],
		       ['$z', '-$x+1/4', '-$y+1/4'],
		       ['-$z+1/4', '-$x+1/4', '$y'],
		       ['-$z+1/4', '$x', '-$y+1/4'],
		       ['$y', '$z', '$x'],
		       ['-$y+1/4', '$z', '-$x+1/4'],
		       ['$y', '-$z+1/4', '-$x+1/4'],
		       ['-$y+1/4', '-$z+1/4', '$x'],
		       ['-$x', '-$y', '-$z'],
		       ['$x+3/4', '$y+3/4', '-$z'],
		       ['$x+3/4', '-$y', '$z+3/4'],
		       ['-$x', '$y+3/4', '$z+3/4'],
		       ['-$z', '-$x', '-$y'],
		       ['-$z', '$x+3/4', '$y+3/4'],
		       ['$z+3/4', '$x+3/4', '-$y'],
		       ['$z+3/4', '-$x', '$y+3/4'],
		       ['-$y', '-$z', '-$x'],
		       ['$y+3/4', '-$z', '$x+3/4'],
		       ['-$y', '$z+3/4', '$x+3/4'],
		       ['$y+3/4', '$z+3/4', '-$x']],
		       },
     'i m -3' => {
	 number => 204,
	 schoenflies => 't_h^5',
	 full => 'i 2/m -3',
	 thirtyfive => 'i m 3',
	 positions => [['$x', '$y', '$z'],
		       ['-$x', '-$y', '$z'],
		       ['-$x', '$y', '-$z'],
		       ['$x', '-$y', '-$z'],
		       ['$z', '$x', '$y'],
		       ['$z', '-$x', '-$y'],
		       ['-$z', '-$x', '$y'],
		       ['-$z', '$x', '-$y'],
		       ['$y', '$z', '$x'],
		       ['-$y', '$z', '-$x'],
		       ['$y', '-$z', '-$x'],
		       ['-$y', '-$z', '$x'],
		       ['-$x', '-$y', '-$z'],
		       ['$x', '$y', '-$z'],
		       ['$x', '-$y', '$z'],
		       ['-$x', '$y', '$z'],
		       ['-$z', '-$x', '-$y'],
		       ['-$z', '$x', '$y'],
		       ['$z', '$x', '-$y'],
		       ['$z', '-$x', '$y'],
		       ['-$y', '-$z', '-$x'],
		       ['$y', '-$z', '$x'],
		       ['-$y', '$z', '$x'],
		       ['$y', '$z', '-$x']],
		       },
     'p a -3' => {
	 number => 205,
	 schoenflies => 't_h^6',
	 full => 'p 2/a -3',
	 thirtyfive => 'p a 3',
	 positions => [['$x', '$y', '$z'],
		       ['-$x+1/2', '-$y', '$z+1/2'],
		       ['-$x', '$y+1/2', '-$z+1/2'],
		       ['$x+1/2', '-$y+1/2', '-$z'],
		       ['$z', '$x', '$y'],
		       ['$z+1/2', '-$x+1/2', '-$y'],
		       ['-$z+1/2', '-$x', '$y+1/2'],
		       ['-$z', '$x+1/2', '-$y+1/2'],
		       ['$y', '$z', '$x'],
		       ['-$y', '$z+1/2', '-$x+1/2'],
		       ['$y+1/2', '-$z+1/2', '-$x'],
		       ['-$y+1/2', '-$z', '$x+1/2'],
		       ['-$x', '-$y', '-$z'],
		       ['$x+1/2', '$y', '-$z+1/2'],
		       ['$x', '-$y+1/2', '$z+1/2'],
		       ['-$x+1/2', '$y+1/2', '$z'],
		       ['-$z', '-$x', '-$y'],
		       ['-$z+1/2', '$x+1/2', '$y'],
		       ['$z+1/2', '$x', '-$y+1/2'],
		       ['$z', '-$x+1/2', '$y+1/2'],
		       ['-$y', '-$z', '-$x'],
		       ['$y', '-$z+1/2', '$x+1/2'],
		       ['-$y+1/2', '$z+1/2', '$x'],
		       ['$y+1/2', '$z', '-$x+1/2']],
		       },
     'i a -3' => {
	 number => 206,
	 schoenflies => 't_h^7',
	 full => 'i 2/a -3',
	 thirtyfive => 'i a 3',
	 positions => [['$x', '$y', '$z'],
		       ['-$x+1/2', '-$y', '$z+1/2'],
		       ['-$x', '$y+1/2', '-$z+1/2'],
		       ['$x+1/2', '-$y+1/2', '-$z'],
		       ['$z', '$x', '$y'],
		       ['$z+1/2', '-$x+1/2', '-$y'],
		       ['-$z+1/2', '-$x', '$y+1/2'],
		       ['-$z', '$x+1/2', '-$y+1/2'],
		       ['$y', '$z', '$x'],
		       ['-$y', '$z+1/2', '-$x+1/2'],
		       ['$y+1/2', '-$z+1/2', '-$x'],
		       ['-$y+1/2', '-$z', '$x+1/2'],
		       ['-$x', '-$y', '-$z'],
		       ['$x+1/2', '$y', '-$z+1/2'],
		       ['$x', '-$y+1/2', '$z+1/2'],
		       ['-$x+1/2', '$y+1/2', '$z'],
		       ['-$z', '-$x', '-$y'],
		       ['-$z+1/2', '$x+1/2', '$y'],
		       ['$z+1/2', '$x', '-$y+1/2'],
		       ['$z', '-$x+1/2', '$y+1/2'],
		       ['-$y', '-$z', '-$x'],
		       ['$y', '-$z+1/2', '$x+1/2'],
		       ['-$y+1/2', '$z+1/2', '$x'],
		       ['$y+1/2', '$z', '-$x+1/2']],
     },
     'p 4 3 2' => {
	 number => 207,
	 schoenflies => 'o^1',
	 thirtyfive => 'p 4 3',
	 positions => [['$x', '$y', '$z'],
		       ['-$x', '-$y', '$z'],
		       ['-$x', '$y', '-$z'],
		       ['$x', '-$y', '-$z'],
		       ['$z', '$x', '$y'],
		       ['$z', '-$x', '-$y'],
		       ['-$z', '-$x', '$y'],
		       ['-$z', '$x', '-$y'],
		       ['$y', '$z', '$x'],
		       ['-$y', '$z', '-$x'],
		       ['$y', '-$z', '-$x'],
		       ['-$y', '-$z', '$x'],
		       ['$y', '$x', '-$z'],
		       ['-$y', '-$x', '-$z'],
		       ['$y', '-$x', '$z'],
		       ['-$y', '$x', '$z'],
		       ['$x', '$z', '-$y'],
		       ['-$x', '$z', '$y'],
		       ['-$x', '-$z', '-$y'],
		       ['$x', '-$z', '$y'],
		       ['$z', '$y', '-$x'],
		       ['$z', '-$y', '$x'],
		       ['-$z', '$y', '$x'],
		       ['-$z', '-$y', '-$x']],
     },
     'p 42 3 2' => {
	 number => 208,
	 schoenflies => 'o^2',
	 thirtyfive => 'p 42 3',
	 positions => [['$x', '$y', '$z'],
		       ['-$x', '-$y', '$z'],
		       ['-$x', '$y', '-$z'],
		       ['$x', '-$y', '-$z'],
		       ['$z', '$x', '$y'],
		       ['$z', '-$x', '-$y'],
		       ['-$z', '-$x', '$y'],
		       ['-$z', '$x', '-$y'],
		       ['$y', '$z', '$x'],
		       ['-$y', '$z', '-$x'],
		       ['$y', '-$z', '-$x'],
		       ['-$y', '-$z', '$x'],
		       ['$y+1/2', '$x+1/2', '-$z+1/2'],
		       ['-$y+1/2', '-$x+1/2', '-$z+1/2'],
		       ['$y+1/2', '-$x+1/2', '$z+1/2'],
		       ['-$y+1/2', '$x+1/2', '$z+1/2'],
		       ['$x+1/2', '$z+1/2', '-$y+1/2'],
		       ['-$x+1/2', '$z+1/2', '$y+1/2'],
		       ['-$x+1/2', '-$z+1/2', '-$y+1/2'],
		       ['$x+1/2', '-$z+1/2', '$y+1/2'],
		       ['$z+1/2', '$y+1/2', '-$x+1/2'],
		       ['$z+1/2', '-$y+1/2', '$x+1/2'],
		       ['-$z+1/2', '$y+1/2', '$x+1/2'],
		       ['-$z+1/2', '-$y+1/2', '-$x+1/2']],
     },
     'f 4 3 2' => {
	 number => 209,
	 schoenflies => 'o^3',
	 thirtyfive => 'f 4 3',
	 positions => [['$x', '$y', '$z'],
		       ['-$x', '-$y', '$z'],
		       ['-$x', '$y', '-$z'],
		       ['$x', '-$y', '-$z'],
		       ['$z', '$x', '$y'],
		       ['$z', '-$x', '-$y'],
		       ['-$z', '-$x', '$y'],
		       ['-$z', '$x', '-$y'],
		       ['$y', '$z', '$x'],
		       ['-$y', '$z', '-$x'],
		       ['$y', '-$z', '-$x'],
		       ['-$y', '-$z', '$x'],
		       ['$y', '$x', '-$z'],
		       ['-$y', '-$x', '-$z'],
		       ['$y', '-$x', '$z'],
		       ['-$y', '$x', '$z'],
		       ['$x', '$z', '-$y'],
		       ['-$x', '$z', '$y'],
		       ['-$x', '-$z', '-$y'],
		       ['$x', '-$z', '$y'],
		       ['$z', '$y', '-$x'],
		       ['$z', '-$y', '$x'],
		       ['-$z', '$y', '$x'],
		       ['-$z', '-$y', '-$x']],
     },
     'f 41 3 2' => {
	 number => 210,
	 schoenflies => 'o^4',
	 thirtyfive => 'f 41 3',
	 positions => [['$x', '$y', '$z'],
		       ['-$x', '-$y+1/2', '$z+1/2'],
		       ['-$x+1/2', '$y+1/2', '-$z'],
		       ['$x+1/2', '-$y', '-$z+1/2'],
		       ['$z', '$x', '$y'],
		       ['$z+1/2', '-$x', '-$y+1/2'],
		       ['-$z', '-$x+1/2', '$y+1/2'],
		       ['-$z+1/2', '$x+1/2', '-$y'],
		       ['$y', '$z', '$x'],
		       ['-$y+1/2', '$z+1/2', '-$x'],
		       ['$y+1/2', '-$z', '-$x+1/2'],
		       ['-$y', '-$z+1/2', '$x+1/2'],
		       ['$y+3/4', '$x+1/4', '-$z+3/4'],
		       ['-$y+1/4', '-$x+1/4', '-$z+1/4'],
		       ['$y+1/4', '-$x+3/4', '$z+3/4'],
		       ['-$y+3/4', '$x+3/4', '$z+1/4'],
		       ['$x+3/4', '$z+1/4', '-$y+3/4'],
		       ['-$x+3/4', '$z+3/4', '$y+1/4'],
		       ['-$x+1/4', '-$z+1/4', '-$y+1/4'],
		       ['$x+1/4', '-$z+3/4', '$y+3/4'],
		       ['$z+3/4', '$y+1/4', '-$x+3/4'],
		       ['$z+1/4', '-$y+3/4', '$x+3/4'],
		       ['-$z+3/4', '$y+3/4', '$x+1/4'],
		       ['-$z+1/4', '-$y+1/4', '-$x+1/4']],
     },
     'i 4 3 2' => {
	 number => 211,
	 schoenflies => 'o^5',
	 thirtyfive => 'i 4 3',
	 positions => [['$x', '$y', '$z'],
		       ['-$x', '-$y', '$z'],
		       ['-$x', '$y', '-$z'],
		       ['$x', '-$y', '-$z'],
		       ['$z', '$x', '$y'],
		       ['$z', '-$x', '-$y'],
		       ['-$z', '-$x', '$y'],
		       ['-$z', '$x', '-$y'],
		       ['$y', '$z', '$x'],
		       ['-$y', '$z', '-$x'],
		       ['$y', '-$z', '-$x'],
		       ['-$y', '-$z', '$x'],
		       ['$y', '$x', '-$z'],
		       ['-$y', '-$x', '-$z'],
		       ['$y', '-$x', '$z'],
		       ['-$y', '$x', '$z'],
		       ['$x', '$z', '-$y'],
		       ['-$x', '$z', '$y'],
		       ['-$x', '-$z', '-$y'],
		       ['$x', '-$z', '$y'],
		       ['$z', '$y', '-$x'],
		       ['$z', '-$y', '$x'],
		       ['-$z', '$y', '$x'],
		       ['-$z', '-$y', '-$x']],
     },
     'p 43 3 2' => {
	 number => 212,
	 schoenflies => 'o^6',
	 thirtyfive => 'p 43 3',
	 positions => [['$x', '$y', '$z'],
		       ['-$x+1/2', '-$y', '$z+1/2'],
		       ['-$x', '$y+1/2', '-$z+1/2'],
		       ['$x+1/2', '-$y+1/2', '-$z'],
		       ['$z', '$x', '$y'],
		       ['$z+1/2', '-$x+1/2', '-$y'],
		       ['-$z+1/2', '-$x', '$y+1/2'],
		       ['-$z', '$x+1/2', '-$y+1/2'],
		       ['$y', '$z', '$x'],
		       ['-$y', '$z+1/2', '-$x+1/2'],
		       ['$y+1/2', '-$z+1/2', '-$x'],
		       ['-$y+1/2', '-$z', '$x+1/2'],
		       ['$y+1/4', '$x+3/4', '-$z+3/4'],
		       ['-$y+1/4', '-$x+1/4', '-$z+1/4'],
		       ['$y+3/4', '-$x+3/4', '$z+1/4'],
		       ['-$y+3/4', '$x+1/4', '$z+3/4'],
		       ['$x+1/4', '$z+3/4', '-$y+3/4'],
		       ['-$x+3/4', '$z+1/4', '$y+3/4'],
		       ['-$x+1/4', '-$z+1/4', '-$y+1/4'],
		       ['$x+3/4', '-$z+3/4', '$y+1/4'],
		       ['$z+1/4', '$y+3/4', '-$x+3/4'],
		       ['$z+3/4', '-$y+3/4', '$x+1/4'],
		       ['-$z+3/4', '$y+1/4', '$x+3/4'],
		       ['-$z+1/4', '-$y+1/4', '-$x+1/4']],
     },
     'p 41 3 2' => {
	 number => 213,
	 schoenflies => 'o^7',
	 thirtyfive => 'p 41 3',
	 positions => [['$x', '$y', '$z'],
		       ['-$x+1/2', '-$y', '$z+1/2'],
		       ['-$x', '$y+1/2', '-$z+1/2'],
		       ['$x+1/2', '-$y+1/2', '-$z'],
		       ['$z', '$x', '$y'],
		       ['$z+1/2', '-$x+1/2', '-$y'],
		       ['-$z+1/2', '-$x', '$y+1/2'],
		       ['-$z', '$x+1/2', '-$y+1/2'],
		       ['$y', '$z', '$x'],
		       ['-$y', '$z+1/2', '-$x+1/2'],
		       ['$y+1/2', '-$z+1/2', '-$x'],
		       ['-$y+1/2', '-$z', '$x+1/2'],
		       ['$y+3/4', '$x+1/4', '-$z+1/4'],
		       ['-$y+1/4', '-$x+1/4', '-$z+1/4'],
		       ['$y+1/4', '-$x+1/4', '$z+3/4'],
		       ['-$y+1/4', '$x+3/4', '$z+1/4'],
		       ['$x+3/4', '$z+1/4', '-$y+1/4'],
		       ['-$x+1/4', '$z+3/4', '$y+1/4'],
		       ['-$x+3/4', '-$z+3/4', '-$y+3/4'],
		       ['$x+1/4', '-$z+1/4', '$y+3/4'],
		       ['$z+3/4', '$y+1/4', '-$x+1/4'],
		       ['$z+1/4', '-$y+1/4', '$x+3/4'],
		       ['-$z+1/4', '$y+3/4', '$x+1/4'],
		       ['-$z+3/4', '-$y+3/4', '-$x+3/4']],
     },
     'i 41 3 2' => {
	 number => 214,
	 schoenflies => 'o^8',
	 thirtyfive => 'i 41 3',
	 positions => [['$x', '$y', '$z'],
		       ['-$x+1/2', '-$y', '$z+1/2'],
		       ['-$x', '$y+1/2', '-$z+1/2'],
		       ['$x+1/2', '-$y+1/2', '-$z'],
		       ['$z', '$x', '$y'],
		       ['$z+1/2', '-$x+1/2', '-$y'],
		       ['-$z+1/2', '-$x', '$y+1/2'],
		       ['-$z', '$x+1/2', '-$y+1/2'],
		       ['$y', '$z', '$x'],
		       ['-$y', '$z+1/2', '-$x+1/2'],
		       ['$y+1/2', '-$z+1/2', '-$x'],
		       ['-$y+1/2', '-$z', '$x+1/2'],
		       ['$y+3/4', '$x+1/4', '-$z+1/4'],
		       ['-$y+1/4', '-$x+1/4', '-$z+1/4'],
		       ['$y+1/4', '-$x+1/4', '$z+3/4'],
		       ['-$y+1/4', '$x+3/4', '$z+1/4'],
		       ['$x+3/4', '$z+1/4', '-$y+1/4'],
		       ['-$x+1/4', '$z+3/4', '$y+1/4'],
		       ['-$x+3/4', '-$z+3/4', '-$y+3/4'],
		       ['$x+1/4', '-$z+1/4', '$y+3/4'],
		       ['$z+3/4', '$y+1/4', '-$x+1/4'],
		       ['$z+1/4', '-$y+1/4', '$x+3/4'],
		       ['-$z+1/4', '$y+3/4', '$x+1/4'],
		       ['-$z+3/4', '-$y+3/4', '-$x+3/4']],
     },
     'p -4 3 m' => {
	 number => 215,
	 schoenflies => 't_d^1',
	 positions => [['$x', '$y', '$z'],
		       ['-$x', '-$y', '$z'],
		       ['-$x', '$y', '-$z'],
		       ['$x', '-$y', '-$z'],
		       ['$z', '$x', '$y'],
		       ['$z', '-$x', '-$y'],
		       ['-$z', '-$x', '$y'],
		       ['-$z', '$x', '-$y'],
		       ['$y', '$z', '$x'],
		       ['-$y', '$z', '-$x'],
		       ['$y', '-$z', '-$x'],
		       ['-$y', '-$z', '$x'],
		       ['$y', '$x', '$z'],
		       ['-$y', '-$x', '$z'],
		       ['$y', '-$x', '-$z'],
		       ['-$y', '$x', '-$z'],
		       ['$x', '$z', '$y'],
		       ['-$x', '$z', '-$y'],
		       ['-$x', '-$z', '$y'],
		       ['$x', '-$z', '-$y'],
		       ['$z', '$y', '$x'],
		       ['$z', '-$y', '-$x'],
		       ['-$z', '$y', '-$x'],
		       ['-$z', '-$y', '$x']],
     },
     'f -4 3 m' => {
	 number => 216,
	 schoenflies => 't_d^2',
	 shorthand => ['zincblende', 'zns'],
	 positions => [['$x', '$y', '$z'],
		       ['-$x', '-$y', '$z'],
		       ['-$x', '$y', '-$z'],
		       ['$x', '-$y', '-$z'],
		       ['$z', '$x', '$y'],
		       ['$z', '-$x', '-$y'],
		       ['-$z', '-$x', '$y'],
		       ['-$z', '$x', '-$y'],
		       ['$y', '$z', '$x'],
		       ['-$y', '$z', '-$x'],
		       ['$y', '-$z', '-$x'],
		       ['-$y', '-$z', '$x'],
		       ['$y', '$x', '$z'],
		       ['-$y', '-$x', '$z'],
		       ['$y', '-$x', '-$z'],
		       ['-$y', '$x', '-$z'],
		       ['$x', '$z', '$y'],
		       ['-$x', '$z', '-$y'],
		       ['-$x', '-$z', '$y'],
		       ['$x', '-$z', '-$y'],
		       ['$z', '$y', '$x'],
		       ['$z', '-$y', '-$x'],
		       ['-$z', '$y', '-$x'],
		       ['-$z', '-$y', '$x']],
     },
     'i -4 3 m' => {
	 number => 217,
	 schoenflies => 't_d^3',
	 positions => [['$x', '$y', '$z'],
		       ['-$x', '-$y', '$z'],
		       ['-$x', '$y', '-$z'],
		       ['$x', '-$y', '-$z'],
		       ['$z', '$x', '$y'],
		       ['$z', '-$x', '-$y'],
		       ['-$z', '-$x', '$y'],
		       ['-$z', '$x', '-$y'],
		       ['$y', '$z', '$x'],
		       ['-$y', '$z', '-$x'],
		       ['$y', '-$z', '-$x'],
		       ['-$y', '-$z', '$x'],
		       ['$y', '$x', '$z'],
		       ['-$y', '-$x', '$z'],
		       ['$y', '-$x', '-$z'],
		       ['-$y', '$x', '-$z'],
		       ['$x', '$z', '$y'],
		       ['-$x', '$z', '-$y'],
		       ['-$x', '-$z', '$y'],
		       ['$x', '-$z', '-$y'],
		       ['$z', '$y', '$x'],
		       ['$z', '-$y', '-$x'],
		       ['-$z', '$y', '-$x'],
		       ['-$z', '-$y', '$x']],
     },
     'p -4 3 n' => {
	 number => 218,
	 schoenflies => 't_d^4',
	 positions => [['$x', '$y', '$z'],
		       ['-$x', '-$y', '$z'],
		       ['-$x', '$y', '-$z'],
		       ['$x', '-$y', '-$z'],
		       ['$z', '$x', '$y'],
		       ['$z', '-$x', '-$y'],
		       ['-$z', '-$x', '$y'],
		       ['-$z', '$x', '-$y'],
		       ['$y', '$z', '$x'],
		       ['-$y', '$z', '-$x'],
		       ['$y', '-$z', '-$x'],
		       ['-$y', '-$z', '$x'],
		       ['$y+1/2', '$x+1/2', '$z+1/2'],
		       ['-$y+1/2', '-$x+1/2', '$z+1/2'],
		       ['$y+1/2', '-$x+1/2', '-$z+1/2'],
		       ['-$y+1/2', '$x+1/2', '-$z+1/2'],
		       ['$x+1/2', '$z+1/2', '$y+1/2'],
		       ['-$x+1/2', '$z+1/2', '-$y+1/2'],
		       ['-$x+1/2', '-$z+1/2', '$y+1/2'],
		       ['$x+1/2', '-$z+1/2', '-$y+1/2'],
		       ['$z+1/2', '$y+1/2', '$x+1/2'],
		       ['$z+1/2', '-$y+1/2', '-$x+1/2'],
		       ['-$z+1/2', '$y+1/2', '-$x+1/2'],
		       ['-$z+1/2', '-$y+1/2', '$x+1/2']],
     },
     'f -4 3 c' => {
	 number => 219,
	 schoenflies => 't_d^5',
	 positions => [['$x', '$y', '$z'],
		       ['-$x', '-$y', '$z'],
		       ['-$x', '$y', '-$z'],
		       ['$x', '-$y', '-$z'],
		       ['$z', '$x', '$y'],
		       ['$z', '-$x', '-$y'],
		       ['-$z', '-$x', '$y'],
		       ['-$z', '$x', '-$y'],
		       ['$y', '$z', '$x'],
		       ['-$y', '$z', '-$x'],
		       ['$y', '-$z', '-$x'],
		       ['-$y', '-$z', '$x'],
		       ['$y+1/2', '$x+1/2', '$z+1/2'],
		       ['-$y+1/2', '-$x+1/2', '$z+1/2'],
		       ['$y+1/2', '-$x+1/2', '-$z+1/2'],
		       ['-$y+1/2', '$x+1/2', '-$z+1/2'],
		       ['$x+1/2', '$z+1/2', '$y+1/2'],
		       ['-$x+1/2', '$z+1/2', '-$y+1/2'],
		       ['-$x+1/2', '-$z+1/2', '$y+1/2'],
		       ['$x+1/2', '-$z+1/2', '-$y+1/2'],
		       ['$z+1/2', '$y+1/2', '$x+1/2'],
		       ['$z+1/2', '-$y+1/2', '-$x+1/2'],
		       ['-$z+1/2', '$y+1/2', '-$x+1/2'],
		       ['-$z+1/2', '-$y+1/2', '$x+1/2']],
     },
     'i -4 3 d' => {
	 number => 220,
	 schoenflies => 't_d^6',
	 positions => [['$x', '$y', '$z'],
		       ['-$x+1/2', '-$y', '$z+1/2'],
		       ['-$x', '$y+1/2', '-$z+1/2'],
		       ['$x+1/2', '-$y+1/2', '-$z'],
		       ['$z', '$x', '$y'],
		       ['$z+1/2', '-$x+1/2', '-$y'],
		       ['-$z+1/2', '-$x', '$y+1/2'],
		       ['-$z', '$x+1/2', '-$y+1/2'],
		       ['$y', '$z', '$x'],
		       ['-$y', '$z+1/2', '-$x+1/2'],
		       ['$y+1/2', '-$z+1/2', '-$x'],
		       ['-$y+1/2', '-$z', '$x+1/2'],
		       ['$y+1/4', '$x+1/4', '$z+1/4'],
		       ['-$y+1/4', '-$x+3/4', '$z+3/4'],
		       ['$y+3/4', '-$x+1/4', '-$z+3/4'],
		       ['-$y+3/4', '$x+3/4', '-$z+1/4'],
		       ['$x+1/4', '$z+1/4', '$y+1/4'],
		       ['-$x+3/4', '$z+3/4', '-$y+1/4'],
		       ['-$x+1/4', '-$z+3/4', '$y+3/4'],
		       ['$x+3/4', '-$z+1/4', '-$y+3/4'],
		       ['$z+1/4', '$y+1/4', '$x+1/4'],
		       ['$z+3/4', '-$y+1/4', '-$x+3/4'],
		       ['-$z+3/4', '$y+3/4', '-$x+1/4'],
		       ['-$z+1/4', '-$y+3/4', '$x+3/4']],
     },
     'p m -3 m' => {
	 number => 221,
	 schoenflies => 'o_h^1',
	 full => 'p 4/m -3 2/m',
	 thirtyfive => 'p m 3 m',
	 shorthand => ['cubic', 'cscl', 'perov', 'perovskite'],
	 positions => [['$x', '$y', '$z'],
		       ['-$x', '-$y', '$z'],
		       ['-$x', '$y', '-$z'],
		       ['$x', '-$y', '-$z'],
		       ['$z', '$x', '$y'],
		       ['$z', '-$x', '-$y'],
		       ['-$z', '-$x', '$y'],
		       ['-$z', '$x', '-$y'],
		       ['$y', '$z', '$x'],
		       ['-$y', '$z', '-$x'],
		       ['$y', '-$z', '-$x'],
		       ['-$y', '-$z', '$x'],
		       ['$y', '$x', '-$z'],
		       ['-$y', '-$x', '-$z'],
		       ['$y', '-$x', '$z'],
		       ['-$y', '$x', '$z'],
		       ['$x', '$z', '-$y'],
		       ['-$x', '$z', '$y'],
		       ['-$x', '-$z', '-$y'],
		       ['$x', '-$z', '$y'],
		       ['$z', '$y', '-$x'],
		       ['$z', '-$y', '$x'],
		       ['-$z', '$y', '$x'],
		       ['-$z', '-$y', '-$x'],
		       ['-$x', '-$y', '-$z'],
		       ['$x', '$y', '-$z'],
		       ['$x', '-$y', '$z'],
		       ['-$x', '$y', '$z'],
		       ['-$z', '-$x', '-$y'],
		       ['-$z', '$x', '$y'],
		       ['$z', '$x', '-$y'],
		       ['$z', '-$x', '$y'],
		       ['-$y', '-$z', '-$x'],
		       ['$y', '-$z', '$x'],
		       ['-$y', '$z', '$x'],
		       ['$y', '$z', '-$x'],
		       ['-$y', '-$x', '$z'],
		       ['$y', '$x', '$z'],
		       ['-$y', '$x', '-$z'],
		       ['$y', '-$x', '-$z'],
		       ['-$x', '-$z', '$y'],
		       ['$x', '-$z', '-$y'],
		       ['$x', '$z', '$y'],
		       ['-$x', '$z', '-$y'],
		       ['-$z', '-$y', '$x'],
		       ['-$z', '$y', '-$x'],
		       ['$z', '-$y', '-$x'],
		       ['$z', '$y', '$x']],
     },
     'p n -3 n' => {
	 number => 222,
	 schoenflies => 'o_h^2',
	 full => 'p 4/n -3 2/n',
	 thirtyfive => 'p n 3 n',
	 shiftvec => [-1/4, -1/4, -1/4],
	 positions => [['$x', '$y', '$z'],
		       ['-$x+1/2', '-$y+1/2', '$z'],
		       ['-$x+1/2', '$y', '-$z+1/2'],
		       ['$x', '-$y+1/2', '-$z+1/2'],
		       ['$z', '$x', '$y'],
		       ['$z', '-$x+1/2', '-$y+1/2'],
		       ['-$z+1/2', '-$x+1/2', '$y'],
		       ['-$z+1/2', '$x', '-$y+1/2'],
		       ['$y', '$z', '$x'],
		       ['-$y+1/2', '$z', '-$x+1/2'],
		       ['$y', '-$z+1/2', '-$x+1/2'],
		       ['-$y+1/2', '-$z+1/2', '$x'],
		       ['$y', '$x', '-$z+1/2'],
		       ['-$y+1/2', '-$x+1/2', '-$z+1/2'],
		       ['$y', '-$x+1/2', '$z'],
		       ['-$y+1/2', '$x', '$z'],
		       ['$x', '$z', '-$y+1/2'],
		       ['-$x+1/2', '$z', '$y'],
		       ['-$x+1/2', '-$z+1/2', '-$y+1/2'],
		       ['$x', '-$z+1/2', '$y'],
		       ['$z', '$y', '-$x+1/2'],
		       ['$z', '-$y+1/2', '$x'],
		       ['-$z+1/2', '$y', '$x'],
		       ['-$z+1/2', '-$y+1/2', '-$x+1/2'],
		       ['-$x', '-$y', '-$z'],
		       ['$x+1/2', '$y+1/2', '-$z'],
		       ['$x+1/2', '-$y', '$z+1/2'],
		       ['-$x', '$y+1/2', '$z+1/2'],
		       ['-$z', '-$x', '-$y'],
		       ['-$z', '$x+1/2', '$y+1/2'],
		       ['$z+1/2', '$x+1/2', '-$y'],
		       ['$z+1/2', '-$x', '$y+1/2'],
		       ['-$y', '-$z', '-$x'],
		       ['$y+1/2', '-$z', '$x+1/2'],
		       ['-$y', '$z+1/2', '$x+1/2'],
		       ['$y+1/2', '$z+1/2', '-$x'],
		       ['-$y', '-$x', '$z+1/2'],
		       ['$y+1/2', '$x+1/2', '$z+1/2'],
		       ['-$y', '$x+1/2', '-$z'],
		       ['$y+1/2', '-$x', '-$z'],
		       ['-$x', '-$z', '$y+1/2'],
		       ['$x+1/2', '-$z', '-$y'],
		       ['$x+1/2', '$z+1/2', '$y+1/2'],
		       ['-$x', '$z+1/2', '-$y'],
		       ['-$z', '-$y', '$x+1/2'],
		       ['-$z', '$y+1/2', '-$x'],
		       ['$z+1/2', '-$y', '-$x'],
		       ['$z+1/2', '$y+1/2', '$x+1/2']],
		       },
     'p m -3 n' => {
	 number => 223,
	 schoenflies => 'o_h^3',
	 thirtyfive => 'p m 3 n',
	 full => 'p 42/m -3 2/n',
	 positions => [['$x', '$y', '$z'],
		       ['-$x', '-$y', '$z'],
		       ['-$x', '$y', '-$z'],
		       ['$x', '-$y', '-$z'],
		       ['$z', '$x', '$y'],
		       ['$z', '-$x', '-$y'],
		       ['-$z', '-$x', '$y'],
		       ['-$z', '$x', '-$y'],
		       ['$y', '$z', '$x'],
		       ['-$y', '$z', '-$x'],
		       ['$y', '-$z', '-$x'],
		       ['-$y', '-$z', '$x'],
		       ['$y+1/2', '$x+1/2', '-$z+1/2'],
		       ['-$y+1/2', '-$x+1/2', '-$z+1/2'],
		       ['$y+1/2', '-$x+1/2', '$z+1/2'],
		       ['-$y+1/2', '$x+1/2', '$z+1/2'],
		       ['$x+1/2', '$z+1/2', '-$y+1/2'],
		       ['-$x+1/2', '$z+1/2', '$y+1/2'],
		       ['-$x+1/2', '-$z+1/2', '-$y+1/2'],
		       ['$x+1/2', '-$z+1/2', '$y+1/2'],
		       ['$z+1/2', '$y+1/2', '-$x+1/2'],
		       ['$z+1/2', '-$y+1/2', '$x+1/2'],
		       ['-$z+1/2', '$y+1/2', '$x+1/2'],
		       ['-$z+1/2', '-$y+1/2', '-$x+1/2'],
		       ['-$x', '-$y', '-$z'],
		       ['$x', '$y', '-$z'],
		       ['$x', '-$y', '$z'],
		       ['-$x', '$y', '$z'],
		       ['-$z', '-$x', '-$y'],
		       ['-$z', '$x', '$y'],
		       ['$z', '$x', '-$y'],
		       ['$z', '-$x', '$y'],
		       ['-$y', '-$z', '-$x'],
		       ['$y', '-$z', '$x'],
		       ['-$y', '$z', '$x'],
		       ['$y', '$z', '-$x'],
		       ['-$y+1/2', '-$x+1/2', '$z+1/2'],
		       ['$y+1/2', '$x+1/2', '$z+1/2'],
		       ['-$y+1/2', '$x+1/2', '-$z+1/2'],
		       ['$y+1/2', '-$x+1/2', '-$z+1/2'],
		       ['-$x+1/2', '-$z+1/2', '$y+1/2'],
		       ['$x+1/2', '-$z+1/2', '-$y+1/2'],
		       ['$x+1/2', '$z+1/2', '$y+1/2'],
		       ['-$x+1/2', '$z+1/2', '-$y+1/2'],
		       ['-$z+1/2', '-$y+1/2', '$x+1/2'],
		       ['-$z+1/2', '$y+1/2', '-$x+1/2'],
		       ['$z+1/2', '-$y+1/2', '-$x+1/2'],
		       ['$z+1/2', '$y+1/2', '$x+1/2']],
     },
     'p n -3 m' => {
	 number => 224,
	 schoenflies => 'o_h^4',
	 full => 'p 42/n -3 2/m',
	 thirtyfive => 'p n 3 m',
	 shiftvec => [-1/4, -1/4, -1/4],
	 positions => [['$x', '$y', '$z'],
		       ['-$x+1/2', '-$y+1/2', '$z'],
		       ['-$x+1/2', '$y', '-$z+1/2'],
		       ['$x', '-$y+1/2', '-$z+1/2'],
		       ['$z', '$x', '$y'],
		       ['$z', '-$x+1/2', '-$y+1/2'],
		       ['-$z+1/2', '-$x+1/2', '$y'],
		       ['-$z+1/2', '$x', '-$y+1/2'],
		       ['$y+1/2', '$z', '$x+1/2'],
		       ['-$y', '$z+1/2', '-$x+1/2'],
		       ['$y+1/2', '-$z+1/2', '-$x'],
		       ['-$y+1/2', '-$z+1/2', '$x'],
		       ['$y+1/2', '$x+1/2', '-$z'],
		       ['-$y', '-$x', '-$z'],
		       ['$y+1/2', '-$x', '$z+1/2'],
		       ['-$y', '$x+1/2', '$z+1/2'],
		       ['$x+1/2', '$z+1/2', '-$y'],
		       ['-$x', '$z+1/2', '$y+1/2'],
		       ['-$x', '-$z', '-$y'],
		       ['$x+1/2', '-$z', '$y+1/2'],
		       ['$z+1/2', '$y+1/2', '-$x'],
		       ['$z+1/2', '-$y', '$x+1/2'],
		       ['-$z', '$y+1/2', '$x+1/2'],
		       ['-$z', '-$y', '-$x'],
		       ['-$x', '-$y', '-$z'],
		       ['$x+1/2', '$y+1/2', '-$z'],
		       ['$x+1/2', '-$y', '$z+1/2'],
		       ['-$x', '$y+1/2', '$z+1/2'],
		       ['-$z', '-$x', '-$y'],
		       ['-$z', '$x+1/2', '$y+1/2'],
		       ['$z+1/2', '$x+1/2', '-$y'],
		       ['$z+1/2', '-$x', '$y+1/2'],
		       ['-$y', '-$z', '-$x'],
		       ['$y+1/2', '-$z', '$x+1/2'],
		       ['-$y', '$z+1/2', '$x+1/2'],
		       ['$y+1/2', '$z+1/2', '-$x'],
		       ['-$y+1/2', '-$x+1/2', '$z'],
		       ['$y', '$x', '$z'],
		       ['-$y+1/2', '$x', '-$z+1/2'],
		       ['$y', '-$x+1/2', '-$z+1/2'],
		       ['-$x+1/2', '-$z+1/2', '$y'],
		       ['$x', '-$z+1/2', '-$y+1/2'],
		       ['$x', '$z', '$y'],
		       ['-$x+1/2', '$z', '-$y+1/2'],
		       ['-$z+1/2', '-$y+1/2', '$x'],
		       ['-$z+1/2', '$y', '-$x+1/2'],
		       ['$z', '-$y+1/2', '-$x+1/2'],
		       ['$z', '$y', '$x']],
     },
     'f m -3 m' => {
	 number => 225,
	 schoenflies => 'o_h^5',
	 full => 'f 4/m -3 2/m',
	 thirtyfive => 'f m 3 m',
	 shorthand => ['fcc', 'salt', 'nacl'],
	 positions => [['$x', '$y', '$z'],
		       ['-$x', '-$y', '$z'],
		       ['-$x', '$y', '-$z'],
		       ['$x', '-$y', '-$z'],
		       ['$z', '$x', '$y'],
		       ['$z', '-$x', '-$y'],
		       ['-$z', '-$x', '$y'],
		       ['-$z', '$x', '-$y'],
		       ['$y', '$z', '$x'],
		       ['-$y', '$z', '-$x'],
		       ['$y', '-$z', '-$x'],
		       ['-$y', '-$z', '$x'],
		       ['$y', '$x', '-$z'],
		       ['-$y', '-$x', '-$z'],
		       ['$y', '-$x', '$z'],
		       ['-$y', '$x', '$z'],
		       ['$x', '$z', '-$y'],
		       ['-$x', '$z', '$y'],
		       ['-$x', '-$z', '-$y'],
		       ['$x', '-$z', '$y'],
		       ['$z', '$y', '-$x'],
		       ['$z', '-$y', '$x'],
		       ['-$z', '$y', '$x'],
		       ['-$z', '-$y', '-$x'],
		       ['-$x', '-$y', '-$z'],
		       ['$x', '$y', '-$z'],
		       ['$x', '-$y', '$z'],
		       ['-$x', '$y', '$z'],
		       ['-$z', '-$x', '-$y'],
		       ['-$z', '$x', '$y'],
		       ['$z', '$x', '-$y'],
		       ['$z', '-$x', '$y'],
		       ['-$y', '-$z', '-$x'],
		       ['$y', '-$z', '$x'],
		       ['-$y', '$z', '$x'],
		       ['$y', '$z', '-$x'],
		       ['-$y', '-$x', '$z'],
		       ['$y', '$x', '$z'],
		       ['-$y', '$x', '-$z'],
		       ['$y', '-$x', '-$z'],
		       ['-$x', '-$z', '$y'],
		       ['$x', '-$z', '-$y'],
		       ['$x', '$z', '$y'],
		       ['-$x', '$z', '-$y'],
		       ['-$z', '-$y', '$x'],
		       ['-$z', '$y', '-$x'],
		       ['$z', '-$y', '-$x'],
		       ['$z', '$y', '$x']],
     },
     'f m -3 c' => {
	 number => 226,
	 schoenflies => 'o_h^6',
	 full => 'f 4/m -3 2/c',
	 thirtyfive => 'f m 3 c',
	 positions => [['$x', '$y', '$z'],
		       ['-$x', '-$y', '$z'],
		       ['-$x', '$y', '-$z'],
		       ['$x', '-$y', '-$z'],
		       ['$z', '$x', '$y'],
		       ['$z', '-$x', '-$y'],
		       ['-$z', '-$x', '$y'],
		       ['-$z', '$x', '-$y'],
		       ['$y', '$z', '$x'],
		       ['-$y', '$z', '-$x'],
		       ['$y', '-$z', '-$x'],
		       ['-$y', '-$z', '$x'],
		       ['$y+1/2', '$x+1/2', '-$z+1/2'],
		       ['-$y+1/2', '-$x+1/2', '-$z+1/2'],
		       ['$y+1/2', '-$x+1/2', '$z+1/2'],
		       ['-$y+1/2', '$x+1/2', '$z+1/2'],
		       ['$x+1/2', '$z+1/2', '-$y+1/2'],
		       ['-$x+1/2', '$z+1/2', '$y+1/2'],
		       ['-$x+1/2', '-$z+1/2', '-$y+1/2'],
		       ['$x+1/2', '-$z+1/2', '$y+1/2'],
		       ['$z+1/2', '$y+1/2', '-$x+1/2'],
		       ['$z+1/2', '-$y+1/2', '$x+1/2'],
		       ['-$z+1/2', '$y+1/2', '$x+1/2'],
		       ['-$z+1/2', '-$y+1/2', '-$x+1/2'],
		       ['-$x', '-$y', '-$z'],
		       ['$x', '$y', '-$z'],
		       ['$x', '-$y', '$z'],
		       ['-$x', '$y', '$z'],
		       ['-$z', '-$x', '-$y'],
		       ['-$z', '$x', '$y'],
		       ['$z', '$x', '-$y'],
		       ['$z', '-$x', '$y'],
		       ['-$y', '-$z', '-$x'],
		       ['$y', '-$z', '$x'],
		       ['-$y', '$z', '$x'],
		       ['$y', '$z', '-$x'],
		       ['-$y+1/2', '-$x+1/2', '$z+1/2'],
		       ['$y+1/2', '$x+1/2', '$z+1/2'],
		       ['-$y+1/2', '$x+1/2', '-$z+1/2'],
		       ['$y+1/2', '-$x+1/2', '-$z+1/2'],
		       ['-$x+1/2', '-$z+1/2', '$y+1/2'],
		       ['$x+1/2', '-$z+1/2', '-$y+1/2'],
		       ['$x+1/2', '$z+1/2', '$y+1/2'],
		       ['-$x+1/2', '$z+1/2', '-$y+1/2'],
		       ['-$z+1/2', '-$y+1/2', '$x+1/2'],
		       ['-$z+1/2', '$y+1/2', '-$x+1/2'],
		       ['$z+1/2', '-$y+1/2', '-$x+1/2'],
		       ['$z+1/2', '$y+1/2', '$x+1/2']],
     },
     'f d -3 m' => {
	 number => 227,
	 schoenflies => 'o_h^7',
	 full => 'f 41/d -3 2/m',
	 thirtyfive => 'f d 3 m',
	 shorthand => ['diamond'],
	 shiftvec => [-1/8, -1/8, -1/8],
	 positions => [['$x', '$y', '$z'],
		       ['-$x+3/4', '-$y+1/4', '$z+1/2'],
		       ['-$x+1/4', '$y+1/2', '-$z+3/4'],
		       ['$x+1/2', '-$y+3/4', '-$z+1/4'],
		       ['$z', '$x', '$y'],
		       ['$z+1/2', '-$x+3/4', '-$y+1/4'],
		       ['-$z+3/4', '-$x+1/4', '$y+1/2'],
		       ['-$z+1/4', '$x+1/2', '-$y+3/4'],
		       ['$y', '$z', '$x'],
		       ['-$y+1/4', '$z+1/2', '-$x+3/4'],
		       ['$y+1/2', '-$z+3/4', '-$x+1/4'],
		       ['-$y+3/4', '-$z+1/4', '$x+1/2'],
		       ['$y+3/4', '$x+1/4', '-$z+1/2'],
		       ['-$y', '-$x', '-$z'],
		       ['$y+1/4', '-$x+1/2', '$z+3/4'],
		       ['-$y+1/2', '$x+3/4', '$z+1/4'],
		       ['$x+3/4', '$z+1/4', '-$y+1/2'],
		       ['-$x+1/2', '$z+3/4', '$y+1/4'],
		       ['-$x', '-$z', '-$y'],
		       ['$x+1/4', '-$z+1/2', '$y+3/4'],
		       ['$z+3/4', '$y+1/4', '-$x+1/2'],
		       ['$z+1/4', '-$y+1/2', '$x+3/4'],
		       ['-$z+1/2', '$y+3/4', '$x+1/4'],
		       ['-$z', '-$y', '-$x'],
		       ['-$x', '-$y', '-$z'],
		       ['$x+1/4', '$y+3/4', '-$z+1/2'],
		       ['$x+3/4', '-$y+1/2', '$z+1/4'],
		       ['-$x+1/2', '$y+1/4', '$z+3/4'],
		       ['-$z', '-$x', '-$y'],
		       ['-$z+1/2', '$x+1/4', '$y+3/4'],
		       ['$z+1/4', '$x+3/4', '-$y+1/2'],
		       ['$z+3/4', '-$x+1/2', '$y+1/4'],
		       ['-$y', '-$z', '-$x'],
		       ['$y+3/4', '-$z+1/2', '$x+1/4'],
		       ['-$y+1/2', '$z+1/4', '$x+3/4'],
		       ['$y+1/4', '$z+3/4', '-$x+1/2'],
		       ['-$y+1/4', '-$x+3/4', '$z+1/2'],
		       ['$y', '$x', '$z'],
		       ['-$y+3/4', '$x+1/2', '-$z+1/4'],
		       ['$y+1/2', '-$x+1/4', '-$z+3/4'],
		       ['-$x+1/4', '-$z+3/4', '$y+1/2'],
		       ['$x+1/2', '-$z+1/4', '-$y+3/4'],
		       ['$x', '$z', '$y'],
		       ['-$x+3/4', '$z+1/2', '-$y+1/4'],
		       ['-$z+1/4', '-$y+3/4', '$x+1/2'],
		       ['-$z+3/4', '$y+1/2', '-$x+1/4'],
		       ['$z+1/2', '-$y+1/4', '-$x+3/4'],
		       ['$z', '$y', '$x']],
     },
     'f d -3 c' => {
	 number => 228,
	 schoenflies => 'o_h^8',
	 full => 'f 41/d -3 2/c',
	 thirtyfive => 'f d 3 c',
	 shiftvec => [ -3/8, -3/8, -3/8 ],
	 positions => [['$x', '$y', '$z'],
		       ['-$x+1/4', '-$y+3/4', '$z+1/2'],
		       ['-$x+3/4', '$y+1/2', '-$z+1/4'],
		       ['$x+1/2', '-$y+1/4', '-$z+3/4'],
		       ['$z', '$x', '$y'],
		       ['$z+1/2', '-$x+1/4', '-$y+3/4'],
		       ['-$z+1/4', '-$x+3/4', '$y+1/2'],
		       ['-$z+3/4', '$x+1/2', '-$y+1/4'],
		       ['$y', '$z', '$x'],
		       ['-$y+3/4', '$z+1/2', '-$x+1/4'],
		       ['$y+1/2', '-$z+1/4', '-$x+3/4'],
		       ['-$y+1/4', '-$z+3/4', '$x+1/2'],
		       ['$y+3/4', '$x+1/4', '-$z'],
		       ['-$y+1/2', '-$x+1/2', '-$z+1/2'],
		       ['$y+1/4', '-$x', '$z+3/4'],
		       ['-$y', '$x+3/4', '$z+1/4'],
		       ['$x+3/4', '$z+1/4', '-$y'],
		       ['-$x', '$z+3/4', '$y+1/4'],
		       ['-$x+1/2', '-$z+1/2', '-$y+1/2'],
		       ['$x+1/4', '-$z', '$y+3/4'],
		       ['$z+3/4', '$y+1/4', '-$x'],
		       ['$z+1/4', '-$y', '$x+3/4'],
		       ['-$z', '$y+3/4', '$x+1/4'],
		       ['-$z+1/2', '-$y+1/2', '-$x+1/2'],
		       ['-$x', '-$y', '-$z'],
		       ['$x+3/4', '$y+1/4', '-$z+1/2'],
		       ['$x+1/4', '-$y+1/2', '$z+3/4'],
		       ['-$x+1/2', '$y+3/4', '$z+1/4'],
		       ['-$z', '-$x', '-$y'],
		       ['-$z+1/2', '$x+3/4', '$y+1/4'],
		       ['$z+3/4', '$x+1/4', '-$y+1/2'],
		       ['$z+1/4', '-$x+1/2', '$y+3/4'],
		       ['-$y', '-$z', '-$x'],
		       ['$y+1/4', '-$z+1/2', '$x+3/4'],
		       ['-$y+1/2', '$z+3/4', '$x+1/4'],
		       ['$y+3/4', '$z+1/4', '-$x+1/2'],
		       ['-$y+1/4', '-$x+3/4', '$z'],
		       ['$y+1/2', '$x+1/2', '$z+1/2'],
		       ['-$y+3/4', '$x', '-$z+1/4'],
		       ['$y', '-$x+1/4', '-$z+3/4'],
		       ['-$x+1/4', '-$z+3/4', '$y'],
		       ['$x', '-$z+1/4', '-$y+3/4'],
		       ['$x+1/2', '$z+1/2', '$y+1/2'],
		       ['-$x+3/4', '$z', '-$y+1/2'],
		       ['-$z+1/4', '-$y+3/4', '$x'],
		       ['-$z+3/4', '$y', '-$x+1/4'],
		       ['$z', '-$y+1/4', '-$x+3/4'],
		       ['$z+1/2', '$y+1/2', '$x+1/2']],
     },
     'i m -3 m' => {
	 number => 229,
	 schoenflies => 'o_h^9',
	 full => 'i 4/m -3 2/m',
	 shorthand => ['bcc'],
	 thirtyfive => 'i m 3 m',
	 positions => [['$x', '$y', '$z'],
		       ['-$x', '-$y', '$z'],
		       ['-$x', '$y', '-$z'],
		       ['$x', '-$y', '-$z'],
		       ['$z', '$x', '$y'],
		       ['$z', '-$x', '-$y'],
		       ['-$z', '-$x', '$y'],
		       ['-$z', '$x', '-$y'],
		       ['$y', '$z', '$x'],
		       ['-$y', '$z', '-$x'],
		       ['$y', '-$z', '-$x'],
		       ['-$y', '-$z', '$x'],
		       ['$y', '$x', '-$z'],
		       ['-$y', '-$x', '-$z'],
		       ['$y', '-$x', '$z'],
		       ['-$y', '$x', '$z'],
		       ['$x', '$z', '-$y'],
		       ['-$x', '$z', '$y'],
		       ['-$x', '-$z', '-$y'],
		       ['$x', '-$z', '$y'],
		       ['$z', '$y', '-$x'],
		       ['$z', '-$y', '$x'],
		       ['-$z', '$y', '$x'],
		       ['-$z', '-$y', '-$x'],
		       ['-$x', '-$y', '-$z'],
		       ['$x', '$y', '-$z'],
		       ['$x', '-$y', '$z'],
		       ['-$x', '$y', '$z'],
		       ['-$z', '-$x', '-$y'],
		       ['-$z', '$x', '$y'],
		       ['$z', '$x', '-$y'],
		       ['$z', '-$x', '$y'],
		       ['-$y', '-$z', '-$x'],
		       ['$y', '-$z', '$x'],
		       ['-$y', '$z', '$x'],
		       ['$y', '$z', '-$x'],
		       ['-$y', '-$x', '$z'],
		       ['$y', '$x', '$z'],
		       ['-$y', '$x', '-$z'],
		       ['$y', '-$x', '-$z'],
		       ['-$x', '-$z', '$y'],
		       ['$x', '-$z', '-$y'],
		       ['$x', '$z', '$y'],
		       ['-$x', '$z', '-$y'],
		       ['-$z', '-$y', '$x'],
		       ['-$z', '$y', '-$x'],
		       ['$z', '-$y', '-$x'],
		       ['$z', '$y', '$x']],
     },
     'i a -3 d' => {
	 number => 230,
	 schoenflies => 'o_h^10',
	 full => 'i 41/a -3 2/d',
	 thirtyfive => 'i a 3 d',
	 positions => [['$x', '$y', '$z'],
		       ['-$x+1/2', '-$y', '$z+1/2'],
		       ['-$x', '$y+1/2', '-$z+1/2'],
		       ['$x+1/2', '-$y+1/2', '-$z'],
		       ['$z', '$x', '$y'],
		       ['$z+1/2', '-$x+1/2', '-$y'],
		       ['-$z+1/2', '-$x', '$y+1/2'],
		       ['-$z', '$x+1/2', '-$y+1/2'],
		       ['$y', '$z', '$x'],
		       ['-$y', '$z+1/2', '-$x+1/2'],
		       ['$y+1/2', '-$z+1/2', '-$x'],
		       ['-$y+1/2', '-$z', '$x+1/2'],
		       ['$y+3/4', '$x+1/4', '-$z+1/4'],
		       ['-$y+3/4', '-$x+3/4', '-$z+3/4'],
		       ['$y+1/4', '-$x+1/4', '$z+3/4'],
		       ['-$y+1/4', '$x+3/4', '$z+1/4'],
		       ['$x+3/4', '$z+1/4', '-$y+1/4'],
		       ['-$x+1/4', '$z+3/4', '$y+1/4'],
		       ['-$x+3/4', '-$z+3/4', '-$y+3/4'],
		       ['$x+1/4', '-$z+1/4', '$y+3/4'],
		       ['$z+3/4', '$y+1/4', '-$x+1/4'],
		       ['$z+1/4', '-$y+1/4', '$x+3/4'],
		       ['-$z+1/4', '$y+3/4', '$x+1/4'],
		       ['-$z+3/4', '-$y+3/4', '-$x+3/4'],
		       ['-$x', '-$y', '-$z'],
		       ['$x+1/2', '$y', '-$z+1/2'],
		       ['$x', '-$y+1/2', '$z+1/2'],
		       ['-$x+1/2', '$y+1/2', '$z'],
		       ['-$z', '-$x', '-$y'],
		       ['-$z+1/2', '$x+1/2', '$y'],
		       ['$z+1/2', '$x', '-$y+1/2'],
		       ['$z', '-$x+1/2', '$y+1/2'],
		       ['-$y', '-$z', '-$x'],
		       ['$y', '-$z+1/2', '$x+1/2'],
		       ['-$y+1/2', '$z+1/2', '$x'],
		       ['$y+1/2', '$z', '-$x+1/2'],
		       ['-$y+1/4', '-$x+3/4', '$z+3/4'],
		       ['$y+1/4', '$x+1/4', '$z+1/4'],
		       ['-$y+3/4', '$x+3/4', '-$z+1/4'],
		       ['$y+3/4', '-$x+1/4', '-$z+3/4'],
		       ['-$x+1/4', '-$z+3/4', '$y+3/4'],
		       ['$x+3/4', '-$z+1/4', '-$y+3/4'],
		       ['$x+1/4', '$z+1/4', '$y+1/4'],
		       ['-$x+3/4', '$z+3/4', '-$y+1/4'],
		       ['-$z+1/4', '-$y+3/4', '$x+3/4'],
		       ['-$z+3/4', '$y+3/4', '-$x+1/4'],
		       ['$z+3/4', '-$y+1/4', '-$x+3/4'],
		       ['$z+1/4', '$y+1/4', '$x+1/4']],
     },

     );

my $thisdir = &identify_self;
my $outfile = File::Spec -> catfile($thisdir, "space_groups.db");
(-e $outfile) && do {
  print "  Removing old space_groups.db.\n";
  unlink($outfile);
};
print "  Saving space_groups.db to disk.\n";
##Storable:
nstore(\%space_groups, $outfile) or die "can't store hash: $!\n";
##end Storable:
##MLDBM:
## tie my %save_hash, 'MLDBM', "space_groups.db", O_CREAT|O_RDWR, 0644 or die $!;
## (tied %save_hash) -> DumpMeth('portable');    # Ask for portable binary
## %save_hash = %space_groups;
## untie %save_hash;
##end MLDBM:

sub identify_self {
  my @caller = caller;
  use File::Basename qw(dirname);
  return dirname($caller[1]);
};

1;

## Local Variables:
## time-stamp-line-limit: 25
## End:
