/*
 * HPTalx
 * A PC<->HP48/49 communication program for Linux
 *f
 * (C) 1999, 2000 by Bruno Barberi Gnecco
 *
 * rev 0.1:	Initial testing
 * rev 0.1.1:	Fixed some bugs
 * 		finished most commands
 *		Added preferences file
 * rev 0.2:	Fixed change_dir bug
 *		Improved preferences
 *		Added file editor support
 *		Added backup/restore memory
 *		Added rom update (49)
 * rev 1.0:	Added progress window in transfer
 *		Corrections due to pseudo terminals
 *		File type is now set in clist_data
 *		Fixed backup and restore
 *		Multiple file selection support
 *		Added un/select all
 * rev 1.0.1:	Fixed ls bug in refresh()
 *		Fixed port setup bug in setup()
 *		Fixed cancel transfer
 *		Fixed delete many files
 *		Fixed hp dir change in FILES
 *		Added timeout when connecting
 *		Fixed translation after transfer
 * rev 1.1.0:	File split from io.c
 *		Fixes to the "Unknown calc" bug
 *		Added kermit/xmodem support
 *		Made refresh(PC) independent of ckermit
 *		ckermit is only run upon connect()
 *		stdin/out not redirected to kermit anymore
 *		ROM update can be accessed without previous connection
 

    Copyright note:

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include "hptalx.h"
#include "icons.h"
#include <string.h>
#include <stddef.h>
#include <stdlib.h>
#include <unistd.h>
#include <fcntl.h>
#include <signal.h>
#include <ctype.h>
#include <dirent.h>
#include <sys/stat.h>
#include <sys/time.h>
#include <sys/types.h>
#include <pty.h>

#define BUFFERSIZE 256

static void clear_xinput ( void ) {
	struct timeval tv;
	fd_set set;

	FD_ZERO( &set );
	FD_SET( fileno(xmodem), &set );
	tv.tv_sec = 0;
	tv.tv_usec = 100000;

	if ( select (FD_SETSIZE, &set, NULL, NULL, &tv) == 1 ) {
		fgetc(xmodem);
		clear_xinput();
	}
}

FILE *xmodem, *kermit;

/* sends header as it is to the serial port, then builds a packet with
   s as data and sends to serial port too. Assures that the packet have been
   received, by trying up to 5 times.

 packet format:
2 byte: packet size (low endian mode)
n bytes: packet data
1 byte: packet chksum (the packet size is not use in the CHK computation)
*/
int send_packet ( const unsigned char *header, const unsigned char *s ) {
	static int calls = 0;
	unsigned char *packet, *t;
	int i, len;
	struct timeval tv;
	fd_set set;

	if ( header ) {
		for ( i = 0; i < strlen(header); i++ )
			fputc( header[i], xmodem);
		fflush( xmodem );
	}

	if ( !s ) 
		return 0;

	len = strlen(s);
	packet = (unsigned char *)malloc(len+4);

	packet[0] = (unsigned char)((len & 0xFF00)>>8);
	packet[1] = (unsigned char)(len & 0x00FF);
	memcpy( packet+2, s, len );
	
	/* checksum */
	packet[len+2] = 0;
	for ( (const unsigned char *)t = s; *t != '\0'; t++ )
		packet[len+2] += *t;
	packet[len+3] = '\0'; 

	/* send the packet */
	for ( i = 0; i < len+4; i++ )
		fputc( packet[i], xmodem);
	fflush(xmodem);
	free( packet );

	/* get response \006 */
	FD_ZERO( &set );
	FD_SET( fileno(xmodem), &set );
	tv.tv_sec = 0;
	tv.tv_usec = 200000;
	
	switch ( select (FD_SETSIZE, &set, NULL, NULL, &tv) ) {
		case -1:
#ifdef DEBUG
			fprintf(stderr, "-1 @ send\n");
#endif
			/* error. send again */
		case 0:
#ifdef DEBUG
			fprintf(stderr, "0 @ send\n");
#endif
			/* timeout. Resend the packet. */
			if ( calls < 5) {
				calls++;
				if ( send_packet( header, s ) == -1 )
					return -1;
			}
			else if ( calls >= 5 ) {
				set_status("Packet error, 5 resents.");
				calls = 0;
				return -1;
			}
			break;
		case 1:
#ifdef DEBUG
			fprintf(stderr, "-1 @ send\n");
#endif
			if ( fgetc( xmodem ) != 6 )
				set_status("OK byte error, ignoring.");
			break;
	}
	return 0;
}

unsigned char *get_packet ( int *size ) {
	unsigned char *data, *t, c = 0;
	int i;

	*size = (fgetc( xmodem )&0xFF)<<8;
	*size += (fgetc( xmodem )&0xFF);
	i = *size;

	data = (unsigned char *) malloc (i+1);

	t = data;
	for ( ; i > 0; i-- )
		c += *t++ = fgetc( xmodem );
	*t = '\0';

	if ( c != (unsigned char) fgetc( xmodem ) )  {
		free(data);
		return NULL;
	}
	fputc( '\006', xmodem );
	return data;
}

unsigned char *get_stack ( int *size ) {
	unsigned char bytes[BUFFERSIZE], *data, *t;
	int i = 0;

	send_packet( "E", "\x8DSTR DUP SIZE SWAP +" );
	send_packet( "E", "XMIT" );

	do {
		bytes[i] = fgetc(xmodem);
	} while ( bytes[i++] != '.' && i < BUFFERSIZE-1 );
	if ( i == BUFFERSIZE-1 )
		return NULL;
	bytes[i] = '\0';
	*size = atoi(bytes);

	data = (unsigned char *) malloc ((*size)+1);

	t = data;
	for ( i = 0; i < *size; i++ )
		*t++ = fgetc( xmodem );
	*t = '\0';
	return data;
}


void refreshall ( GtkWidget *w, gpointer data ) {
	if ( connected == ONLINE ) 
		refresh( w, GINT_TO_POINTER(BOTH));
	else 
		refresh( w, GINT_TO_POINTER(PC));
}

void refresh ( GtkWidget *w, gpointer data ) {
	char temp[BUFFERSIZE], **dir, *file[3];
	unsigned char *t;

	/* this ghost allocation is due to gtk */
	dir = (char **)malloc(1);
	dir[0] = (char *)malloc(BUFFERSIZE);

	if ( (HPANED)GPOINTER_TO_INT(data) == PC || 
			(HPANED)GPOINTER_TO_INT(data) == BOTH ) {
		int i = 0, dot = 0;
		struct dirent **ep;
		int n;
		char filenamewithpath[BUFFERSIZE];

		set_status( "Refreshing PC lists..." );

		gtk_clist_freeze( GTK_CLIST(files_pc) );
		gtk_clist_freeze( GTK_CLIST(dir_pc) );
		gtk_clist_clear( GTK_CLIST(files_pc) );
		gtk_clist_clear( GTK_CLIST(dir_pc) );

		file[0] = NULL;

		gtk_clist_set_column_title( GTK_CLIST(dir_pc), 0, data_pc.current );

		n = scandir(data_pc.current, &ep, 0, alphasort);
		if ( n < 0 ) {
			message_box( "Error", "scandir\nCannot list directory");
			return;
		}

		for ( i = 0; i < n; i++ ) {
			struct stat st;
			
			file[1] = ep[i]->d_name;
			sprintf( filenamewithpath, "%s/%s", data_pc.current,
					file[1] );
			
			if ( stat(filenamewithpath, &st) != 0 ) {
				message_box("Cannot open", filenamewithpath);
				continue;
			}
			file[2] = malloc(20);
			sprintf( file[2], "%d", (int)st.st_size );
			gtk_clist_append( GTK_CLIST(files_pc), file );
			
			if ( S_ISDIR(st.st_mode) ) {
				gtk_clist_append( GTK_CLIST(dir_pc), &file[1] );
				gtk_clist_set_pixmap( GTK_CLIST(files_pc),
						i, 0, dir_icon, dir_mask );
				gtk_clist_set_row_data( GTK_CLIST(files_pc), i,
						"Directory" );
			}
			else {
				gtk_clist_set_pixmap( GTK_CLIST(files_pc),
						i, 0, file_icon, file_mask );
				gtk_clist_set_row_data( GTK_CLIST(files_pc), i,
						"File" );
			}
			
			if ( preferences.start_after_dot == TRUE )
				if ( file[1][0] != '.' && dot == 0 )
					dot = i;
		}
		gtk_clist_sort( GTK_CLIST(files_pc) );
		gtk_clist_unselect_all( GTK_CLIST(files_pc) );
		gtk_clist_thaw( GTK_CLIST(files_pc) );
		gtk_clist_thaw( GTK_CLIST(dir_pc) );
		if ( preferences.start_after_dot ) {
			gtk_clist_moveto( GTK_CLIST(files_pc), dot, 0, 0.0, 0.0 );
		}
	}
	if ( ((HPANED)GPOINTER_TO_INT(data) == HP || 
			(HPANED)GPOINTER_TO_INT(data) == BOTH ) && 
			connected == ONLINE ) {
		int i = 0;
		struct timeval tv;
		fd_set rfds;

		FD_ZERO(&rfds);
		FD_SET(0, &rfds);

		set_status( "Refreshing HP lists..." );
		
		file[0] = NULL;
		/* clear kermit cache and HP */
		if ( preferences.protocol == KERMIT ) {
			fprintf( kermit, "REMOTE HOST 3 TRANSIO\n" );
			fprintf( kermit, "REMOTE HOST CLEAR\n" );
			clear_kinput();

			sleep(2);
			fprintf( kermit, "REMOTE DIR\n" );
			fgets( temp, BUFFERSIZE-1, kermit );

			tv.tv_sec = 2;
			tv.tv_usec = 0;
			/* directory information: { HOME DIR SUBDIR } CRC? */
			do {
				fgets( temp, BUFFERSIZE-1, kermit );
			} while ( temp[0] != '{' );
			t = strchr( temp, '}' ) +1;
			free( data_hp.current );
			data_hp.current = (char *)malloc(strlen(temp)-strlen(t)+1);
			strncpy( data_hp.current, temp, strlen(temp)-strlen(t));
			data_hp.current[strlen(temp)-strlen(t)] = '\0';
			gtk_clist_set_column_title( GTK_CLIST(dir_hp), 0, data_hp.current );

			/* now files */
			gtk_clist_freeze( GTK_CLIST(files_hp) );
			gtk_clist_freeze( GTK_CLIST(dir_hp) );
			gtk_clist_clear( GTK_CLIST(files_hp) );
			gtk_clist_clear( GTK_CLIST(dir_hp) );
			if ( !strstr(data_hp.current, "HOME }") ) {
				strcpy( dir[0], "..");
				gtk_clist_append( GTK_CLIST(dir_hp), dir );
			}

			fgets( temp, BUFFERSIZE-1, kermit );
			while ( !strstr( temp, "Kermit>" ) && 
					!strstr( temp, "Press the X" ) ) {

				/* format is: filename size type CRC? */
				if ( !(t = strtok( temp, " " )) ) {
					break;
				}
				file[1] = strdup( t );
				if ( !(t = strtok( NULL, " " )) ) {
					break;
				}
				file[2] = strdup( t );
				if ( !(t = strtok( NULL, "1234567890" )) ) {
					break;
				}

				/* ignoring last field */
			
				gtk_clist_append( GTK_CLIST(files_hp), file );

				if ( strstr( t, "Directory" ) ) {
					gtk_clist_append( GTK_CLIST(dir_hp), &file[1] );
					gtk_clist_set_pixmap( GTK_CLIST(files_hp),
							i, 0, dir_icon, dir_mask );
					gtk_clist_set_row_data( GTK_CLIST(files_hp), i, 
							"Directory" );
				}
				else if ( strstr( t, "List" ) ) { 
					gtk_clist_set_pixmap( GTK_CLIST(files_hp),
							i, 0, list_icon, list_mask );
					gtk_clist_set_row_data( GTK_CLIST(files_hp), i, 
						"List" );
				}
				else if ( strstr( t, "Library" ) ) {
					gtk_clist_set_pixmap( GTK_CLIST(files_hp),
							i, 0, library_icon, library_mask );
					gtk_clist_set_row_data( GTK_CLIST(files_hp), i,
							"Library" );
				}
				else if ( strstr( t, "Program" ) ) {
					gtk_clist_set_pixmap( GTK_CLIST(files_hp),
							i, 0, program_icon, program_mask );
					gtk_clist_set_row_data( GTK_CLIST(files_hp), i,
							"Program" );
				}
				else if ( strstr( t, "String" ) ) {
					gtk_clist_set_pixmap( GTK_CLIST(files_hp),
							i, 0, string_icon, string_mask );
					gtk_clist_set_row_data( GTK_CLIST(files_hp), i,
							"String" );
				}
				else if ( strstr( t, "Algebraic" ) ) {
					gtk_clist_set_pixmap( GTK_CLIST(files_hp),
							i, 0, algebraic_icon, algebraic_mask );
					gtk_clist_set_row_data( GTK_CLIST(files_hp), i,
							"Algebraic" );
				}
				else if ( strstr( t, "rray" ) ) {
					gtk_clist_set_pixmap( GTK_CLIST(files_hp),
							i, 0, array_icon, array_mask );
					gtk_clist_set_row_data( GTK_CLIST(files_hp), i,
							"Array" );
				}
				else if ( strstr( t, "Real Number" ) )  {
					gtk_clist_set_pixmap( GTK_CLIST(files_hp),
						i, 0, real_icon, real_mask );
					gtk_clist_set_row_data( GTK_CLIST(files_hp), i,
							"Real Number" );
				}
				else if ( strstr( t, "Complex Number" ) ) {
					gtk_clist_set_pixmap( GTK_CLIST(files_hp),
							i, 0, complex_icon, complex_mask );
					gtk_clist_set_row_data( GTK_CLIST(files_hp), i,
							"Complex Number" );
				}
				else {
					gtk_clist_set_pixmap( GTK_CLIST(files_hp),
							i, 0, file_icon, file_mask );
					gtk_clist_set_row_data( GTK_CLIST(files_hp), i,
							"File" );
				}
				i++;
				free( file[1] );	free( file[2] );
	
				/* now we must wait until data arrives, as communication
				is not instantaneous. This probably may crash if the
				comm is too slow. Anyway, at 9600bps I think it's
				likely to give few problems. 
				while ( !select( 1, &rfds, NULL, NULL, &tv ) ) {
					tv.tv_sec = 2;
					tv.tv_usec = 0;
				}*/
				fgets( temp, BUFFERSIZE-1, kermit );
			}
		}
		else { /* XMODEM */
			unsigned char *list, c, *s;
			int size, i = 0, filesize, prolog = 0;

			gtk_clist_freeze( GTK_CLIST(files_hp) );
			gtk_clist_freeze( GTK_CLIST(dir_hp) );
			gtk_clist_clear( GTK_CLIST(files_hp) );
			gtk_clist_clear( GTK_CLIST(dir_hp) );

			do {
				send_packet( "L", NULL );
				list = get_packet( &size );
			} while ( list == NULL );

			/* parse list */
			t = list;
			file[1] = temp;
			file[2] = (char *)malloc(10);
			while ( size > 0 ) {
				/* how many chars? */
				c = *t++;
				size -= 8+c;
				/* filename */
				for ( s = temp; c > 0; c-- )
					*s++ = *t++;
				*s = '\0';
				/* prolog */
				prolog = *t++;
				prolog += ((*t++)<<8)&0xFF00;
				/* file size */
				filesize = *t++;
				filesize += ((*t++)<<8)&0xFF00;
				filesize += ((*t++)<<16)&0xFF00;
				/* CRC */
				t++;
				t++;

				sprintf( file[2], "%.1f", ((float)filesize)/2.0 );
				gtk_clist_append( GTK_CLIST(files_hp), file );

				/* icon */
				if ( prolog == 0x2A96 ) {
					gtk_clist_append( GTK_CLIST(dir_hp), &file[1] );
					gtk_clist_set_pixmap( GTK_CLIST(files_hp),
							i, 0, dir_icon, dir_mask );
					gtk_clist_set_row_data( GTK_CLIST(files_hp), i, 
							"Directory" );
				}
				else if ( prolog == 0x2A74 ) { 
					gtk_clist_set_pixmap( GTK_CLIST(files_hp),
						i, 0, list_icon, list_mask );
					gtk_clist_set_row_data( GTK_CLIST(files_hp), i, 
						"List" );
				}
				else if ( prolog == 0x2B40 ) {
					gtk_clist_set_pixmap( GTK_CLIST(files_hp),
							i, 0, library_icon, library_mask );
					gtk_clist_set_row_data( GTK_CLIST(files_hp), i,
							"Library" );
				}
				else if ( prolog == 0x2D9D ) {
					gtk_clist_set_pixmap( GTK_CLIST(files_hp),
							i, 0, program_icon, program_mask );
					gtk_clist_set_row_data( GTK_CLIST(files_hp), i,
							"Program" );
				}
				else if ( prolog == 0x2A2C ) {
					gtk_clist_set_pixmap( GTK_CLIST(files_hp),
							i, 0, string_icon, string_mask );
					gtk_clist_set_row_data( GTK_CLIST(files_hp), i,
							"String" );
				}
				else if ( prolog == 0x2AB8 ) {
					gtk_clist_set_pixmap( GTK_CLIST(files_hp),
							i, 0, algebraic_icon, algebraic_mask );
					gtk_clist_set_row_data( GTK_CLIST(files_hp), i,
							"Algebraic" );
				}
				else if ( prolog == 0x29E8 ) {
					gtk_clist_set_pixmap( GTK_CLIST(files_hp),
							i, 0, array_icon, array_mask );
					gtk_clist_set_row_data( GTK_CLIST(files_hp), i,
							"Array" );
				}
				else if ( prolog == 0x2614 )  {
					gtk_clist_set_pixmap( GTK_CLIST(files_hp),
						i, 0, real_icon, real_mask );
					gtk_clist_set_row_data( GTK_CLIST(files_hp), i,
							"Real Number" );
				}
				else if ( prolog == 0x2977 ) {
					gtk_clist_set_pixmap( GTK_CLIST(files_hp),
							i, 0, complex_icon, complex_mask );
					gtk_clist_set_row_data( GTK_CLIST(files_hp), i,
							"Complex Number" );
				}
				else {
					gtk_clist_set_pixmap( GTK_CLIST(files_hp),
							i, 0, file_icon, file_mask );
					gtk_clist_set_row_data( GTK_CLIST(files_hp), i,
							"File" );
				}
				i++;
			}

			free( list );
			send_packet( "E", "PATH" );
			if ( data_hp.current ) 
				free( data_hp.current );
//			clear_xinput();
			data_hp.current = get_stack(&size);
			gtk_clist_set_column_title( GTK_CLIST(dir_hp), 0, data_hp.current );

			send_packet( "E", "CLEAR" );
		}

		if ( !strstr(data_hp.current, "HOME }") ) {
			strcpy( dir[0], "..");
			gtk_clist_append( GTK_CLIST(dir_hp), dir );
		}
		gtk_clist_sort( GTK_CLIST(files_hp) );
		gtk_clist_select_row( GTK_CLIST(files_hp), 0, 0 );
		gtk_clist_moveto( GTK_CLIST(files_hp), 0, 0, 0.0, 0.0 );
		gtk_clist_unselect_all( GTK_CLIST(files_hp) );
		gtk_clist_thaw( GTK_CLIST(files_hp) );
		gtk_clist_thaw( GTK_CLIST(dir_hp) );
	}
	free(dir[0]);
	free(dir);
	if ( preferences.protocol == KERMIT && connected == ONLINE )
		clear_kinput();
	set_status( "Done" );
}


static const char *on_items[] = { "<main>/Utilities/Command...", 
	"<main>/Connect/Disconnect", "<main>/File/Transfer",  
	"<main>/File/Move", "<main>/Utilities/Backup...",
	"<main>/Utilities/Restore", NULL };
static const char *off_items[] = { "<main>/Connect/Connect", NULL };
static const char *hp49_items[] = { 
	/*"<main>/Utilities/Protocol/XModem",*/ NULL };
	
void hp_connect ( GtkWidget *w, gpointer data ) {
	char temp[BUFFERSIZE];
	char const *t;
	GtkWidget *tw;
	int i = 0, master, r;
	time_t start, end;
	char serial[10];
	char stty[] = "stty 1:0:8bd:0:3:1c:7f:15:4:5:1:0:11:13:1a:0:12:f:17:16:0:0:73:0:0:0:0:0:0:0:0:0:0:0:0:0 < /dev/ttyS? > /dev/ttyS?";

	if ( connected == ONLINE )
		return;
	connected = ONLINE;

	set_status( "Connecting..." );

if ( preferences.protocol == KERMIT ) {
	if ( (r = forkpty( &master, NULL, NULL, NULL )) == -1 ) { /* error */
		message_box("Error", "Couldn't open pseudo terminal!" );
		connected = OFFLINE;
		return;
	}
	else if ( r == 0 ) { /* child */
		execlp("kermit", "kermit", "-z", NULL);
		/* if we failed */
		message_box("Error", "Make sure kermit is on the path." );
		connected = OFFLINE;
		return;
	}
	else { /* parent */
		/* read doesn't block */
		if ( fcntl( master, F_SETFL, O_NONBLOCK ) == -1 ) {
			message_box( "Error", "fcntl SETFL NONBLOCK failed\nDon't you love these cryptic messages? :)" );
			message_box( "Still error", "Anyway, sorry,\n but that error is fatal." );
			connected = OFFLINE;
			return;
		}

		if ( !(kermit = fdopen( master, "r+" )) ) {
			message_box( "Error", "Cannot open pipe, kermit");
			connected = OFFLINE;
			return;
		}

		/* not buffer in output */
		setvbuf( kermit, NULL, _IONBF, 0 );
		
		/* prompt can be read by fgets */
		fprintf( kermit, "SET PROMPT (\\v(dir)) C-Kermit>\\010\n" );
		fgets( temp, BUFFERSIZE-1, kermit );  

		fprintf( kermit, "SET MODEM TYPE DIRECT\n" );
		fprintf( kermit, "SET PORT /dev/ttyS%d\n", preferences.port );
		fprintf( kermit, "SET SPEED 9600\n" );
		fprintf( kermit, "SET CARRIER-WATCH OFF\n" );
		fprintf( kermit, "SET FLOW NONE\n" );
		fprintf( kermit, "SET PARITY NONE\n" );
		fprintf( kermit, "SET SEND TIMEOUT 20\n" );
		fprintf( kermit, "SET SEND PAUSE 100\n" );
		fprintf( kermit, "SET BLOCK 3\n" );
		fprintf( kermit, "SET CONTROL PREFIX ALL\n" );
		fprintf( kermit, "SET FILE TYPE BINARY\n" );
		fprintf( kermit, "SET FILE DISPLAY CRT\n" );
		sleep(preferences.wait_connect);
		clear_kinput();
		fprintf( kermit, "REMOTE HOST 3 TRANSIO\n" );
		time( &start );
		do {
			fgets( temp, BUFFERSIZE-1, kermit );
			if ( strstr( temp, "Kermit>" ) )
				break;
			time( &end );
		} while (difftime( end, start ) < 6);
		if ( !strstr( temp, "Kermit>") ) {
			set_status("Timeout!");
			fputc( '\003', kermit );
			message_box( "Timeout", "Timeout.\nCheck the cable, if the "
				"calculator\nis in server mode, and the port settings.");
			connected = OFFLINE;
			return;
		}
	
		clear_kinput();
		fprintf( kermit, "REMOTE HOST CLEAR\n" );
		sleep(1);
		wait_prompt();
	
		clear_kinput();
		set_status( "Getting HP type..." );
		fprintf( kermit, "REMOTE HOST VERSION\n" );
		sleep(1);
		do {
			while ( !fgets( temp, BUFFERSIZE-1, kermit ) ) ;
			if ( strstr( temp, "HP48" ) ) {
				/* we suppose a 48G */
				calc = HP48G;
				set_status( "HP 48G found" );
			}
			else if ( strstr( temp, "HP49" ) ) {
				calc = HP49;
			set_status( "HP 49 found" );
		}
		} while ( !strstr( temp, "Kermit>" ) );
		if ( calc == UNKNOWN_CALC ) {
			message_box( "Unknown calculator",
				"Calculator is unknown.\n"
				"Make sure it's connected, the port settings\n"
				"are correct and the calculator is supported\n"
				"by this program." );
			set_status( "Unknown calc!" );
			connected = OFFLINE;
			/* something may have gone wrong, so it's better clean the
			line */
			putchar( '\003' );
			return;
		}
		fprintf( kermit, "REMOTE HOST CLEAR\n" );
	}
}
else { /* XMODEM */
	char *stack;

	/* as dirty as hpflash :-) */
	sprintf( serial, "/dev/ttyS%d", preferences.port);
	stty[100] = stty[113] = serial[9];
	system(stty);

	if ( !(xmodem = fopen( serial, "r+" )) ) {
		message_box( "ERROR", "Cannot open port.\n"
				"Make sure you have priviledges to use it." );
		connected = OFFLINE;
		return;
	}
	setvbuf( xmodem, NULL, _IONBF, 0 );

	/* we know it's a 49, but we test the connection */
/*	send_packet("E", "VERSION");
	get_stack(
	if ( (stack = get_packet()) ){
		if ( strstr( temp, "HP49" ) ) { */
			calc = HP49;
			set_status("HP49 found");
/*		}
		/* goto is not spaghetti here 
		else goto xerror;
	}
	else {
		xerror:
		message_box( "Error",
			"An error occured.\n"
			"Make sure calculator it's connected and in\n"
			"XModem Server, the port settings\n"
			"are correct and the calculator is supported\n"
			"by this program." );
		set_status( "Couldn't connect!" );
		connected = OFFLINE;
		fclose( xmodem );
		return;
	} */

}
	data_hp.filename = data_hp.dir = data_hp.current = NULL;
	
	/* make menu correct */
	t = on_items[0];
	while ( t != NULL ) {
		tw = gtk_item_factory_get_widget( item_factory, t );
		gtk_widget_set_sensitive( tw, TRUE );
		t = on_items[++i];
	}
	t = off_items[0];	i = 0;
	while ( t != NULL ) {
		tw = gtk_item_factory_get_widget( item_factory, t );
		gtk_widget_set_sensitive( tw, FALSE );
		t = off_items[++i];
	}
	if ( calc == HP49 ) {
		t = hp49_items[0];	i = 0;
		while ( t != NULL ) {
			tw = gtk_item_factory_get_widget( item_factory, t );
			gtk_widget_set_sensitive( tw, TRUE );
			t = hp49_items[++i];
		}
	}

	/* now pop_menu */
	get_pop_menu( NULL );

	refresh( w, GINT_TO_POINTER(HP) );
}

void disconnect ( GtkWidget *w, gpointer data ) {
	/* just clean it all :) */
	char const *t;
	GtkWidget *temp;
	int i = 0;

	if ( connected == OFFLINE )
		return;
	connected = OFFLINE;
	calc = UNKNOWN_CALC;
	set_status( "Disconnecting..." );

	if ( preferences.protocol == KERMIT && data == NULL )
		fprintf( kermit, "QUIT" );

	gtk_clist_freeze( GTK_CLIST(files_hp) );
	gtk_clist_freeze( GTK_CLIST(dir_hp) );
	gtk_clist_clear( GTK_CLIST(files_hp) );
	gtk_clist_clear( GTK_CLIST(dir_hp) );
	gtk_clist_set_column_title( GTK_CLIST(dir_hp), 0, "Not connected" );
	gtk_clist_thaw( GTK_CLIST(files_hp) );
	gtk_clist_thaw( GTK_CLIST(dir_hp) );

	/* make menu correct */
	t = off_items[0];
	while ( t != NULL ) {
		temp = gtk_item_factory_get_widget( item_factory, t );
		gtk_widget_set_sensitive( temp, TRUE );
		t = off_items[++i];
	}
	t = on_items[0];	i = 0;
	while ( t != NULL ) {
		temp = gtk_item_factory_get_widget( item_factory, t );
		gtk_widget_set_sensitive( temp, FALSE );
		t = on_items[++i];
	}
	t = hp49_items[0];	i = 0;
	while ( t != NULL ) {
		temp = gtk_item_factory_get_widget( item_factory, t );
		gtk_widget_set_sensitive( temp, FALSE );
		t = hp49_items[++i];
	}

	/* now pop_menu */
	get_pop_menu( NULL );

	if ( data_hp.filename )
		free( data_hp.filename );
	if ( data_hp.dir )
		free( data_hp.dir );
	if ( data_hp.current )
		free( data_hp.current );
	set_status( "Done" );
}
