/*
 * Decompiled with CFR 0.152.
 */
package htsjdk.samtools.cram.build;

import htsjdk.samtools.SAMException;
import htsjdk.samtools.SAMFileHeader;
import htsjdk.samtools.SAMTextHeaderCodec;
import htsjdk.samtools.cram.io.ByteBufferUtils;
import htsjdk.samtools.cram.io.CountingInputStream;
import htsjdk.samtools.cram.io.ExposedByteArrayOutputStream;
import htsjdk.samtools.cram.structure.Block;
import htsjdk.samtools.cram.structure.BlockCompressionMethod;
import htsjdk.samtools.cram.structure.BlockContentType;
import htsjdk.samtools.cram.structure.CompressionHeaderBLock;
import htsjdk.samtools.cram.structure.Container;
import htsjdk.samtools.cram.structure.ContainerHeaderIO;
import htsjdk.samtools.cram.structure.CramHeader;
import htsjdk.samtools.cram.structure.Slice;
import htsjdk.samtools.cram.structure.SliceIO;
import htsjdk.samtools.seekablestream.SeekableBufferedStream;
import htsjdk.samtools.seekablestream.SeekableFTPStream;
import htsjdk.samtools.seekablestream.SeekableFileStream;
import htsjdk.samtools.seekablestream.SeekableHTTPStream;
import htsjdk.samtools.seekablestream.SeekableStream;
import htsjdk.samtools.seekablestream.UserPasswordInput;
import htsjdk.samtools.util.BufferedLineReader;
import htsjdk.samtools.util.Log;
import java.io.BufferedInputStream;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.DataInputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.io.RandomAccessFile;
import java.net.MalformedURLException;
import java.net.SocketException;
import java.net.URISyntaxException;
import java.net.URL;
import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.nio.MappedByteBuffer;
import java.nio.channels.FileChannel;
import java.util.ArrayList;
import java.util.Arrays;

public class CramIO {
    public static int DEFINITION_LENGTH = 26;
    private static Log log = Log.getInstance(CramIO.class);
    public static byte[] ZERO_B_EOF_MARKER = ByteBufferUtils.bytesFromHex("0b 00 00 00 ff ff ff ff ff e0 45 4f 46 00 00 00 00 01 00 00 01 00 06 06 01 00 01 00 01 00");

    public static String getFileName(String urlString) {
        URL url = null;
        try {
            url = new URL(urlString);
            return new File(url.getFile()).getName();
        }
        catch (MalformedURLException e) {
            return new File(urlString).getName();
        }
    }

    public static InputStream openInputStreamFromURL(String source) throws SocketException, IOException, URISyntaxException {
        URL url = null;
        try {
            url = new URL(source);
        }
        catch (MalformedURLException e) {
            File file = new File(source);
            return new SeekableBufferedStream(new SeekableFileStream(file));
        }
        String protocol = url.getProtocol();
        if ("ftp".equalsIgnoreCase(protocol)) {
            return new SeekableBufferedStream(new NamedSeekableFTPStream(url));
        }
        if ("http".equalsIgnoreCase(protocol)) {
            return new SeekableBufferedStream(new SeekableHTTPStream(url));
        }
        if ("file".equalsIgnoreCase(protocol)) {
            File file = new File(url.toURI());
            return new SeekableBufferedStream(new SeekableFileStream(file));
        }
        throw new RuntimeException("Uknown protocol: " + protocol);
    }

    public static InputStream openCramInputStream(String cramURL, boolean decrypt, String password) throws IOException, URISyntaxException {
        InputStream is = null;
        is = cramURL == null ? new BufferedInputStream(System.in) : CramIO.openInputStreamFromURL(cramURL);
        if (decrypt) {
            throw new SAMException("Encryption not supported in this version.");
        }
        if (is instanceof SeekableStream) {
            CramHeader cramHeader = CramIO.readFormatDefinition(is, new CramHeader());
            SeekableStream s = (SeekableStream)is;
            if (!CramIO.hasZeroB_EOF_marker(s)) {
                CramIO.eofNotFound(cramHeader.getMajorVersion(), cramHeader.getMinorVersion());
            }
            s.seek(0L);
        } else {
            log.warn("CRAM file/stream completion cannot be verified.");
        }
        return is;
    }

    private static void eofNotFound(byte major, byte minor) {
        if (major >= 2 && minor >= 1) {
            log.error("Incomplete data: EOF marker not found.");
            System.exit(1);
        } else {
            log.warn("EOF marker not found, possibly incomplete file/stream.");
        }
    }

    public static Container readContainer(CramHeader cramHeader, InputStream is) throws IOException {
        Container c = CramIO.readContainer(is);
        if (c == null) {
            CramIO.eofNotFound(cramHeader.getMajorVersion(), cramHeader.getMinorVersion());
            return CramIO.readContainer(new ByteArrayInputStream(ZERO_B_EOF_MARKER));
        }
        if (c.isEOF()) {
            log.debug("EOF marker found, file/stream is complete.");
        }
        return c;
    }

    public static long issueZeroB_EOF_marker(OutputStream os) throws IOException {
        os.write(ZERO_B_EOF_MARKER);
        return ZERO_B_EOF_MARKER.length;
    }

    public static boolean hasZeroB_EOF_marker(SeekableStream s) throws IOException {
        byte[] tail = new byte[ZERO_B_EOF_MARKER.length];
        s.seek(s.length() - (long)ZERO_B_EOF_MARKER.length);
        ByteBufferUtils.readFully(tail, s);
        tail[8] = (byte)(tail[8] | 0xF0);
        return Arrays.equals(tail, ZERO_B_EOF_MARKER);
    }

    public static boolean hasZeroB_EOF_marker(File file) throws IOException {
        byte[] tail = new byte[ZERO_B_EOF_MARKER.length];
        RandomAccessFile raf = new RandomAccessFile(file, "r");
        try {
            raf.seek(file.length() - (long)ZERO_B_EOF_MARKER.length);
            raf.readFully(tail);
        }
        catch (IOException e) {
            throw e;
        }
        finally {
            raf.close();
        }
        tail[8] = (byte)(tail[8] | 0xF0);
        return Arrays.equals(tail, ZERO_B_EOF_MARKER);
    }

    public static long writeCramHeader(CramHeader h, OutputStream os) throws IOException {
        os.write("CRAM".getBytes("US-ASCII"));
        os.write(h.getMajorVersion());
        os.write(h.getMinorVersion());
        os.write(h.id);
        for (int i = h.id.length; i < 20; ++i) {
            os.write(0);
        }
        long len = CramIO.writeContainerForSamFileHeader(h.getSamFileHeader(), os);
        return (long)DEFINITION_LENGTH + len;
    }

    private static CramHeader readFormatDefinition(InputStream is, CramHeader header) throws IOException {
        for (byte b : CramHeader.magick) {
            if (b == is.read()) continue;
            throw new RuntimeException("Unknown file format.");
        }
        header.setMajorVersion((byte)is.read());
        header.setMinorVersion((byte)is.read());
        DataInputStream dis = new DataInputStream(is);
        dis.readFully(header.id);
        return header;
    }

    public static CramHeader readCramHeader(InputStream is) throws IOException {
        CramHeader header = new CramHeader();
        CramIO.readFormatDefinition(is, header);
        header.setSamFileHeader(CramIO.readSAMFileHeader(new String(header.id), is));
        return header;
    }

    public static int writeContainer(Container c, OutputStream os) throws IOException {
        int i;
        long time1 = System.nanoTime();
        ExposedByteArrayOutputStream baos = new ExposedByteArrayOutputStream();
        CompressionHeaderBLock block = new CompressionHeaderBLock(c.h);
        block.write(baos);
        c.blockCount = 1;
        ArrayList<Integer> landmarks = new ArrayList<Integer>();
        SliceIO sio = new SliceIO();
        for (i = 0; i < c.slices.length; ++i) {
            Slice s = c.slices[i];
            landmarks.add(baos.size());
            sio.write(s, baos);
            ++c.blockCount;
            ++c.blockCount;
            if (s.embeddedRefBlock != null) {
                ++c.blockCount;
            }
            c.blockCount += s.external.size();
        }
        c.landmarks = new int[landmarks.size()];
        for (i = 0; i < c.landmarks.length; ++i) {
            c.landmarks[i] = (Integer)landmarks.get(i);
        }
        c.containerByteSize = baos.size();
        CramIO.calculateSliceOffsetsAndSizes(c);
        ContainerHeaderIO chio = new ContainerHeaderIO();
        int len = chio.writeContainerHeader(c, os);
        os.write(baos.getBuffer(), 0, baos.size());
        long time2 = System.nanoTime();
        log.debug("CONTAINER WRITTEN: " + c.toString());
        c.writeTime = time2 - time1;
        return len += baos.size();
    }

    public static Container readContainer(InputStream is) throws IOException {
        return CramIO.readContainer(is, 0, Integer.MAX_VALUE);
    }

    public static Container readContainerHeader(InputStream is) throws IOException {
        ContainerHeaderIO chio = new ContainerHeaderIO();
        Container c = new Container();
        if (!chio.readContainerHeader(c, is)) {
            return null;
        }
        return c;
    }

    private static Container readContainer(InputStream is, int fromSlice, int howManySlices) throws IOException {
        long time1 = System.nanoTime();
        Container c = CramIO.readContainerHeader(is);
        if (c == null) {
            return null;
        }
        CompressionHeaderBLock chb = new CompressionHeaderBLock(is);
        c.h = chb.getCompressionHeader();
        howManySlices = Math.min(c.landmarks.length, howManySlices);
        if (fromSlice > 0) {
            is.skip(c.landmarks[fromSlice]);
        }
        SliceIO sio = new SliceIO();
        ArrayList<Slice> slices = new ArrayList<Slice>();
        int s = fromSlice;
        while (s < howManySlices - fromSlice) {
            Slice slice = new Slice();
            slice.index = s++;
            sio.readSliceHeadBlock(slice, is);
            sio.readSliceBlocks(slice, true, is);
            slices.add(slice);
        }
        c.slices = slices.toArray(new Slice[slices.size()]);
        CramIO.calculateSliceOffsetsAndSizes(c);
        long time2 = System.nanoTime();
        log.debug("READ CONTAINER: " + c.toString());
        c.readTime = time2 - time1;
        return c;
    }

    private static void calculateSliceOffsetsAndSizes(Container c) {
        if (c.slices.length == 0) {
            return;
        }
        for (int i = 0; i < c.slices.length - 1; ++i) {
            Slice s = c.slices[i];
            s.offset = c.landmarks[i];
            s.size = c.landmarks[i + 1] - s.offset;
        }
        Slice lastSlice = c.slices[c.slices.length - 1];
        lastSlice.offset = c.landmarks[c.landmarks.length - 1];
        lastSlice.size = c.containerByteSize - lastSlice.offset;
    }

    public static byte[] toByteArray(SAMFileHeader samFileHeader) {
        ExposedByteArrayOutputStream headerBodyOS = new ExposedByteArrayOutputStream();
        OutputStreamWriter w = new OutputStreamWriter(headerBodyOS);
        new SAMTextHeaderCodec().encode(w, samFileHeader);
        try {
            w.close();
        }
        catch (IOException e) {
            throw new RuntimeException(e);
        }
        ByteBuffer buf = ByteBuffer.allocate(4);
        buf.order(ByteOrder.LITTLE_ENDIAN);
        buf.putInt(headerBodyOS.size());
        buf.flip();
        byte[] bytes = new byte[buf.limit()];
        buf.get(bytes);
        ByteArrayOutputStream headerOS = new ByteArrayOutputStream();
        try {
            headerOS.write(bytes);
            headerOS.write(headerBodyOS.getBuffer(), 0, headerBodyOS.size());
        }
        catch (IOException e) {
            throw new RuntimeException(e);
        }
        return headerOS.toByteArray();
    }

    private static long writeContainerForSamFileHeader(SAMFileHeader samFileHeader, OutputStream os) throws IOException {
        byte[] data = CramIO.toByteArray(samFileHeader);
        return CramIO.writeContainerForSamFileHeaderData(data, 0, Math.max(1024, data.length + data.length / 2), os);
    }

    private static long writeContainerForSamFileHeaderData(byte[] data, int offset, int len, OutputStream os) throws IOException {
        Block block = new Block();
        byte[] blockContent = new byte[len];
        System.arraycopy(data, 0, blockContent, offset, Math.min(data.length - offset, len));
        block.setRawContent(blockContent);
        block.method = BlockCompressionMethod.RAW;
        block.contentId = 0;
        block.contentType = BlockContentType.FILE_HEADER;
        block.compress();
        Container c = new Container();
        c.blockCount = 1;
        c.blocks = new Block[]{block};
        c.landmarks = new int[0];
        c.slices = new Slice[0];
        c.alignmentSpan = 0;
        c.alignmentStart = 0;
        c.bases = 0L;
        c.globalRecordCounter = 0L;
        c.nofRecords = 0;
        c.sequenceId = 0;
        ExposedByteArrayOutputStream baos = new ExposedByteArrayOutputStream();
        block.write(baos);
        c.containerByteSize = baos.size();
        ContainerHeaderIO chio = new ContainerHeaderIO();
        int containerHeaderByteSize = chio.writeContainerHeader(c, os);
        os.write(baos.getBuffer(), 0, baos.size());
        return containerHeaderByteSize + baos.size();
    }

    public static SAMFileHeader readSAMFileHeader(String id, InputStream is) throws IOException {
        CramIO.readContainerHeader(is);
        Block b = new Block(is, true, true);
        is = new ByteArrayInputStream(b.getRawContent());
        ByteBuffer buf = ByteBuffer.allocate(4);
        buf.order(ByteOrder.LITTLE_ENDIAN);
        for (int i = 0; i < 4; ++i) {
            buf.put((byte)is.read());
        }
        buf.flip();
        int size = buf.asIntBuffer().get();
        DataInputStream dis = new DataInputStream(is);
        byte[] bytes = new byte[size];
        dis.readFully(bytes);
        BufferedLineReader r = new BufferedLineReader(new ByteArrayInputStream(bytes));
        SAMTextHeaderCodec codec = new SAMTextHeaderCodec();
        SAMFileHeader header = codec.decode(r, id);
        return header;
    }

    public static boolean replaceCramHeader(File file, CramHeader newHeader) throws IOException {
        int MAP_SIZE = (int)Math.min(0x100000L, file.length());
        FileInputStream inputStream = new FileInputStream(file);
        CountingInputStream cis = new CountingInputStream(inputStream);
        CramHeader header = new CramHeader();
        CramIO.readFormatDefinition(cis, header);
        if (header.getMajorVersion() != newHeader.getMajorVersion() && header.getMinorVersion() != newHeader.getMinorVersion()) {
            log.error(String.format("Cannot replace CRAM header because format versions differ: ", header.getMajorVersion(), header.getMinorVersion(), newHeader.getMajorVersion(), header.getMinorVersion(), file.getAbsolutePath()));
            cis.close();
            return false;
        }
        CramIO.readContainerHeader(cis);
        Block b = new Block(cis, false, false);
        long dataStart = cis.getCount();
        cis.close();
        byte[] data = CramIO.toByteArray(newHeader.getSamFileHeader());
        if (data.length > b.getRawContentSize()) {
            log.error("Failed to replace CRAM header because the new header is bigger.");
            return false;
        }
        RandomAccessFile raf = new RandomAccessFile(file, "rw");
        FileChannel channelOut = raf.getChannel();
        MappedByteBuffer mapOut = channelOut.map(FileChannel.MapMode.READ_WRITE, dataStart, (long)MAP_SIZE - dataStart);
        mapOut.put(data);
        mapOut.force();
        channelOut.close();
        raf.close();
        return true;
    }

    private static class NamedSeekableFTPStream
    extends SeekableFTPStream {
        private URL source;

        public NamedSeekableFTPStream(URL url) throws IOException {
            super(url);
            this.source = url;
        }

        public NamedSeekableFTPStream(URL url, UserPasswordInput userPasswordInput) throws IOException {
            super(url, userPasswordInput);
            this.source = url;
        }

        @Override
        public String getSource() {
            return this.source.toString();
        }
    }
}

