/*
 * @(#)ReadOnlyPacket.java	1.39 05/03/21
 *
 * Copyright 2000-2002 Sun Microsystems, Inc. All Rights Reserved
 *
 */

package com.sun.messaging.jmq.io;

import java.util.Hashtable;
import java.io.*;

import com.sun.messaging.jmq.net.IPAddress;

/**
 * This class is a mostly immutable encapsulation of a JMQ packet.
 * It is only "mostly" immutable because it can be modified by calling
 * the readPacket() method which modifies the state of the object to reflect
 * the packet just read.
 *
 * WARNING! This class emphasizes performance over safety. In particular
 * the readPacket() method is NOT synchronized. You must only call
 * readPacket() in a single threaded environment (or do the synchronization
 * yourself). Once the object is initialized via a readPacket() you can safely
 * use the accessors and the writePacket() methods in a multi-threaded
 * envornment.
 *
 * @version 1.39 05/03/21
 *
 */
public class ReadOnlyPacket implements Cloneable {

    // Packet magic number. NEVER change this
    public static final int   MAGIC = 469754818;


    // Packet version number
    public static final short VERSION1 = 103;

    public static final short VERSION2 = 200;

    public static final short VERSION3 = 301;

    public static short defaultVersion = VERSION3;

    // Number of bytes in fixed header portion of packet
    protected static final int  HEADER_SIZE = 72;

    // Buffer to hold header
    protected byte[]	headerBuffer  = new byte[HEADER_SIZE];

    // Buffer to hold remainder of packet (strings, properties, message body)
    // rop == remainder of packet!
    protected byte[]	ropBuffer = null;
    protected int	ropLength = 0;    // Number of valid bytes in ropBuffer

    // Values in fixed header
    protected int	version        = defaultVersion;
    protected int	magic          = MAGIC;
    protected int	packetType     = 0;
    protected int	packetSize     = 0;
    protected long	expiration     = 0;
    protected int	propertyOffset = 0;
    protected int	propertySize   = 0;
    protected int	encryption     = 0;
    protected long	transactionID  = 0;
    protected int	priority       = 5;
    protected int	bitFlags       = 0;
    protected long	consumerID     = 0;

    // Holds sequence, IPaddr, port and timestamp
    protected SysMessageID sysMessageID = new SysMessageID();

    // Values in variable string section
    protected String	destination        = null;
    protected String	destinationClass   = null;
    protected String	messageID          = null;
    protected String	correlationID      = null;
    protected String	replyTo            = null;
    protected String	replyToClass       = null;
    protected String  	messageType        = null;
    protected long      producerID         = 0;

    // True if string fields have been parsed out of buffer
    protected boolean   	stringsParsed  = false;

    // Properties from packet
    protected Hashtable	properties    = null;

    public ReadOnlyPacket() {
	this.reset();
    }

    /**
     * this method MUST be called before a packet is created
     */
    public static void setDefaultVersion(short version) {
        defaultVersion = version;
    }

    /**
     * Read packet from an InputStream. This method reads one packet
     * from the InputStream and sets the state of this object to
     * reflect the packet read. This method is not synchronized
     * and should only be called by non-concurrent code.
     *
     * If we read a packet with a bad magic number (ie it looks like
     * bogus data), we give up and throw a StreamCorruptedException. 
     *
     * If we read a packet that does not match our packet version
     * we attempt to swallow the packet and then throw an
     * IllegalArgumentException.
     *
     * If this method throws an OutOfMemoryError, the caller can try
     * to free memory and call retryReadPacket().
     *
     * @param is        the InputStream to read the packet from
     */
    public void readPacket(InputStream is)
	throws IOException, EOFException, StreamCorruptedException, IllegalArgumentException {

        // Read the fixed header
        readFixedHeader(is);

        // Read the rest of the packet
        readRestOfPacket(is);

	return;
    }

    /**
     * Attempt to complete the reading of a packet. This is of limited
     * usefulness. It depends on the fixed portion of a packet having been
     * read successfully already. In practice this method should only be
     * used if readPacket() throws an OutOfMemoryError() and the caller has
     * freed some memory and wants to try completing the read of the packet.
     */
    public void retryReadPacket(InputStream is) 
	throws IOException, EOFException, StreamCorruptedException, IllegalArgumentException {

        if (packetSize == 0) {
            // Header was not read. Try to read an entire packet
            readPacket(is);
        } else {
            // Header was probably read and parsed. Try to read remainder
            // of packet
            readRestOfPacket(is);
        }

        return;
    }

    // Read the fixed header portion of a packet and parse the information
    // into fields.
    // Must be called before readRestOfPacket
    private void readFixedHeader(InputStream is)
	throws IOException, EOFException, StreamCorruptedException {

	this.reset();

	// Read fixed header. In this case if the read times out and
	// reads 0 bytes we can abort the read.
	readFully(is, headerBuffer, 0, HEADER_SIZE, false);

	// Parse fixed header into instance variables
        try {
	    parseHeader(new ByteArrayInputStream(headerBuffer));
        } catch (StreamCorruptedException e) {
            // Bad packet magic number. We got garbage. All we can
	    // do is throw an exception.
	    throw e;
        }
    }

    // Read the remaining portion of the packet.
    // Must be called after readFixedHeader.
    private void readRestOfPacket(InputStream is)
	throws IOException, EOFException, IllegalArgumentException {

	// Read remainder of packet into a buffer
	// We try to reuse ropBuffer when possible. We reallocate
	// it if it is too small, or is significantly too large.
        // This is the most likely spot for the packet code to get
        // an OutOfMemoryError.
	ropLength = packetSize - HEADER_SIZE;
	if (ropBuffer == null || ropBuffer.length < ropLength || 
	    ropBuffer.length > 2 * ropLength) {

	    ropBuffer = null;
	    try {
	        ropBuffer = new byte[ropLength];
	    } catch (OutOfMemoryError e) {
		// Could not allocate buffer to hold rest of packet.
                // Just rethrow the exception. Caller may try to free memory
                // and re-read packet.
		throw e;
	    }
	}

	// Read the rest of the packet. In this case we want the read
	// to continue to retry until it has read all bytes.
	readFully(is, ropBuffer, 0, ropLength, true);

        // If it's a version we don't understand throw exception
	if (version != VERSION1 && version != VERSION2 && version != VERSION3) {
	    throw new IllegalArgumentException("Bad packet version number: " +
		version + ". Expecting: " + VERSION1 + " or " + VERSION2 +
                      " or " + VERSION3);
	}

	// Parse strings into instance variables
	// [PERFORMANCE] Could delay this until an accessor is called,
	// but for simplicity we do it now.
        if (propertyOffset != HEADER_SIZE) {
	    parseVariableFields(new ByteArrayInputStream(ropBuffer));
        }

	// We don't parse the properties yet since we may not need to
	return;
    }

    /**
     * Our own version of readFully(). This is identical to DataInputStream's
     * except that we handle InterruptedIOException by doing a yield, and
     * continuing trying to read. This is to handle the case where the
     * socket has an SO_TIMEOUT set.
     *
     * If retry is false we abandon the read if it times out and we haven't
     * read anything. If it is true we continue to retry the read.
     */
    private void readFully(InputStream in, byte b[], int off, int len, 
			boolean retry)
	throws IOException, EOFException, InterruptedIOException {

        if (len < 0)
            throw new IndexOutOfBoundsException();

        int n = 0;
	int count;
        while (n < len) {
	    count = 0;
            try {
                count = in.read(b, off + n, len - n);
            } catch (InterruptedIOException e) {
                // if we really have read nothing .. throw an ex
                if (!retry && n == 0 && count == 0 && e.bytesTransferred == 0) {
                    throw new InterruptedIOException("no data available");
		}

		count = e.bytesTransferred;

		Thread.currentThread().yield();
	    }
            if (count < 0)
                throw new EOFException();
            n += count;
        }
    }

    /**
     * Parse the fixed packet header into instance variables.
     */
    private void parseHeader(InputStream is)
	throws IOException {

	DataInputStream d = new DataInputStream(is);

	magic   = d.readInt();

	if (magic != MAGIC) {
	    throw new StreamCorruptedException("Bad packet magic number: " +
			magic + ". Expecting: " + MAGIC);
	}
	version = d.readShort();
	packetType = d.readShort();
	packetSize = d.readInt();

        // Up to this point the packet header stays the same between 
        // versions. After this point it may be different depending
        // on the version. If the version is something we don't understand
        // then the following will generate garbage values, but we will
        // catch the version error later.


        // In VERSION1 the transactionID is a 32 bit value in the header.
        // In VERSION2 It became a 64 bit value in the variable portion
        // of the packet.
	if (version == VERSION1) {
	    transactionID  = d.readInt();
        }

	expiration = d.readLong();

	// Reads timestamp, source IP address, source port and sequence number
	sysMessageID.readID(d);

	propertyOffset = d.readInt();
	propertySize   = d.readInt();

	priority   = d.readByte();
	encryption = d.readUnsignedByte();

	bitFlags   = d.readUnsignedShort();

        // In VERSION2 the consumerID is a long
        if (version == VERSION1) {
	    consumerID = d.readInt();
	} else {
	    consumerID = d.readLong();
        }

	return;
    }

    public boolean getFlag(int flag) {
	return((bitFlags & flag) == flag);
    }

    /**
     * Parse variable portion of the packet
     */
    private void parseVariableFields(InputStream is) throws IOException {

	DataInputStream d = new DataInputStream(is);

	int type, len;
	String s;

	type = d.readShort();
	while (type != PacketString.NULL) {
	    switch (type) {
		case PacketString.DESTINATION:
	            s = d.readUTF();
		    destination = s;
		    break;

                case PacketString.TRANSACTIONID:
                    // In VERSION2 transactionID is in this part of packet
                    // Skip length. Should be 8 bytes.
                    len = d.readUnsignedShort();
                    transactionID = d.readLong();
                    break;

                case PacketString.PRODUCERID:
                    // Skip length. ProducerID is a long
                    len = d.readUnsignedShort();
                    producerID = d.readLong();
                    break;

		case PacketString.MESSAGEID:
	            s = d.readUTF();
    		    messageID = s;
		    break;

		case PacketString.CORRELATIONID:
	            s = d.readUTF();
    		    correlationID = s;
		    break;

		case PacketString.REPLYTO:
	            s = d.readUTF();
    		    replyTo = s;
		    break;

		case PacketString.TYPE:
	            s = d.readUTF();
    		    messageType = s;
		    break;

		case PacketString.DESTINATION_CLASS:
	            s = d.readUTF();
    		    destinationClass = s;
		    break;

		case PacketString.REPLYTO_CLASS:
	            s = d.readUTF();
    		    replyToClass = s;
		    break;
	
		default:
                    // Silently ignore fields we don't understand
                    d.skipBytes(d.readUnsignedShort());
                    break;
	    }
	    type = d.readShort();
	}

	stringsParsed = true;

    }

    /**
     * Parse message properties into the hashtable.
     */
    private void parseProperties(InputStream is)
	throws IOException, ClassNotFoundException {

        if (version == VERSION3) {

	    properties = PacketProperties.parseProperties(is);

        } else {
	    ObjectInputStream p = new ObjectInputStream(is);

	    properties = (Hashtable)p.readObject();
        }

	return;
    }

    /**
     * Write the packet to an OutputStream
     *
     * @param os                The OutputStream to write the packet to
     *
     */
    public void writePacket(OutputStream os)
	throws IOException {

	// Write the header
	os.write(headerBuffer, 0, HEADER_SIZE);

	// Write rest of packet
	if (ropBuffer != null) {
	    os.write(ropBuffer, 0, ropLength);
	}

	os.flush();
    }

    /**
     * Write the packet to an OutputStream
     *
     * @param os                The OutputStream to write the packet to
     *
     */
    public void writePacket(OutputStream os, int new_version)
	throws IOException {

        // If version isn't different, then just write packet as-is
        if (new_version == this.version) {
            writePacket(os);
        } else {
            writePacket(os, new_version,
                this.consumerID,
                getFlag(PacketFlag.P_FLAG),
                getFlag(PacketFlag.R_FLAG),
                getFlag(PacketFlag.L_FLAG));
        }
    }

    /**
     * Write the packet to an OutputStream with some modified fields.
     *
     * @param os                The OutputStream to write the packet to
     * @param new_consumerID    New value for the consumerID
     * @param new_pauseFlow     New value for the pause flow flag (F bit)
     * @param new_redelivered   New value for the redelivered flag (R bit)
     * @param new_lastPkt       New value for the last message flag (L bit)
     *
     */
    public void writePacket(OutputStream os, 
	long new_consumerID,
	boolean new_pauseFlow,
	boolean new_redelivered,
	boolean new_lastPkt)
	throws IOException {

        writePacket(os, this.version, new_consumerID, new_pauseFlow,
            new_redelivered, new_lastPkt);
    }

    /**
     * Write the packet to an OutputStream with some modified fields.
     *
     * @param os                The OutputStream to write the packet to
     * @param new_version       New version value
     * @param new_consumerID    New value for the consumerID
     * @param new_pauseFlow     New value for the pause flow flag (F bit)
     * @param new_redelivered   New value for the redelivered flag (R bit)
     * @param new_lastPkt       New value for the last message flag (L bit)
     *
     */
    public void writePacket(OutputStream os, int new_version,
	long new_consumerID,
	boolean new_pauseFlow,
	boolean new_redelivered,
	boolean new_lastPkt)
	throws IOException {

	if (headerBuffer == null) {
	    throw new IOException("No buffer to write");
        }

	DataOutputStream dos = new DataOutputStream(os);

	// Since this class is imutable we know the content of 
	// the buffers and the size of the packet has not changed
	// We only have to worry about updating the passed parameters
	// which is why this method can be unsynchronized
	
        if (new_version == this.version) {
            // Packet version is not changing, we can re-use most of header
            if (new_version == VERSION1) {
	        // Write the header, less bitflags and consumerID.
	        dos.write(headerBuffer, 0, HEADER_SIZE - 6);
            } else {
	        // Write the header, less bitflags and consumerID.
	        dos.write(headerBuffer, 0, HEADER_SIZE - 10);
            }
        } else {
            this.version = new_version;

            // We need to rebuild first part of header buffer :-(
            // This duplicates code in ReadWritePacket.updateHeaderBuffer()
            dos.writeInt(MAGIC);
	    dos.writeShort(version);
	    dos.writeShort(packetType);
	    dos.writeInt(packetSize);

            /*
             * In VERSION1 transactionID was a 32bit value in the header
             * In VERSION2 it is in the variable portion of the packet
             *
             * XXX BUG 11/02/2001 dipol: When going from VERSION1 to VERSION2
             * the transaction ID is lost since we do not update the ropBuffer.
             * We don't worry about this because this method is only used
             * by the broker and only when forwarding a packet to a client.
             * In this situation the transaction ID is never used by the client.
             */
            if (version == VERSION1) {
	        dos.writeInt((int)transactionID);
            }

	    dos.writeLong(expiration);

	    // Writes timestamp, source IP addr, source port,
            // and sequence number
	    sysMessageID.writeID(dos);

	    dos.writeInt(propertyOffset);
	    dos.writeInt(propertySize);
	    dos.writeByte(priority);
	    dos.writeByte(encryption);
        }

	// Write bit flags and interest ID

	int flags = bitFlags;

	if (new_redelivered) {
	    flags = flags | PacketFlag.R_FLAG;
	} else {
	    flags = flags & ~PacketFlag.R_FLAG;
	}

	if (new_lastPkt) {
	    flags = flags | PacketFlag.L_FLAG;
	} else {
	    flags = flags & ~PacketFlag.L_FLAG;
	}

	if (new_pauseFlow) {
	    flags = flags | PacketFlag.F_FLAG;
	} else {
	    flags = flags & ~PacketFlag.F_FLAG;
	}

	dos.writeShort(flags);

        // In VERSION1 consumer ID is 32 bits. In VERSION2 it is 64 bits
        if (new_version == VERSION1) {
	    dos.writeInt((int)new_consumerID);
        } else {
	    dos.writeLong(new_consumerID);
        }

	// Write rest of packet
	if (ropBuffer != null) {
	    dos.write(ropBuffer, 0, ropLength);
 	}

	dos.flush();
    }

    /**
     * Check if this packet matches the specified system message id.
     *
     * @return    "true" if the packet matches the specified id, else "false"
     */
    public boolean isEqual(SysMessageID id) {
	return sysMessageID.equals(id);
    }

    /*
     * Accessors for packet fields
     */

    public int getVersion() {
	return version;
    }

    public int getMagic() {
	return magic;
    }

    public int getPacketType() {
	return packetType;
    }

    public int getPacketSize() {
	return packetSize;
    }

    public long getTimestamp() {
	return sysMessageID.timestamp;
    }

    public long getExpiration() {
	return expiration;
    }

    public int getPort() {
	return sysMessageID.port;
    }

    public String getIPString() {
	return sysMessageID.toString();
    }

    public byte[] getIP() {
	return sysMessageID.getIPAddress();
    }

    public int getSequence() {
	return sysMessageID.sequence;
    }

    public int getPropertyOffset() {
	return propertyOffset;
    }

    public int getPropertySize() {
        return propertySize;
    }

    public int getEncryption() {
	return encryption;
    }

    public int getPriority() {
	return priority;
    }

    public long getTransactionID() {
	return transactionID;
    }

    public long getProducerID() {
	return producerID;
    }

    public long getConsumerID() {
	return consumerID;
    }

    // backwards compatibility
    public int getInterestID() {
	return (int)getConsumerID();
    }

    public boolean getPersistent() {
	return getFlag(PacketFlag.P_FLAG);
    }

    public boolean getRedelivered() {
	return getFlag(PacketFlag.R_FLAG);
    }

    public boolean getIsQueue() {
	return getFlag(PacketFlag.Q_FLAG);
    }

    public boolean getSelectorsProcessed() {
	return getFlag(PacketFlag.S_FLAG);
    }

    public boolean getSendAcknowledge() {
	return getFlag(PacketFlag.A_FLAG);
    }

    public boolean getIsLast() {
	return getFlag(PacketFlag.L_FLAG);
    }

    public boolean getFlowPaused() {
	return getFlag(PacketFlag.F_FLAG);
    }

    public boolean getIsTransacted() {
	return getFlag(PacketFlag.T_FLAG);
    }

    public boolean getConsumerFlow() {
	return getFlag(PacketFlag.C_FLAG);
    }

    public boolean getIndempotent() {
	return getFlag(PacketFlag.I_FLAG);
    }

    public String getDestination() {
	return destination;
    }

    public String getDestinationClass() {
	return destinationClass;
    }

    /**
     * Get the MessageID for the packet. If the client has set a MessageID
     * then that is what is returned. Otherwise the system message ID is 
     * returned (see getSysMessageID())
     *
     * @return     The packet's MessageID
     */
    public String getMessageID() {
	if (messageID == null) {
	    return sysMessageID.toString();
        } else {
	    return messageID;
	}
    }

    public String getCorrelationID() {
	return correlationID;
    }

    public String getReplyTo() {
	return replyTo;
    }

    public String getReplyToClass() {
	return replyToClass;
    }

    public String getMessageType() {
	return messageType;
    }

    /**
     * Get the system message ID. Note that this is not the JMS MessageID
     * set by the client. Rather this is a system-wide unique message ID
     * generated from the timestamp, sequence number, port number and
     * IP address of the packet.
     * <P>
     * WARNING! This returns a references to the Packet's SysMessageID
     * not a copy.
     * 
     * @return     The packet's system MessageID
     *
     */
    public SysMessageID getSysMessageID() {

	return sysMessageID;
    }

    /**
     * Return the size of the message body in bytes
     *
     * @return    Size of message body in bytes
     */
    public int getMessageBodySize() {
	return (packetSize - propertyOffset - propertySize);
    }


    /**
     * Return an InputStream that contains the contents of the
     * message body.
     *
     * @return    An InputStream from which the message body can
     *            be read from. Or null if no message body.
     */
    public InputStream getMessageBodyStream() {

	int offset = (propertyOffset - HEADER_SIZE) + propertySize;
	int size   = ropLength - offset;

	if (ropBuffer == null || size <= 0) {
	    return null;
        } else {
	    return new ByteArrayInputStream(ropBuffer, offset, size);
	}
    }


    /**
     * Return the property hashtable for this packet.
     *
     * WARNING! This method emphasizes performance over safety. The
     * HashTable object returned is a reference to the HashTable object
     * in the object -- it is NOT a copy. Modifying the contents of
     * the HashTable will have non-deterministic results so don't do it!
     */
    public Hashtable getProperties()
	throws IOException, ClassNotFoundException {

	if (properties == null && propertySize > 0) {
	    // Properties haven't been parsed from buffer. Do so now.
	    synchronized(this) {
		if (properties == null) {
		    // We subtract HEADER_SIZE since ropBuffer doesn't
		    // include the fixed header
		    parseProperties(new ByteArrayInputStream(ropBuffer,
			propertyOffset - HEADER_SIZE, propertySize));
		}
	    }
	}

	return properties;
    }

    /**
     * Reset packet to initial values
     */
    protected void reset() {
	version        = defaultVersion;
	magic          = MAGIC;
        ropLength      = 0;
        packetType     = 0;
        packetSize     = 0;
        expiration     = 0;
        propertyOffset = 0;
        propertySize   = 0;
        encryption     = 0;
        priority       = 5;
        bitFlags       = 0;
        consumerID     = 0;
        producerID     = 0;
        transactionID  = 0;

	sysMessageID.clear();

        destination      = null;
        destinationClass = null;
        messageID        = null;
        correlationID    = null;
        replyTo          = null;
        replyToClass     = null;
        messageType      = null;

        stringsParsed    = false;

        properties       = null;
    }

    /**
     * Make a shallow copy of this packet. Warning! This does not
     * copy any of the internal buffers.
     */
    public Object cloneShallow() {
	try {
	    // Call Object.clone() which does a shallow copy
	    return(super.clone());
	} catch (CloneNotSupportedException e) {
	    // Should never get this, but don't fail silently
	    System.out.println("ReadOnlyPacket: Could not clone: " + e);
	    return null;
	}
    }

    /**
     * Make a deep copy of this packet. This will be slow.
     */
    public Object clone() {
        // Make a shallow copy first
        ReadOnlyPacket newPkt = (ReadOnlyPacket)this.cloneShallow();

        // Now do deep copy of buffers and other classes. We don't need to
	// copy Strings since they are immutable -- having two packets
	// share the same Strings is not a problem.
	if (this.headerBuffer != null) {
	    newPkt.headerBuffer = (byte[])this.headerBuffer.clone();
	}
	if (this.ropBuffer != null) {
	    newPkt.ropBuffer    = (byte[])this.ropBuffer.clone();
        }

	// XXX BUG 05/11/00 dipol This does not do a deep copy of
	// the Hashtable. Will fix this soon.
	if (this.properties != null) {
	    newPkt.properties   = (Hashtable)this.properties.clone();
        }

	newPkt.sysMessageID = (SysMessageID)this.sysMessageID.clone();

	return (newPkt);
    }


    /**
     * Return a unique string that identifies the packet
     */
    public String toString() {
	return
	    PacketType.getString(packetType) + ":" +
	    sysMessageID.toString();
    }

    /**
     * Return a string containing the contents of the packet in a human
     * readable form.
     */
    public String toVerboseString() {
        ByteArrayOutputStream bos = new ByteArrayOutputStream();

	this.dump(new PrintStream(bos));

	return bos.toString();
    }

    /**
     * Dump the contents of the packet in human readable form to
     * the specified OutputStream.
     *
     * @param    os    OutputStream to write packet contents to
     */
    public void dump(PrintStream os) {

	os.println(

	"******** Packet: " + this.toString() + "\n" +


	"  Magic/Version: " + magic + "/" + version + "\tSize: " + packetSize +
		"\t Type: " + PacketType.getString(packetType) + "\n" +

	"     Expiration: " + expiration +
        ((version == VERSION1) ? ("   TransactionID: " + transactionID) : "\t") +
	"       Timestamp: " + sysMessageID.timestamp +
	"\n" +

	"      Source IP: " + sysMessageID.ip.toString() +
	      "\t  Source Port: " + sysMessageID.port +
		"\tSequence: " + sysMessageID.sequence + "\n" +
	
	"Property Offset: " + propertyOffset +
		"\t\t\tProperty Size: " + propertySize + "\n" +
	
	"     Encryption: " + encryption + "\tPriority: " +
		priority + "\n" +

	
	"          Flags: " + PacketFlag.getString(bitFlags) +
		"\t\t\t   consumerID: " +
						consumerID);

        if (version >= VERSION2) {
	    os.println("  TransactionID: " + getTransactionID());
        }
        if (getProducerID() != 0) {
	    os.println("      ProducerID: " + getProducerID());
        }
	if (destination != null)
	    os.println("     Destination: " + destination);
	if (destinationClass != null)
	    os.println("DestinationClass: " + destinationClass);
	if (messageID != null)
	    os.println("       MessageID: " + messageID);
	if (correlationID != null)
	    os.println("   CorrelationID: " + correlationID);
	if (replyTo != null)
	    os.println("         ReplyTo: " + replyTo);
	if (replyToClass != null)
	    os.println("    ReplyToClass: " + replyToClass);
	if (messageType != null)
	    os.println("     MessageType: " + messageType);

	Hashtable props = null;
	try {
	    props = getProperties();
	} catch (Exception e) {
	    os.println("Exception getting properties: " + e.getMessage());
	}
	if (props == null) {
	    os.println("     Properties: null");
        } else {
	    os.println("     Properties: " + props);
	}

        Packet.dumpBody(os, packetType,
            getMessageBodyStream(), getMessageBodySize(), props);

	os.println("Internal headerBuffer: " + headerBuffer + " " +
		headerBuffer.length + " bytes");
	if (ropBuffer != null) {
	    os.println("   Internal ropBuffer: " + ropBuffer + " " +
		ropBuffer.length + " bytes" +
		", content: " + ropLength + " bytes");
	} else {
	    os.println("   Internal ropBuffer: null");
	}

    }
}
