/*
 * @(#)BrokerProcess.java	1.12 02/08/06
 *
 * Copyright 2005 Sun Microsystems, Inc. All Rights Reserved
 * SUN PROPRIETARY/CONFIDENTIAL. Use is subject to license terms. 
 *
 */

package com.sun.messaging.jmq.jmsserver;

import java.util.*;

/**
 * Wrapper used to start the broker. It wraps a singleton class
 * (only one broker can be running in any process).<P>
 *
 * <u>Example</u><P>
 * <code><PRE>
 *      BrokerProcess bp = BrokerProcess.getBrokerProcess();
 *      try {
 *      
 *          Properties ht = BrokerProcess.convertArgs(args);
 *          int exitcode = bp.start(true, ht, null);
 *          System.out.println("Broker exited with " + exitcode);
 *
 *      } catch (IllegalArgumentException ex) {
 *          System.err.println("Bad Argument " + ex.getMessage());
 *          System.out.println(BrokerProcess.usage());
 *      }
 * </PRE></code>
 */
public class BrokerProcess
{
    Broker b = null;
    private static BrokerProcess bp = null;

    /**
     * Constructor
     */
    public BrokerProcess() {
        b = Broker.getBroker();
    }

    /**
     * Retrieve the singleton BrokerProcess object
     */
    public static BrokerProcess getBrokerProcess()
    {
        synchronized (BrokerProcess.class) {
            if (bp == null)
                bp = new BrokerProcess();
        }

        return bp;
    }

    /**
     * Change command line args into a hashtable format
     *<P>
     * Additional arguments are:
     *    <UL>
     *       <LI> -varhome</LI>
     *       <LI> -imqhome</LI>
     *    </UL>
     *
     * @param args arguments in broker format
     */
    public static Properties convertArgs(String[] args)
        throws IllegalArgumentException
    {
        Properties props = new Properties();

        // first look for var home and the like
        for (int i =0; i < args.length; i ++) {
            String arg = args[i];
            if (arg.equals("-varhome")) {
                props.setProperty("imq.varhome",
                        args[i+1]);
                i ++;
            } else if (arg.equals("-imqhome")) {
                props.setProperty("imq.home",
                        args[i+1]);
                i ++;
            } else if (arg.equals("-libhome")) {
                props.setProperty("imq.libhome",
                        args[i+1]);
                i ++;
            }
        }
        Globals.pathinit(props);
        return getBrokerProcess().b.convertArgs(args);
    }

    /**
     * Usage string
     *
     */
    public static String usage()
    {
        return getBrokerProcess().b.usage();
    }

    /**
     * Checks the state of the Broker
     *
     * @return the state of the broker
     */
    public boolean isRunning() {
        return true;
    }

    /**
     * Start the broker (only one broker can be running in a given
     * vm).<p>This call returns as soon as the broker sucessfully starts.
     * @param inProcess - indicates that the broker is running inprocess
     *                    and the shutdown hook and memory management
     *                    code should not be used.
     * @param properties - configuration setttings for the broker
     *
     * @param bn - optional class to notify when a broker has completed
     *             starting or has been shutdown.
     *
     * @return the exit code what would be returned by the broker if it
     *       was running as a standalone process. (or 0 if it sucessfully
     *       started).
     *
     * @throws OutOfMemoryError - if the broker can not allocate enough 
     *          memory to continue running
     * @throws IllegalStateException - the broker is already running.  
     * @throws IllegalArgumentException - an invalid value for a property
     *                was passed on the command line
     */
    public int start(boolean inProcess, Properties properties, BrokerNotification bn)
        throws OutOfMemoryError, IllegalStateException, IllegalArgumentException
    {
        return getBrokerProcess().b.start(inProcess, properties, bn);
        
    }
    /**
     * init the broker (only one broker can be running in a given
     * vm).<p>This call returns as soon as the broker sucessfully initializes.
     * Equivalent to starting the broker with -init.
     * @param inProcess - indicates that the broker is running inprocess
     *                    and the shutdown hook and memory management
     *                    code should not be used.
     * @param properties - configuration setttings for the broker
     *
     * @param bn - optional class to notify when a broker has completed
     *             starting or has been shutdown.
     *
     * @return the exit code what would be returned by the broker if it
     *       was running as a standalone process. (or 0 if it sucessfully
     *       started).
     *
     * @throws OutOfMemoryError - if the broker can not allocate enough 
     *          memory to continue running
     * @throws IllegalStateException - the broker is already running.  
     * @throws IllegalArgumentException - an invalid value for a property
     *                was passed on the command line
     */
    public int init(boolean inProcess, Properties properties, BrokerNotification bn)
        throws OutOfMemoryError, IllegalStateException, IllegalArgumentException
    {
        return getBrokerProcess().b.init(inProcess, properties, bn);
        
    }

    /**
     * Stop the broker (only one broker can be running in a given
     * vm).<p>
     * @param cleanup - if false, the code does not need to worry about freeing
     *                  unused resources. (broker is about to exit)
     * @throws IllegalStateException - the broker is already stopped.  
     */
    public void stop(boolean cleanup)
        throws IllegalStateException
    {

        b.destroyBroker(cleanup);
        b = null;
        bp = null;
    }

    public static void main(String args[]) {
try {

        BrokerProcess bp = getBrokerProcess();

        BrokerNotification bn = new BrokerNotification()
            {
                public boolean exitRequested(int reason,
                   String info, Throwable thr) {
System.out.println("EXIT REQUESTED " + reason);
                      return false;
                }

                public void brokerShutdown() {
                   System.err.println("XXX should never be called");
                }

                public void brokerStarted() {
                   System.err.println("XXX hey broker started");
                }
            };

        String newargs[] = new String[args.length + 4];
        int i =0;
        for (i =0; i < args.length; i ++)
            newargs[i] = args[i];
        newargs[i++] = "-varhome";
        newargs[i++] = "../../../solaris/opt/var";
        newargs[i++] = "-imqhome";
        newargs[i++] = "../../../solaris/opt/";
       
        Properties p =bp.convertArgs(newargs);

System.err.println("HEY INIT");
        int exit = bp.init(true, p, bn);
System.err.println("HEY START");
        exit = bp.start(true, p, bn);
System.err.println("HEY STARTED");
        Thread.currentThread().sleep(10*1000);
        System.out.println("HEY SHUTTING DOWN");
        bp.stop(true);
        System.out.println("Sleeping");
        Thread.currentThread().sleep(1000*10);
        System.out.println("Restarting");
        exit = bp.start(true, p, null);
        System.out.println("Sleeping");
        Thread.currentThread().sleep(10*1000);
        bp.stop(true);
        System.out.println("HEY SHUTTING DOWN");
        System.out.println("Sleeping");
        Thread.currentThread().sleep(1000*60);
        System.exit(0);
} catch (Exception ex) {
ex.printStackTrace();
}

    }
}


