
/*
 * @(#)MultibrokerRouter.java	1.22 02/08/06
 *
 * Copyright 2005 Sun Microsystems, Inc. All Rights Reserved
 * SUN PROPRIETARY/CONFIDENTIAL
 * Use is subject to license terms. 
 *
 */

package com.sun.messaging.jmq.jmsserver.core.cluster;

import javax.transaction.xa.XAResource;
import java.util.*;
import java.io.*;
import com.sun.messaging.jmq.util.log.*;
import com.sun.messaging.jmq.util.*;
import com.sun.messaging.jmq.jmsserver.Globals;
import com.sun.messaging.jmq.jmsserver.DMQ;
import com.sun.messaging.jmq.jmsserver.core.*;
import com.sun.messaging.jmq.jmsserver.data.TransactionUID;
import com.sun.messaging.jmq.jmsserver.data.PacketRouter;
import com.sun.messaging.jmq.jmsserver.data.handlers.TransactionHandler;
import com.sun.messaging.jmq.jmsserver.data.TransactionState;
import com.sun.messaging.jmq.jmsserver.data.AutoRollbackType;
import com.sun.messaging.jmq.jmsserver.data.TransactionList;
import com.sun.messaging.jmq.jmsserver.multibroker.ClusterGlobals;
import com.sun.messaging.jmq.jmsserver.multibroker.Protocol;
import com.sun.messaging.jmq.jmsserver.util.BrokerException;
import com.sun.messaging.jmq.jmsserver.service.ConnectionUID;
import com.sun.messaging.jmq.jmsserver.util.lists.*;
import com.sun.messaging.jmq.jmsserver.util.*;
import com.sun.messaging.jmq.io.*;
import java.lang.ref.*;
import com.sun.messaging.jmq.jmsserver.util.lists.*;
import com.sun.messaging.jmq.util.lists.*;
import com.sun.messaging.jmq.util.selector.*;

/**
 * This class handles the processing of messages from other brokers
 * in the cluster.
 */
public class MultibrokerRouter implements ClusterRouter
{
    public static boolean DEBUG = true;
    private static Logger logger = Globals.getLogger();

    ClusterBroadcast cb = null;
    Protocol p = null;

    BrokerConsumers bc = null;

    public MultibrokerRouter(ClusterBroadcast cb)
    {
        this.cb = cb;
        this.p = cb.getProtocol();
        bc = new BrokerConsumers(p);
    }

    public static String msgToString(int id)
    {
        switch(id) {
            case ClusterBroadcaster.MSG_DELIVERED:
                return "MSG_DELIVERED";
            case ClusterBroadcaster.MSG_ACKNOWLEDGED:
                return "MSG_ACKNOWLEDGED";
            case ClusterBroadcaster.MSG_TXN_ACKNOWLEDGED:
                return "MSG_TXN_ACKNOWLEDGED";
            case ClusterBroadcaster.MSG_PREPARED:
                return "MSG_PREPARED";
            case ClusterBroadcaster.MSG_ROLLEDBACK:
                return "MSG_ROLLEDBACK";
            case ClusterBroadcaster.MSG_IGNORED:
                return "MSG_IGNORED";
            case ClusterBroadcaster.MSG_UNDELIVERABLE:
                return "MSG_UNDELIVERABLE";
            case ClusterBroadcaster.MSG_DEAD:
                return "MSG_DEAD";
        }
        return "UNKNOWN";
    }

    public void addConsumer(Consumer c) 
       throws BrokerException, IOException, SelectorFormatException
    {
        bc.addConsumer(c);
    }

    public void removeConsumer(com.sun.messaging.jmq.jmsserver.core.ConsumerUID c)
       throws BrokerException, IOException
    {
        bc.removeConsumer(c);
    }


    public void removeConsumers(ConnectionUID uid)
       throws BrokerException, IOException
    {
        bc.removeConsumers(uid);
    }


    public void brokerDown(com.sun.messaging.jmq.jmsserver.core.BrokerAddress ba)
       throws BrokerException, IOException
    {
        bc.brokerDown(ba);
    }

    public void forwardMessage(PacketReference ref, Collection consumers) {
        bc.forwardMessageToRemote(ref, consumers);
    }



    public void handleJMSMsg(Packet p, List consumers, BrokerAddress sender,
              boolean sendMsgDeliveredAck)
        throws BrokerException
    {
        ArrayList targetVector = new ArrayList();
        ArrayList ignoreVector = new ArrayList();

        Iterator itr = consumers.iterator();

        while (itr.hasNext()) {
            ConsumerUID uid = (ConsumerUID)itr.next();
            Consumer interest = Consumer.getConsumer(uid);

            if (interest != null) {
                // we need the interest for updating the ref
                targetVector.add(interest);
            } else {
                ignoreVector.add(uid);
            }
        }

        boolean exists = false;
        PacketReference ref = Destination.get(p.getSysMessageID());
        if (ref != null) {
                exists = true;
                ref.setBrokerAddress(sender);
                if (p.getRedelivered()) ref.overrideRedeliver();
        } else {
                ref = PacketReference.createReference(p, null);
                ref.setBrokerAddress(sender);
        }
        // XXX - note, we send a reply for all message delivered
        //       acks, that is incorrect
        //       really, we should have a sendMsgDeliveredAck per
        //       consumer
        if (sendMsgDeliveredAck) {
            for (int i=0; i < targetVector.size(); i ++) {
                Consumer c = (Consumer)targetVector.get(i);
                //ref.addMessageDeliveredAck(c.getStoredConsumerUID());
                ref.addMessageDeliveredAck(c.getConsumerUID());
            }
        }
        try {
            if (ref == null)  {
                return;
            }
            Destination d = Destination.getDestination(
                ref.getDestinationUID().getName(), 
                ref.getDestinationUID().isQueue() ? DestType.DEST_TYPE_QUEUE
                   : DestType.DEST_TYPE_TOPIC, true, true);
            if (d == null) {
               ignoreVector.addAll(targetVector); 
            } else {
                if (!exists && !targetVector.isEmpty()) {
                    ref.setNeverStore(true);
                    // OK .. we dont need to route .. its already happened
                    ref.store(targetVector);
                    d.queueMessage(ref, false); // add to message count
                } else if (exists) {
                    ref.add(targetVector);
                }
            }
        } catch (IOException ex) {
            logger.logStack(logger.INFO,"Internal Exception ", ex);
            ignoreVector.addAll(targetVector); 
        } catch (BrokerException ex) {
            // unable to store
            ignoreVector.addAll(targetVector); 
        }

        // Now deliver the message...
        String debugString = "\n";

        int i;
        for (i = 0; i < targetVector.size(); i++) {
            Consumer interest = (Consumer)targetVector.get(i);

            if (!interest.routeMessage(ref, false)) {
                // it disappeard on us, take care of it
               try {
                    if (ref.acknowledged(interest.getConsumerUID(),
                          interest.getStoredConsumerUID(),
                          true, false)) {
                        Destination d=Destination.getDestination(ref.getDestinationUID());
                        d.removeMessage(ref.getSysMessageID(),
                               RemoveReason.ACKNOWLEDGED);
                    }
                } catch (Exception ex) {
                    logger.log(logger.INFO,"Internal error processing ack",
                           ex);
                }
            }



            if (DEBUG) {
                debugString = debugString +
                    "\t" + interest.getConsumerUID() + "\n";
            }
        }

        if (DEBUG) {
            logger.log(logger.DEBUGHIGH,
                "MessageBus: Delivering message to : {0}",
                debugString);
        }

        debugString = "\n";
        // Finally, send  ClusterGlobals.MB_MSG_IGNORED acks if any...
        Object o = null;
        ConsumerUID cuid = null;
        for (i = 0; i < ignoreVector.size(); i++) {
            try {
                o = ignoreVector.get(i);
                if (o instanceof Consumer) cuid = ((Consumer)o).getConsumerUID(); 
                else cuid = (ConsumerUID)o;
                cb.acknowledgeMessage(sender, ref.getSysMessageID(), cuid,
                    ClusterBroadcast.MSG_IGNORED,
                    false /* dont wait for ack */, null, null /* txnID */);
            } catch (Exception e) {
                logger.log(logger.WARNING, "sendMessageAck IGNORE failed to"+sender); //XXX
            }

            if (DEBUG) {
                debugString = debugString +
                    "\t" + ignoreVector.get(i) + "\n";
            }
        }

        if (DEBUG) {
            if (ignoreVector.size() > 0)
                logger.log(logger.DEBUGHIGH,
                    "MessageBus: Invalid targets : {0}",
                    debugString);
        }

    }


   /**
     * txnID must not be null for ack types
     *     MSG_TXN_ACKNOWLEDGED
     *     MSG_PREPARE
     *     MSG_ROLLEDBACK
     *     MSG_ACKNOWLEDGED (transacted)
     *
     * msgBrokerSession and msgStoreSession are passed in for all ack types
     *     msgBrokerSession null if the ack is from < 4.0 broker
     *     msgStoreSession  null if non-HA
     */
    public void handleAck(int type, SysMessageID id, ConsumerUID uid,
                          Map optionalProps, Long txnID, 
                          UID msgBrokerSession, UID msgStoreSession) 
                          throws BrokerException 
    {

       logger.log(logger.DEBUG,"RECEIVED " + msgToString(type) + " for " +
                   id + ":" + uid + " TXN=" + txnID);
       TransactionUID tuid = null;
       if (txnID != null)
           tuid = new TransactionUID(txnID.longValue());

       TransactionList tl = Globals.getTransactionList();
       switch (type)
       {
           // ok .. I hate the fact that this doesn't use the existing
           // txn commit, rollback, etc classes
           // but it only has to do SOME of it so this is easier for
           // now
           // next release the code should be unified somehow
           case ClusterBroadcast.MSG_TXN_ACKNOWLEDGED:
           {
                 // create txn if necessary (load from DB if HA)
                 // we only store if NOT HA (otherwise its already stored)
                 // add ACK (save to file)
               TransactionState state= tl.retrieveState(tuid);
               if (state == null) {
                   tl.addRemoteTxn(uid.getBrokerAddress(),
                       tuid);
               }
               // ok add ack
               Consumer c = Consumer.getConsumer(uid);
               Consumer p = c.getSubscription();
               Consumer usedC = (p == null) ?  c : p;
               tl.addAcknowledgement(tuid, id, usedC.getConsumerUID(), 
                       usedC.getStoredConsumerUID(), !Globals.getHAEnabled());
               
               break;
           }
           case ClusterBroadcast.MSG_PREPARED:
           {
               // ok if we end up here CID and SYSID better be null
               if (id != null || uid != null)
                   throw new BrokerException("INTERNAL ERROR:" + 
                          " bad protocol");
               // check state, if in-memory not prepared, set
               // if not HA, store
               TransactionState state= tl.retrieveState(tuid);
               if (state == null) {
                   logger.log(Logger.DEBUG,"XXX PREPARE - state is not there for " + tuid);
               } else if (state.getState() != TransactionState.PREPARED) {
                   // ok
                   state.setState(TransactionState.PREPARED);
               }
               break;
           }
           case ClusterBroadcast.MSG_ROLLEDBACK:
           {
               // ok if we end up here CID and SYSID better be null
               if (id != null || uid != null)
                   throw new BrokerException("INTERNAL ERROR:" + 
                          " bad protocol");

               tl.rollbackRemoteTxn(tuid);

               break;
           }
           case ClusterBroadcast.MSG_ACKNOWLEDGED:
               if (tuid != null) {
                   // hey just commit and ack
                   if (id != null || uid != null)
                       throw new BrokerException("INTERNAL ERROR:" + 
                              " bad protocol");
                   tl.commitRemoteTxn(tuid);
                   break;
               }
           case ClusterBroadcast.MSG_DELIVERED:
           case ClusterBroadcast.MSG_IGNORED:
           case ClusterBroadcast.MSG_UNDELIVERABLE:
           case ClusterBroadcast.MSG_DEAD:
               bc.acknowledgeMessageFromRemote(id, uid, type, optionalProps, txnID);
               break;
       }

       //handleAcks(type, ids, uids, txnID);
    }


    public void handleCtrlMsg(int type, HashMap props)
        throws BrokerException
    {
        // not implemented
    }

}


/**
 * This class represents the remote Consumers associated with
 * the brokers in this cluster.
 */



class BrokerConsumers  implements Runnable,       com.sun.messaging.jmq.util.lists.EventListener
{
    public static boolean DEBUG = false;

    Thread thr = null;

    Logger logger = Globals.getLogger();
    Protocol protocol = null;
    boolean valid = true;
    Set activeConsumers= Collections.synchronizedSet(new HashSet());
    Map consumers= Collections.synchronizedMap(new HashMap());
    Map listeners = Collections.synchronizedMap(new HashMap());
    Set txnUIDs = Collections.synchronizedSet(new HashSet());


    public static int BTOBFLOW = Globals.getConfig().getIntProperty(
               Globals.IMQ + ".cluster.consumerFlowLimit",1000); 

    Map deliveredMessages = new LinkedHashMap();

    public BrokerConsumers(Protocol p)
    {
        this.protocol = p;
        Thread thr =new MQThread(this,"Broker Monitor");
        thr.setDaemon(true);
        thr.start();
    }

    class ackEntry
    {
        com.sun.messaging.jmq.jmsserver.core.ConsumerUID uid = null;
        com.sun.messaging.jmq.jmsserver.core.ConsumerUID storedcid = null;
        WeakReference pref = null;
        SysMessageID id = null;
        com.sun.messaging.jmq.jmsserver.core.BrokerAddress address = null;

        public ackEntry(SysMessageID id,
              com.sun.messaging.jmq.jmsserver.core.ConsumerUID uid,
              com.sun.messaging.jmq.jmsserver.core.BrokerAddress address) 
        { 
             assert id != null;
             assert uid != null;
             this.id = id;
             this.uid = uid;
             this.address = address;
             pref = null;
        }

        public com.sun.messaging.jmq.jmsserver.core.BrokerAddress 
               getBrokerAddress() 
        {
             return address;
        }

        public com.sun.messaging.jmq.jmsserver.core.ConsumerUID 
               getConsumerUID() 
        {
            return uid;
        }
        public SysMessageID getSysMessageID() {
            return id;
        }
        public PacketReference getReference() {
            return (PacketReference)pref.get();
        }
        

        public ackEntry(PacketReference ref, 
               com.sun.messaging.jmq.jmsserver.core.ConsumerUID uid, 
               com.sun.messaging.jmq.jmsserver.core.ConsumerUID storedUID) 
        {
            pref = new WeakReference(ref);
            id = ref.getSysMessageID();
            storedcid = storedUID;
            this.uid = uid;
        }

        public void acknowledged(boolean notify, Long txnid) 
            throws BrokerException
        {

            assert pref != null;

            PacketReference ref = 
                (PacketReference)pref.get();

            try {
                if (ref == null) {
                    ref = Destination.get(id);
                }
                if (ref == null) { // nothing we can do ?
                    return;
                }
                if (ref.acknowledged(uid, storedcid, !uid.isDupsOK(), notify)) {
                    Destination d=Destination.getDestination(ref.getDestinationUID());
                    d.removeMessage(ref.getSysMessageID(),
                      RemoveReason.ACKNOWLEDGED);
                }
            } catch (Exception ex) {
                logger.logStack(Logger.DEBUG,"Unable to process acknowledgement, Ignoring", ex);
                if (ex instanceof BrokerException) 
                    throw (BrokerException)ex;
                throw new BrokerException(ex.getMessage(), ex, Status.ERROR);
            }
        }

        public boolean equals(Object o) {
            if (! (o instanceof ackEntry)) {
                return false;
            }
            ackEntry ak = (ackEntry)o;
            return uid.equals(ak.uid) &&
                   id.equals(ak.id);
        }
        public int hashCode() {
            // uid is 4 bytes
            return id.hashCode()*15 + uid.hashCode();
        }
    }


    public void destroy() {
        valid = false;
        synchronized(activeConsumers) {
            activeConsumers.notify();
        }
    }


    public void eventOccured(EventType type,  Reason r,
            Object target, Object oldval, Object newval, 
            Object userdata) 
    {
        if (type != EventType.BUSY_STATE_CHANGED) {
            assert false; // bad type
        }
        // OK .. add to busy list
        Consumer c = (Consumer)target;

        synchronized(activeConsumers) {
            if (c.isBusy() ) {
                activeConsumers.add(c);
            }
            activeConsumers.notify();
        }
    }

    public void brokerDown
         (com.sun.messaging.jmq.jmsserver.core.BrokerAddress address) 
        throws BrokerException
    {

        Set removedConsumers = new HashSet();
        // get list of consumers
        synchronized(consumers) {
            Iterator itr = consumers.keySet().iterator();
            while (itr.hasNext()) {
                com.sun.messaging.jmq.jmsserver.core.ConsumerUID cuid 
                        = (com.sun.messaging.jmq.jmsserver.core.ConsumerUID)itr.next();
                if (address.equals(cuid.getBrokerAddress()) &&
                    address.getBrokerSessionUID().equals(
                                     cuid.getBrokerAddress().getBrokerSessionUID())) {
                    // found one
                    removedConsumers.add(cuid);
                }
            }
        }
        // destroy consumers
        Iterator itr = removedConsumers.iterator();
        while (itr.hasNext()) {
            com.sun.messaging.jmq.jmsserver.core.ConsumerUID cuid = (com.sun.messaging.jmq.jmsserver.core.ConsumerUID)itr.next();
            removeConsumer(cuid);
        }
        if (Globals.getHAMonitorService() != null) {
            Globals.getHAMonitorService().brokerDown(address);
        }
    }

    // topic send
    public void forwardMessageToRemote(PacketReference ref, Collection cons)
    {
         Iterator itr = cons.iterator();
         while (itr.hasNext()) {
             // hey create an ack entry since we are bypassing consumer
            Consumer consumer = (Consumer)itr.next();
            com.sun.messaging.jmq.jmsserver.core.ConsumerUID sid = consumer.getStoredConsumerUID();

            // if we dont have an ack, go on
            if (consumer.getConsumerUID().isNoAck()) continue;
            ackEntry entry = new ackEntry(ref, consumer.getConsumerUID(), sid);
            synchronized(deliveredMessages) {
                deliveredMessages.put(entry, entry);
            }
         }
         protocol.sendMessage(ref, cons, false);
    }

    public void removeConsumers(ConnectionUID uid) 
        throws BrokerException
    {
        Set removedConsumers = new HashSet();
        // get list of consumers
        synchronized(consumers) {
            Iterator itr = consumers.keySet().iterator();
            while (itr.hasNext()) {
                com.sun.messaging.jmq.jmsserver.core.ConsumerUID cuid = (com.sun.messaging.jmq.jmsserver.core.ConsumerUID)itr.next();
                if (uid.equals(cuid.getConnectionUID())) {
                    // found one
                    removedConsumers.add(cuid);
                }
            }
        }
        // destroy consumers
        Iterator itr = removedConsumers.iterator();
        while (itr.hasNext()) {
            com.sun.messaging.jmq.jmsserver.core.ConsumerUID cuid = (com.sun.messaging.jmq.jmsserver.core.ConsumerUID)itr.next();
            removeConsumer(cuid);
        }
    }

    public void removeConsumer(
       com.sun.messaging.jmq.jmsserver.core.ConsumerUID uid) 
        throws BrokerException
    {
          Consumer c = (Consumer)consumers.remove(uid);

          if (c == null) return;

          c.pause("MultiBroker - removing consumer");
          // remove it from the destination
          Destination d= Destination.getDestination(
              c.getDestinationUID());

          // quit listening for busy events
          Object listener= listeners.remove(uid);
          if (listener != null) {
              c.removeEventListener(listener);
          }

          // remove it from the active list
          activeConsumers.remove(c);

          Set destroySet = new LinkedHashSet();
          // OK .. get the acks .. if its a FalconRemote
          // we sent the messages directly and must explicitly ack
          synchronized(deliveredMessages) {
              Iterator itr = deliveredMessages.values().iterator();
              while (itr.hasNext()) {
                  ackEntry e = (ackEntry)itr.next();
                  if (e.getConsumerUID() == uid) {
                     itr.remove();
                     if (c.isFalconRemote()) {
                         e.acknowledged(false, null);
                     } else {
                         destroySet.add(e.getReference());
                     }
                  }
              }

          }

          c.destroyConsumer(destroySet, false, false);
          if (d != null)  {
              d.removeConsumer(uid, false);
          }
    }

    public void acknowledgeMessageFromRemote(SysMessageID id,
                com.sun.messaging.jmq.jmsserver.core.ConsumerUID uid, int ackType,
                Map optionalProps, Long txnID)
        throws BrokerException
    {
        if (ackType == ClusterBroadcast.MSG_DELIVERED) {
            Consumer c = Consumer.getConsumer(uid);
            if (c != null) c.resumeFlow();
            return;
        } else if (ackType == ClusterBroadcast.MSG_IGNORED) {
           /* dont do anything .. we will soon be closing the consumer and that
            * will cause the right things to happen 
            */
            if (DEBUG) {
                logger.log(Logger.DEBUG, "got message ignored ack, can not process [" +
                                          id + "," + uid +"]" + ackType);
            }
            return;
        } else if (ackType == ClusterBroadcast.MSG_PREPARED) {
            // OK .. this should never happen
            // PREPARE should go through the other method call
            // handleAcks
            logger.log(Logger.INFO,"UNEXPECTED PREPARE [ " + id + "," + uid + "] " + txnID);
            return;

        } else if (ackType == ClusterBroadcast.MSG_ROLLEDBACK) {
            // ok, these messages were produced here and are not being rolled back
            // handle it just like a normal rollback
            logger.log(Logger.DEBUG,"ROLLEDBACK [ " + id + "," + uid + "] " + txnID);
            return;
        } else if (ackType != ClusterBroadcast.MSG_ACKNOWLEDGED && 
            ackType != ClusterBroadcast.MSG_UNDELIVERABLE &&
            ackType != ClusterBroadcast.MSG_DEAD) {
            logger.log(Logger.WARNING, "XXXI18n Unknown ack type[" +
                                        id + "," + uid +"]" + ackType);
            throw new BrokerException("XXXI18N Unknown ack type internal error",
                     Status.ERROR);
        }

        // NOW handle COMMIT
        // OK, if we have a txnid ... make an effort to commit it
        if (txnID != null) {
            TransactionUID tuid = new TransactionUID(txnID.longValue());

            // OK, get the transaction state to see if it exists
            TransactionList tl = Globals.getTransactionList();
            TransactionState ts = tl.retrieveState(tuid);
            
            if (ts != null && !Globals.getHAEnabled()) {
                JMQXid xid = tl.UIDToXid(tuid);
                PacketRouter pr = Globals.getPacketRouter(0);
                TransactionHandler thandler = (TransactionHandler)
                                pr.getHandler(PacketType.ROLLBACK_TRANSACTION);
                thandler.doCommit(tuid, xid, new Integer(XAResource.TMNOFLAGS), ts, null,
                                false, null, null);
            } else {
            }
            // doCommit
            return;
        }

        // NOW HANDLE ACK
        ackEntry entry = new ackEntry(id, uid, null);
        synchronized(deliveredMessages) {
            ackEntry value = (ackEntry)deliveredMessages.remove(entry);
            if (ackType == ClusterBroadcast.MSG_DEAD ||
                ackType == ClusterBroadcast.MSG_UNDELIVERABLE) { 

                PacketReference ref = Destination.get(id);
                if (ref == null) return;

                Destination d= ref.getDestination();
                if (d == Destination.DeadMessageQueue) {
                    // already gone, ignore
                    return;
                }
                // first pull out properties
                String comment = null;
                RemoveReason reason = null;
                Exception ex = null;
                Integer deliverCnt = null;
                Integer reasonInt = null;
                String deadbkr = null;
                if (optionalProps != null) {
                    comment = (String)optionalProps.get(DMQ.UNDELIVERED_COMMENT);
                    ex = (Exception)optionalProps.get(DMQ.UNDELIVERED_EXCEPTION);
                    deliverCnt = (Integer)optionalProps.get(Destination.TEMP_CNT);
                    reasonInt = (Integer)optionalProps.get("REASON");
                    deadbkr = (String)optionalProps.get(DMQ.DEAD_BROKER);
                }
                RemoveReason rr = null;
                if (ackType == ClusterBroadcast.MSG_UNDELIVERABLE) {
                    rr = RemoveReason.UNDELIVERABLE;
                } else {
                    rr = RemoveReason.ERROR;
                    if (reasonInt != null)
                        rr = RemoveReason.findReason(reasonInt.intValue());
                }
                if (comment == null)
                    comment = "none";
                // OK

                if (uid.longValue() == 0) { // marking @ high level
                   // we either only delivered the msg to one person
                   // OR it was destroyed @ a high level
                   d.removeMessage(id, rr, 
                         (optionalProps == null ? null
                           : new Hashtable(optionalProps)));
                } else { // single ref
                    ref.markDead(uid, comment, ex, rr,
                         (deliverCnt == null ? 0 : deliverCnt.intValue()),
                         deadbkr);
                }
                return;
            } else {
                if (value == null) { 
                    throw new BrokerException("Notfound: " +ackType+" "
                           +id+" "+uid, Status.NOT_FOUND);
                }
                value.acknowledged(false, txnID);
            }
        }
    }

    public void addConsumer(Consumer c) 
        throws BrokerException
    {
        com.sun.messaging.jmq.jmsserver.core.ConsumerUID cuid = 
                c.getConsumerUID();

        if (! (c instanceof Subscription)) {
            consumers.put(cuid, c);
            listeners.put(cuid, c.addEventListener(this, 
                 EventType.BUSY_STATE_CHANGED, null));
        }

        DestinationUID duid = c.getDestinationUID();

        int type = (duid.isQueue() ? DestType.DEST_TYPE_QUEUE :
                     DestType.DEST_TYPE_TOPIC);

        Destination d= null;

        try {
            // ok handle adding a reference count
            // we'll try at most 2 times to get a
            // destination
            for (int i=0; i < 2 && d == null ; i ++) {
                d = Destination.getDestination(duid.getName(),
                  type, true, true);

                try {
                    // increment the reference count
                    // this make sure that the destination
                    // is not removed by autocreate prematurely    
                    if (d != null) {
                        d.incrementRefCount();

                        break; // well we should break anyway, but
                               // this makes it explicit
                    }

                } catch (BrokerException ex) {
                    // incrementRefCount throws a BrokerException
                    // if the destination was destroyed
                    // if we are here then the destination went away
                    // try to get the destination again
                    d = null;
                }
           }
           if (d == null)
               throw new BrokerException("Unable to attach to destination "
                    + duid);
        } catch (IOException ex) {
            throw new BrokerException("Unable to autocreate destination " +
                   duid , ex);
        }

        try {
            // OK, if we made it here, we have incremented the reference
            // count on the object, we need to make sure we decrement the RefCount
            // before we exit (so cleanup can occur)
            if (!c.getDestinationUID().isQueue() && 
               (! (c instanceof Subscription)) &&
               c.getSubscription() == null) {
                // directly send messages
                c.setFalconRemote(true);
             } else {
                int mp = 100;
                if (d != null) {
                    mp = d.getMaxPrefetch();
                } else {
                    logger.log(Logger.INFO,"Internal Error: Unknown " +
                               "destination for consumer " + c.getDestinationUID() 
                               + " can not determine prefetch ");
                }
                if (mp > BTOBFLOW)
                    mp = BTOBFLOW;
                c.setPrefetch(mp);
             }

             try {
                 if (c.getSubscription() == null) {
                     d.addConsumer(c, false);
                 }
             } catch (SelectorFormatException ex) {
                 throw new BrokerException("unable to add destination " + d,
                       ex);
             }
        
            if (! (c instanceof Subscription)) {
                if (c.isBusy()) {
                    synchronized (activeConsumers) {
                        activeConsumers.add(c);
                        activeConsumers.notify();
                    }
                }
            }
        } finally {
            // decrement the ref count since we've finished
            // processing the add consumer
            d.decrementRefCount();
        }
    }

    public void run() {
        while (valid) {
            Consumer c = null;
            synchronized(activeConsumers) {
                while (valid && activeConsumers.isEmpty()) {
                    try {
                        activeConsumers.wait();
                    } catch (InterruptedException ex) {
                    }
                }
                if (valid) {
                    Iterator itr = activeConsumers.iterator();
                    c = (Consumer) itr.next();
                    itr.remove();
                    if (c.isBusy()) 
                        activeConsumers.add(c);
                }
            }
            if (c == null)
                continue;
            PacketReference ref = c.getAndFillNextPacket(null);
            if (ref == null)
                continue;

            HashSet s = new HashSet();
            s.add(c);
            boolean cb = ref.getMessageDeliveredAck(c.getConsumerUID()) 
                           || c.isPaused();

            if (!c.getConsumerUID().isNoAck()) {
                ackEntry entry = new ackEntry(ref, c.getConsumerUID(), 
                   c.getStoredConsumerUID());
                synchronized(deliveredMessages) {
                    deliveredMessages.put(entry, entry);
                }
            }
            protocol.sendMessage(ref, s, cb);
                
            
        }
    }
    
}

    


