/*
 * @(#)AckHandler.java	1.69 01/20/06
 *
 * Copyright 2000 Sun Microsystems, Inc. All Rights Reserved
 * SUN PROPRIETARY/CONFIDENTIAL
 * Use is subject to license terms. 
 *
 */

package com.sun.messaging.jmq.jmsserver.data.handlers;

import java.io.*;
import java.util.*;

import com.sun.messaging.jmq.jmsserver.resources.*;
import com.sun.messaging.jmq.jmsserver.data.PacketHandler;
import com.sun.messaging.jmq.jmsserver.core.ConsumerUID;
import com.sun.messaging.jmq.jmsserver.core.Destination;
import com.sun.messaging.jmq.jmsserver.core.ClusterBroadcast;
import com.sun.messaging.jmq.jmsserver.core.Consumer;
import com.sun.messaging.jmq.jmsserver.data.TransactionList;
import com.sun.messaging.jmq.jmsserver.core.PacketReference;
import com.sun.messaging.jmq.jmsserver.core.Session;
import com.sun.messaging.jmq.jmsserver.core.SessionUID;
import com.sun.messaging.jmq.jmsserver.util.PacketUtil;
import com.sun.messaging.jmq.jmsserver.data.TransactionUID;
import com.sun.messaging.jmq.jmsserver.util.BrokerException;
import com.sun.messaging.jmq.jmsserver.util.lists.RemoveReason;
import com.sun.messaging.jmq.io.*;
import com.sun.messaging.jmq.jmsserver.service.Connection;
import com.sun.messaging.jmq.jmsserver.service.imq.IMQConnection;

import com.sun.messaging.jmq.util.log.Logger;
import com.sun.messaging.jmq.jmsserver.Globals;
import com.sun.messaging.jmq.jmsserver.BrokerStateHandler;
import com.sun.messaging.jmq.jmsserver.FaultInjection;



/**
 * Handler class which deals with recording message acknowldegements
 */
public class AckHandler extends PacketHandler 
{
    // An Ack block is a Long ConsumerUID and a SysMessageID
    static final int ACK_BLOCK_SIZE =  8 + SysMessageID.ID_SIZE;

    private int ackProcessCnt = 0; // used for fault injection
    private FaultInjection fi = null;

    private TransactionList translist = null;
    private final Logger logger = Globals.getLogger();
    public static boolean DEBUG = false;

    public static int ACKNOWLEDGE_REQUEST=0;
    public static int UNDELIVERABLE_REQUEST=1;
    public static int DEAD_REQUEST=2;

    public static void checkRequestType(int ackType)
        throws BrokerException
    {
        if (ackType > DEAD_REQUEST || ackType < ACKNOWLEDGE_REQUEST) {
            // L10N .. localize internal error
            throw new BrokerException("Internal Error: unknown ackType "
                   + ackType);
        }
    }

    public AckHandler( TransactionList translist) 
    {
        this.translist = translist;
        fi = FaultInjection.getInjection();
    }

    /**
     * Method to handle Acknowledgement messages
     */
    public boolean handle(IMQConnection con, Packet msg) 
        throws BrokerException
    {
        int size = msg.getMessageBodySize();
        int ackcount = size/ACK_BLOCK_SIZE;
        int mod = size%ACK_BLOCK_SIZE;
        int status = Status.OK;
        String reason = null;


        if (DEBUG) {
            logger.log(Logger.DEBUG,"AckHandler: processing message {0} {1}",
                     msg.toString(), 
                     con.getConnectionUID().toString());
        }
        if (!con.isAdminConnection() && fi.FAULT_INJECTION) {
           ackProcessCnt ++; // for fault injection
        } else {
           ackProcessCnt = 0;
        }

        if (ackcount == 0 ) {
            logger.log(Logger.ERROR,BrokerResources.E_INTERNAL_BROKER_ERROR,
                     "Internal Error: Empty Ack Message "+
                     msg.getSysMessageID().toString());
            reason = "Empty ack message";
            status = Status.ERROR;
        }
        if (mod != 0) {
            logger.log(Logger.ERROR, BrokerResources.E_INTERNAL_BROKER_ERROR,
                     "Internal Error: Invalid Ack Message Size "
                     + String.valueOf(size) +  " for message " +
                     msg.getSysMessageID().toString());
            reason = "corrupted ack message";
            status = Status.ERROR;
        }

        TransactionUID tid = null;
        if (msg.getTransactionID() != 0) { // HANDLE TRANSACTION
            try {
                tid = new TransactionUID(msg.getTransactionID());
            } catch (Exception ex) {
               logger.log(Logger.ERROR, BrokerResources.E_INTERNAL_BROKER_ERROR,
                     "Internal Error: can not create transactionID for "
                     + msg, ex);
               status = Status.ERROR;
            }
        }
        boolean markDead = false;
        Hashtable props = null;
        Throwable deadthr = null;
        String deadcmt = null;
        int deliverCnt = 0;
        int ackType = ACKNOWLEDGE_REQUEST;
        boolean JMQValidate = false;
        try {
            props = msg.getProperties();
            Integer iackType = (props == null ? null 
                  :(Integer)props.get("JMQAckType"));
            ackType = (iackType == null ? ACKNOWLEDGE_REQUEST
                 : iackType.intValue());

            Boolean validateFlag = (props == null ? null 
                  :(Boolean)props.get("JMQValidate"));

            JMQValidate = (validateFlag == null ? false
                 : validateFlag.booleanValue());

            checkRequestType(ackType);
            if (ackType == DEAD_REQUEST ) {
                deadthr = (Throwable)props.get("JMQException");
                deadcmt = (String)props.get("JMQComment");
                Integer itr = (Integer)props.get("JMSXDeliveryCount");
                deliverCnt = (itr == null ? -1 : itr.intValue());
            }   
        } catch (Exception ex) {
            // assume not dead
            logger.logStack(Logger.INFO, "Internal Error: bad protocol", ex);
            ackType = ACKNOWLEDGE_REQUEST;
        }

       
        // OK .. handle Fault Injection
        if (!con.isAdminConnection() && fi.FAULT_INJECTION) {
            Map m = new HashMap();
            if (props != null)
                m.putAll(props);
            m.put("mqAckCount", new Integer(ackProcessCnt));
            m.put("mqIsTransacted", new Boolean(tid != null));
            fi.checkFaultAndExit(FaultInjection.FAULT_ACK_MSG_1,
                 m, 2, false);
        }

        List cleanList = null;
        try {
            if (status == Status.OK) {
                DataInputStream is = new DataInputStream(
		            msg.getMessageBodyStream());

                // pull out the messages into two lists
                SysMessageID ids[] = new SysMessageID[ackcount];
                ConsumerUID cids[] = new ConsumerUID[ackcount];
                for (int i = 0; i < ackcount; i ++) {
                    long newid = is.readLong();
                    cids[i] = new ConsumerUID(newid);
                    cids[i].setConnectionUID(con.getConnectionUID());
                    ids[i] =  new SysMessageID();
                    ids[i].readID(is);
                }
                if (JMQValidate) {
                    if (ackType == DEAD_REQUEST || 
                           ackType == UNDELIVERABLE_REQUEST) {
                        status = Status.BAD_REQUEST;
                        reason = "Can not use JMQValidate with ackType of "
                                  + ackType;
                    } else if (tid == null) {
                        status = Status.BAD_REQUEST;
                        reason = "Can not use JMQValidate with no tid ";
                    } else if (!validateMessages(tid, ids, cids)) {
                        status = Status.NOT_FOUND;
                        reason = "Acknowledgement not processed";
                    }
                } else if (ackType == DEAD_REQUEST) {
                     cleanList = handleDeadMsgs(ids, cids, deadthr, deadcmt, 
                               deliverCnt);
                } else if (ackType == UNDELIVERABLE_REQUEST) {
                     cleanList = handleUndeliverableMsgs(ids, cids);
                } else {
                    if (tid != null) {
                        cleanList= handleTransaction(tid, ids, cids);
                     } else  {
                        cleanList = handleAcks(con, ids, cids);
                     }
                }
           } 

        } catch (Throwable thr) {
            status = Status.ERROR;
            if (thr instanceof BrokerException)
                status = ((BrokerException)thr).getStatusCode();
            reason = thr.getMessage();
            if (status == Status.ERROR) { // something went wrong
                logger.logStack(Logger.ERROR,  BrokerResources.E_INTERNAL_BROKER_ERROR,
                    "-------------------------------------------" +
                    "Internal Error: Invalid Acknowledge Packet processing\n" +
                    " " + 
                    (msg.getSendAcknowledge() ? " notifying client\n"
                          : " can not notify the client" )
                    + com.sun.messaging.jmq.jmsserver.util.PacketUtil.dumpPacket(msg)
                    + "--------------------------------------------",
                    thr);
            }
        }

        // OK .. handle Fault Injection
        if (!con.isAdminConnection() && fi.FAULT_INJECTION) {
            Map m = new HashMap();
            if (props != null)
                m.putAll(props);
            m.put("mqAckCount", new Integer(ackProcessCnt));
            m.put("mqIsTransacted", new Boolean(tid != null));
            fi.checkFaultAndExit(FaultInjection.FAULT_ACK_MSG_2,
                 m, 2, false);
        }

          // send the reply (if necessary)
        if (msg.getSendAcknowledge()) {
 
             Packet pkt = new Packet(con.useDirectBuffers());
             pkt.setPacketType(PacketType.ACKNOWLEDGE_REPLY);
             pkt.setConsumerID(msg.getConsumerID());
             Hashtable hash = new Hashtable();
             hash.put("JMQStatus", new Integer(status));
             if (reason != null)
                 hash.put("JMQReason", reason);
             if (con.DUMP_PACKET || con.OUT_DUMP_PACKET)
                 hash.put("JMQReqID", msg.getSysMessageID().toString());
             pkt.setProperties(hash);
             con.sendControlMessage(pkt);

        }

        // OK .. handle Fault Injection
        if (!con.isAdminConnection() && fi.FAULT_INJECTION) {
            Map m = new HashMap();
            if (props != null)
                m.putAll(props);
            m.put("mqAckCount", new Integer(ackProcessCnt));
            m.put("mqIsTransacted", new Boolean(tid != null));
            fi.checkFaultAndExit(FaultInjection.FAULT_ACK_MSG_3,
                 m, 2, false);
        }

        // we dont need to clear the memory up until after we reply
        // to the ack
        if (cleanList != null && !cleanList.isEmpty()) {
            Iterator itr = cleanList.iterator();
            while (itr.hasNext()) {
                PacketReference ref = (PacketReference)itr.next();
                Destination d= ref.getDestination();
                try {
                    if (ref.isDead()) {
                        d.removeDeadMessage(ref);
                    } else {
                        d.removeMessage(ref.getSysMessageID(),
                           RemoveReason.ACKNOWLEDGED);
                    }
                } catch (Exception ex) {
                    logger.log(Logger.INFO,"Internal Error: unable to "
                          + "clean up message " + ref + " after ack "
                          + "processing");
                }
            }
        }
        return true;
    }

    public List handleAcks(IMQConnection con, SysMessageID[] ids, ConsumerUID[] cids) 
        throws BrokerException, IOException
    {
        List l = new LinkedList();
        // XXX - we know that everything on this message
        // we could eliminate on of the lookups
        for (int i=0; i < ids.length; i ++) {
            Session s = Session.getSession(cids[i]);
            if (s == null) { // consumer does not have session
               Consumer c = Consumer.getConsumer(cids[i]);
               if (c == null) {
                   if (!con.isValid() || con.isBeingDestroyed()) {
                        logger.log(logger.DEBUG,"Received ack for consumer " 
                             + cids[i] +  " on closing connection " + con);
                        continue;
                   }
                   if (BrokerStateHandler.shutdownStarted)
                       throw new BrokerException(
                           BrokerResources.I_ACK_FAILED_BROKER_SHUTDOWN);
                   throw new BrokerException("Internal Error: Unable to complete processing acks:"
                          + " Unknown consumer " + cids[i]);
                } else if (c.getConsumerUID().getBrokerAddress() !=
                           Globals.getClusterBroadcast().getMyAddress())
                {
                     // remote consumer
                     PacketReference ref = Destination.get(ids[i]);
                     ConsumerUID cuid = c.getConsumerUID();
                     // right now we dont store messages for remote
                     // brokers so the sync flag is unnecessary
                     // but who knows if that will change
                     if (ref.acknowledged(cuid, 
                         c.getStoredConsumerUID(),
                         !cuid.isDupsOK(), true, null)) {
                           l.add(ref);
                     }

                 } else {
                     logger.log(Logger.INFO,
                         Globals.getBrokerResources().getString(
                         BrokerResources.E_INTERNAL_BROKER_ERROR,
                         "local consumer does not have "
                         + "associated session " + c));
                     throw new BrokerException("Unknown local consumer " 
                         + cids[i]);
                }
            } else { // we have a session
               PacketReference ref = s.ackMessage(cids[i],
                     ids[i], null);
               if (ref != null) {
                     l.add(ref);
               }
            } 
        }
        return l; 
    }

    public List handleTransaction(TransactionUID tid, SysMessageID[] ids, ConsumerUID[] cids) 
        throws BrokerException
    {
        for (int i=0; i < ids.length; i ++) {
            try {
                // lookup the session by consumerUID

                Session s = Session.getSession(cids[i]);

                // look up the consumer by consumerUID
                Consumer consumer = Consumer.getConsumer(cids[i]);

                // try and find the stored consumerUID
                ConsumerUID sid = (consumer != null) ? consumer.getStoredConsumerUID() :
                          null;

                // if we still dont have a session, attempt to find one
                if (s == null && consumer != null) {
                    SessionUID suid = consumer.getSessionUID();
                    s = Session.getSession(suid);
                }
                if (s == null)  {
                   if (BrokerStateHandler.shutdownStarted)
                       throw new BrokerException(
                           BrokerResources.I_ACK_FAILED_BROKER_SHUTDOWN);
                   throw new BrokerException("Internal Error: Unable to complete processing transaction:"
                          + " Unknown consumer/session " + cids[i]);
                }
                translist.addAcknowledgement(tid, ids[i], cids[i], sid);
                s.acknowledgeInTransaction(cids[i], ids[i], tid);

            } catch (Exception ex) {
                logger.log(Logger.INFO,"Internal Exception processing trans "
                     + "acknowledge [" + tid + "," + ids[i] + "," + cids[i]
                     + "]", ex);
                throw new BrokerException("Internal Error: Unable to " +
                    " complete processing acknowledgements in a tranaction: " 
                     +ex, ex);
            }
        }
        return null;

    }


    public boolean validateMessages(TransactionUID tid,
            SysMessageID[] ids, ConsumerUID[] cids)
        throws BrokerException
    {

// LKS - XXX need to revisit this
// I'm not 100% sure if we still need this or not (since its really
// targeted at supporting the NEVER rollback option
//
// putting in a mimimal support for the feature

        // OK, get a status on the transaction

        boolean openTransaction = false;
        try {
            translist.getTransactionMap(tid, false);
            openTransaction = true;
        } catch (BrokerException ex) {
        }

       
        // if transaction exists we need to check its information
        if (openTransaction) {
            for (int i=0; i < ids.length; i ++) {
                Consumer c = Consumer.getConsumer(cids[i]);
                if (c == null) { // unknown consumer
                    throw new BrokerException("Internal Error, " +
                       "unknown consumer " + cids[i], 
                       Status.BAD_REQUEST);
                }
                if (!translist.checkAcknowledgement(tid, ids[i], 
                   c.getConsumerUID())) {
                    return false;
                }
           }
        } else { // check packet reference
            for (int i=0; i < ids.length; i ++) {
                Consumer c = Consumer.getConsumer(cids[i]);
                if (c == null) { // unknown consumer
                    throw new BrokerException("Internal Error, " +
                       "unknown consumer " + cids[i], 
                       Status.BAD_REQUEST);
                }
                PacketReference ref = Destination.get(ids[i]);
                if (ref == null) {
                    logger.log(Logger.DEBUG, "in validateMessages: Could not find " + ids[i]);
                    continue;
                }
                if (!ref.hasConsumerAcked(c.getStoredConsumerUID())) {
                    return false;
                }
           }
        }
         
        return true;
    }

    public List handleDeadMsgs(SysMessageID[] ids, ConsumerUID[] cids, 
            Throwable thr, String comment, int deliverCnt) 
        throws BrokerException
    {
        List l = new ArrayList();
        for (int i=0; i < ids.length; i ++) {
            Session s = Session.getSession(cids[i]);
            if (s == null) { // consumer does not have session
                 // really nothing to do, ignore
                logger.log(Logger.DEBUG,"Dead message for Unknown Consumer/Session"
                        + cids[i]);
                continue;
            }
            if (DEBUG)
                logger.log(Logger.DEBUG,"Handling Dead Message " + cids[i] +
                     ":" + ids[i]);
            PacketReference ref = s.handleDead(cids[i], ids[i],
                    thr, comment, deliverCnt);

            // if we return the reference, we could not re-deliver it ...
            // no consumers .. so clean it up
            if (ref != null) {
                l.add(ref);
            }
        }
        return l;
    }


    public List handleUndeliverableMsgs(SysMessageID[] ids, ConsumerUID[] cids) 
        throws BrokerException
    {

        List l = new ArrayList();
        for (int i=0; i < ids.length; i ++) {
            Session s = Session.getSession(cids[i]);
            if (s == null) { // consumer does not have session
                 // really nothing to do, ignore
                logger.log(Logger.DEBUG,"Undeliverable message for Unknown Consumer/Session"
                        + cids[i]);
            }
            if (DEBUG)
                logger.log(Logger.DEBUG,"Handling Undeliverable Message " + cids[i] +
                     ":" + ids[i]);
            PacketReference ref = s.handleUndeliverable(cids[i], ids[i]);

            // if we return the reference, we could not re-deliver it ...
            // no consumers .. so clean it up
            if (ref != null) {
                l.add(ref);
            }
        }
        return l;
    }
}
