/*
 * @(#)ConsumerHandler.java	1.91 01/23/06
 *
 * Copyright 2000-2004 Sun Microsystems, Inc. All Rights Reserved
 * SUN PROPRIETARY/CONFIDENTIAL
 * Use is subject to license terms. 
 *
 */

package com.sun.messaging.jmq.jmsserver.data.handlers;

import java.util.*;
import java.io.*;
import com.sun.messaging.jmq.jmsserver.data.PacketHandler;
import com.sun.messaging.jmq.util.DestType;
import com.sun.messaging.jmq.io.*;
import com.sun.messaging.jmq.jmsserver.service.Connection;
import com.sun.messaging.jmq.jmsserver.service.imq.IMQConnection;
import com.sun.messaging.jmq.jmsserver.util.BrokerException;
import com.sun.messaging.jmq.jmsserver.core.Destination;
import com.sun.messaging.jmq.jmsserver.core.DestinationUID;
import com.sun.messaging.jmq.jmsserver.core.SessionUID;
import com.sun.messaging.jmq.jmsserver.core.Session;
import com.sun.messaging.jmq.jmsserver.core.Consumer;
import com.sun.messaging.jmq.jmsserver.core.Subscription;
import com.sun.messaging.jmq.jmsserver.core.ConsumerUID;
import com.sun.messaging.jmq.jmsserver.util.PacketUtil;
import com.sun.messaging.jmq.jmsserver.resources.BrokerResources;
import com.sun.messaging.jmq.util.log.Logger;
import com.sun.messaging.jmq.jmsserver.Globals;
import com.sun.messaging.jmq.util.selector.Selector;
import com.sun.messaging.jmq.util.lists.OutOfLimitsException;
import com.sun.messaging.jmq.util.selector.SelectorFormatException;
import com.sun.messaging.jmq.jmsserver.license.*;
import com.sun.messaging.jmq.jmsserver.management.agent.Agent;




/**
 * Handler class which deals with adding and removing interests from the RouteTable
 */
public class ConsumerHandler extends PacketHandler 
{
    private Logger logger = Globals.getLogger();
    public static boolean DEBUG = false;
  

    public ConsumerHandler() {
    }

    boolean ALWAYS_WAIT_ON_DESTROY = Globals.getConfig().getBooleanProperty(Globals.IMQ + ".waitForConsumerDestroy");


    /**
     * Method to handle Consumer(add or delete) messages
     */
    public boolean handle(IMQConnection con, Packet msg) 
        throws BrokerException
    {
        boolean sessionPaused = false;
        boolean conPaused = false;
        Hashtable props = null;
        try {
            props = msg.getProperties();
        } catch (Exception ex) {
            logger.log(Logger.INFO,"Internal Error: unable to retrieve "+
                " properties from consumer message " + msg, ex);
            props = new Hashtable();
        }

        Long lsessionid = (Long)props.get("JMQSessionID");


        Session session = null;


        String err_reason = null;

        Boolean blockprop = (Boolean)props.get("JMQBlock");

        Consumer newc = null;

        assert blockprop == null || msg.getPacketType() == PacketType.DELETE_CONSUMER
               : msg;

        boolean blockprop_bool = (blockprop != null && blockprop.booleanValue());


        boolean isIndemp = msg.getIndempontent();


        // OK ... set up the reply packet
        Packet pkt = new Packet(con.useDirectBuffers());
        pkt.setConsumerID(msg.getConsumerID()); // correlation ID
        Hashtable hash = new Hashtable();
        pkt.setPacketType(msg.getPacketType() + 1);

        int status = Status.OK;
        String warning = BrokerResources.W_ADD_CONSUMER_FAILED;

        ConsumerUID uid = null;
        Integer destType = null;
        Integer oldid = null;
        Subscription sub = null;
        try {
            con.suspend();
            conPaused = true;
            if (msg.getPacketType() == PacketType.ADD_CONSUMER) {
                if (DEBUG) {
                    logger.log(Logger.DEBUGHIGH, "ConsumerHandler: handle() "
                      + "[ Received AddConsumer message {0}]", msg.toString());
                }
                pkt.setPacketType(PacketType.ADD_CONSUMER_REPLY);
                if (lsessionid == null) {
                    if (DEBUG)
                    logger.log(Logger.DEBUG,"not Raptor consumer packet (no session id)");
                    // assign session same # as consumer
                    SessionUID sessionID = new SessionUID(
                               con.getConnectionUID().longValue());
                    // single threaded .. we dont have to worry about
                    // someone else creating it
                    session = con.getSession(sessionID);
                    if (session == null) {
                       session = Session.createSession(sessionID, 
                                  con.getConnectionUID(), null);
                       con.attachSession(session);
                    }
                } else {
                    SessionUID sessionID = new SessionUID(lsessionid.longValue());
                    session = con.getSession(sessionID);
                    if (session == null) {
                        throw new BrokerException("Internal Error: client set invalid"
                         + " sessionUID " + sessionID + " session does not exist");
                    }
                }

                if (blockprop_bool) { // turn off all processing
                   session.pause("Consumer - Block flag");
                   sessionPaused = true;
                }


                /* XXX-LKS KLUDGE FOR 2.0 compatibility */
                
                // for now, we just pass the consumer ID back on the old
                // packet .. I need to revisit this in the future
                oldid = (Integer)props.get("JMQConsumerID"); // old consumer ID
                if (oldid != null) 
                    hash.put("JMQOldConsumerID", oldid);


                
                if (props == null) {
                    throw new BrokerException(Globals.getBrokerResources().getString(
                   BrokerResources.X_INTERNAL_EXCEPTION,"no properties in addConsumer "
                      + "packet - client does not match protocol"));
                }
                Integer inttype = (Integer )props.get("JMQDestType");
                int type = (inttype == null ? -1 : inttype.intValue());
                if (type == -1) {
                    throw new BrokerException(Globals.getBrokerResources().getString(
                   BrokerResources.X_INTERNAL_EXCEPTION,"Client is not sending DestType, "
                         + "unable to add interest"));
                }

                boolean queue = DestType.isQueue(type) ;

                String destination = (String)props.get("JMQDestination");
                String selector =  (String)props.get("JMQSelector");
                Boolean nolocal = (Boolean)props.get("JMQNoLocal");
                String durablename = (String)props.get("JMQDurableName");
                String clientid = getClientID(props, con);
                Boolean reconnect = (Boolean)props.get("JMQReconnect");
                Boolean share = (Boolean)props.get("JMQShare");
                Integer size = (Integer)props.get("JMQSize");

                if (queue && nolocal != null && nolocal.booleanValue()) {
                    Globals.getLogger().log(Logger.ERROR, BrokerResources.E_INTERNAL_BROKER_ERROR,
                            "NoLocal is not supported on Queue Receivers");
                   throw new BrokerException("Unsupported property on queues JMQNoLocal "
                        + "is set to " + nolocal, Status.ERROR);
                }
                if (reconnect != null && reconnect.booleanValue()) {
                    Globals.getLogger().log(Logger.ERROR,
                        BrokerResources.E_INTERNAL_BROKER_ERROR,
                        "JMQReconnect not implemented");
                }


                if (share != null && share.booleanValue() && 
                       ! ClientIDHandler.CAN_USE_SHARED_CONSUMERS) {
                    throw new BrokerException(
                        Globals.getBrokerResources().getKString(
                            BrokerResources.X_FEATURE_UNAVAILABLE,
                            Globals.getBrokerResources().getKString(
                                BrokerResources.M_SHARED_CONS), destination),
                            BrokerResources.X_FEATURE_UNAVAILABLE,
                            (Throwable) null,
                            Status.NOT_ALLOWED);
                }

                Destination d = null;

                while (true ) {
                   d =  Destination.getDestination(destination,
                                 type, true /* autocreate if possible*/,
                                 !con.isAdminConnection());
                   if (d.isAutoCreated())
                       warning = BrokerResources.W_ADD_AUTO_CONSUMER_FAILED;
                   try {
                       if (d != null)
                           d.incrementRefCount();
                   } catch (BrokerException ex) {
                       continue; // was destroyed in process
                   } catch (IllegalStateException ex) {
                        throw new BrokerException(
                            Globals.getBrokerResources().getKString(
                            BrokerResources.X_SHUTTING_DOWN_BROKER),
                            BrokerResources.X_SHUTTING_DOWN_BROKER,
                            ex,
                            Status.ERROR);
                   }
                   break; // we got one
                }


                if (d == null) {
                    // unable to autocreate destination
                    status  = Status.NOT_FOUND;
                    // XXX error
                    throw new BrokerException(
                        Globals.getBrokerResources().getKString(
                            BrokerResources.X_DESTINATION_NOT_FOUND, destination),
                            BrokerResources.X_DESTINATION_NOT_FOUND,
                            null,
                            Status.NOT_FOUND);
                }

                // Must have a clientID to add a durable
                if (durablename != null && clientid == null) {
                    throw new BrokerException(
                        Globals.getBrokerResources().getKString(
                            BrokerResources.X_NO_CLIENTID, durablename),
                            BrokerResources.X_NO_CLIENTID,
                            null,
                            Status.PRECONDITION_FAILED);
                }
                DestinationUID dest_uid = d.getDestinationUID();

                Consumer c = null;
                
                try { 
                    // need to deal w/ reconnection AND ackType

                    int prefetch = -1;
                    if (isIndemp) { // see if we already created it
                        c = Consumer.getConsumer(msg.getSysMessageID());
                        if (c != null) 
                            prefetch = c.getPrefetch();
                    }

                    if (c == null) {
                        c = new Consumer(dest_uid, selector,
                                 (nolocal == null ? false : nolocal.booleanValue()), 
                                 con.getConnectionUID());
                        c.setCreator((SysMessageID)msg.getSysMessageID().clone());
                        newc = c;
                        newc.pause("Consumer: new consumer");

                        int cprefetch = (size == null ? -1 : size.intValue());
                        int dprefetch = (share == null || !share.booleanValue())
                                ? d.getMaxPrefetch()
                                  : d.getSharedConsumerFlowLimit();
                                       
                        prefetch = (dprefetch == -1) ?
                              cprefetch :
                                (cprefetch == -1 ? cprefetch  
                                   : (cprefetch > dprefetch?
                                       dprefetch : cprefetch));
                        c.setPrefetch(prefetch);

                        // actual subscription added to the destination
    
                        if (durablename != null) {
    
                            // get the subscription ... this may throw
                            // an exception IF we cant 
                            sub = Subscription.
                                 findCreateDurableSubscription(clientid,
                                durablename, dest_uid, selector,
                                nolocal == null ? false :
                                nolocal.booleanValue(),
                                true);
                            sub.pause("Consumer attaching to durable");

                            boolean shared = (share == null ? false :
                                  share.booleanValue());
                            sub.setShared(shared);
                            // add the consumer .. this may throw an
                            // exception IF
                            sub.attachConsumer(c);

                            Subscription oldsub = (Subscription)d.addConsumer(sub, true);
                            if (oldsub != null) {
                                oldsub.purge();
                            }
                            sub.sendCreateSubscriptionNotification(c);
                        } else if (!d.isQueue() && share != null && share.booleanValue()) {
                            if (clientid == null) {
                                throw new BrokerException(
                                    Globals.getBrokerResources().getKString(
                                        BrokerResources.X_NO_CLIENTID, durablename),
                                        BrokerResources.X_NO_CLIENTID,
                                        null,
                                        Status.PRECONDITION_FAILED);
                            }
                            // shared
                            logger.log(Logger.DEBUG,"Creating shared non-durable "
                                    + c);
                            sub = Subscription.createAttachNonDurableSub(c, con);
                            if (sub != null) {
                                sub.pause("Consumer: attaching to nondurable");
                                sub.setShared(true);
                                d.addConsumer(sub, true);
                            }
                            c.attachToConnection(con.getConnectionUID());

                            //c.sendCreateConsumerNotification();
                            if (sub != null)
                                sub.sendCreateSubscriptionNotification(c);

                        } else {
                            d.addConsumer(c, true);
                            c.attachToConnection(con.getConnectionUID());
                            c.sendCreateConsumerNotification();
                        }
                    }
                    if (prefetch != -1 || size != null)
                        hash.put("JMQSize", new Integer(prefetch));

                } catch (SelectorFormatException ex) {
                      throw new BrokerException(
                            Globals.getBrokerResources().getKString(
                            BrokerResources.W_SELECTOR_PARSE, selector),
                            BrokerResources.W_SELECTOR_PARSE,
                            ex,
                            Status.BAD_REQUEST);
                } catch (OutOfLimitsException ex) {
                    if (d.isQueue()) {
                        String args[] = { dest_uid.getName(),
                            String.valueOf(d.getActiveConsumerCount()),
                            String.valueOf(d.getFailoverConsumerCount()) };
                        throw new BrokerException(
                            Globals.getBrokerResources().getKString(
                            BrokerResources.X_S_QUEUE_ATTACH_FAILED, args),
                            BrokerResources.X_S_QUEUE_ATTACH_FAILED,
                            ex,
                            Status.CONFLICT);
                    } else { // durable
                        String args[] = { dest_uid.getName(),
                            durablename, clientid,
                            String.valueOf(ex.getLimit()) };
                        throw new BrokerException(
                            Globals.getBrokerResources().getKString(
                            BrokerResources.X_S_DUR_ATTACH_FAILED, args),
                            BrokerResources.X_S_DUR_ATTACH_FAILED,
                            ex,
                            Status.CONFLICT);
                    }
                } finally {
                    if (d != null)
                        d.decrementRefCount();
                }

                // add the consumer to the session
                session.attachConsumer(c);
        
                Integer acktype = (Integer)props.get("JMQAckMode");
                if (acktype != null) {
                    c.getConsumerUID().setAckType(acktype.intValue());
                }

                uid = c.getConsumerUID();

            } else { // removing Interest
                if (DEBUG) {
                    logger.log(Logger.DEBUGHIGH, 
                        "ConsumerHandler: handle() [ Received DestroyConsumer message {0}]", 
                         msg.toString());
                }

                warning = BrokerResources.W_DESTROY_CONSUMER_FAILED;
                pkt.setPacketType(PacketType.DELETE_CONSUMER_REPLY);

                String durableName = (String)props.get("JMQDurableName");
                String clientID = getClientID(props, con);
                Long cid = (Long)props.get("JMQConsumerID");
                uid = (cid == null ? null :  new ConsumerUID( cid.longValue()));

                if (lsessionid != null) { //  passed on in
                    SessionUID sessionID = new SessionUID(lsessionid.longValue());
                    session = con.getSession(sessionID);
                } else {
                    session = Session.getSession(uid);
                }
                if (session == null && durableName == null && !isIndemp) {
                     logger.log(Logger.ERROR,"Internal error processing"+
                           " delete consumer\n"+
                            com.sun.messaging.jmq.jmsserver.util
                              .PacketUtil.dumpPacket(msg));
                     Session.dumpAll();

                }

                // retrieve the LastDelivered property
                Integer bodytype = (Integer)props.get("JMQBodyType");
                int btype = (bodytype == null ? 0 : bodytype.intValue());
                
                SysMessageID lastid = null;

                if (btype == PacketType.SYSMESSAGEID) {
                    int size = msg.getMessageBodySize();
                    if (size == 0) {
                        logger.log(Logger.INFO,"Warning, bad body in destroy consumer");
                     } else {
                         DataInputStream is = new DataInputStream(
                                msg.getMessageBodyStream());
                         lastid = new SysMessageID();
                         lastid.readID(is);
                     }
                }
                if (DEBUG && lastid != null) {
                    logger.log(Logger.DEBUG,"Sent lastID [" + lastid + "]"
                       + " for consumer " + uid + Destination.get(lastid));
                }

                if (!sessionPaused && session != null) {
                    sessionPaused = true;
                    session.pause("Consumer removeconsumer");
                }
                if (durableName != null) {
                    Subscription usub = Subscription.unsubscribe(durableName,
                                 clientID);
                    

                    if (usub == null) { // already destroyed
                        throw new BrokerException(
                            Globals.getBrokerResources().getString(
                               BrokerResources.X_UNKNOWN_DURABLE_INTEREST,
                               durableName, clientID),
                          Status.NOT_FOUND);
                    }
                    DestinationUID dest_uid = usub.getDestinationUID();
                    Destination d = Destination.getDestination(dest_uid);
                    assert d != null;
                    if (d != null)
                        d.removeConsumer(uid, true);
                } else {
                    assert session != null;
                    boolean redeliver = false;
                    if (con.getClientProtocolVersion() < Connection.RAPTOR_PROTOCOL ) {
                        redeliver = true;
                    }
                    if (session != null) { // should only be null w/ indemp
                        Consumer c = session.detatchConsumer(uid, lastid, redeliver);
                        DestinationUID dest_uid = c.getDestinationUID();
                        Destination d = Destination.getDestination(dest_uid);
                        if (d != null)
                            d.removeConsumer(uid, true);
                    }
                }

            }

        } catch (BrokerException ex) {

            status = ex.getStatusCode();
            String consumid = null;
            String destination = null;
            try {
                destination = (String) props.get("JMQDestination");
                if (destination == null && msg.getPacketType() 
                     != PacketType.ADD_CONSUMER)
                     destination = "";
                if (oldid != null)
                    consumid = oldid.toString();
                else
                    consumid = "";
            } catch (Exception ex1) {}
            String args[] = {consumid, con.getRemoteConnectionString(), destination};

            err_reason = ex.getMessage();

            if (ex.getStatusCode() == Status.PRECONDITION_FAILED
                  || ex.getStatusCode() == Status.CONFLICT ) {
                logger.log(Logger.WARNING, warning, args, ex);
            } else if (ex.getStatusCode() == Status.BAD_REQUEST) {
                // Probably a bad selector
                logger.log(Logger.WARNING, warning, args, ex);
                if (ex.getCause() != null) {
                    logger.log(Logger.INFO, ex.getCause().toString());
                }
            } else {
                if (isIndemp && msg.getPacketType() == PacketType.DELETE_CONSUMER) {
                    logger.logStack(Logger.DEBUG, "Reprocessing Indempotent message for "
                          + "{0} on destination {2} from {1}",args, ex);
                    status = Status.OK;
                    err_reason = null;
                } else {
                    logger.logStack(Logger.WARNING, warning,args, ex);
                }
            }
        } catch (IOException ex) {
            logger.log(Logger.INFO,"Internal Error: unable to process "+
                " consumer request " + msg, ex);
            props = new Hashtable();
            err_reason = ex.getMessage();
            assert false;
        } catch (SecurityException ex) {
            status = Status.FORBIDDEN;
            err_reason = ex.getMessage();
            String destination = null;
            String consumid = null;
            try {
                destination = (String) props.get("JMQDestination");
                if (oldid != null)
                    consumid = oldid.toString();
            } catch (Exception ex1) {}
            logger.log(Logger.WARNING, warning, destination, consumid,ex);
        } finally {
            if (conPaused)
                con.resume();
        }
        hash.put("JMQStatus", new Integer(status));

        if (err_reason != null)
            hash.put("JMQReason", err_reason);

        if (uid != null)
            hash.put("JMQConsumerID", new Long(uid.longValue()));

        if (destType != null)
            hash.put("JMQDestType", destType);

        if (con.DUMP_PACKET || con.OUT_DUMP_PACKET)
            hash.put("JMQReqID", msg.getSysMessageID().toString());

        pkt.setProperties(hash);
        con.sendControlMessage(pkt);

        if (sessionPaused)
             session.resume("Consumer - session was paused");

        if (newc != null)
            newc.resume("Consumer - new consumer");

        if (sub != null)
            sub.resume("Consumer - added to sub");

        return true;
    }

    public  String getClientID(Hashtable props, Connection con)
    {
        String clientid = (String)con.getClientData(IMQConnection.CLIENT_ID);
        if (clientid == null) {
            clientid = (String)props.get("JMQClientID");
            if (clientid != null) {
                logger.log(Logger.ERROR,BrokerResources.E_INTERNAL_BROKER_ERROR, "client did not send setCLIENTID method before adding/removing a consumer, retrieved clientid " + clientid + " from packet properties");
            }
        }
        return clientid;
        
    } 


}
