/*
 * @(#)QBrowseHandler.java	1.46 08/26/05
 *
 * Copyright 2000 Sun Microsystems, Inc. All Rights Reserved
 * SUN PROPRIETARY/CONFIDENTIAL
 * Use is subject to license terms. 
 *
 */

package com.sun.messaging.jmq.jmsserver.data.handlers;

import java.io.*;
import java.util.*;
import com.sun.messaging.jmq.jmsserver.data.PacketHandler;
import com.sun.messaging.jmq.jmsserver.data.TransactionState;
import com.sun.messaging.jmq.jmsserver.data.TransactionList;
import com.sun.messaging.jmq.util.DestType;
import com.sun.messaging.jmq.io.*;
import com.sun.messaging.jmq.jmsserver.service.Connection;
import com.sun.messaging.jmq.jmsserver.service.ConnectionUID;
import com.sun.messaging.jmq.jmsserver.core.ConsumerUID;
import com.sun.messaging.jmq.jmsserver.core.DestinationUID;
import com.sun.messaging.jmq.jmsserver.core.Destination;
import com.sun.messaging.jmq.jmsserver.util.BrokerException;
import com.sun.messaging.jmq.jmsserver.util.SelectorFilter;
import com.sun.messaging.jmq.jmsserver.core.PacketReference;
import com.sun.messaging.jmq.util.selector.*;
import com.sun.messaging.jmq.util.lists.*;

import com.sun.messaging.jmq.jmsserver.resources.BrokerResources;
import com.sun.messaging.jmq.util.log.Logger;
import com.sun.messaging.jmq.jmsserver.Globals;

import com.sun.messaging.jmq.jmsserver.service.imq.IMQConnection;


/**
 * Handler class which deals with requests for the current queue state
 */
public class QBrowseHandler extends PacketHandler 
{

    private Logger logger = Globals.getLogger();
    public static boolean DEBUG = false;

    public QBrowseHandler() {
    }

    /**
     * Method to handle Destination (create or delete) messages
     */
    public boolean handle(IMQConnection con, Packet msg) 
        throws BrokerException
    {

        int status = Status.OK;
        String reason = null;

        // XXX - REVISIT 2/25/00 racer
        // do we need to create a reply packet each time ?

        Packet pkt = new Packet(con.useDirectBuffers());
        pkt.setConsumerID(msg.getConsumerID());

        pkt.setPacketType(PacketType.BROWSE_REPLY);

        Hashtable hash = new Hashtable();

        // pass back a generated ID
        ConsumerUID uid = new ConsumerUID();
        hash.put("JMQConsumerID", new Long(uid.longValue()));

        Hashtable props = null;
        byte[] body = null;

        String destination = null;

        String selectorstr = null;
        int type = 0;
        try {
            props = msg.getProperties();
            destination = (String )props.get("JMQDestination");
            type = DestType.DEST_TYPE_QUEUE;

            selectorstr = (String )props.get("JMQSelector");

            if (DEBUG) {
                logger.log(Logger.DEBUG, "QBrowse request: destination =  " + destination + "  selector = " + selectorstr);
            }
        
            Destination d = Destination.getDestination(destination,
                     DestType.isQueue(type));

            Boolean getMetrics = (Boolean)props.get("JMQMetrics");

            if (d == null) {
                status = Status.NOT_FOUND;
                reason = "Destination " + destination + " not found";
                logger.log(Logger.WARNING,BrokerResources.W_QUEUE_BROWSE_FAILED_NODEST, 
                     (destination==null? "unknown" : destination));
            } else if (getMetrics != null && getMetrics == Boolean.TRUE) {
                logger.log(Logger.INFO,"Getting destination metrics on " + d);
                ByteArrayOutputStream bos = new ByteArrayOutputStream();
                try {
                    ObjectOutputStream oos = new ObjectOutputStream(bos);
                    oos.writeObject(d.getMetrics());
                    oos.close();
                    body = bos.toByteArray();
                } catch (Exception e) {
                    // Programing error. Do not need to localize
                    logger.log(Logger.ERROR, BrokerResources.E_INTERNAL_BROKER_ERROR,
                        this.getClass().getName() +
                        " : Got exception writing metrics to browse reply message:\n" +
                        e );
                }
                hash.put("JMQMetrics", getMetrics);

               
            } else {

                Collection msgs = null;

                if (selectorstr == null) {
                    msgs = d.getAll((Filter)null).values();
                } else {
                    SelectorFilter f = new SelectorFilter(selectorstr);
                    Map m =  d.getAll(f);
                    msgs = m.values();
                }

                // sort the messages
                ArrayList sorted = new ArrayList(msgs);
                Collections.sort(sorted, new RefCompare());   

               TransactionList tlist = Globals.getTransactionList();

               // remove any expired messages or messages in open txn
               Iterator itr = sorted.iterator();
               while (itr.hasNext()) {
                   PacketReference p = (PacketReference)itr.next();
                   if (p.isExpired())
                       itr.remove();
                   if (p.getTransactionID() != null) {
                      // look up txn
                      TransactionState ts = tlist.retrieveState(p.getTransactionID());
                      if (ts != null && ts.getState() != TransactionState.COMMITTED) {
                          // open txn, remove
                          itr.remove();
                      }
                   }
               }
    
            
                if (DEBUG) {
                    logger.log(Logger.DEBUG, "QBrowse request: current queue size is " + sorted.size());
                }
                ByteArrayOutputStream bos = new ByteArrayOutputStream();
                for (int i =0; i < sorted.size(); i ++) {
                    PacketReference ref = (PacketReference)sorted.get(i);
                    if (ref == null) continue;
                    SysMessageID id = (SysMessageID) ref.getSysMessageID();
                    if (DEBUG) {
                        logger.log(Logger.DEBUG, "\t[" + i + "] id = " + id);
                     }
                    id.writeID(new DataOutputStream(bos));
                }
                bos.flush();
                body = bos.toByteArray();
                bos.close();            
          }
        } catch (IOException ex) {
            logger.logStack(Logger.ERROR, BrokerResources.E_INTERNAL_BROKER_ERROR, "Unable to verify destination - no properties",ex);
            reason = ex.getMessage();
            status = Status.ERROR;
        } catch (ClassNotFoundException ex) {
            logger.logStack(Logger.ERROR, BrokerResources.E_INTERNAL_BROKER_ERROR, "Unable to verify destination -bad class",ex);
            reason = ex.getMessage();
            status = Status.ERROR;
        } catch (SelectorFormatException ex) {
            reason = ex.getMessage();
            status = Status.BAD_REQUEST;
            logger.log(Logger.WARNING,BrokerResources.W_SELECTOR_PARSE, 
                 selectorstr, ex);
        } catch (BrokerException ex) {
            reason = ex.getMessage();
            status = ex.getStatusCode();
            logger.log(Logger.WARNING,BrokerResources.W_QUEUE_BROWSE_FAILED, 
                 (destination==null? "unknown" : destination), ex);
        } catch (SecurityException ex) {
            reason = ex.getMessage();
            status = Status.FORBIDDEN;
            logger.logStack(Logger.DEBUG, BrokerResources.E_INTERNAL_BROKER_ERROR, " access to destination " + destination + " is forbidden ",ex);
            reason = ex.getMessage();
            // XXX - log message
        }

        hash.put("JMQStatus", new Integer(status));

        if (reason != null) {
            hash.put("JMQReason", reason);
        }
        if (con.DUMP_PACKET || con.OUT_DUMP_PACKET)
            hash.put("JMQReqID", msg.getSysMessageID().toString());

        if (status == Status.NOT_FOUND) {
            hash.put("JMQCanCreate", new Boolean(
                Destination.canAutoCreate(DestType.isQueue(type))));
        }

        pkt.setProperties(hash);
        if (body != null)
            pkt.setMessageBody(body);
        con.sendControlMessage(pkt);
        return true;
    }



}
