/*
 * @(#)ClusterBroadcaster.java	1.47 09/01/05
 *
 * Copyright 2003-5 Sun Microsystems, Inc. All Rights Reserved
 * SUN PROPRIETARY/CONFIDENTIAL
 * Use is subject to license terms. 
 *
 */


package com.sun.messaging.jmq.jmsserver.multibroker;

import java.util.*;
import java.io.*;
import com.sun.messaging.jmq.io.Packet;
import com.sun.messaging.jmq.io.SysMessageID;
import com.sun.messaging.jmq.util.UID;
import com.sun.messaging.jmq.util.ServiceType;
import com.sun.messaging.jmq.jmsserver.core.BrokerAddress;
import com.sun.messaging.jmq.jmsserver.Globals;
import com.sun.messaging.jmq.jmsserver.core.*;
import com.sun.messaging.jmq.jmsserver.core.ClusterRouter; 
import com.sun.messaging.jmq.jmsserver.util.*;
import com.sun.messaging.jmq.jmsserver.resources.*;
import com.sun.messaging.jmq.jmsserver.config.*;
import com.sun.messaging.jmq.jmsserver.service.*;
import com.sun.messaging.jmq.util.log.*;
import com.sun.messaging.jmq.io.Status;

/**
 * this class implements the ClusterBroadcast interface for
 * the broker.
 */

public  class ClusterBroadcaster implements ClusterBroadcast, 
                MessageBusCallback
{

    public static boolean DEBUG = false;

    Logger logger = Globals.getLogger();    
    BrokerConfig config = Globals.getConfig();

    BrokerResources br = Globals.getBrokerResources();
    private int version = 0;
    private com.sun.messaging.jmq.jmsserver.core.BrokerAddress selfAddress = null;
    private String driver = null;
    private Cluster c = null;
    private HashMap brokerList = null;

    private int connLimit = 0;

    private Protocol protocol = null;

    public ClusterBroadcaster(Integer connLimit, Integer version)
        throws BrokerException
    {
        this(connLimit.intValue(),version.intValue());
    }
    

    public ClusterBroadcaster(int connLimit, int version) 
        throws BrokerException
    {
        this.version = version;
        brokerList = new HashMap();

        // Create the cluster topology
        this.connLimit = connLimit;
        driver = config.getProperty(ClusterGlobals.TOPOLOGY_PROPERTY);
        if (driver == null)
            driver = "fullyconnected";

        // TBD: JMQ2.1 : Load the topology driver class dynamically...
        if (driver.equals("fullyconnected")) {
            c = (Cluster) new com.sun.messaging.jmq.jmsserver.
                     multibroker.fullyconnected.ClusterImpl(this.connLimit);

            logger.log(Logger.INFO, br.I_CLUSTER_INITIALIZED);
        }
        else {
            driver = "standalone";
        }

        if (driver.equals("standalone")) {
            c = (Cluster) new
                   com.sun.messaging.jmq.jmsserver
                   .multibroker.standalone.ClusterImpl();

            logger.log(Logger.INFO, br.I_STANDALONE_INITIALIZED);
        }
        selfAddress = c.getSelfAddress();

        protocol = new CommonProtocol(this, c, selfAddress);
        /*
        if (version == ClusterGlobals.VERSION_300) {
            protocol = new com.sun.messaging.jmq.jmsserver.multibroker
                      .falcon.FalconProtocol(this, c , selfAddress);
        } else {
            protocol = new com.sun.messaging.jmq.jmsserver.multibroker
                      .raptor.RaptorProtocol(this, c , selfAddress);
        }
        */

        if (version != protocol.getHighestSupportedVersion()) {
            throw  new BrokerException("The version "+version+
                              " is not supported by the ClusterBroadcaster");
        }
        c.setCallback(protocol);
    }

    public Protocol getProtocol()
    {
        return protocol;
    }

    public int getClusterVersion() {
        return version;
    }

    public void startClusterIO() {
        protocol.startClusterIO();
    }

    public void stopClusterIO(boolean requestTakeover) {
        protocol.stopClusterIO(requestTakeover);
    }

    /* 
    public void shutdown() {    
        protocol.shutdown();
    }
    */

    /**
     * Handle jmq administration command to reload and update
     * the cluster configuration.
     */
    public void reloadCluster() {
        protocol.reloadCluster();
    }

    public void pauseMessageFlow() throws IOException {
        protocol.stopMessageFlow();
    }

    public void resumeMessageFlow() throws IOException {
        protocol.resumeMessageFlow();
    }

    /**
     * Set the matchProps for the cluster.
     */
    public void setMatchProps(Properties matchProps) {
        protocol.setMatchProps(matchProps);
    }

    /**
     *
     */
    public boolean waitForConfigSync() {
        return protocol.waitForConfigSync();
    }

    /**
     * Returns the address of this broker.
     * @return <code> BrokerAddress </code> object representing this
     * broker.
     */
    public com.sun.messaging.jmq.jmsserver.core.BrokerAddress getMyAddress() {
        return selfAddress;
    }

    public boolean lockSharedResource(String resource, Object owner) {
        if (DEBUG) {
            logger.log(Logger.INFO, "lockSharedResource : " + resource);
        }
        return (protocol.lockSharedResource(resource,
            (ConnectionUID) owner) == ClusterGlobals.MB_LOCK_SUCCESS);
    }

    public boolean lockDestination(DestinationUID uid, Object owner) {
        if (DEBUG) {
            logger.log(Logger.INFO,"lockDestination " + uid);
        }
        return (protocol.lockResource("destCreate:"+uid.toString(), 0, (ConnectionUID)owner)
            == ClusterGlobals.MB_LOCK_SUCCESS);
    }

    public void unlockDestination(DestinationUID uid, Object owner) {
        if (DEBUG) {
            logger.log(Logger.INFO,"unlockDestination " + uid);
        }
        protocol.unlockResource("destCreate:"+uid.toString());
    }

    public boolean lockClientID(String clientid, Object owner, boolean shared)
    {
        if (DEBUG) {
            logger.log(Logger.INFO,"lockClientID " + clientid);
        }
        if (shared) {
            return (protocol.lockSharedResource("clientid:" +
                   clientid.toString(), (ConnectionUID)owner) 
                   == ClusterGlobals.MB_LOCK_SUCCESS);
        }
        return (protocol.lockResource("clientid:"+clientid.toString(), 0, (ConnectionUID)owner)
            == ClusterGlobals.MB_LOCK_SUCCESS);
    }

    public void unlockClientID(String clientid, Object owner) {
        if (DEBUG) {
            logger.log(Logger.INFO,"unlockClientID " + clientid);
        }
        protocol.unlockResource("clientid:"+clientid.toString());
    }

    public boolean getConsumerLock(com.sun.messaging.jmq.jmsserver.core.ConsumerUID uid,
                    DestinationUID duid, int position,
                    int maxActive, Object owner)
            throws BrokerException
    {
        if (DEBUG) {
            logger.log(Logger.INFO,"getConsumerLock " + uid);
        }
        if (maxActive > 1 && protocol.getClusterVersion() < VERSION_350) {
            throw new BrokerException("Feature not support in this"
               + " cluster protocol");
        }
        return (protocol.lockResource("queue:"+duid.getName() + "_" + position, 0, (ConnectionUID)owner)
            == ClusterGlobals.MB_LOCK_SUCCESS);
    }

    public void unlockConsumer(com.sun.messaging.jmq.jmsserver.core.ConsumerUID uid, 
              DestinationUID duid, int position)
    {
        if (DEBUG) {
            logger.log(Logger.INFO,"unlockConsumer " + uid);
        }
        protocol.unlockResource("queue:"+duid.getName() + "_" + position);
    }
    
    private int convertToClusterType(int type) {
        switch (type)
        {
            case MSG_DELIVERED:
                return ClusterGlobals.MB_MSG_DELIVERED;
            case MSG_ACKNOWLEDGED:
                return ClusterGlobals.MB_MSG_CONSUMED;
            case MSG_IGNORED:
                return ClusterGlobals.MB_MSG_IGNORED;
            case MSG_UNDELIVERABLE:
                return ClusterGlobals.MB_MSG_UNDELIVERABLE;
            case MSG_DEAD:
                return ClusterGlobals.MB_MSG_DEAD;
            case MSG_TXN_ACKNOWLEDGED:
                return ClusterGlobals.MB_MSG_TXN_ACK;
            case MSG_PREPARED:
                return ClusterGlobals.MB_MSG_TXN_PREPARE;
            case MSG_ROLLEDBACK:
                return ClusterGlobals.MB_MSG_TXN_ROLLEDBACK;
        }
        return ClusterGlobals.MB_MSG_SENT;
    }

    private int convertToLocalType(int type) {
        switch (type)
        {
            case ClusterGlobals.MB_MSG_DELIVERED:
                return MSG_DELIVERED;
            case ClusterGlobals.MB_MSG_CONSUMED:
                return MSG_ACKNOWLEDGED;
            case ClusterGlobals.MB_MSG_IGNORED:
                return MSG_IGNORED;
            case ClusterGlobals.MB_MSG_UNDELIVERABLE:
                return MSG_UNDELIVERABLE;
            case ClusterGlobals.MB_MSG_DEAD:
                return MSG_DEAD;
            case ClusterGlobals.MB_MSG_TXN_ACK:
                return MSG_TXN_ACKNOWLEDGED;
            case ClusterGlobals.MB_MSG_TXN_PREPARE:
                return MSG_PREPARED;
            case ClusterGlobals.MB_MSG_TXN_ROLLEDBACK:
                return MSG_ROLLEDBACK;
        }
        return -1;
    }

    public void acknowledgeMessage(com.sun.messaging.jmq.jmsserver.core.BrokerAddress address,
                                   SysMessageID mid,
                                   com.sun.messaging.jmq.jmsserver.core.ConsumerUID cid,
                                   int ackType, boolean ackack, Map optionalProps, Long txnID) 
                                   throws BrokerException
    {
        if (address != null && address != selfAddress) {
            try {
            protocol.sendMessageAck(address, mid, cid, convertToClusterType(ackType),
                                    ackack, optionalProps, txnID);
            return;

            } catch (BrokerException e) {
            logger.log(logger.WARNING, 
            "XXI18N-acknowledgeMessage "+mid+" cid="+cid+" ackType="+ackType+" to "
            +address+ " failed:"+e.getMessage(), e);
            throw e;
            }
        }
        logger.log(logger.ERROR, "XXXI18N InternalError: bad address "+address+
                   " in acknowledge message "+ mid+" cid="+cid+" ackType="+ackType);
        throw new BrokerException("XXXI18N Internal Error Bad address "+address, Status.ERROR);
    }


    public void recordUpdateDestination(Destination dest)
        throws BrokerException {
        if (DEBUG) {
            logger.log(Logger.INFO,"recordUpdateDestination : " + dest);
        }
        // check compatibility with version of system

        protocol.recordUpdateDestination(dest);
    }

    public void recordRemoveDestination(Destination dest)
        throws BrokerException {
        if (DEBUG) {
            logger.log(Logger.INFO,"recordRemoveDestination : " + dest);
        }
        // check compatibility with version of system

        protocol.recordRemoveDestination(dest);
    }

    public void createDestination(Destination dest)
            throws BrokerException
    {
        if (DEBUG) {
            logger.log(Logger.INFO,"createDestination " + dest);
        }
        // check compatibility with version of system

        protocol.sendNewDestination(dest);

    }

    public void recordCreateSubscription(Subscription sub)
        throws BrokerException {
        if (DEBUG) {
            logger.log(Logger.INFO,"recordCreateSubscription " + sub);
        }
        protocol.recordCreateSubscription(sub);
    }

    public void recordUnsubscribe(Subscription sub)
        throws BrokerException {
        if (DEBUG) {
            logger.log(Logger.INFO,"recordUnsubscribe " + sub);
        }
        protocol.recordUnsubscribe(sub);
    }

    public void createSubscription(Subscription sub, Consumer cons)
            throws BrokerException
    {
        if (DEBUG) {
            logger.log(Logger.INFO,"createSubscription " + sub);
        }
        protocol.sendNewSubscription(sub, cons, false);
    }

    public void createConsumer(Consumer con)
            throws BrokerException
    {
        if (DEBUG) {
            logger.log(Logger.INFO,"createConsumer " + con);
        }
        protocol.sendNewConsumer(con, true /* XXX */);
    }

    public void updateDestination(Destination dest)
            throws BrokerException
    {
        if (DEBUG) {
            logger.log(Logger.INFO,"updateDestination " + dest);
        }
        protocol.sendUpdateDestination(dest);
    }


    public void updateSubscription(Subscription sub)
            throws BrokerException
    {
    }

    public void updateConsumer(Consumer con)
            throws BrokerException
    {
    }


    public void destroyDestination(Destination dest)
            throws BrokerException
    {
        if (DEBUG) {
            logger.log(Logger.INFO,"destroyDestination " + dest);
        }
        protocol.sendRemovedDestination(dest);
    }

    public void destroySubscription(Subscription sub)
            throws BrokerException
    {
        if (DEBUG) {
            logger.log(Logger.INFO,"destroySubscription " + sub);
        }
        protocol.sendRemovedConsumer(sub);
    }

    public void destroyConsumer(Consumer con)
            throws BrokerException
    {
        if (DEBUG) {
            logger.log(Logger.INFO,"destroyConsumer " + con);
        }
        protocol.sendRemovedConsumer(con);
    }

    public void connectionClosed(ConnectionUID uid, boolean admin)
    {
        if (DEBUG) {
            logger.log(Logger.INFO,"connectionClosed " + uid);
        }
       if (!admin)
           protocol.clientClosed(uid, true);
    }

    public void messageDelivered(SysMessageID id, 
         com.sun.messaging.jmq.jmsserver.core.ConsumerUID uid,
         com.sun.messaging.jmq.jmsserver.core.BrokerAddress address)
    {
        if (DEBUG) {
            logger.log(Logger.INFO,"messageDelivered - XXX not implemented");
        }
    }

    public void forwardMessage(PacketReference ref, Collection consumers)
    {
         Globals.getClusterRouter().forwardMessage(ref, consumers);
    }

    public boolean lockUIDPrefix(short p) {
        if (DEBUG) {
            logger.log(Logger.INFO, "lockUIDPrefix " + p);
        }
        return (protocol.lockResource("uidprefix:" +
            Short.toString(p), 0, new ConnectionUID(0)) ==
            ClusterGlobals.MB_LOCK_SUCCESS);
    }

    //-----------------------------------------------
    //-      MessageBusCallback                     -
    //-----------------------------------------------

    
    public void configSyncComplete() {
       ServiceManager sm = Globals.getServiceManager();
        try {
            sm.resumeAllActiveServices(ServiceType.NORMAL);
        }
        catch (Exception e) {
            logger.logStack(Logger.ERROR,
                BrokerResources.E_INTERNAL_BROKER_ERROR,
                "during broker initialization",
                e);
        }

        // Generate a unique short prefix for the UID. This must be
        // done after startClusterIO()..
        Random r = new Random();
        boolean uidInit = false;
        for (int i = 0; i < 5; i++) {
            short p = (short) r.nextInt(Short.MAX_VALUE);
            if (lockUIDPrefix(p)) {
                com.sun.messaging.jmq.util.UID.setPrefix(p);
                uidInit = true;
                break;
            }
        }

        if (! uidInit) {
            // This should never happen.
            logger.log(Logger.WARNING,
                BrokerResources.E_INTERNAL_BROKER_ERROR,
                "Could not generate a unique prefix for UIDs.");
        }
    }

    /**
     * Interest creation notification. This method is called when
     * any remote interest is created.
     */
    public void interestCreated(Consumer intr) {
        try {
            Globals.getClusterRouter().addConsumer(intr);
        } catch (Exception ex) {
           logger.log(Logger.INFO,"Internal Error:  unable to add remote consumer "+
                intr, ex);
        }
    }

    public void unsubscribe(Subscription sub) {
        if (DEBUG) {
            logger.log(Logger.DEBUG,"callback unsubscribe : " + sub);
        }

        assert sub != null;

        try {
            Subscription.remoteUnsubscribe(sub.getDurableName(), sub.getClientID());
        } catch (Exception ex) {

           int loglevel = Logger.ERROR;
           if (ex instanceof BrokerException) {
              if (((BrokerException)ex).getStatusCode() == Status.PRECONDITION_FAILED
                  || ((BrokerException)ex).getStatusCode() == Status.NOT_FOUND) {
                 loglevel = Logger.WARNING;
              }
           }
           if (loglevel == Logger.ERROR || DEBUG) {
              String args[] = { sub.getDurableName(), sub.getClientID(), ex.getMessage() };
              logger.logStack(loglevel, BrokerResources.E_CLUSTER_UNSUBSCRIBE_EXCEPTION, args, ex);
           } else {
              logger.log(loglevel, ex.getMessage());
           }

        }
    }

    /**
     * Interest removal notification. This method is called when
     * any remote interest is removed.
     */
    public void interestRemoved(Consumer cuid){
        if (DEBUG) {
            logger.log(Logger.INFO,"callback interestRemoved " + cuid);
        }
        try {
            Globals.getClusterRouter().removeConsumer(cuid.getConsumerUID());
        } catch (Exception ex) {
           logger.log(Logger.INFO,"Internal Error:  unable to remove remote consumer "+
                cuid, ex);
        }
    }



    /**
     * Primary interest change notification. This method is called when
     * a new interest is chosen as primary interest for a failover queue.
     */
    public void activeStateChanged(Consumer intr) {
        // does nothing
        if (DEBUG) {
            logger.log(Logger.INFO,"callback activeStateChanged " + intr );
        }
    }


    /**
     * Client down notification. This method is called when a local
     * or remote client connection is closed.
     */
    public void clientDown(ConnectionUID conid){
        if (DEBUG) {
            logger.log(Logger.INFO,"clientDown " + conid );
        }
        try {
            Globals.getClusterRouter().removeConsumers(conid);
        } catch (Exception ex) {
           logger.log(Logger.INFO,"Internal Error: unable to remove remote consumers "+
                conid, ex);
        }
    }


    /**
     * Broker down notification. This method is called when any broker
     * in this cluster goes down.
     */
    public void brokerDown(com.sun.messaging.jmq.jmsserver.core.BrokerAddress broker){
        if (DEBUG) {
            logger.log(Logger.INFO,"brokerDown " + broker );
        }
        try {
            Globals.getClusterRouter().brokerDown(broker);
        } catch (Exception ex) {
           logger.log(Logger.INFO,"Internal Error: unable to remove remote consumers "+
                broker, ex);
        }
    }


    /**
     * A new destination was created by the administrator on a remote
     * broker.  This broker should also add the destination if it is
     * not already present.
     */
    public void notifyCreateDestination(Destination d){
        try  {
            Destination.addDestination(d);
            d.store();
        } catch (Exception ex) {
            logger.log(Logger.DEBUG,"Received exception adding new destination"
                  + " is caused because the destination " + d 
                  + " is being autocreated on both sides", ex);
 
        }
    }


    /**
     * A destination was removed by the administrator on a remote
     * broker. This broker should also remove the destination, if it
     * is present.
     */
    public void notifyDestroyDestination(DestinationUID uid){
        try {
            Destination.removeDestination(uid, false, 
               Globals.getBrokerResources().getString(
                   BrokerResources.M_ADMIN_REMOTE));
        } catch (Exception ex) {
            logger.log(Logger.DEBUG,"Internal Error: unable to remove stored destination "
               + uid , ex);
        }
    }


    /**
     * A destination was updated
     */
    public void notifyUpdateDestination(DestinationUID uid, Map changes) {
       Destination d = Destination.getDestination(uid);
       if (d != null) {
           try {
               d.setDestinationProperties(changes);
           } catch (Exception ex) {
               logger.log(Logger.INFO,"Internal Error, unable to update destination " +
                    uid.toString(), ex);
           }
       }
    }

    public void processRemoteMessage(Packet msg, List consumers, BrokerAddress home,
                                     boolean sendMsgRedeliver) throws BrokerException {

        Globals.getClusterRouter().handleJMSMsg(msg, consumers, home, sendMsgRedeliver); 
    }

    public void processRemoteAck(SysMessageID sysid, 
                                 com.sun.messaging.jmq.jmsserver.core.ConsumerUID cid,
                                 int ackType, Map optionalProps, Long txnID,
                                 UID msgBrokerSession, UID msgStoreSession) 
                                 throws BrokerException {
        if (DEBUG) {
        logger.log(Logger.INFO, "processRemoteAck: " + sysid + ":" + cid + ":" + 
               ackType + "(" + convertToLocalType(ackType)+")"+", brokerSession:"+
               msgBrokerSession+", storeSession:"+msgStoreSession);
        }

        Globals.getClusterRouter().handleAck(convertToLocalType(ackType), sysid, cid, 
                              optionalProps, txnID, msgBrokerSession, msgStoreSession);
    }

    /**
     * Switch to HA_ACTIVE state.
     *
     * Falcon HA: Complete the initialization process, start all the
     * ServiceType.NORMAL services and start processing client work.
     */
    public void goHAActive(){
        //XXX  does nothing right now
    }


}

/*
 * EOF
 */
