/*
 * @(#)ClusterConsumerInfo.java	1.7 08/09/05
 *
 * Copyright 2005 Sun Microsystems, Inc. All Rights Reserved
 * SUN PROPRIETARY/CONFIDENTIAL
 * Use is subject to license terms. 
 *
 */

package com.sun.messaging.jmq.jmsserver.multibroker.raptor;

import java.util.*;
import java.io.*;
import java.nio.*;
import com.sun.messaging.jmq.io.GPacket;
import com.sun.messaging.jmq.util.log.Logger;
import com.sun.messaging.jmq.util.selector.SelectorFormatException;
import com.sun.messaging.jmq.jmsserver.Globals;
import com.sun.messaging.jmq.jmsserver.util.BrokerException;
import com.sun.messaging.jmq.jmsserver.core.Consumer;
import com.sun.messaging.jmq.jmsserver.core.Subscription;
import com.sun.messaging.jmq.jmsserver.core.Destination;
import com.sun.messaging.jmq.jmsserver.core.ConsumerUID;
import com.sun.messaging.jmq.jmsserver.core.DestinationUID;
import com.sun.messaging.jmq.jmsserver.core.BrokerAddress;
import com.sun.messaging.jmq.jmsserver.service.ConnectionUID;
import com.sun.messaging.jmq.jmsserver.multibroker.Cluster;
import com.sun.messaging.jmq.jmsserver.multibroker.raptor.ProtocolGlobals;

/**
 * An instance of this class is intended to be used one direction only 
 * either Consumers -> GPacket or GPacket -> Consumers (see assertions)
 */

public class ClusterConsumerInfo
{
    private Logger logger = Globals.getLogger();
	private static final long ConsumerVersionUID = 99353142765567461L;

    private Cluster c;
    private Collection consumers = null;
    private GPacket pkt = null;

    private ClusterConsumerInfo(Collection consumers, Cluster c) {
        this.consumers = consumers;
        this.c = c;
    }

    private ClusterConsumerInfo(Consumer consumer, Cluster c) {
        Set s = new HashSet();
        s.add(consumer);
        this.consumers = s;
        this.c = c;
    }
    
    private  ClusterConsumerInfo(GPacket pkt, Cluster c) {
        this.pkt = pkt;
        this.c = c;
    }

    public static ClusterConsumerInfo newInstance(Collection consumers, Cluster c) {
        return new ClusterConsumerInfo(consumers, c);
    }

    public static ClusterConsumerInfo newInstance(Consumer consumer, Cluster c) {
        return new ClusterConsumerInfo(consumer, c);
    }

    public static ClusterConsumerInfo newInstance(GPacket pkt, Cluster c) { 
        return new ClusterConsumerInfo(pkt, c);
    }

    public GPacket getGPacket(short protocol) {
        return getGPacket(protocol, -1); 
    }

    public GPacket getGPacket(short protocol, int subtype) {
        assert ( consumers != null );
        assert ( protocol == ProtocolGlobals.G_NEW_INTEREST ||
                 protocol == ProtocolGlobals.G_INTEREST_UPDATE );

        if (protocol == ProtocolGlobals.G_INTEREST_UPDATE) {
        assert ( subtype == ProtocolGlobals.G_NEW_PRIMARY_INTEREST || // not effectively used ?
                 subtype == ProtocolGlobals.G_REM_INTEREST ||
                 subtype == ProtocolGlobals.G_DURABLE_DETACH );
        }

        GPacket gp = GPacket.getInstance();
        gp.setType(protocol);
        gp.putProp("C", new Integer(consumers.size()));
        if (c != null) c.marshalBrokerAddress(c.getSelfAddress(), gp);

        ByteArrayOutputStream bos = new ByteArrayOutputStream();
        DataOutputStream dos = new DataOutputStream(bos);

        switch (protocol) {
            case ProtocolGlobals.G_NEW_INTEREST:
            try {
                Iterator itr = consumers.iterator();
                while (itr.hasNext()) {
                    Consumer c = (Consumer) itr.next();
                    writeConsumer(c, dos);
                }
                dos.flush();
                bos.flush();
            }
            catch (IOException e) { /* Ignore */ }

            gp.setPayload(ByteBuffer.wrap(bos.toByteArray()));
            break;

            case ProtocolGlobals.G_INTEREST_UPDATE:

            gp.putProp("T", new Integer(subtype));
            try {
                Iterator itr = consumers.iterator();
                while (itr.hasNext()) {
                    Consumer c = (Consumer)itr.next();
                    writeConsumerUID(c.getConsumerUID(), dos);
                }
                dos.flush();
                bos.flush();
            }
            catch (IOException e) { /* Ignore */ }

            gp.setPayload(ByteBuffer.wrap(bos.toByteArray()));
            break;
        }

        return gp;
    }

    public int getConsumerCount() {
        assert ( pkt !=  null ); 
		return ((Integer)pkt.getProp("C")).intValue();
    }

    public int getSubtype() {
        assert ( pkt != null );

        short type = pkt.getType();
        assert ( type == ProtocolGlobals.G_INTEREST_UPDATE );

        return ((Integer)pkt.getProp("T")).intValue();
    }

    public Iterator getConsumers() throws Exception {
        assert ( pkt !=  null ); 

        short type = pkt.getType();
        assert ( type == ProtocolGlobals.G_NEW_INTEREST );

        return new ConsumerIterator(pkt.getPayload().array(), getConsumerCount(), 
                                    c.unmarshalBrokerAddress(pkt));
    }

    public Iterator getConsumerUIDs() throws Exception {
        assert ( pkt !=  null ); 

        short type = pkt.getType();
        assert ( type == ProtocolGlobals.G_INTEREST_UPDATE );

        return new ProtocolConsumerUIDIterator(pkt.getPayload().array(), getConsumerCount(),
                                               c.unmarshalBrokerAddress(pkt));
    }

    public boolean isConfigSyncResponse() {
        assert ( pkt != null );

        boolean b = false;
        if (pkt.getProp("M") != null) {
            b = ((Boolean) pkt.getProp("M")).booleanValue();
        }
        return b;
    }

    public boolean needReply() {
        assert ( pkt != null );
        return pkt.getBit(pkt.A_BIT);
    }


    public static void writeConsumer(Consumer consumer, DataOutputStream dos)
                       throws IOException
    {
        String destName = consumer.getDestinationUID().getName();
        ConsumerUID id = consumer.getConsumerUID();
        String durableName = null;
        String clientID = null;
        String selstr = consumer.getSelectorStr();
        boolean noLocalDelivery = consumer.getNoLocal();
        boolean isQueue = consumer.getDestinationUID().isQueue();
        boolean isReady = true;
        boolean setMaxCnt = false;
        int maxcnt = 1;
 
        if (consumer instanceof Subscription ) {
            Subscription s = (Subscription)consumer;
            maxcnt = s.getMaxNumActiveConsumers();
            setMaxCnt = true;
            durableName = s.getDurableName();
            clientID = s.getClientID();
            if (! s.isActive())
                isReady = false;
        }
        dos.writeLong(ConsumerVersionUID); // version
        dos.writeUTF(destName);
        dos.writeBoolean(id != null);
        if (id != null) {
            writeConsumerUID(id, dos);
        }
        dos.writeBoolean(clientID != null);
        if (clientID != null) {
            dos.writeUTF(clientID);
        }
        dos.writeBoolean(durableName != null);
        if (durableName != null) {
            dos.writeUTF(durableName);
        }
        dos.writeBoolean(selstr != null);
        if (selstr != null) {
            dos.writeUTF(selstr);
        }
        dos.writeBoolean(isQueue);
        dos.writeBoolean(noLocalDelivery);
        dos.writeBoolean(isReady);
        dos.writeBoolean(setMaxCnt);
        if (setMaxCnt)
            dos.writeInt(maxcnt);
    }

    public static Consumer readConsumer(DataInputStream dis) throws IOException
    {
        Logger logger = Globals.getLogger();
        ConsumerUID id = null;
        String destName = null;
        String clientID = null;
        String durableName = null;
        String selstr = null;
        boolean isQueue;
        boolean noLocalDelivery;
        boolean consumerReady;
        int sharedcnt;

        long ver = dis.readLong(); // version
        if (ver != ConsumerVersionUID) {
            throw new IOException("Wrong Consumer Version " + ver + " expected " + ConsumerVersionUID);
        }
        destName = dis.readUTF();
        boolean hasId = dis.readBoolean();
        if (hasId) {
            id = readConsumerUID(dis);
        }
        boolean hasClientID = dis.readBoolean();
        if (hasClientID) {
            clientID = dis.readUTF();
        }
        boolean hasDurableName = dis.readBoolean();
        if (hasDurableName) {
            durableName = dis.readUTF();
        }

        boolean hasSelector = dis.readBoolean();
        if (hasSelector) {
            selstr = dis.readUTF();
        }

        isQueue = dis.readBoolean();
        noLocalDelivery = dis.readBoolean();
        consumerReady = dis.readBoolean();

        boolean sharedSet = false;
        sharedcnt = 1;
        try {
            sharedSet = dis.readBoolean();
            if (sharedSet == true) {
                sharedcnt = dis.readInt();
            } 
        } catch (Exception ex) {
            // do nothing prevents failures with old brokers
        }

        DestinationUID dest = DestinationUID.getUID(destName, isQueue);

        try {
            if (durableName != null) {
                Subscription sub = Subscription.findCreateDurableSubscription
                       (clientID,durableName, dest, selstr, noLocalDelivery, false,  id);
                if (sub != null)
                    sub.setMaxNumActiveConsumers(sharedcnt);
                return sub;
            }
            else {
                if (sharedSet) { /* non-durable subscriber */
                    Subscription sub = Subscription.findCreateNonDurableSubscription(
                              clientID, selstr, dest, noLocalDelivery, id ); 
                    if (sub != null)
                        sub.setMaxNumActiveConsumers(sharedcnt);
                    return sub;
                } else {
                    Consumer c = new Consumer(dest, selstr, noLocalDelivery, id);
                    return c;
                }
            }
         } catch (SelectorFormatException ex) {
             logger.log(Logger.INFO,"Internal Error: Got bad selector["+selstr + "] " , ex);
             throw new IOException("bad selector " + selstr);
         } catch (BrokerException ex) {
             logger.logStack(Logger.INFO,"Internal Error: error creating consumer ", ex);
             throw new IOException("error creating consumer ");
         }

    }


    public static void writeConsumerUID(ConsumerUID uid, DataOutputStream dos)
                                                            throws IOException
    {
        dos.writeLong(uid.longValue()); // UID write
        dos.writeLong((uid.getConnectionUID() == null ? 0 :
                                   uid.getConnectionUID().longValue()));
        BrokerAddress brokeraddr= uid.getBrokerAddress();
        if (brokeraddr == null) brokeraddr = Globals.getMyAddress();

        if (brokeraddr == null) {
            // XXX Revisit and cleanup : This method may be called
            // before cluster initialization only during persistent
            // store upgrade. i.e. from -
            // FalconProtocol.upgradeConfigChangeRecord()
            // At that time, Globals.getMyAddress() returns null.
            // Hence this kludge...
            try {
            brokeraddr =
            new com.sun.messaging.jmq.jmsserver.multibroker.fullyconnected.BrokerAddressImpl();
            }
            catch (Exception e) {}
        }

        brokeraddr.writeBrokerAddress(dos); // UID write
    }


    public static ConsumerUID readConsumerUID(DataInputStream dis)
          throws IOException
    {
        long id = dis.readLong(); // UID write
        ConnectionUID conuid = new ConnectionUID(dis.readLong());
        BrokerAddress tempaddr = Globals.getMyAddress();
        BrokerAddress brokeraddr = (BrokerAddress)tempaddr.clone();
        brokeraddr.readBrokerAddress(dis); // UID write
        ConsumerUID cuid = new ConsumerUID(id);
        cuid.setConnectionUID(conuid);
        cuid.setBrokerAddress(brokeraddr);
        return cuid;
    }

    public static GPacket getReplyGPacket(short protocol, int status) {
        GPacket gp = GPacket.getInstance();
        gp.setType(protocol);
        gp.putProp("S", new Integer(status));
        return gp;
    }

}

class ConsumerIterator implements Iterator
{
    private int count = 0;
    private int count_read = 0;
    private DataInputStream dis = null;
    private BrokerAddress from = null;

    public ConsumerIterator(byte[] payload, int count, BrokerAddress from) {
        ByteArrayInputStream bis = new ByteArrayInputStream(payload);
        dis = new DataInputStream(bis);
        this.count = count;
        this.count_read = 0;
        this.from = from;
    }

    public boolean hasNext() { 
        if (count_read < 0) throw new IllegalStateException("ConsumerUID");  
        return count_read < count;
    }

    /**
     * Caller must catch RuntimeException and getCause
     */
    public Object next() throws RuntimeException {
        try {
        Consumer c =  ClusterConsumerInfo.readConsumer(dis);
        if (from != null) c.getConsumerUID().setBrokerAddress(from);
        count_read++;
        return c;
        } catch (IOException e) {
        count_read = -1;
        throw new RuntimeException(e);
        }
    }

    public void remove() {
        throw new UnsupportedOperationException("Not supported");
    }
}

