/*
 * @(#)ClusterMessageInfo.java	1.4 05/20/05
 *
 * Copyright 2005 Sun Microsystems, Inc. All Rights Reserved
 * SUN PROPRIETARY/CONFIDENTIAL
 * Use is subject to license terms. 
 *
 */

package com.sun.messaging.jmq.jmsserver.multibroker.raptor;

import java.io.*;
import java.util.*;
import java.nio.*;
import com.sun.messaging.jmq.io.GPacket;
import com.sun.messaging.jmq.io.Packet;
import com.sun.messaging.jmq.util.log.Logger;
import com.sun.messaging.jmq.jmsserver.Globals;
import com.sun.messaging.jmq.jmsserver.core.Consumer;
import com.sun.messaging.jmq.jmsserver.core.ConsumerUID;
import com.sun.messaging.jmq.jmsserver.core.PacketReference;
import com.sun.messaging.jmq.jmsserver.core.BrokerAddress;
import com.sun.messaging.jmq.jmsserver.resources.BrokerResources;
import com.sun.messaging.jmq.jmsserver.multibroker.Cluster;
import com.sun.messaging.jmq.jmsserver.multibroker.raptor.ProtocolGlobals;

/**
 * An instance of this class is intended to be used one direction only
 */

public class ClusterMessageInfo 
{
    protected Logger logger = Globals.getLogger();

    private PacketReference ref = null;
    private ArrayList consumers =  null; 
    private boolean sendMessageDeliveredAck = false;
    private Cluster c = null;

    private GPacket pkt = null;
    private DataInputStream dis = null;

    private ClusterMessageInfo(PacketReference ref, 
                               ArrayList consumers,
                               boolean sendMessageDeliveredAck, Cluster c) {
        this.ref = ref;
        this.consumers = consumers;
        this.sendMessageDeliveredAck = sendMessageDeliveredAck;
        this.c = c;
    }

    private ClusterMessageInfo(GPacket pkt, Cluster c) {
        this.pkt = pkt;
        this.c = c;
    }

    /**
     * Destination to GPacket
     *
     * @param d The Destination to be marshaled to GPacket
     */
    public static ClusterMessageInfo newInstance(
                                           PacketReference ref,
                                           ArrayList consumers,
                                           boolean sendMessageDeliveredAck,
                                           Cluster c) {
        return new ClusterMessageInfo(ref, consumers, sendMessageDeliveredAck, c);
    }

    /**
     * GPacket to Destination 
     *
     * @param pkt The GPacket to be unmarsheled
     */
    public static ClusterMessageInfo newInstance(GPacket pkt, Cluster c) {
        return new ClusterMessageInfo(pkt, c);
    }

    public GPacket getGPacket() { 
        assert ( ref !=  null );
        assert ( consumers !=  null );

        GPacket gp = GPacket.getInstance();
        gp.setType(ProtocolGlobals.G_MESSAGE_DATA);
        gp.putProp("D", new Boolean(sendMessageDeliveredAck));
        gp.putProp("C", new Integer(consumers.size()));
        c.marshalBrokerAddress(c.getSelfAddress(), gp);

        ByteArrayOutputStream bos = new ByteArrayOutputStream();
        DataOutputStream dos = new DataOutputStream(bos);


        Packet roPkt = null;
        try {
            for (int i = 0; i < consumers.size(); i++) {
                ConsumerUID intid = ((Consumer) consumers.get(i)).getConsumerUID();
                ClusterConsumerInfo.writeConsumerUID(intid, dos);
            }
            if (roPkt == null) roPkt = ref.getPacket();
            roPkt.generateTimestamp(false);
            roPkt.generateSequenceNumber(false);

            roPkt.writePacket(dos);
            dos.flush();
            bos.flush();
           
        } catch (Exception e) {
            logger.logStack(Logger.WARNING,"Exception writing packet ", e);
        }

        byte[] buf = bos.toByteArray();
        gp.setPayload(ByteBuffer.wrap(buf));

        return gp;
    }

    public String toString() {
        if (consumers == null || ref == null) return super.toString();
        StringBuffer buf = new StringBuffer("\n");
        if (consumers !=  null && ref != null) {
            for (int i = 0; i < consumers.size(); i++) {
                ConsumerUID intid = ((Consumer) consumers.get(i)).getConsumerUID();
                buf.append("\t").append(intid).append("\n");
            }
        }
        return buf.toString();
    }

    public BrokerAddress getHomeBrokerAddress() throws Exception {
        assert (pkt != null);
        return c.unmarshalBrokerAddress(pkt);
    }

    public boolean getSendMessageDeliveredAck() {
        assert (pkt != null);
        return ((Boolean) pkt.getProp("D")).booleanValue();
    }

    public int getConsumerCount() {
        assert (pkt != null);
        return ((Integer) pkt.getProp("C")).intValue();
    }

    /**
     * must called in the following order: 
     *
     * initPayloadRead()
     * readPayloadConsumerUIDs()
     * readPayloadMessage()
     */
    public void initPayloadRead() {
        assert ( pkt != null );
        ByteArrayInputStream bis = new ByteArrayInputStream(pkt.getPayload().array());
        dis = new DataInputStream(bis);
    }
    public Iterator readPayloadConsumerUIDs() {
        assert ( pkt !=  null );
        assert ( dis !=  null );

        return new ProtocolConsumerUIDIterator(dis, getConsumerCount());
    }
    public Packet readPayloadMessage() throws IOException {
        assert ( pkt !=  null );
        assert ( dis !=  null );

        Packet roPkt = new Packet(false);
        roPkt.generateTimestamp(false);
        roPkt.generateSequenceNumber(false);
        roPkt.readPacket(dis);
        return roPkt;
    }

    public boolean needReply() {
        assert ( pkt != null );
        return pkt.getBit(pkt.A_BIT);
    }

    public GPacket getReplyGPacket(int status) {
        assert ( pkt != null );

        GPacket gp = GPacket.getInstance();
        gp.setType(ProtocolGlobals.G_MESSAGE_DATA_REPLY);
        gp.putProp("S", new Integer(status));
        // TBD: ADD SysMessageID as property?

        return gp;
    }

}

