/*
 * @(#)ClusterTakeoverInfo.java	1.7 09/29/05
 *
 * Copyright 2005 Sun Microsystems, Inc. All Rights Reserved
 * SUN PROPRIETARY/CONFIDENTIAL
 * Use is subject to license terms. 
 *
 */

package com.sun.messaging.jmq.jmsserver.multibroker.raptor;

import java.io.*;
import java.util.*;
import java.nio.*;
import com.sun.messaging.jmq.io.GPacket;
import com.sun.messaging.jmq.io.Status;
import com.sun.messaging.jmq.util.UID;
import com.sun.messaging.jmq.jmsserver.Globals;
import com.sun.messaging.jmq.jmsserver.cluster.ClusteredBroker;
import com.sun.messaging.jmq.jmsserver.cluster.ha.HAClusteredBroker;
import com.sun.messaging.jmq.jmsserver.multibroker.raptor.ProtocolGlobals;
import com.sun.messaging.jmq.jmsserver.util.BrokerException;

/**
 * An instance of this class is intended to be used one direction only
 */

public class ClusterTakeoverInfo 
{
    public static boolean DEBUG = false;

    private GPacket pkt = null;

    private String brokerID = null;
    private UID storeSession = null;
    private UID brokerSession = null;
    private String brokerHost = null;
    private Long xid =   null;
    private boolean fromTaker = true;
    private String taker = null;
    private Long timestamp = null;
    private boolean timedout = false;

    private ClusterTakeoverInfo(String brokerID, UID storeSession, 
                                String brokerHost, UID brokerSession,
                                Long xid, boolean fromTaker, boolean timedout) {
        this.brokerID = brokerID;
        this.storeSession = storeSession;
        this.brokerHost = brokerHost;
        this.brokerSession = brokerSession;
        this.xid = xid;
        this.fromTaker = fromTaker;
        this.timedout = timedout;
    }

    private ClusterTakeoverInfo(GPacket pkt) {
        assert ( pkt.getType() == ProtocolGlobals.G_TAKEOVER_COMPLETE ||
                 pkt.getType() == ProtocolGlobals.G_TAKEOVER_PENDING ||
                 pkt.getType() == ProtocolGlobals.G_TAKEOVER_ABORT );

        this.pkt = pkt;
        brokerID = (String)pkt.getProp("brokerID");
        storeSession = new UID(((Long)pkt.getProp("storeSession")).longValue());
        Long v = (Long)pkt.getProp("brokerSession");
        if (v != null) brokerSession = new UID(v.longValue());
        brokerHost =(String)pkt.getProp("brokerHost");
        taker = (String)pkt.getProp("taker");
        fromTaker = (taker != null);
        timestamp = (Long)pkt.getProp("timestamp");
        if (timestamp != null) timedout = (timestamp.equals(new Long(0)));
        xid = (Long)pkt.getProp("X");
    }

    /**
     */
    public static ClusterTakeoverInfo newInstance(String brokerID, UID storeSession) {
        return new ClusterTakeoverInfo(brokerID, storeSession, null, null, null, false, false);
    }

    public static ClusterTakeoverInfo newInstance(String brokerID, UID storeSession, 
                                                  String brokerHost, UID brokerSession,
                                                  Long xid, boolean fromTaker) {
        return new ClusterTakeoverInfo(brokerID, storeSession, brokerHost,
                                       brokerSession, xid, fromTaker, false);
    }

    public static ClusterTakeoverInfo newInstance(String brokerID, UID storeSession, 
                                        String brokerHost, UID brokerSession, Long xid,
                                        boolean fromTaker, boolean timedout) {
        return new ClusterTakeoverInfo(brokerID, storeSession, brokerHost,
                                       brokerSession, xid, fromTaker, timedout);
    }

    /**
     *
     * @param pkt The GPacket to be unmarsheled
     */
    public static ClusterTakeoverInfo newInstance(GPacket pkt) {
        return new ClusterTakeoverInfo(pkt);
    }

    public GPacket getGPacket(short protocol) throws BrokerException { 
        assert ( protocol == ProtocolGlobals.G_TAKEOVER_COMPLETE ||
                 protocol == ProtocolGlobals.G_TAKEOVER_PENDING ||
                 protocol == ProtocolGlobals.G_TAKEOVER_ABORT );
        if (!Globals.getHAEnabled()) throw new BrokerException("XXXI18N - not HA");

        if (pkt != null) {
            assert ( pkt.getType() == protocol );
           return pkt;
        }

        GPacket gp = GPacket.getInstance();
        gp.putProp("brokerID", brokerID);
        gp.putProp("storeSession", new Long(storeSession.longValue()));

        if (protocol == ProtocolGlobals.G_TAKEOVER_COMPLETE) {
            gp.setType(protocol);
            gp.setBit(gp.A_BIT, false);
            return gp;
        }

        HAClusteredBroker cb = (HAClusteredBroker)Globals.getClusterManager().getLocalBroker();
        if (protocol == ProtocolGlobals.G_TAKEOVER_PENDING) {
            gp.setType(ProtocolGlobals.G_TAKEOVER_PENDING);
            gp.setBit(gp.A_BIT, false);
            gp.putProp("brokerSession", new Long(brokerSession.longValue()));
            gp.putProp("brokerHost",  brokerHost);
            if (fromTaker) {
                gp.putProp("taker", cb.getBrokerName());
                gp.putProp("timestamp", new Long(cb.getHeartbeat()));
                gp.setBit(gp.A_BIT, true);
            } else if (timedout) { 
                gp.putProp("timestamp", new Long(0));
            }
            gp.putProp("X", xid);
            return gp;
        }

        if (protocol == ProtocolGlobals.G_TAKEOVER_ABORT) {
            gp.setType(ProtocolGlobals.G_TAKEOVER_ABORT);
            if (fromTaker) {
            gp.putProp("taker", cb.getBrokerName());
            }
            gp.setBit(gp.A_BIT, false);
            gp.putProp("X", xid);
            return gp;
        }
        throw new BrokerException("Unknown protocol: "+protocol);
    }

    public String getBrokerID() {
        return brokerID;
    }

    public UID getStoreSession() {
        return storeSession;
    }

    public Long getXid() {
        return xid;
    }

    public UID getBrokerSession() {
        return brokerSession;
    }

    public String getBrokerHost() {
        return brokerHost;
    }

    public String getTaker() {
        assert (pkt != null);
        return taker;
    }

    public Long getTimestamp() {
        assert (pkt != null);
        return timestamp;
    }

    public boolean isTimedout() {
        assert (pkt != null);
        return timedout;
    }

    public boolean isFromTaker() {
        assert (pkt != null);
        return fromTaker;
    }

    public String toString() {
        assert ( pkt != null );
        if (pkt.getType() == ProtocolGlobals.G_TAKEOVER_COMPLETE) {
        return "[brokerID="+getBrokerID()+", storeSession="+getStoreSession()+"]";
        }
        if (pkt.getType() == ProtocolGlobals.G_TAKEOVER_PENDING) {
        String prefix = "";
        if (DEBUG) {
        if (getTaker() != null) prefix = getTaker()+"("+ ((timestamp == null) ? "":""+timestamp)+":";
        return prefix+ "[brokerID="+getBrokerID()+
                       ", storeSession="+getStoreSession()+"]-"+
                       getBrokerHost()+", "+getBrokerSession()+", "+xid;
        }
        return prefix+ "[brokerID="+getBrokerID()+
                       ", storeSession="+getStoreSession()+"],"+xid;
        }
        return getTaker()+
              ":[brokerID="+getBrokerID()+
               ", storeSession="+getStoreSession()+"], "+xid;
    }

    public boolean needReply() {
        assert ( pkt != null );
        return pkt.getBit(pkt.A_BIT);
    }

    public GPacket getReplyGPacket(short protocol, int status, String reason) {
        assert ( pkt != null );
        assert ( protocol == ProtocolGlobals.G_TAKEOVER_PENDING_REPLY );

        GPacket gp = GPacket.getInstance();
        gp.setType(protocol);
        gp.putProp("X", (Long)pkt.getProp("X"));
        gp.putProp("brokerID", pkt.getProp("brokerID"));
        gp.putProp("storeSession", pkt.getProp("storeSession"));
        gp.putProp("taker", pkt.getProp("taker"));
        gp.putProp("S", new Integer(status));
        if (reason != null) gp.putProp("reason", reason);

        return gp;
    }

    public static int getReplyStatus(GPacket reply) {
        return ((Integer)reply.getProp("S")).intValue();
    }
    public static String getReplyStatusReason(GPacket reply) {
        return (String)reply.getProp("reason");
    }
    public static Long getReplyXid(GPacket reply) {
        return (Long)reply.getProp("X");
    }
    public static String toString(GPacket reply) {
        StringBuffer buf = new StringBuffer();
        buf.append("\n\treplyStatus = ").append(Status.getString(getReplyStatus(reply)));
        if (reply.getProp("reason") != null) {
        buf.append("\n\treason = ").append(getReplyStatusReason(reply));
        }
        buf.append("[brokerID = ").append((String)reply.getProp("brokerID"));
        buf.append(", storeSession = ").append((Long)reply.getProp("storeSession"));
        buf.append("], ").append((Long)reply.getProp("X"));
        buf.append(" - ").append((String)reply.getProp("taker"));
        return buf.toString();
    }
}
