/*
 * @(#)BrokerDAO.java	1.10 07/14/05
 *
 * Copyright 2004 Sun Microsystems, Inc. All Rights Reserved
 * SUN PROPRIETARY/CONFIDENTIAL
 * Use is subject to license terms.
 *
 */

package com.sun.messaging.jmq.jmsserver.persist.jdbc;

import com.sun.messaging.jmq.jmsserver.util.BrokerException;
import com.sun.messaging.jmq.jmsserver.persist.HABrokerInfo;
import com.sun.messaging.jmq.jmsserver.cluster.BrokerState;

import java.sql.Connection;
import java.util.HashMap;

/**
 * This class is an interface for the Broker table which will be implemented
 * by database specific code.
 *
 * @version	1.10
 */
public interface BrokerDAO extends BaseDAO {

    /**
     * Broker table:
     * Holds all the broker info in a HA cluster.
     *
     * CREATE TABLE MQBKR<schemaVersion>[C<clusterID>|S<brokerID>] (
     * 		ID                  VARCHAR(100) NOT NULL,
     * 		URL                 VARCHAR(100) NOT NULL,
     * 		VERSION             INTEGER NOT NULL,
     * 		STATE               INTEGER NOT NULL,
     * 		SESSION_ID          BIGINT,
     * 		TAKEOVER_BROKER     VARCHAR(100),
     * 		HEARTBEAT_TS        BIGINT,
     * 		PRIMARY KEY(ID)
     * );
     *
     * ID - Unique ID of the broker
     * URL - The URL of the broker (i.e. includes hostname and port)
     * VERSION - Current version of the borker
     * STATE - State of the broker
     * SESSION_ID - Unique ID associated with this run of the broker
     * TAKEOVER_BROKER - Name of broker that has taken over the store
     * HEARTBEAT_TS - Timestamp periodically updated by a running borker
     */
    public static final String TABLE_NAME_PREFIX = "MQBKR" + DBConstants.SCHEMA_VERSION;
    public static final String ID_COLUMN = "ID";
    public static final String URL_COLUMN = "URL";
    public static final String VERSION_COLUMN = "VERSION";
    public static final String STATE_COLUMN = "STATE";
    public static final String SESSION_ID_COLUMN = "SESSION_ID";
    public static final String TAKEOVER_BROKER_COLUMN = "TAKEOVER_BROKER";
    public static final String HEARTBEAT_TS_COLUMN = "HEARTBEAT_TS";

    void insert( Connection conn, String id, String url, int version,
        BrokerState state, long sessionID, long heartbeat ) throws BrokerException;

    void update( Connection conn, String id, String takeoverID, String url,
        int version, BrokerState state, long sessionID ) throws BrokerException;

    boolean updateHeartbeat( Connection conn, String id, long heartbeat )
        throws BrokerException;

    boolean updateHeartbeat( Connection conn, String id, long heartbeat,
        long lastHeartbeat, BrokerState expectedState ) throws BrokerException;

    boolean updateState( Connection conn, String id, BrokerState newState,
        BrokerState expectedState) throws BrokerException;

    void delete( Connection conn, String id ) throws BrokerException;

    HABrokerInfo takeover( Connection conn, String brokerID,
        String targetBrokerID, long lastHeartbeat, BrokerState expectedState,
        long newHeartbeat, BrokerState newState) throws BrokerException;

    long getHeartbeat( Connection conn, String id ) throws BrokerException;

    HashMap getAllHeartbeats( Connection conn ) throws BrokerException;

    BrokerState getState( Connection conn, String id ) throws BrokerException;

    Object[] getAllStates( Connection conn ) throws BrokerException;

    HABrokerInfo getBrokerInfo( Connection conn, String id ) throws BrokerException;

    HashMap getAllBrokerInfos( Connection conn ) throws BrokerException;

    HashMap getAllBrokerInfosByState( Connection conn, BrokerState state )
        throws BrokerException;
}
