/*
 * @(#)EventBroadcastHelper.java	1.13 12/18/03
 *
 * Copyright 2002 Sun Microsystems, Inc. All Rights Reserved
 * SUN PROPRIETARY/CONFIDENTIAL
 * Use is subject to license terms.
 *
 */

package com.sun.messaging.jmq.util.lists;

import java.util.*;
import java.io.*;


/**
 * this is a helper class to be used by
 * lists that implement EventBroadcaster
 */

public class EventBroadcastHelper implements EventBroadcaster
{
    Collection c[] = new Collection[EventType.EVENT_TYPE_NUM];
    boolean busy[] = new boolean[EventType.EVENT_TYPE_NUM];
    int start[] = null;
    int cnt = 0;
    boolean orderMaintained = true;
   
    // we change the order to address bug 4939969
    // I'm keeping the old behavior in the system incase
    // we ever need the system to respond in a more definitive
    // manner 
    /**
     * determines if listeners should always be called in
     * the same order or the system should change the order on
     * each call. Added to address bug 4939969.
     * @param order if true, order will be maintained
     */
    public synchronized void setOrderMaintained(boolean order) {
        orderMaintained = order;
        if (!orderMaintained) {
            start = new int[EventType.EVENT_TYPE_NUM];
        }
    }

    /**
     * creates a new EventBroadcastHelper
     */
    public EventBroadcastHelper() {
    }

    /**
     * clears all listeners from the helper
     */
    public synchronized void clear() {
        c = new Collection[EventType.EVENT_TYPE_NUM];
        for (int i=0; i < EventType.EVENT_TYPE_NUM; i ++)
            busy[i] = false;
    }


    /**
     * dumps the state of the helper
     * @param ps stream to write the state to
     */
    public void dump(PrintStream ps) {
        ps.println(toString());
    }

    /**
     * converts the state of the object to a string
     * @returns the object as a string
     */
    public String toString() {
        String str = "EventBroadcastHelper {\n";
        for (int i =0; i < c.length; i++) {
            boolean indent = false;
            str += "\t"+i+ " { ";
            if (c[i] == null) {
                str += "null";
            } else {
                Iterator itr = c[i].iterator();
                boolean first = true;
                int indx = 0;
                while (itr.hasNext()) {
                    ListenerInfo li = (ListenerInfo)itr.next();
                    indent = true;
                    if (!first) {
                        str += "\t    ";
                    }
                    first = false;
                    str+= indx + ":  "+li.getListener() 
                            + "\n\t        "
                            + li.getType() 
                            + "\n\t        " 
                            + li.getReason() 
                            + "\n\t        "
                            + li.getUserData() 
                            + "\n";
                    indx ++;
                }
            }
            if (indent) {
                str += "\t  }\n";
            } else { 
                str += " }\n";
            }
        }
        return str;
    }

    /**
     * Request notification when the specific event occurs.
     * @param listener object to notify when the event occurs
     * @param type event which must occur for notification
     * @param userData optional data queued with the notification
     * @return an id associated with this notification
     */
    public Object addEventListener(EventListener listener,
                        EventType type, Object userData) {
        return addEventListener(listener, type, null, userData);
    }

    /**
     * Request notification when the specific event occurs AND
     * the reason matched the passed in reason.
     * @param listener object to notify when the event occurs
     * @param type event which must occur for notification
     * @param userData optional data queued with the notification
     * @param reason reason which must be associated with the
     *               event (or null for all events)
     * @return an id associated with this notification
     */
    public Object addEventListener(EventListener listener,
                        EventType type, Reason reason,
                        Object userData) {
        ListenerInfo li = new ListenerInfo(listener,
                type, reason, userData);
        int indx = type.getEvent();
 
        // OK .. assuming adding & removing listeners are a rare
        // event so it can be slow (limit locks later)
        synchronized (this) {
            if (c[indx] == null) {
                c[indx] = new ArrayList();
                c[indx].add(li);
            } else {
                ArrayList ls = new ArrayList(c[indx]);
                ls.add(li);
                c[indx] = ls;
            }
            busy[indx]=true;
            cnt ++;
        }
        return li;
    }

    /**
     * remove the listener registered with the passed in
     * id.
     * @return the listener callback which was removed
     */
    public synchronized Object removeEventListener(Object id) {
        if (id == null) return null;
        ListenerInfo li = (ListenerInfo) id;
        int indx = li.getType().getEvent();
        Collection s = c[indx];
        if (s == null) return null;
        ArrayList newset = new ArrayList(s);
        newset.remove(li);
        busy[indx]=!newset.isEmpty();
        c[indx] = newset;
        EventListener l = li.getListener();
        li.clear();
        cnt --;
        return l;
    }


    /**
     * method which notifies all listeners an event
     * has occured.
     * @param type of event that has occured
     * @param reason why the event occured (may be null)
     * @param object the event occured on
     * @param oldval value before the event
     * @param newval value after the event
     */
    public void notifyChange(EventType type,  Reason r, 
               Object target,
               Object oldval, Object newval) 
    {
        ArrayList l = null;
        synchronized(this) {
            l = (ArrayList)c[type.getEvent()];
        }
        if (l == null || l.isEmpty()) {
            return;
        }

        int offset = 0;
        int index = 0;
        synchronized (l) {

           int size = l.size();
           if (!orderMaintained && start != null && size > 1) {
                offset = start[type.getEvent()];
                start[type.getEvent()] = (offset>= size-1) ? 0 :
                            offset + 1;
           }
           for (int count = 0; count < size; count ++) {
                // OK .. this code seems to be very timing senstive
                // on mq880 ... dont know why
                // this obscure calculation insures:
                //     offset = 0, index goes from 0-size
                //     offset = n, index wraps from n -> n-1
                index = (offset == 0 ? count :
                        ((count + offset)%size));
                ListenerInfo info = (ListenerInfo)l.get(index);
                if (info == null) continue;
                Reason lr = null;
                EventListener ll = null;
                Object ud = null;
                synchronized(info) {
                    ll = info.getListener();
                    lr = info.getReason();
                    ud = info.getUserData();
                }
                if (ll != null && (lr == null || lr == r )) {
                    ll.eventOccured(type, r, target, oldval,
                            newval, ud);
                }
            }
        }

    }

    /**
     * quick check to determine if the broadcaster
     * has any listeners of a specific type
     * @param type type of event to look at
     * @returns true if the broadcaster has listeners of that
     *          type
     */
    public boolean hasListeners(EventType type) {
        return busy[type.getEvent()];
    }

    /**
     * quick check to determine if the broadcaster
     * has any listeners of a any type
     * @returns true if the broadcaster has any listeners
     */
    public synchronized boolean hasListeners() {
        return cnt > 0;
    }



    /**
     * class maintaining event listener information
     */
    class ListenerInfo {
        EventListener l;
        EventType type;
        Object userData;
        Reason reason;

        public ListenerInfo(EventListener l, EventType t, Reason r, Object ud) 
        {
            this.l = l;
            this.type = t;
            this.userData =ud;
            this.reason = r;
        }

        synchronized Reason getReason() {
            return reason;
        }
        synchronized Object getUserData() {
            return userData;
        }
        synchronized EventType getType() {
            return type;
        }
        synchronized EventListener getListener() {
            return l;
        }
        synchronized void clear() {
            l = null;
            userData = null;
            reason = null;
            type = null;
        }
    }

}


