/*
 * @(#)OnMessageRunner.java	1.13 06/02/09
 *
 * Copyright 2005 Sun Microsystems, Inc. All Rights Reserved.
 * SUN PROPRIETARY/CONFIDENTIAL.
 * Use is subject to license terms.
 *
 */

package com.sun.messaging.jms.ra;

import java.util.logging.Logger;

import javax.jms.*;

import javax.resource.spi.UnavailableException;
import javax.resource.spi.work.*;
import javax.resource.spi.endpoint.*;

import javax.transaction.xa.XAResource;

import java.lang.reflect.Method;

/**
 * Runs a single message through an application server endpoint
 * 
 * @author George Tharakan
 */

public class OnMessageRunner implements Work {

    /** the id of this OnMessageRunner */
    protected int omrId;

    /** the message being processed */
    private Message message;

    /** the thread processing the message */
    private Thread omrThread;

    /** The OnMessageRunnerPool this belongs to */
    private OnMessageRunnerPool omrPool;

    com.sun.messaging.jmq.jmsclient.MessageImpl mqmsg;
    com.sun.messaging.jmq.jmsclient.SessionImpl mqsess;

    /** The MessageEndpoint for this MessageListener instance */
    private MessageEndpoint msgEndpoint = null;

    /** The MessageEndpointFactory for this onMessageRunner instance */
    private MessageEndpointFactory epFactory;

    /** The EndpointConsumer for this onMessageRunner instance */
    private EndpointConsumer epConsumer;

    /** The ActivationSpec for this MessageListener instance */
    private com.sun.messaging.jms.ra.ActivationSpec spec = null;

    /** The XAResource for this MessageListener instance */
    private com.sun.messaging.jmq.jmsclient.XAResourceForRA xar = null;

    /** The onMessage Method that this MessageListener instance will call */
    private Method onMessage = null;

    /** Whether delivery is transacted or not */
    private boolean transactedDelivery = false;

    /** Whether the endpoint acquired is valid or not */
    private boolean endpointValid = false;

    /* Loggers */
    private static transient final String _className = "com.sun.messaging.jms.ra.OnMessageRunner";
    protected static transient Logger _loggerIM;
    protected static transient final String _lgrNameInboundMessage = "javax.resourceadapter.mqjmsra.inbound.message";
    protected static transient final String _lgrMIDPrefix = "MQJMSRA_MR";
    protected static transient final String _lgrMID_EET = _lgrMIDPrefix + "1001: ";
    protected static transient final String _lgrMID_INF = _lgrMIDPrefix + "1101: ";
    protected static transient final String _lgrMID_WRN = _lgrMIDPrefix + "2001: ";
    protected static transient final String _lgrMID_ERR = _lgrMIDPrefix + "3001: ";
    protected static transient final String _lgrMID_EXC = _lgrMIDPrefix + "4001: ";
 
    static {
        _loggerIM = Logger.getLogger(_lgrNameInboundMessage);
    }
 
    /** Constructs an OnMessageRunner */
    public OnMessageRunner(int omrId, OnMessageRunnerPool omrPool,
            MessageEndpointFactory epFactory,
            EndpointConsumer epConsumer,
            ActivationSpec spec)
    {
        Object params[] = new Object[4];
        params[0] = omrPool;
        params[1] = epFactory;
        params[2] = epConsumer;
        params[3] = spec;

        _loggerIM.entering(_className, "constructor()", params);

        this.omrId = omrId;
        this.omrPool = omrPool;
        this.epFactory = epFactory;
        this.epConsumer = epConsumer;
        this.spec = spec;

        while (xar == null) {
        try {
            xar = new com.sun.messaging.jmq.jmsclient.XAResourceForRA(this, epConsumer.ra.xac, epConsumer.xac);
            break;
        } catch (JMSException jmse) {
            System.err.println("MQRA:OMR:Exception on XAResource creation-"+jmse.getMessage());
            try {
                Thread.sleep(2000);
            } catch (Exception e) {
            }
            //System.err.println("MQRA:OMR:Running as non-transacted");
        }
        }
        ////Acquire endpoint from epFactory here
        onMessage = epConsumer.ra._getOnMessageMethod();

        try {
            transactedDelivery = epFactory.isDeliveryTransacted(onMessage);
        } catch (java.lang.NoSuchMethodException nsme) {
            //assume non-transactedDelivery
        }
        for (int i = 1; i < 6; i++ ) {
            try {
                //If it's not deactivated
                if (epConsumer.deactivated != true) {
                    msgEndpoint = epFactory.createEndpoint(xar);
                    break;
                }
            } catch (UnavailableException ue) {
                try {
                    System.err.println("MQRA:OMR:createEndpoint-UnavailableException:Sleeping for:"+i*200);
                    Thread.sleep(i * 200);
                } catch (InterruptedException ie) {
                }
            }    
        }
        if (msgEndpoint == null) {
            //Could not acquire - shut down delivery of messages in this session
            System.err.println("MQRA:OMR:Endpoint Unavailable:Marking OMR as invalid-omrId="+omrId+" for:"+spec.toString());
            //_logger.log(Level.SEVERE, "MQRA:ML:Endpoint Unavailable:Shutting down delivery for "+spec.toString());
            //mqmsg._getSession().closeFromRA();
            //endpoint should be shutdown normally by AS via RA.endpointDeactivation()
            return;
        } else {
            endpointValid = true;
        }
    }

    public void release() {
        //System.out.println("MQRA:OMR:release():tName="+Thread.currentThread().getName());
    }

    public void releaseEndpoint() {
        //System.out.println("MQRA:OMR:releaseEndpoint():omrId="+omrId+" tName="+Thread.currentThread().getName());
        if (msgEndpoint!= null) {
            msgEndpoint.release();
        }
    }

    public void invalidate() {
        //System.out.println("MQRA:OMR:invalidateEndpoint():omrId="+omrId+" tName="+Thread.currentThread().getName());
        endpointValid = false;
        //if (msgEndpoint!= null) {
            //msgEndpoint.release();
        //}
    }


    public void run() {
        synchronized (xar) {
        boolean sendUndeliverableMsgsToDMQ = spec.getSendUndeliverableMsgsToDMQ();
        if (!endpointValid) {
            System.err.println("MQRA:OMR:run:msgEP invalid-Ack Undeliverable & remove OMR fr pool-omrId="+omrId+":msg="+mqmsg.toString());
            omrPool.removeOnMessageRunner(this);
            try {
                mqsess.acknowledgeUndeliverableFromRAEndpoint(mqmsg, xar, sendUndeliverableMsgsToDMQ);
            } catch (JMSException jmse) {
                System.err.println("MQRA:OMR:run:msgEP invalid:JMSException on Ack Undeliverable-"+jmse.getMessage());
                jmse.printStackTrace();
            }
            return;
        }
        //System.err.println("MQRA:OMR:run:omrId="+omrId+" in thread name="+Thread.currentThread().getName()+" transacted="+transactedDelivery);

        ClassLoader cl = spec.getContextClassLoader();
        int exRedeliveryAttempts = spec.getEndpointExceptionRedeliveryAttempts();
        int exRedeliveryInterval = spec.getEndpointExceptionRedeliveryInterval();
        //Deliver message to msgEndpoint
        boolean redeliver = true;
        while (redeliver == true) {
            try {
                if (transactedDelivery) {
                    //System.err.println("MQRA:OMR:run:beforeDelivery()");
                    msgEndpoint.beforeDelivery(onMessage);
                }
                try {
                    //System.out.println("MQRA:OMR:run:Delivering to onMessage()");
                    if (cl != null) {
                        //System.out.println("MQRA:OMR:run:Setting ContextClassLoader:"+cl.toString());
                        try {
                            Thread.currentThread().setContextClassLoader(cl);
                        } catch (Exception sccle) {
                            System.err.println("MQRA:ML:run:Exception setting ContextClassLoader:"+sccle.getMessage());
                        }
                    }
                    //System.err.println("MQRA:OMR:run:Deliver Msg:JMSRedeliver="+message.getJMSRedelivered()+" Msg="+message.toString());
                    ((javax.jms.MessageListener)msgEndpoint).onMessage(message);
                    redeliver = false;
                    //System.err.println("MQRA:OMR:run:Delivered successfully-Msg="+message.toString());
                    try {
                        mqsess.acknowledgeFromRAEndpoint(mqmsg, xar);
                        //System.err.println("MQRA:OMR:run:Acknowledged successfully");
                        //System.err.println("MQRA:OMR:run:omrId="+omrId+" msg acknowledged-msg="+mqmsg.toString());
                        xar.setRollback(false, null);
                    } catch (JMSException jmse) {
                        System.err.println("MQRA:OMR:run:JMSException on message acknowledgement:Rolling back if in txn");
                        xar.setRollback(true, jmse);
                        jmse.printStackTrace();
                    }
                } catch (Exception rte) {
                    System.err.println("MQRA:OMR:run:Caught Exception from onMessage():Redelivering:"+rte.getMessage());
                    try {
                        message.setJMSRedelivered(true);
                    } catch (Exception jmsesr) {
                        System.err.println("MQRA:OMR:run:Exception on setJMSRedelivered():"+jmsesr.getMessage());
                        jmsesr.printStackTrace();
                    }
                    if (exRedeliveryAttempts > 0) {
                        try {
                            //System.out.println("MQRA:OMR:run:RedeliverInterval-start");
                            Thread.sleep(exRedeliveryInterval);
                            //System.out.println("MQRA:OMR:run:RedeliverInterval-stop");
                        } catch (InterruptedException ie) {
                            //System.out.println("MQRA:OMR:run:RedeliverInterval-interrupted");
                        }
                        exRedeliveryAttempts -= 1;
                    } else {
                        System.err.println("MQRA:OMR:run:Exhausted redeliveryAttempts-msg="+message.toString());
                        System.err.println("MQRA:OMR:run:Exhausted redeliveryAttempts-spec="+spec.toString());
                        //if (false) { //ackSendToDMQ not tested  - disabling for alpha
                        if (sendUndeliverableMsgsToDMQ) {
                            System.err.println("MQRA:OMR:run:Message returned & marked for routing to the DMQ");
                        } else {
                            System.err.println("MQRA:OMR:run:Message returned & marked for redelivery by the broker");
                        }
                        try {
                            mqsess.acknowledgeUndeliverableFromRAEndpoint(mqmsg, xar, sendUndeliverableMsgsToDMQ);
                            System.err.println("MQRA:OMR:run:omrId="+omrId+":Acked Undeliverable-Msg="+mqmsg.toString());
                        } catch (JMSException jmse) {
                            System.err.println("MQRA:OMR:run:JMSException on Acked Undeliverable-"+jmse.getMessage());
                            jmse.printStackTrace();
                        }
                        //}
                        redeliver = false;
                        //XXX:XXX:need to finalize behavior here
                        //concurrent behavior cannot close the session - must shunt the message and continue
                        ////mqmsg._getSession().closeFromRA();
                    }
                }
                if (transactedDelivery) {
                    //System.err.println("MQRA:OMR:run:afterDelivery()");
                    msgEndpoint.afterDelivery();
                }
            } catch (Throwable t) {
                System.err.println("MQRA:OMR:run:onMessage caught Throwable-before/on/afterDelivery:Class="
                        +t.getClass().getName()
                        +"Msg="+t.getMessage());
                redeliver = false;
                t.printStackTrace();
                //////mqmsg._getSession().closeFromRA();
            }
        }
        message = null;
        //System.out.println("MQRA:OMR:run:putting back omrId="+omrId);
        omrPool.putOnMessageRunner(this);
        }
    }

    public boolean equals(Object other) {
        if (other instanceof OnMessageRunner) {
            if (omrId == ((OnMessageRunner)other).getId()) {
                return true;
            } else {
                return false;
            }
        } else {
            return false;
        }
    }

    public int hashCode() {
        return omrId;
    }

    public void onMessage(Message msg) {

        //System.err.println("MQRA:OMR:onMessage()");
        message = msg;
        mqmsg = (com.sun.messaging.jmq.jmsclient.MessageImpl)message;
        mqsess = (com.sun.messaging.jmq.jmsclient.SessionImpl)epConsumer.getXASession();
        mqmsg._setConsumerInRA();

        //System.out.println("MQRA:OMR:onMessage():starting Work on omrId="+omrId);
        try {
            epConsumer.ra.workMgr.startWork(this);
        } catch (WorkException we) {
            System.err.println("MQRA:OMR:onMessage:WorkException-"+we.getMessage()+" on omrId="+omrId);
            we.printStackTrace();
        }
////////Replaced with Work above
        //omrThread = new Thread(this);
        //omrThread.setName("imqOnMessageRunner-Id#"+omrId);
        //
        //omrThread.start();
////////

    }

    public int getId() {
        return omrId;
    }

    public boolean isValid() {
        return endpointValid;
    }

}
