/*
 * @(#)OnMessageRunnerPool.java	1.6 06/02/09
 *
 * Copyright 2005 Sun Microsystems, Inc. All Rights Reserved.
 * SUN PROPRIETARY/CONFIDENTIAL.
 * Use is subject to license terms.
 *
 */

package com.sun.messaging.jms.ra;


import javax.jms.*;

import java.util.Vector;
import java.util.ArrayList;
import java.util.logging.Logger;

import javax.resource.spi.endpoint.MessageEndpointFactory;

/**
 * Holds a pool of OnMessageRunner objects.
 * 
 * @author George Tharakan
 */

public class OnMessageRunnerPool {

    /** The maximum number of OnMessageRunner objects allocated */
    private int max;

    /** The minimum number of OnMessageRunner objects allocated */
    private int min;
    
    /** The number of free OnMessageRunner objects */
    private int freeCount;

    /** The slack - number of OnMessageRunner objects that can and have not been created */
    private int slackCount;

    /** The MessageEndpointFactory for this onMessageRunnerPool */
    private MessageEndpointFactory epFactory;

    /** The EndpointConsumer for this onMessageRunnerPool */
    private EndpointConsumer epConsumer;

    /** The ActivationSpec for this MessageListener instance */
    private ActivationSpec spec = null;

    private ArrayList onMessageRunners;

    /** The list of available OnMessageRunner objects */
    private Vector available;

    /* Loggers */
    private static transient final String _className = "com.sun.messaging.jms.ra.OnMessageRunnerPool";
    protected static transient Logger _loggerIM;
    protected static transient final String _lgrNameInboundMessage = "javax.resourceadapter.mqjmsra.inbound.message";
    protected static transient final String _lgrMIDPrefix = "MQJMSRA_RP";
    protected static transient final String _lgrMID_EET = _lgrMIDPrefix + "1001: ";
    protected static transient final String _lgrMID_INF = _lgrMIDPrefix + "1101: ";
    protected static transient final String _lgrMID_WRN = _lgrMIDPrefix + "2001: ";
    protected static transient final String _lgrMID_ERR = _lgrMIDPrefix + "3001: ";
    protected static transient final String _lgrMID_EXC = _lgrMIDPrefix + "4001: ";

    static {
        _loggerIM = Logger.getLogger(_lgrNameInboundMessage);
    }


    /** Constructs an OnMessagePoolRunner */
    public OnMessageRunnerPool(MessageEndpointFactory epFactory,
                EndpointConsumer epConsumer,
                ActivationSpec spec)
    {
        Object params[] = new Object[3];
        params[0] = epFactory;
        params[1] = epConsumer;
        params[2] = spec;

        _loggerIM.entering(_className, "constructor()", params);

        int minimum, maximum;

        this.epFactory = epFactory;
        this.epConsumer = epConsumer;
        this.spec = spec;

        minimum = spec.getEndpointPoolSteadySize();
        maximum = spec.getEndpointPoolMaxSize();

        if (maximum < 1) { this.max = 10; }
        else { this.max = maximum; }
        if (minimum < 1 || minimum > max) { this.min = 1; }
        else { this.min = minimum; }

        available = new Vector(min);

        /* list of all onMessageRunner objects created */
        onMessageRunners = new ArrayList(min);

        OnMessageRunner omr;

        for (int i=0; i<min; i++) {
            omr = new OnMessageRunner(i, this, epFactory, epConsumer, spec);
            onMessageRunners.add(omr);
            available.addElement(omr);
        }
        freeCount = min;
        slackCount = max - min;
    }


    /**
     * Get an OnMessageRunner from the pool
     *
     *
     */
    public synchronized OnMessageRunner
    getOnMessageRunner() throws JMSException {

        OnMessageRunner omr;

        //System.out.println("MQRA:OMRP:getOMR()");
        if (available.size() == 0) {
            //System.out.println("MQRA:OMRP:getOMR:size=0");
            if (slackCount > 0) {
                //System.out.println("MQRA:OMRP:getOMR:adding from slack");
                omr = new OnMessageRunner(onMessageRunners.size(), this, epFactory, epConsumer, spec);
                onMessageRunners.add(omr);
                slackCount--;
                //System.out.println("MQRA:OMRP:getOMR:slack-Getting omr Id="+omr.omrId);
                return omr;
            }
        }
            
        while (available.size() == 0) {
            try {
                //System.out.println("MQRA:OMRP:getOMR:Waiting...");
                wait();
            } catch (InterruptedException ie) {
                //System.out.println("MQRA:OMRP:getOMR:Interrupted while waiting...throwing exception-"+ie.getMessage());
                JMSException jmse = new com.sun.messaging.jms.JMSException(
                    "MQRA:OMRP:Unable to get OMR from pool:"+ie.getMessage());
                jmse.setLinkedException(ie);
                throw jmse;
            }
        }
        omr = (OnMessageRunner)available.elementAt(available.size()-1);
        //System.out.println("MQRA:OMRP:getOMR:Getting omr at index="+(available.size()-1)+" with omrId="+omr.omrId);
        available.removeElementAt(available.size()-1);
        freeCount--;
        return omr;
    }


    /**
     * Put an OnMessageRunner back to the pool
     *
     */
    public synchronized void
    putOnMessageRunner(OnMessageRunner omr) {

        //System.out.println("MQRA:OMRP:Putting back omrId="+omr.omrId+" at index="+available.size());
        available.addElement(omr);

        //XXX:reduction logic - here

        freeCount++;
        notifyAll();
    }

    public synchronized void
    removeOnMessageRunner(OnMessageRunner omr) {
        int index;

        //System.out.println("MQRA:OMRP:removeOMR:Id="+omr.getId());
        index = onMessageRunners.indexOf(omr);
        if (index != -1) {
            //System.out.println("MQRA:OMRP:removeOMR:Id="+omr.getId()+" at index="+index);
            onMessageRunners.remove(index);
            freeCount++;
            if (slackCount < (max-min)) {
                slackCount++;
            }
        }
    }

    public synchronized void
    waitForAllOnMessageRunners() throws JMSException {
        //System.out.println("MQRA:OMRP:wfaOMRs():allocated="+onMessageRunners.size()+" available="+available.size());
        while (available.size() < onMessageRunners.size()) {
            try {
                //System.out.println("(MQRA:OMRP:wfaOMRs:Waiting...");
                wait();
            } catch (InterruptedException ie) {
                //System.out.println("MQRA:OMRP:wfaOMRs:Interrupted while waiting...throwing exception-"+ie.getMessage());
                JMSException jmse = new com.sun.messaging.jms.JMSException(
                    "MQRA:OMRP:Didnot finish waiting for OMRs to return:"+ie.getMessage());
                jmse.setLinkedException(ie);
                throw jmse;
            }
        }
        //System.out.println("MQRA:OMRP:wfaOMRs:DONE:allocated="+onMessageRunners.size()+" available="+available.size());
    }

    public synchronized void
    releaseOnMessageRunners() {
        //System.out.println("MQRA:OMRP:releaseOMRs()");
        for (int i= 0; i<onMessageRunners.size(); i++) {
            ((OnMessageRunner)onMessageRunners.get(i)).releaseEndpoint();
        }
        onMessageRunners.clear();
        available.removeAllElements();
        freeCount = 0;
        slackCount = max;
        //System.out.println("MQRA:OMRP:releaseOMRs-done");
    }

    public synchronized void
    invalidateOnMessageRunners() {
        //System.err.println("MQRA:OMRP:invalidateOMRs()");
        for (int i= 0; i<onMessageRunners.size(); i++) {
            ((OnMessageRunner)onMessageRunners.get(i)).invalidate();
        }
        releaseOnMessageRunners();
        //System.err.println("MQRA:OMRP:invalidateOMRs-done");
    }
}

