/*
 * @(#)ResourceAdapter.java	1.42 06/02/09
 *
 * Copyright 2005 Sun Microsystems, Inc. All Rights Reserved.
 * SUN PROPRIETARY/CONFIDENTIAL.
 * Use is subject to license terms.
 *
 */

package com.sun.messaging.jms.ra;

import javax.resource.*;
import javax.resource.spi.*;
import javax.resource.spi.work.*;
import javax.resource.spi.endpoint.*;

import javax.jms.Connection;
import javax.jms.JMSException;
import javax.jms.JMSSecurityException;
import javax.jms.ExceptionListener;
import javax.transaction.xa.XAResource;

import java.util.Map;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Collection;
import java.util.StringTokenizer;
import java.lang.reflect.Method;

import java.net.InetAddress;
import java.net.UnknownHostException;

import java.io.File;
import java.io.IOException;

import java.util.Vector;
import java.util.Properties;
import java.util.logging.Logger;
import java.util.logging.Level;

import java.security.AccessController;
import java.security.PrivilegedAction;

import com.sun.messaging.jmq.Version;
import com.sun.messaging.ConnectionConfiguration;
import com.sun.messaging.AdminConnectionConfiguration;

/**
 * Sun Java System Message Queue Java EE Resource Adapter for JMS.
 * 
 * @author George Tharakan
 */

public class ResourceAdapter
implements javax.resource.spi.ResourceAdapter,
           javax.jms.ExceptionListener,
           java.io.Serializable,
           com.sun.messaging.jms.notification.EventListener
{
    /* Unique ID acquired from the broker that this RA connects to */
    // This ID is used to identify the 'Namespace' that this RA is generating connections from.
    // The MQ Broker will enforce JMS Semantics of ClientID within 'Namespace'
    //     but will allow sharing of the same ClientID in other 'Namespaces'.
    private String raUID = null;

    /* passed by the Java EE server; used by RA to acquire WorkManager etc. */
    private transient BootstrapContext b_context = null;

    /* WorkManager used to execute message delivery. */
    protected transient WorkManager workMgr = null;

    /* Indicates that RA was successfully started, else ep Activation etc. should fail. */
    private transient boolean started;
    private transient boolean stopping;

    /* hashmaps */
    private transient HashMap epFactories = null;
    private transient HashMap epConsumers = null;
    private transient HashMap epFactoryToConsumer = null;

    /* IDs */
    private transient int _factoryID = 0;
    private transient int _consumerID = 0;

    /* the AdminConnectionFactory for this RA */
    private transient com.sun.messaging.AdminConnectionFactory acf = null;

    /* the ConnectionFactory for this RA */
    private transient com.sun.messaging.XAConnectionFactory xacf = null;

    /* the Connection held by this RA */
    protected transient com.sun.messaging.jmq.jmsclient.XAConnectionImpl xac = null;

    /* the message listener method */
    private transient Method onMessage = null;

    /* flag to control logging during long reconnect attempts */
    private transient boolean logRCFailures = true;

    /* Globally started at least once data */
    private static transient boolean _startedAtLeastOnce;
    private static transient int _rmiRegistryPort;

    /* Loggers */
    private static transient final String _className = "com.sun.messaging.jms.ra.ResourceAdapter";
    protected static transient Logger _loggerB;
    protected static transient Logger _loggerL;
    protected static transient Logger _loggerIM;
    protected static transient final String _lgrNameBase = "javax.resourceadapter.mqjmsra";
    protected static transient final String _lgrNameLifecycle = "javax.resourceadapter.mqjmsra.lifecycle";
    protected static transient final String _lgrNameInboundMessage = "javax.resourceadapter.mqjmsra.inbound.message";
    protected static transient final String _lgrMIDPrefix = "MQJMSRA_RA";
    protected static transient final String _lgrMID_EET = _lgrMIDPrefix + "1001: ";
    protected static transient final String _lgrMID_INF = _lgrMIDPrefix + "1101: ";
    protected static transient final String _lgrMID_WRN = _lgrMIDPrefix + "2001: ";
    protected static transient final String _lgrMID_ERR = _lgrMIDPrefix + "3001: ";
    protected static transient final String _lgrMID_EXC = _lgrMIDPrefix + "4001: ";

    /* Broker Lifecycle */
    private transient EmbeddedBrokerRunner ebr = null;
    private transient LocalBrokerRunner lbr = null;

    /* Non-Managed Connection Manager */
    protected static transient com.sun.messaging.jms.ra.ConnectionManager _cm;

    /* Version Info */
    protected static transient Version _version;

    // Configurable attributes of the Sun ONE MQ RA //
    /** The ConnectionURL for the MQ RA */
    private String connectionURL = "";

    /** The UserName to be used for the MQ RA Connection */
    private String userName = "guest";

    /** The Password to be used for the MQ RA Connection */
    private String password = "guest";

    /* RA Failover controls */
    private boolean reconnectEnabled = false;
    private int reconnectInterval = 5000;
    private int reconnectAttempts = 6;
    private String addressListBehavior = "PRIORITY";
    private int addressListIterations = 1;
    private int maxLoopDelay = 120000;

    /* Indicate whether the ra is being used in the application client container */
    private boolean inAppClientContainer = false;

    /* Indicate whether the ra is being used in a clustered container */
    private boolean inClusteredContainer = false;

    public static final String BROKER_TYPE_REMOTE   = "REMOTE";
    public static final String BROKER_TYPE_LOCAL    = "LOCAL";
    public static final String BROKER_TYPE_EMBEDDED = "EMBEDDED";

    /* Indicate type for broker lifecycle control */
    private String brokerType = BROKER_TYPE_REMOTE;

    /* Indicate the instance name of the broker when its lifecycle is controlled by the RA */
    private String brokerInstanceName = "imqbroker";    // Default is 'imqbroker'

    /* Indicate specific address that broker should bind to when its lifecycle is controlled by the RA */
    private String brokerBindAddress = null;            // Default is 'all addresses'

    /* Indicate main port for broker when its lifecycle is controlled by the RA */
    private int brokerPort = 7676;

    /* Indicate Home Directory for broker when its lifecycle is controlled by the RA */
    private String brokerHomeDir = null;

    /* Indicate Var Directory for broker when its lifecycle is controlled by the RA */
    private String brokerVarDir = null;

    /* Indicate Lib Directory for broker when its lifecycle is controlled by the RA */
    private String brokerLibDir = null;

    /* Indicate Java Directory for broker when its lifecycle is controlled by the RA */
    private String brokerJavaDir = null;

    /* Additional cmdline args for broker when its lifecycle is controlled by the RA */
    private String brokerArgs = null;

    /* The masterBroker for broker when its lifecycle is controlled by the RA */
    private String masterBroker = null;

    /* Indicate the start timeout in milliseconds for broker when its lifecycle is controlled by the RA */
    private int brokerStartTimeout = 10000;             // Default is 10 seconds

    /** The admin userName to be used for JMX connections */
    private String adminUsername = "admin";

    /** The admin password to be used for JMX connections */
    private String adminPassword = "admin";

    /** The admin password file to be used when starting the broker w/ admin user/password checking */
    private String adminPassFile = null;

    /* Indicate whether the JNDI form of the JMXServiceURL is to be used in the broker when its lifecycle is controlled by the RA */
    private boolean useJNDIRmiServiceURL = true;

    /* Indicate whether the broker should start its own RMI Registry its lifecycle is controlled by the RA */
    private boolean startRmiRegistry = false;

    /* Indicate the port for the Rmi Registry in the broker when its lifecycle is controlled by the RA */
    private int rmiRegistryPort = 1099;

    /* Indicate whether SSL must be used for the JMXConnector in the broker when its lifecycle is controlled by the RA */
    private boolean useSSLJMXConnector = true;

    /* Indicate whether the broker must enable High Availability functionality */
    private boolean brokerEnableHA = false;

    /* The ClusterId identifies the cluster of MQ brokers that the broker belongs to when its lifecycle is controlled by the RA
    ** All brokers of an AS server cluster should belong to the same broker clusterId
    */
    private String clusterId = null;

    /* The BrokerId identifies the broker in an HA store when its lifecycle is controlled by the RA
    ** Every broker registers in the HA store with a different brokerId
    */
    private String brokerId = null;

    public static final String DB_TYPE_DERBY      = "derby";
    public static final String DB_TYPE_HADB       = "hadb";
    public static final String DB_TYPE_ORACLE     = "oracle";
    public static final String DB_TYPE_POINTBASE  = "pointbase";
    public static final String DB_TYPE_CLOUDSCAPE = "cloudscape";

    /* Indicate database type for broker lifecycle control */
    private String dbType = null;

    /* The properties object that holds database type-specific config properties for the broker when its lifecycle is controlled by the RA */
    private Properties dbProps = null;

    /* The properties object that holds DataSource-specific properties for the broker when its lifecycle is controlled by the RA */
    private Properties dsProps = null;

    /*
    ** The Group Name assigned to this Resource Adapter.
    ** Supports deployment scenarios where multiple instances of a group
    ** are acting as a single logical unit for JMS semantics.
    ** All ResourceAdapter instances in the group must have the same Group Name.
    */
    private String groupName = null;

    /** The JMXServiceURL String that can be used to acquire JMX connections */
    private transient String jmxServiceURL = null;

    /** The JMXServiceURLList String that can be used to acquire JMX connections to all brokers specified on connectionURL */
    private transient String jmxServiceURLList = null;

    /* Indicate whether the JMXServiceURLList is valid or needs to be re-acquired */
    private boolean isJMXServiceURLListValid = false;

    /** The JMXConnectorEnv HashMap that is used to acquire JMX connections */
    private transient HashMap jmxConnectorEnv = null;

    private transient boolean doInitOnlyOnStart = false;


    //Remove for 4.0
    //Indicate whether High Availability functionality is required on Broker Connections
    //private boolean haRequired = false;


    /* private constants */ //Broker common defs
    private static transient String IMQ_BROKERID = "imq.brokerid";
    private static transient String IMQ_JDBC_VENDOR = "imq.persist.jdbc.dbVendor";
    private static transient String HADB_USER = DB_TYPE_HADB+".user";
    private static transient String HADB_PASSWORD = DB_TYPE_HADB+".password";
    private static transient String HADB_SERVERLIST = DB_TYPE_HADB+".serverList";
    private static transient String IMQ_HADB = "imq.persist.jdbc.hadb";
    private static transient String IMQ_HADB_DSPROP = IMQ_HADB+".property";
    private static transient String IMQ_HADB_USER = IMQ_HADB+".user";
    private static transient String IMQ_HADB_PASSWORD = IMQ_HADB+".password";

    private static transient String IMQ_HADB_DSPROP_SERVERLIST = IMQ_HADB_DSPROP+".serverList";


    static {
        _startedAtLeastOnce = false;
        _version = new Version();
        _loggerB = Logger.getLogger(_lgrNameBase);
        _loggerL = Logger.getLogger(_lgrNameLifecycle);
        _loggerIM = Logger.getLogger(_lgrNameInboundMessage);
        _cm = new com.sun.messaging.jms.ra.ConnectionManager();
    }

    /* Must declare public empty constructor */
    public ResourceAdapter() {
        _loggerL.entering(_className, "constructor()");
        dbProps = new Properties();
        dsProps = new Properties();
        started = false;
    }

    //ResourceAdapter interface methods //
    //

    /** Starts this instance of the Resource Adapter.
     *  Called by the Application Server to start this instance of the Resource Adapter.
     *  Note that this does not control the MQ Server lifecycle
     * {@inheritDoc}
     */
    public synchronized void
    start(BootstrapContext ctx) 
    throws ResourceAdapterInternalException
    {
        _loggerL.entering(_className, "start()", ctx);
        stopping = false;
        if (started) {
            _loggerL.warning(_lgrMID_WRN+"start:Previously started:Ignoring");
        } else {
            if (doInitOnlyOnStart == true) {
                if (BROKER_TYPE_REMOTE.equals(brokerType)) {
                    _loggerL.warning(_lgrMID_WRN+"start:Invalid to perform doInitOnlyOnStart for REMOTE brokerType:Ignoring & Resetting doInitOnlyOnStart");
                    doInitOnlyOnStart = false;
                    return;
                }
                if (_startedAtLeastOnce) {
                    _loggerL.warning(_lgrMID_WRN+"start:Invalid to perform doInitOnlyOnStart if RA has been started previously and stopped:Ignoring & Resetting doInitOnlyOnStart");
                    doInitOnlyOnStart = false;
                    return;
                }
                try {
                    ebr = new EmbeddedBrokerRunner(brokerInstanceName, brokerBindAddress, brokerPort,
                                    brokerHomeDir, brokerLibDir, brokerVarDir, brokerJavaDir, brokerArgs,
                                    useJNDIRmiServiceURL, _rmiRegistryPort, startRmiRegistry, useSSLJMXConnector,
                                    brokerStartTimeout, adminUsername, adminPassword, adminPassFile,
                                    _getEffectiveBrokerProps());
                    ebr.init();
                    doInitOnlyOnStart = false;
                    if (BROKER_TYPE_LOCAL.equals(brokerType)) {
                        ebr.stop();
                        ebr = null;
                    }
                    return;
                } catch (Exception ebse) {
                    ResourceAdapterInternalException raie = new ResourceAdapterInternalException(_lgrMID_EXC+
                                "start:Aborting:Exception performing doInitOnlyOnStart on broker="+ebse.getMessage());
                    raie.initCause(ebse);
                    _loggerL.severe(raie.getMessage());
                    ebse.printStackTrace();
                    _loggerL.throwing(_className, "start()doInitOnlyOnStart", raie);
                    doInitOnlyOnStart = false;
                    throw raie;
                }
            }
            if (BROKER_TYPE_EMBEDDED.equals(brokerType)) {
                _loggerL.info(_lgrMID_INF+"SJSMQ JMS Resource Adapter starting...");
            } else {
                _loggerL.info(_lgrMID_INF+"SJSMQ JMS Resource Adapter starting...\n" +_version.getBanner(false, Version.MINI_COPYRIGHT));
            }

            this.b_context = ctx;

            if (b_context != null) {
                workMgr = b_context.getWorkManager();
            }

            //Try to set inAppClientContainer correctly
            _adjustInAppClientContainer();

            this.xacf = new com.sun.messaging.XAConnectionFactory();
            this.acf = new com.sun.messaging.AdminConnectionFactory();
            if (!inAppClientContainer) {
                AccessController.doPrivileged(
                    new PrivilegedAction()
                    {
                        public Object run() {
                            System.setProperty("imq.DaemonThreads", "true");
                            return null;
                        }
                    }
                );
            }

            if (BROKER_TYPE_LOCAL.equals(brokerType)) {
                try {
                    lbr = new LocalBrokerRunner(brokerInstanceName, brokerBindAddress, brokerPort,
                                    brokerHomeDir, brokerLibDir, brokerVarDir, brokerJavaDir, brokerArgs,
                                    useJNDIRmiServiceURL, rmiRegistryPort, startRmiRegistry, useSSLJMXConnector,
                                    brokerStartTimeout, adminUsername, adminPassword, adminPassFile,
                                    _getEffectiveBrokerProps());
                    lbr.start();
                } catch (Exception lbse) {
                    ResourceAdapterInternalException raie = new ResourceAdapterInternalException(_lgrMID_EXC+
                            "start:Aborting:Exception starting LOCAL broker="+lbse.getMessage());
                    raie.initCause(lbse);
                    _loggerL.severe(raie.getMessage());
                    lbse.printStackTrace();
                    _loggerL.throwing(_className, "start()", raie);
                    throw raie;
                }
            } else {
                if (BROKER_TYPE_EMBEDDED.equals(brokerType)) {
                    try {
                        if (!_startedAtLeastOnce) {
                            _rmiRegistryPort = rmiRegistryPort;
                        }
                        if (ebr == null) {
                            ebr = new EmbeddedBrokerRunner(brokerInstanceName, brokerBindAddress, brokerPort,
                                            brokerHomeDir, brokerLibDir, brokerVarDir, brokerJavaDir, brokerArgs,
                                            useJNDIRmiServiceURL, _rmiRegistryPort, startRmiRegistry, useSSLJMXConnector,
                                            brokerStartTimeout, adminUsername, adminPassword, adminPassFile,
                                            _getEffectiveBrokerProps());
                        }
                        ebr.start();
                        _startedAtLeastOnce = true;
                    } catch (Exception ebse) {
                        ResourceAdapterInternalException raie = new ResourceAdapterInternalException(_lgrMID_EXC+
                                "start:Aborting:Exception starting EMBEDDED broker="+ebse.getMessage());
                        raie.initCause(ebse);
                        _loggerL.severe(raie.getMessage());
                        ebse.printStackTrace();
                        _loggerL.throwing(_className, "start()", raie);
                        throw raie;
                    }
                }
            }
            try {
                //Configure RA ConnectionFactory
                configureFactory();
                //Create Connection
                xac = (com.sun.messaging.jmq.jmsclient.XAConnectionImpl)xacf.createXAConnection();
                _loggerL.fine(_lgrMID_INF+"start:cID="+xac._getConnectionID());
                //Init the connection; set ExceptionListener etc.
                init();
            } catch (JMSException jmse) {
                ResourceAdapterInternalException raie = new ResourceAdapterInternalException(_lgrMID_EXC+
                                "start:Aborting:JMSException on createConnection="+jmse.getMessage());
                raie.initCause(jmse);
                _loggerL.severe(raie.getMessage());
                jmse.printStackTrace();
                _loggerL.throwing(_className, "start()", raie);
                throw raie;
            }

            /* init hash maps */
            epFactories = new HashMap(10);
            epConsumers = new HashMap(10);
            epFactoryToConsumer = new HashMap(10);

            _setOnMessageMethod();
            started = true;
            _loggerL.info(_lgrMID_INF+toString());
            _loggerL.info(_lgrMID_INF+"start:SJSMQ JMSRA Connection Factory Config="+xacf.getCurrentConfiguration());
            _loggerL.info(_lgrMID_INF+"SJSMQ JMSRA Started");
        }
        _loggerL.exiting(_className, "start()");
    }
    
    /** Stops this instance of the Resource Adapater.
     *  Called by the Application Server to stop this instance of the Resource Adapter.
     *  Note that this does not control the MQ server lifecycle
     * {@inheritDoc}
     */
    public synchronized void
    stop()
    {
        _loggerL.entering(_className, "stop()");
        stopping = true;
        if (!started) {
            _loggerL.warning(_lgrMID_WRN+"stop:Previously stopped:Ignoring");
            //Thread.dumpStack();
        } else {
            _loggerL.info(_lgrMID_INF+"SJSMQ JMSRA stopping...");
            //remove all epConsumers
            removeAllConsumers();

            if (xac != null) {

                //close the RA connection.
                try{
                    if (_loggerL.isLoggable(Level.FINER)) {
                        _loggerL.finer(_lgrMID_INF+"stop:close:cID="+xac._getConnectionID());
                    }
                    xac.close();
                } catch (JMSException jmse) {
                    _loggerL.logp(Level.WARNING, _className, "stop()", _lgrMID_WRN+"Exception on close:Ignoring:", jmse);
                    //jmse.printStackTrace();
                }
            }
            if (_cm != null) {
                _cm.destroyConnections();
            }
            if (ebr != null) {
                ebr.stop();
                ebr = null;
            }
            if (lbr != null) {
                lbr.stop();
                lbr = null;
            }
            started = false;
        }
        _loggerL.info(_lgrMID_INF+"SJSMQ JMSRA stopped.");
        _loggerL.exiting(_className, "stop()");
    }
    
    /** Activate a message endpoint.
     *  Called by the Application Server when a message-driven bean is deployed.
     *
     *  {@inheritDoc}
     */
    public void
    endpointActivation(MessageEndpointFactory endpointFactory,
                       javax.resource.spi.ActivationSpec spec)
    throws NotSupportedException
    {
        Object params[] = new Object[2];
        params[0] = endpointFactory;
        params[1] = spec;

        _loggerIM.entering(_className, "endpointActivation()", params);
        if (!started) {
            _loggerIM.logp(Level.SEVERE, _className, "endpointActivation()", _lgrMID_EXC+"MQJMSRA not started:Aborting:", params);
            NotSupportedException nse = new NotSupportedException(
                "MQJMSRA-endpointActivation:Error:RA not started:aborting");
            _loggerIM.throwing(_className, "endpointActivation()", nse);
            throw nse;
        }
        EndpointConsumer ec = new EndpointConsumer(this);
                            
        try {
            ec.createMessageConsumer(endpointFactory, spec);
            if (_loggerIM.isLoggable(Level.FINER)) {
                _loggerIM.finer(_lgrMID_INF+"endpointActivation:createMessageConsumer:DONE:"+"fID="+ec.getFactoryID()+" cID="+ec.getConsumerID());
            }
        } catch (Exception ex) {
            _loggerIM.logp(Level.SEVERE, _className, "endpointActivation()", _lgrMID_EXC+":Failed due to:"+ex.getMessage(), params);
            NotSupportedException nse = new NotSupportedException("MQJMSRA-endpointActivation:Exception on createMessageConsumer:");
            nse.initCause(ex);
            _loggerIM.throwing(_className, "endpointActivation()", nse);
            throw nse;
        }
        _loggerIM.exiting(_className, "endpointActivation()");
    }
    
    /** Deactivates a message endpoint
     *  Called by the Application Server when a message-driven bean is undeployed.
     *
     *  {@inheritDoc}
     */
    public void
    endpointDeactivation(MessageEndpointFactory endpointFactory, 
                         javax.resource.spi.ActivationSpec spec)
    {
        Object params[] = new Object[2];
        params[0] = endpointFactory;
        params[1] = spec;

        _loggerIM.entering(_className, "endpointDeactivation()", params);
        if (!started) {
            _loggerIM.logp(Level.SEVERE, _className, "endpointDeactivation()", _lgrMID_EXC+"MQJMSRA not started:Aborting:", params);
        } else {
            int factoryID = matchMessageFactory(endpointFactory);
            if (factoryID != -1) {
                int consumerID = _getConsumerIDbyFactoryID(factoryID);
                EndpointConsumer ec = _getEndpointConsumer(consumerID);
                //System.out.println("MQJMSRA-endpointDeactivation:setting deactivated");
                ec.setDeactivated();

                try{
                    //System.out.println("MQJMSRA-endpointDeactivation:stopping MessageConsumerfID="+factoryID+" cID="+consumerID+" spec=\n"+spec.toString());
                    if (_loggerIM.isLoggable(Level.FINER)) {
                        _loggerIM.finer(_lgrMID_INF+"endpointDeactivation:stopMessageConsumer:fID="+factoryID+" cID="+consumerID);
                    }
                    ec.stopMessageConsumer(spec);
                    //System.out.println("MQJMSRA-endpointDeactivation:stopped MessageConsumer");
                } catch(Exception ex) {
                    //No exception for this method, print
                    //System.err.println("MQJMSRA-endpointDeactivation:Error:stopMessageConsumer exception:ignoring");
                    _loggerIM.logp(Level.WARNING, _className, "endpointDeactivation()", _lgrMID_WRN+"Exception on stopMessageConsumer:Ignoring:", ex);
                    ex.printStackTrace();
                }
                //System.out.println("MQJMSRA-endpointDeactivation:removing from maps-fID="+factoryID);
                removeFromMaps(factoryID);
            } else {
                _loggerIM.log(Level.WARNING, _lgrMID_WRN+"endpointDeactivation:Ignoring:Not found:"+spec.toString());
            }
        }
        _loggerIM.exiting(_className, "endpointDeactivation()");
    }
    
    /** Returns the XAResource array that correspond to the
     *  ActivationSpec instances parameter.
     *  Called by the Application Server when it needs
     *  to determine transaction status for these message endpoints
     *  from the Resource Adapter.
     *
     *  {@inheritDoc}
     */
    public javax.transaction.xa.XAResource[]
    getXAResources(javax.resource.spi.ActivationSpec[] specs)
    throws ResourceException
    {
        _loggerL.entering(_className, "getXAResources()");
        XAResource[] xar = new XAResource[0];
        _loggerL.exiting(_className, "getXAResources()");
        return xar;
    }

    //javax.jms.ExceptionListener interface method
    public void onException(JMSException exception) {
        _loggerL.entering(_className, "onException()", exception);
        _loggerL.warning(_lgrMID_WRN+"onException:Connection Failed:"+exception.getMessage());
        logRCFailures = true;
        int loopDelay = reconnectInterval;
        int loopCount = 0;

        while (started && !stopping) {
            //wait till initial interval expires
            try {
                Thread.sleep(loopDelay);
            } catch (Exception e) {}
            try {
                loopCount += 1;
                _loggerL.warning(_lgrMID_WRN+"onException:Reconnecting...Loop Attempt# "+loopCount+":Delayed "+loopDelay+" milliseconds.");
                xac = (com.sun.messaging.jmq.jmsclient.XAConnectionImpl)xacf.createXAConnection();
                //Init the connection; set ExceptionListener etc.
                init();
                _loggerL.warning(_lgrMID_WRN+"onException:Reconnect successfull on loop# "+loopCount);
                break;
            } catch (JMSException jmse) {
                if (logRCFailures) {
                    _loggerL.severe(_lgrMID_WRN+"onException:Reconnect unsuccessfull on loop# "+loopCount+":"+jmse.getMessage());
                    jmse.printStackTrace();
                } else {
                    _loggerL.severe(_lgrMID_WRN+"onException:Reconnect unsuccessfull on loop# "+loopCount);
                }
                logRCFailures = false;
                if (loopDelay < maxLoopDelay) {
                    loopDelay *= 3;
                    if (loopDelay > maxLoopDelay) loopDelay = maxLoopDelay;
                }
            }
        }
        _loggerL.exiting(_className, "onException()");
    }

    //com.sun.messaging.jms.notification.EventListener interface method
    public void
    onEvent(com.sun.messaging.jms.notification.Event evnt)
    {
        _loggerL.entering(_className, "onEvent()", evnt);
        _loggerL.info(_lgrMID_INF+"onEvent:Connection Event:"+(evnt == null ? "null" : evnt.toString()));
    }


    // ResourceAdapter Javabean configuration methods //
    // These Methods can throw java.lang.RuntimeException or subclasses //

    /** Sets the ConnectionURL for this ResourceAdapter instance
     *   
     *  @param connectionURL The ConnectionURL
     */  
    public synchronized void  
    setConnectionURL(String connectionURL) 
    {
        //XXX: work around this Logger API stripping the String after the 1st 'space'
        String tConnectionURL = connectionURL;
        _loggerL.entering(_className, "setConnectionURL()", tConnectionURL);
        this.connectionURL = connectionURL;
        this.isJMXServiceURLListValid = false;
    }
 
    /** Return the ConnectionURL for this ResourceAdapter instance
     *   
     *  @return The ConnectionURL
     */
    public String
    getConnectionURL()
    {
        _loggerL.entering(_className, "getConnectionURL()", connectionURL);
        if ("".equals(connectionURL)) {
            _loggerL.fine(_lgrMID_INF+"getConnectionURL:returning default of 'localhost' for empty connectionURL");
            return "localhost";
        } else {
            return connectionURL;
        }
    }
 
    /** Sets the UserName for this ResourceAdapter instance
     *   
     *  @param userName The UserName
     */  
    public synchronized void  
    setUserName(String userName) 
    {
        _loggerL.entering(_className, "setUserName()", userName);
        this.userName = userName;
    }
 
    /** Return the UserName for this ResourceAdapter instance
     *   
     *  @return The UserName
     */
    public String
    getUserName()
    {
        _loggerL.entering(_className, "getUserName()", userName);
        return userName;
    }
 
    /** Sets the Password for this ResourceAdapter instance
     *   
     *  @param password The Password
     */  
    public synchronized void  
    setPassword(String password) 
    {
        _loggerL.entering(_className, "setPassword()");
        this.password = password;
    }
 
    /** Return the Password for this ResourceAdapter instance
     *   
     *  @return The Password
     */
    public String
    getPassword()
    {
        _loggerL.entering(_className, "getPassword()");
        return password;
    }

    /** Sets the Reconnect behavior for this Resource Adapter
     *   
     *  @param reconnectEnabled if true, enables reconnect behavior
     */  
    public synchronized void  
    setReconnectEnabled(boolean reconnectEnabled) 
    {
        _loggerL.entering(_className, "setReconnectEnabled()", Boolean.toString(reconnectEnabled));
        this.reconnectEnabled = reconnectEnabled;
    }
 
    /** Returns the Reconnect behavior for this Resource Adapter
     *   
     *  @return the Reconnect behavior for this Resource Adapter
     */
    public boolean
    getReconnectEnabled()
    {
        _loggerL.entering(_className, "getReconnectEnabled()", Boolean.toString(reconnectEnabled));
        return reconnectEnabled;
    }

    /** Sets the Reconnect interval for this Resource Adapter
     *   
     *  @param reconnectInterval The reconnect interval in milliseconds
     *                  when reconnect is enabled
     */  
    public synchronized void  
    setReconnectInterval(int reconnectInterval) 
    {
        _loggerL.entering(_className, "setReconnectInterval()", Integer.toString(reconnectInterval));
        this.reconnectInterval = reconnectInterval;
    }
 
    /** Returns the Reconnect interval for this Resource Adapter
     *   
     *  @return the Reconnect interval for this Resource Adapter
     */
    public int
    getReconnectInterval()
    {
        _loggerL.entering(_className, "getReconnectInterval()", Integer.toString(reconnectInterval));
        return reconnectInterval;
    }

    /** Sets the reconnectAttempts for this Resource Adapter
     *   
     *  @param reconnectAttempts The number of reconnect attempts
     *                  when reconnect is enabled
     */  
    public synchronized void  
    setReconnectAttempts(int reconnectAttempts) 
    {
        _loggerL.entering(_className, "setReconnectAttempts()", Integer.toString(reconnectAttempts));
        this.reconnectAttempts = reconnectAttempts;
    }
 
    /** Returns the Reconnect attempts for this Resource Adapter
     *   
     *  @return the reconnectAttempts for this Resource Adapter
     */
    public int
    getReconnectAttempts()
    {
        _loggerL.entering(_className, "getReconnectAttempts()", Integer.toString(reconnectAttempts));
        return reconnectAttempts;
    }

    /** Sets the addressListBehavior for this Resource Adapter
     *   
     *  @param addressListBehavior The behavior of connectionURL
     *                 or addressList on connection establishment
     */  
    public synchronized void  
    setAddressListBehavior(String addressListBehavior) 
    {
        _loggerL.entering(_className, "setAddressListBehavior()", addressListBehavior);
        if ("RANDOM".equalsIgnoreCase(addressListBehavior)) {
            this.addressListBehavior = "RANDOM";;
        } else {
            this.addressListBehavior = "PRIORITY";
        }
    }
 
    /** Returns the addressListBehavior for this Resource Adapter
     *   
     *  @return the addressListBehavior for this Resource Adapter
     */
    public String
    getAddressListBehavior()
    {
        _loggerL.entering(_className, "getAddressListBehavior()", addressListBehavior);
        return addressListBehavior;
    }

    /** Sets the addressListIterations for this Resource Adapter
     *   
     *  @param addressListIterations The number of iterations on
     *         addressList to be attempted on connection establishment
     */  
    public synchronized void  
    setAddressListIterations(int addressListIterations) 
    {
        _loggerL.entering(_className, "setAddressListIterations()", Integer.toString(addressListIterations));
        if (addressListIterations < 1) {
            _loggerL.warning(_lgrMID_WRN+"setAddressListIterations:Invalid value:"+Integer.toString(addressListIterations)+":Setting to 1");
            this.addressListIterations = 1;
        } else {
            this.addressListIterations = addressListIterations;
        }
    }
 
    /** Returns the addressListIterations for this Resource Adapter
     *   
     *  @return the addressListIterations for this Resource Adapter
     */
    public int
    getAddressListIterations()
    {
        _loggerL.entering(_className, "getAddressListIterations()", Integer.toString(addressListIterations));
        return addressListIterations;
    }

    /** Sets the Resource Adapter for use in the Application Client Container
     *   
     *  @param inAppClientContainer if true, indicates that it is in the ACC
     */  
    public synchronized void  
    setInAppClientContainer(boolean inAppClientContainer) 
    {
        _loggerL.entering(_className, "setInAppClientContainer()", Boolean.toString(inAppClientContainer));
        this.inAppClientContainer = inAppClientContainer;
        _adjustInAppClientContainer();
    }
 
    /** Return whether the Resource Adapter is being used in the App Client Cont.
     *   
     *  @return true if being used in the App Client Cont. 
     */
    public boolean
    getInAppClientContainer()
    {
        _loggerL.entering(_className, "getInAppClientContainer()", Boolean.toString(inAppClientContainer));
        return inAppClientContainer;
    }

    /** Sets the RA for use in a Clustered Java EE Container
     *   
     *  @param inClusteredContainer if true, indicates that it is in a Clustered Java EE Containe.
     */  
    public synchronized void  
    setInClusteredContainer(boolean inClusteredContainer) 
    {
        _loggerL.entering(_className, "setInClusteredContainer()", Boolean.toString(inClusteredContainer));
        this.inClusteredContainer = inClusteredContainer;
    }
 
    /** Return whether this RA is being used in a Clustered Java EE Container.
     *   
     *  @return true if being used in a Clustered Java EE Container. 
     */
    public boolean
    getInClusteredContainer()
    {
        _loggerL.entering(_className, "getInClusteredContainer()", Boolean.toString(inClusteredContainer));
        return inClusteredContainer;
    }

    /** Sets the groupName for this ResourceAdapter
     *   
     *  @param groupName The Group Name
     */  
    public synchronized void
    setGroupName(String groupName)
    {
        _loggerL.entering(_className, "setGroupName()", groupName);
        this.groupName = groupName;
    }

    /** Returns the groupName for this ResourceAdapter
     *   
     *  @return The groupName
     */  
    public String
    getGroupName()
    {
        _loggerL.entering(_className, "getGroupName()", groupName);
        return groupName;
    }
 
    /** Sets whether ra.start() does initialization only or not for this ResourceAdapter
     *   
     *  @param doInitOnlyOnStart true if ra.start() performs initialization only
     *                           false if ra.start() performs a normal RA start
     */  
    public synchronized void
    setDoInitOnlyOnStart(boolean doInitOnlyOnStart)
    {
        _loggerL.entering(_className, "setDoInitOnlyOnStart()", Boolean.toString(doInitOnlyOnStart));
        this.doInitOnlyOnStart = doInitOnlyOnStart;
    }

    /** Returns the setting of doInitOnlyOnStart for this ResourceAdapter
     *   
     *  @return doInitOnlyOnStart
     */  
    public boolean
    getDoInitOnlyOnStart()
    {
        _loggerL.entering(_className, "getDoInitOnlyOnStart()", Boolean.toString(doInitOnlyOnStart));
        return doInitOnlyOnStart;
    }
 
    /** Sets the brokerType for this ResourceAdapter
     *   
     *  @param brokerType The type of the broker that this RA associates with
     */  
    public synchronized void
    setBrokerType(String brokerType)
    {
        _loggerL.entering(_className, "setBrokerType()", brokerType);
        if (started) {
            _loggerL.warning(_lgrMID_WRN+"setBrokerType:RA already started:Disallowing change from:"+this.brokerType+":to:"+brokerType);
            return;
        }
        if (BROKER_TYPE_EMBEDDED.equals(brokerType) ||
            BROKER_TYPE_LOCAL.equals(brokerType)) {
                this.brokerType = brokerType;
                //this.connectionURL="";
                return;
        }
        if (BROKER_TYPE_REMOTE.equals(brokerType)) {
            this.brokerType = brokerType;
        } else {
            _loggerL.warning(_lgrMID_WRN+"setBrokerType:Invalid value:"+brokerType);
        }
    }

    /** Returns the brokerType for this ResourceAdapter
     *   
     *  @return The brokerType
     */  
    public String
    getBrokerType()
    {
        _loggerL.entering(_className, "getBrokerType()", brokerType);
        return brokerType;
    }
 
    /** Sets the brokerInstanceName for this ResourceAdapter
     *   
     *  @param brokerInstanceName The Instance Name that the broker must use
     */  
    public synchronized void
    setBrokerInstanceName(String brokerInstanceName)
    {
        _loggerL.entering(_className, "setBrokerInstanceName()", brokerInstanceName);
        if (isNameValidAlphaNumeric_(brokerInstanceName)) {
            this.brokerInstanceName = brokerInstanceName;
        } else {
            _loggerL.warning(_lgrMID_WRN+"setBrokerInstanceName:Invalid value:"+brokerInstanceName);
        }
    }

    /** Returns the brokerInstanceName for this ResourceAdapter
     *   
     *  @return The brokerInstanceName
     */  
    public String
    getBrokerInstanceName()
    {
        _loggerL.entering(_className, "getBrokerInstanceName()", brokerInstanceName);
        return brokerInstanceName;
    }
 
    /** Sets the brokerBindAddress for this ResourceAdapter
     *   
     *  @param brokerBindAddress The network address that the broker must bind to
     */  
    public synchronized void
    setBrokerBindAddress(String brokerBindAddress)
    {
        _loggerL.entering(_className, "setBrokerBindAddress()", brokerBindAddress);
        if (started) {
            _loggerL.warning(_lgrMID_WRN+"setBrokerBindAddress:RA already started:Disallowing change from:"+this.brokerBindAddress+":to:"+brokerBindAddress);
            return;
        }
        try {
            InetAddress ia = InetAddress.getByName(brokerBindAddress);
            this.brokerBindAddress = brokerBindAddress;
        } catch (UnknownHostException e) {
            _loggerL.warning(_lgrMID_WRN+"setBrokerBindAddress:Ignoring Invalid Address:"+brokerBindAddress+":ExceptionMsg="+e.getMessage());
        }
    }

    /** Returns the brokerBindAddress for this ResourceAdapter
     *   
     *  @return The brokerBindAddress
     */  
    public String
    getBrokerBindAddress()
    {
        _loggerL.entering(_className, "getBrokerBindAddress()", brokerBindAddress);
        return brokerBindAddress;
    }
 
    /** Sets the brokerPort for this ResourceAdapter
     *   
     *  @param brokerPort The Broker Main Port
     */  
    public synchronized void
    setBrokerPort(int brokerPort)
    {
        _loggerL.entering(_className, "setBrokerPort()", new Integer(brokerPort));
        if (started) {
            _loggerL.warning(_lgrMID_WRN+"setBrokerPort:RA already started:Disallowing change from:"+this.brokerPort+":to:"+brokerPort);
            return;
        }
        this.brokerPort = brokerPort;
    }

    /** Returns the brokerPort for this ResourceAdapter
     *   
     *  @return The brokerPort
     */  
    public int
    getBrokerPort()
    {
        _loggerL.entering(_className, "getBrokerPort()", new Integer(brokerPort));
        return brokerPort;
    }
 
    /** Sets the brokerHomeDir for this ResourceAdapter
     *   
     *  @param brokerHomeDir The Broker Home Directory
     */  
    public synchronized void
    setBrokerHomeDir(String brokerHomeDir)
    {
        _loggerL.entering(_className, "setBrokerHomeDir()", brokerHomeDir);
        if (started) {
            _loggerL.warning(_lgrMID_WRN+"setBrokerHomeDir:RA already started:Disallowing change from:"+this.brokerHomeDir+":to:"+brokerHomeDir);
            return;
        }
        try {
            String path = new File(brokerHomeDir).getCanonicalPath();
            this.brokerHomeDir = brokerHomeDir;
        } catch (IOException e) {
            _loggerL.warning(_lgrMID_WRN+"setBrokerHomeDir:Invalid value:"+brokerHomeDir+":Exception Message="+e.getMessage());
        }
    }

    /** Returns the brokerHomeDir for this ResourceAdapter
     *   
     *  @return The brokerHomeDir
     */  
    public String
    getBrokerHomeDir()
    {
        _loggerL.entering(_className, "getBrokerHomeDir()", brokerHomeDir);
        return brokerHomeDir;
    }
 
    /** Sets the brokerVarDir for this ResourceAdapter
     *   
     *  @param brokerVarDir The Broker VarDir Directory
     */  
    public synchronized void
    setBrokerVarDir(String brokerVarDir)
    {
        _loggerL.entering(_className, "setBrokerVarDir()", brokerVarDir);
        if (started) {
            _loggerL.warning(_lgrMID_WRN+"setBrokerVarDir:RA already started:Disallowing change from:"+this.brokerVarDir+":to:"+brokerVarDir);
            return;
        }
        try {
            String path = new File(brokerVarDir).getCanonicalPath();
            this.brokerVarDir = brokerVarDir;
        } catch (IOException e) {
            _loggerL.warning(_lgrMID_WRN+"setBrokerVarDir:Invalid value:"+brokerVarDir+":Exception Message="+e.getMessage());
        }
    }

    /** Returns the brokerVarDir for this ResourceAdapter
     *   
     *  @return The brokerVarDir
     */  
    public String
    getBrokerVarDir()
    {
        _loggerL.entering(_className, "getBrokerVarDir()", brokerVarDir);
        return brokerVarDir;
    }
 
    /** Sets the brokerLibDir for this ResourceAdapter
     *   
     *  @param brokerLibDir The Broker LibDir Directory
     */  
    public synchronized void
    setBrokerLibDir(String brokerLibDir)
    {
        _loggerL.entering(_className, "setBrokerLibDir()", brokerLibDir);
        if (started) {
            _loggerL.warning(_lgrMID_WRN+"setBrokerLibDir:RA already started:Disallowing change from:"+this.brokerLibDir+":to:"+brokerLibDir);
            return;
        }
        try {
            String path = new File(brokerLibDir).getCanonicalPath();
            this.brokerLibDir = brokerLibDir;
        } catch (IOException e) {
            _loggerL.warning(_lgrMID_WRN+"setBrokerLibDir:Invalid value:"+brokerLibDir+":Exception Message="+e.getMessage());
        }
    }

    /** Returns the brokerLibDir for this ResourceAdapter
     *   
     *  @return The brokerLibDir
     */  
    public String
    getBrokerLibDir()
    {
        _loggerL.entering(_className, "getBrokerLibDir()", brokerLibDir);
        return brokerLibDir;
    }
 
    /** Sets the brokerJavaDir for this ResourceAdapter
     *   
     *  @param brokerJavaDir The Broker JavaDir Directory
     */  
    public synchronized void
    setBrokerJavaDir(String brokerJavaDir)
    {
        _loggerL.entering(_className, "setBrokerJavaDir()", brokerJavaDir);
        if (started) {
            _loggerL.warning(_lgrMID_WRN+"setBrokerJavaDir:RA already started:Disallowing change from:"+this.brokerJavaDir+":to:"+brokerJavaDir);
            return;
        }
        try {
            String path = new File(brokerJavaDir).getCanonicalPath();
            this.brokerJavaDir = brokerJavaDir;
        } catch (IOException e) {
            _loggerL.warning(_lgrMID_WRN+"setBrokerJavaDir:Invalid value:"+brokerJavaDir+":Exception Message="+e.getMessage());
        }
    }

    /** Returns the brokerJavaDir for this ResourceAdapter
     *   
     *  @return The brokerJavaDir
     */  
    public String
    getBrokerJavaDir()
    {
        _loggerL.entering(_className, "getBrokerJavaDir()", brokerJavaDir);
        return brokerJavaDir;
    }
 
    /** Sets the brokerArgs for this ResourceAdapter
     *   
     *  @param brokerArgs The extra cmd line args to be used for the broker that this RA controls
     */  
    public synchronized void
    setBrokerArgs(String brokerArgs)
    {
        _loggerL.entering(_className, "setBrokerArgs()", brokerArgs);
        if (started) {
            _loggerL.warning(_lgrMID_WRN+"setBrokerArgs:RA already started:Disallowing change from:"+this.brokerArgs+":to:"+brokerArgs);
            return;
        }
        this.brokerArgs = brokerArgs;
    }

    /** Returns the brokerArgs for this ResourceAdapter
     *   
     *  @return The brokerArgs
     */  
    public String
    getBrokerArgs()
    {
        _loggerL.entering(_className, "getBrokerArgs()", brokerArgs);
        return brokerArgs;
    }

    /** Sets the masterBroker for this ResourceAdapter
     *   
     *  @param masterBroker The masterBroker that the Resource Adapter must use
     */  
    public synchronized void
    setMasterBroker(String masterBroker)
    {
        _loggerL.entering(_className, "setMasterBroker()", masterBroker);
        if (started) {
            _loggerL.warning(_lgrMID_WRN+"setMasterBroker:RA already started:Disallowing change from:"+this.masterBroker+":to:"+masterBroker);
            return;
        }
        this.masterBroker = masterBroker;
        /*
        if (isNameValidAlphaNumeric_(brokerId)) {
            this.brokerId = brokerId;
        } else {
            _loggerL.warning(_lgrMID_WRN+"setMasterBroker:Invalid value:"+brokerId);
        }
        */
    }

    /** Returns the masterBroker for this ResourceAdapter
     *   
     *  @return The masterBroker
     */  
    public String
    getMasterBroker()
    {
        _loggerL.entering(_className, "getMasterBroker()", masterBroker);
        return brokerId;
    }
 
    /** Sets the brokerStartTimeout for this ResourceAdapter
     *   
     *  @param brokerStartTimeout The brokerStartTimeout that the Resource Adapter must use
     */  
    public synchronized void
    setBrokerStartTimeout(int brokerStartTimeout)
    {
        _loggerL.entering(_className, "setBrokerStartTimeout()", new Integer(brokerStartTimeout));
        if (started) {
            _loggerL.warning(_lgrMID_WRN+"setBrokerStartTimeout:RA already started:Disallowing change from:"+this.brokerStartTimeout+":to:"+brokerStartTimeout);
            return;
        }
        this.brokerStartTimeout = brokerStartTimeout;
    }

    /** Returns the brokerStartTimeout for this ResourceAdapter
     *   
     *  @return The brokerStartTimeout
     */  
    public int
    getBrokerStartTimeout()
    {
        _loggerL.entering(_className, "getBrokerStartTimeout()", new Integer(brokerStartTimeout));
        return brokerStartTimeout;
    }
 
    /** Sets the admin Username for this ResourceAdapter instance
     *   
     *  @param adminUsername The adminUsername
     */  
    public synchronized void  
    setAdminUsername(String adminUsername) 
    {
        _loggerL.entering(_className, "setAdminUsername()", adminUsername);
        this.adminUsername = adminUsername;
    }
 
    /** Return the adminUsername for this ResourceAdapter instance
     *   
     *  @return The adminUsername
     */
    public String
    getAdminUsername()
    {
        _loggerL.entering(_className, "getAdminUsername()", adminUsername);
        return adminUsername;
    }
 
    /** Sets the admin Password for this ResourceAdapter instance
     *   
     *  @param adminPassword The adminPassword
     */  
    public synchronized void  
    setAdminPassword(String adminPassword) 
    {
        _loggerL.entering(_className, "setAdminPassword()");
        this.adminPassword = adminPassword;
    }
 
    /** Return the adminPassword for this ResourceAdapter instance
     *   
     *  @return The adminPassword
     */
    public String
    getAdminPassword()
    {
        _loggerL.entering(_className, "getAdminPassword()");
        return adminPassword;
    }
 
    /** Sets the admin Password File for this ResourceAdapter instance
     *   
     *  @param adminPassFile The adminPassFile
     */  
    public synchronized void  
    setAdminPassFile(String adminPassFile) 
    {
        _loggerL.entering(_className, "setAdminPassFile()", adminPassFile);
        this.adminPassFile = adminPassFile;
    }
 
    /** Return the admin Password File for this ResourceAdapter instance
     *   
     *  @return The adminPassFile
     */
    public String
    getAdminPassFile()
    {
        _loggerL.entering(_className, "getAdminPassFile()", adminPassFile);
        return adminPassFile;
    }
 
    /** Return the JMXConnectorEnv for this ResourceAdapter instance
     *   
     *  @return The JMXConnectorEnv
     */
    public synchronized HashMap
    getJMXConnectorEnv()
    {
        _loggerL.entering(_className, "getJMXConnectorEnv()");
        if (jmxConnectorEnv == null) {
            jmxConnectorEnv = new HashMap();
            String[] credentials = new String[] { adminUsername, adminPassword };
            jmxConnectorEnv.put(javax.management.remote.JMXConnector.CREDENTIALS, credentials);
        }
        return jmxConnectorEnv;
    }
 
    /** Return the JMXServiceURL for this ResourceAdapter instance
     *   
     *  @return The JMXServiceURL
     */
    public synchronized String
    getJMXServiceURL()
    {
        _loggerL.entering(_className, "getJMXServiceURL()");
        if (!started) {
            _loggerL.warning(_lgrMID_WRN+"getJMXServiceURL:RA not started:Returning null");
            return null;
        }
        if ((jmxServiceURL == null) && !BROKER_TYPE_REMOTE.equals(brokerType)) {
            try {
                jmxServiceURL = acf.getJMXServiceURL().toString();
            } catch (Exception e) {
                _loggerL.warning(_lgrMID_EXC+"getJMXServiceURL:Exception:Message="+e.getMessage());
            }
        }
        _loggerL.exiting(_className, "getJMXServiceURL()", jmxServiceURL);
        return jmxServiceURL;
    }

    /** Return the JMXServiceURLList for this ResourceAdapter instance
     *   
     *  @return The JMXServiceURLList
     */
    public synchronized String
    getJMXServiceURLList()
    {
        _loggerL.entering(_className, "getJMXServiceURLList()", "For addressList = "+connectionURL);
        if (isJMXServiceURLListValid) {
            _loggerL.exiting(_className, "getJMXServiceURLList()", jmxServiceURLList);
            return jmxServiceURLList;
        }
        com.sun.messaging.AdminConnectionFactory tacf = new com.sun.messaging.AdminConnectionFactory();
        try {
            tacf.setProperty(AdminConnectionConfiguration.imqDefaultAdminUsername, adminUsername);
            tacf.setProperty(AdminConnectionConfiguration.imqDefaultAdminPassword, adminPassword);
        } catch (Exception e) {
            _loggerL.warning(_lgrMID_EXC+"getJMXServiceURLList:Exception configuring AdminConnectionFactory:Message="+e.getMessage());
        }

        //XXXJava_5 Use StringBuilder
        StringBuffer jb = new StringBuffer(256);
        jb.append("");

        String jurl = null;
        StringTokenizer st = new StringTokenizer(connectionURL, " ,");
        while (st.hasMoreTokens()) {
            String t = st.nextToken().trim();
            if (_loggerL.isLoggable(Level.FINER)) {
                _loggerL.finer(_lgrMID_INF+"getJMXServiceURLList:addressList component = "+t);
            }
            //XXX:tharakan:Needs to be resolved when ACF is updated
            //XXX:         ACF currently takes a normal MQ address and defaults the service name to jmxrmi
         /* if (t.indexOf(":") == -1) {
                t = "mq://" + t + ":7676";
            } else {
                if (t.indexOf("://") == -1) {
                    t = "mq://" + t;
                } else {
                    t = t + ":7676";
                }
            } */
            try {
                //XXX:tharakan:This depends on AdminConnectionFactory defaulting the serviceName to jmxrmi
                tacf.setProperty(AdminConnectionConfiguration.imqAddress, t );
                //tacf.setProperty(AdminConnectionConfiguration.imqAddress, t + "/jmxrmi" );
                if (_loggerL.isLoggable(Level.FINER)) {
                    _loggerL.finer(_lgrMID_INF+"getJMXServiceURLList:address="+t);
                }
                jurl = tacf.getJMXServiceURL().toString();
                if (_loggerL.isLoggable(Level.FINER)) {
                    _loggerL.finer(_lgrMID_INF+"getJMXServiceURLList:JMXServiceURL string for addressList component "+t+" = "+jurl);
                }
                jb.append(jurl + " ");
            } catch (Exception e) {
                _loggerL.warning(_lgrMID_EXC+"getJMXServiceURLList:Exception:Message="+e.getMessage());
            }
        }
        jmxServiceURLList = jb.toString();
        isJMXServiceURLListValid = true;
        _loggerL.exiting(_className, "getJMXServiceURLList()", jmxServiceURLList);
        return jmxServiceURLList;
    }

    /** Sets useJNDIRmiServiceURL for this ResourceAdapter instance
     *   
     *  @param useJNDIRmiServiceURL The boolean value of useJNDIRmiServiceURL
     */  
    public synchronized void  
    setUseJNDIRmiServiceURL(boolean useJNDIRmiServiceURL) 
    {
        _loggerL.entering(_className, "setUseJNDIRmiServiceURL()", new Boolean(useJNDIRmiServiceURL));
        if (started) {
            _loggerL.warning(_lgrMID_WRN+"setUseJNDIRmiServiceURL:RA already started:Disallowing change from:"+this.useJNDIRmiServiceURL+":to:"+useJNDIRmiServiceURL);
            return;
        }
        this.useJNDIRmiServiceURL = useJNDIRmiServiceURL;
    }
 
    /** Return useJNDIRmiServiceURL of this ResourceAdapter instance
     *   
     *  @return The useJNDIRmiServiceURL
     */
    public boolean
    getUseJNDIRmiServiceURL()
    {
        _loggerL.entering(_className, "getUseJNDIRmiServiceURL()", new Boolean(useJNDIRmiServiceURL));
        return useJNDIRmiServiceURL;
    }

    /* XXX Temporarily here for testing w/older AS that uses this interface
           Remove when they have switched over to StartRmiRegistry
    public void setUseExternalRMIRegistry (boolean ext) {
        if (ext) {
            setStartRmiRegistry(false);
        } else {
            setStartRmiRegistry(true);
        }
    }

    public boolean getUseExternalRMIRegistry() {
        return (!startRmiRegistry);
    }
    */

    /** Sets the startRmiRegistry for this ResourceAdapter
     *   
     *  @param startRmiRegistry The RMI Registry Port
     */  
    public synchronized void
    setStartRmiRegistry(boolean startRmiRegistry)
    {
        _loggerL.entering(_className, "setStartRmiRegistry()", new Boolean(startRmiRegistry));
        if (started || (_startedAtLeastOnce && BROKER_TYPE_EMBEDDED.equals(brokerType))) {
            _loggerL.warning(_lgrMID_WRN+"setStartRmiRegistry:RA already started OR run once as EMBEDDED:Disallowing change from:"+this.startRmiRegistry+":to:"+startRmiRegistry);
            return;
        }
        this.startRmiRegistry = startRmiRegistry;
    }

    /** Returns the startRmiRegistry for this ResourceAdapter
     *   
     *  @return startRmiRegistry
     */  
    public boolean
    getStartRmiRegistry()
    {
        _loggerL.entering(_className, "getStartRmiRegistry()", new Boolean(startRmiRegistry));
        return startRmiRegistry;
    }

    /** Sets the rmiRegistryPort for this ResourceAdapter
     *   
     *  @param rmiRegistryPort The RMI Registry Port
     */  
    public synchronized void
    setRmiRegistryPort(int rmiRegistryPort)
    {
        _loggerL.entering(_className, "setRmiRegistryPort()", new Integer(rmiRegistryPort));
        if (started || (_startedAtLeastOnce && BROKER_TYPE_EMBEDDED.equals(brokerType))) {
            _loggerL.warning(_lgrMID_WRN+"setRmiRegistryPort:RA already started OR run once as EMBEDDED:Disallowing change from:"+this.rmiRegistryPort+":to:"+rmiRegistryPort);
            return;
        }
        this.rmiRegistryPort = rmiRegistryPort;
    }

    /** Returns the rmiRegistryPort for this ResourceAdapter
     *   
     *  @return The rmiRegistryPort
     */  
    public int
    getRmiRegistryPort()
    {
        _loggerL.entering(_className, "getRmiRegistryPort()", new Integer(rmiRegistryPort));
        return rmiRegistryPort;
    }

    /** Sets useSSLJMXConnector for this ResourceAdapter instance
     *   
     *  @param useSSLJMXConnector The boolean value of useSSLJMXConnector
     */  
    public synchronized void  
    setUseSSLJMXConnector(boolean useSSLJMXConnector) 
    {
        _loggerL.entering(_className, "setUseSSLJMXConnector()", new Boolean(useSSLJMXConnector));
        if (started) {
            _loggerL.warning(_lgrMID_WRN+"setUseSSLJMXConnector:RA already started:Disallowing change from:"+this.useSSLJMXConnector+":to:"+useSSLJMXConnector);
            return;
        }
        this.useSSLJMXConnector = useSSLJMXConnector;
    }
 
    /** Return useSSLJMXConnector of this ResourceAdapter instance
     *   
     *  @return The useSSLJMXConnector
     */
    public boolean
    getUseSSLJMXConnector()
    {
        _loggerL.entering(_className, "getUseSSLJMXConnector()", new Boolean(useSSLJMXConnector));
        return useSSLJMXConnector;
    }

    /** Sets brokerEnableHA for this ResourceAdapter instance
     *   
     *  @param brokerEnableHA The boolean value of brokerEnableHA
     */  
    public synchronized void  
    setBrokerEnableHA(boolean brokerEnableHA) 
    {
        _loggerL.entering(_className, "setBrokerEnableHA()", new Boolean(brokerEnableHA));
        if (started) {
            _loggerL.warning(_lgrMID_WRN+"setBrokerEnableHA:RA already started:Disallowing change from:"+this.brokerEnableHA+":to:"+brokerEnableHA);
            return;
        }
        this.brokerEnableHA = brokerEnableHA;
    }
 
    /** Return brokerEnableHA of this ResourceAdapter instance
     *   
     *  @return The brokerEnableHA
     */
    public boolean
    getBrokerEnableHA()
    {
        _loggerL.entering(_className, "getBrokerEnableHA()", new Boolean(brokerEnableHA));
        return brokerEnableHA;
    }

    /** Sets the clusterId for this ResourceAdapter
     *   
     *  @param clusterId The cluster Id that the Resource Adapter must use
     */  
    public synchronized void
    setClusterId(String clusterId)
    {
        _loggerL.entering(_className, "setClusterId()", clusterId);
        if (started) {
            _loggerL.warning(_lgrMID_WRN+"setClusterId:RA already started:Disallowing change from:"+this.clusterId+":to:"+clusterId);
            return;
        }
        if (isNameValidAlphaNumeric_(clusterId)) {
            this.clusterId = clusterId;
        } else {
            _loggerL.warning(_lgrMID_WRN+"setClusterId:Invalid value:"+clusterId);
        }
    }

    /** Returns the clusterId for this ResourceAdapter
     *   
     *  @return The clusterId
     */  
    public String
    getClusterId()
    {
        _loggerL.entering(_className, "getClusterId()", clusterId);
        return clusterId;
    }
 
    /** Sets the brokerId for this ResourceAdapter
     *   
     *  @param brokerId The brokerId that the Resource Adapter must use
     */  
    public synchronized void
    setBrokerId(String brokerId)
    {
        _loggerL.entering(_className, "setBrokerId()", brokerId);
        if (started) {
            _loggerL.warning(_lgrMID_WRN+"setBrokerId:RA already started:Disallowing change from:"+this.brokerId+":to:"+brokerId);
            return;
        }
        if (isNameValidAlphaNumeric_(brokerId)) {
            this.brokerId = brokerId;
        } else {
            _loggerL.warning(_lgrMID_WRN+"setBrokerId:Invalid value:"+brokerId);
        }
    }

    /** Returns the brokerId for this ResourceAdapter
     *   
     *  @return The brokerId
     */  
    public String
    getBrokerId()
    {
        _loggerL.entering(_className, "getBrokerId()", brokerId);
        return brokerId;
    }
 
    /** Sets the dbType for this ResourceAdapter
     *   
     *  @param dbType The type of the broker that this RA associates with
     */  
    public synchronized void
    setDBType(String dbType)
    {
        _loggerL.entering(_className, "setDBType()", dbType);
        if (started) {
            _loggerL.warning(_lgrMID_WRN+"setDBType:RA already started:Disallowing change from:"+this.dbType+":to:"+dbType);
            return;
        }
        if (DB_TYPE_HADB.equals(dbType) ||
            DB_TYPE_ORACLE.equals(dbType) ||
            DB_TYPE_POINTBASE.equals(dbType) ||
            DB_TYPE_CLOUDSCAPE.equals(dbType) ||
            DB_TYPE_DERBY.equals(dbType)) {
            this.dbType = dbType;
        } else {
            _loggerL.warning(_lgrMID_WRN+"setDBType:Invalid value:"+dbType);
        }
    }

    /** Returns the dbType for this ResourceAdapter
     *   
     *  @return The dbType
     */  
    public String
    getDBType()
    {
        _loggerL.entering(_className, "getDBType()", dbType);
        return dbType;
    }
 
    /** Sets the dbProps for this ResourceAdapter
     *   
     *  @param dbProps The dbProps that the Resource Adapter must use
     */  
    public synchronized void
    setDBProps(Properties dbProps)
    {
        _loggerL.entering(_className, "setDBProps()", dbProps);
        if (started) {
            _loggerL.warning(_lgrMID_WRN+"setDBProps:RA already started:Disallowing change from:"+this.dbProps.toString()+":to:"+dbProps.toString());
            return;
        }
        this.dbProps = dbProps;
    }

    /** Returns the dbProps for this ResourceAdapter
     *   
     *  @return The dbProps
     */  
    public Properties
    getDBProps()
    {
        _loggerL.entering(_className, "getDBProps()", dbProps);
        return dbProps;
    }
 
    /** Sets the dsProps for this ResourceAdapter
     *   
     *  @param dsProps The dsProps that the Resource Adapter must use
     */  
    public synchronized void
    setDSProps(Properties dsProps)
    {
        _loggerL.entering(_className, "setDSProps()", dsProps);
        if (started) {
            _loggerL.warning(_lgrMID_WRN+"setDSProps:RA already started:Disallowing change from:"+this.dsProps.toString()+":to:"+dsProps.toString());
            return;
        }
        this.dsProps = dsProps;
    }

    /** Returns the dsProps for this ResourceAdapter
     *   
     *  @return The dsProps
     */  
    public Properties
    getDSProps()
    {
        _loggerL.entering(_className, "getDSProps()", dsProps);
        return dsProps;
    }
 
    /** Sets haRequired for this ResourceAdapter instance
     *   
     *  @param haRequired The boolean value of haRequired
    public synchronized void  
    setHARequired(boolean haRequired) 
    {
        _loggerL.entering(_className, "setHARequired()", new Boolean(haRequired));
        if (started) {
            _loggerL.warning(_lgrMID_WRN+"setHARequired:RA already started:Disallowing change from:"+this.haRequired+":to:"+haRequired);
            return;
        }
        this.haRequired = haRequired;
    }
     */  
 
    /** Return haRequired of this ResourceAdapter instance
     *   
     *  @return The haRequired
    public boolean
    getHARequired()
    {
        _loggerL.entering(_className, "getHARequired()", new Boolean(haRequired));
        return haRequired;
    }
     */

 
    //Protected methods //
    //
    /** Returns the effective connectionURL based on brokerType
    */ 
    protected String _getEffectiveConnectionURL()
    {
        _loggerL.entering(_className, "_getEffectiveConnectionURL()");
        String eConnectionURL = null;
        if (!BROKER_TYPE_REMOTE.equals(brokerType)) {
            eConnectionURL = ((brokerBindAddress != null) ? brokerBindAddress : "localhost" )
                           + ":"+Integer.toString(brokerPort);
            if (!("".equals(connectionURL))) {
                eConnectionURL = eConnectionURL + "," + connectionURL;
            }
        } else {
            eConnectionURL = connectionURL;
        }
        _loggerL.exiting(_className, "_getEffectiveConnectionURL()", eConnectionURL);
        return eConnectionURL;
        /*
        _loggerL.exiting(_className, "_getEffectiveConnectionURL()", connectionURL);
        return connectionURL;
        */
    }

    /** Returns the XAConnectionFactory for this ResourceAdapter
    */
    protected com.sun.messaging.XAConnectionFactory
    _getXACF()
    {
        return xacf;
    }

    /** Returns the raUID for this ResourceAdapter
    */
    protected String _getRAUID()
    {
        return raUID;
    }

    /** Returns the method that is called in the MessageListener
    * to deliver messages to the endpoint consumer
    */
    protected Method
    _getOnMessageMethod()
    {
        if (onMessage == null) {
            _setOnMessageMethod();
        }
        return onMessage;
    }


    //Private & Protected Methods //
    //
    /* configure the xacf */
    private void configureFactory()
    throws JMSException
    {
        String lcConnectionURL = null;

        if (!BROKER_TYPE_REMOTE.equals(brokerType)) {
            xacf.setProperty(ConnectionConfiguration.imqAddressList, _getEffectiveConnectionURL());
            xacf.setProperty(ConnectionConfiguration.imqAddressListBehavior, "PRIORITY");

            acf.setProperty(AdminConnectionConfiguration.imqAddress, "mq://" +
                    (((brokerBindAddress != null) ? brokerBindAddress : "localhost" )
                           + ":"+Integer.toString(brokerPort)) + "/jmxrmi" );

            acf.setProperty(AdminConnectionConfiguration.imqDefaultAdminUsername, adminUsername);
            acf.setProperty(AdminConnectionConfiguration.imqDefaultAdminPassword, adminPassword);
        } else {
            xacf.setProperty(ConnectionConfiguration.imqAddressList, connectionURL);
            xacf.setProperty(ConnectionConfiguration.imqAddressListBehavior, addressListBehavior);
        }
        xacf.setProperty(ConnectionConfiguration.imqDefaultUsername, userName);
        xacf.setProperty(ConnectionConfiguration.imqDefaultPassword, password);
        xacf.setProperty(ConnectionConfiguration.imqAddressListIterations, Integer.toString(addressListIterations));

        //Force reconnectEnabled to false for XAR succes
        xacf.setProperty(ConnectionConfiguration.imqReconnectEnabled, Boolean.toString(false)); //reconnectEnabled

        xacf.setProperty(ConnectionConfiguration.imqReconnectInterval, Integer.toString(reconnectInterval));
        xacf.setProperty(ConnectionConfiguration.imqReconnectAttempts, Integer.toString(reconnectAttempts));
    }

    /* init the connection */
    private void init()
    throws JMSException
    {
        if (xac != null) {
            //Needs to be done once only, even across a reconnect
            if (inClusteredContainer && (raUID == null)) {
                raUID = Long.toString(xac.generateUID());
            }
            xac.setExceptionListener(this);
            ((com.sun.messaging.jms.Connection)xac).setEventListener(this);
        }
        if (_loggerL.isLoggable(Level.FINER)) {
           _loggerL.finer(_lgrMID_INF+"init:"+"DONE:"+"raUID="+raUID);
        }
    }

    /* Keys for factory in epFactories hashmap */
    private int createFactoryID()
    {
        return ++_factoryID;
    }

    /* Keys for consumer in epConsumers hashmap */
    private int createConsumerID()
    {
        return ++_consumerID;
    }

    /** Adds a MessageEndpointFactory to the epFactories hashmap
     * 
     * @param endpointFactory The MessageEndpointFactory to be added
     * @return The ID assigned to the MessageEndpoitnFactory added
     */
    protected int
    addMessageFactory(MessageEndpointFactory endpointFactory)
    {
        int factoryID = createFactoryID();
        synchronized (epFactories) {
            epFactories.put(new Integer(factoryID), endpointFactory);
        }
        return factoryID;
    }
       

    /** Removes a MessageEndpointFactory by ID from the epFactories hashmap
     * 
     * @param factoryID The ID of the MessageEndpointFactory to be removed
     */
    protected void
    removeMessageFactory(int factoryID)
    {
	synchronized (epFactories) {
	    epFactories.remove(new Integer(factoryID));
	}
    }

    /** Returns the MessageEndpointFactory corresponding to ID from the hashmap
     *  
     * @param factoryID The ID of the MessageEndpointFactory to be returned
     * 
     * @return MessageEndpointFactory
     */
    protected MessageEndpointFactory
    _getMessageFactory(int factoryID)
    {
        synchronized(epFactories) {
            MessageEndpointFactory  epFactory = 
                (MessageEndpointFactory) epFactories.get(new Integer(factoryID));
            return epFactory;
        }
    }
    
    /** Searches for a MessageEndpointFactory in the epFactories hashmap
     *
     * @param endpointFactory The MessageEndpointFactory to find
     * @return The ID of the MessageEndpoitnFactory in the epFactories hashmap
     */
    private int
    matchMessageFactory(MessageEndpointFactory endpointFactory)
    {
        int key = -1;

        if (endpointFactory != null) {
            synchronized(epFactories) {

                Collection factories = epFactories.entrySet();
                if (factories != null) {
                    Iterator    iter = factories.iterator();
                    while (iter.hasNext()) {
                        /* Get the map's entries */
                        Map.Entry map = (Map.Entry) iter.next();              

                        /* If the endpointFactory matches an entry
                        * in the map, we have a match. Get the
                        * key for that match.
                        */
                        if(endpointFactory.equals(map.getValue())) {
                            Integer iKey = (Integer) map.getKey();
                            key = iKey.intValue();
                            break;
                        }
                    }
                }
            }
        }
        return key;
    }

    /** Adds a link between factoryID for MessageEndpointFactory and
     *  consumerID (endpointConsumer) in the hashmaps
     */
    protected void
    addFactorytoConsumerLink(int factoryID, int consumerID)
    {
        synchronized (epFactoryToConsumer) {
            epFactoryToConsumer.put(new Integer(factoryID), new Integer(consumerID));
        }
    }
      
    /** Removes Link between factoryID (MessageEndpointFactory) and 
     *  consumerID (EndpointConsumer) in the epFactoryToConsumer hashmap
     * 
     *  @param factoryID MessageEndpointFactory ID for which linked
     *         consumerID (EndpointConsumer) is to be removed
     */
    private void
    removeFactorytoConsumerLink(int factoryID)
    {
	synchronized (epFactoryToConsumer) {
	    epFactoryToConsumer.remove(new Integer(factoryID));
	}
    }

    /** Returns the ID for the EndpointConsumer that is linked by
     * the ID for the MessageEndpointFactory
     *  
     * @param factoryID The ID for the MessageEndpointFactory
     * 
     * @return consumerID for the EndpointConsumer linked by this factoryID
     */
    private int
    _getConsumerIDbyFactoryID(int factoryID)
    {
        return ((Integer)epFactoryToConsumer.get(new Integer(factoryID))).intValue();
    }
    
    /** Adds an EndpointConsumer in the hashmap
     * 
     * @param endpointConsumer The EndpointConsumer to be added
     * @return The ID assigned to the EndpointConsumer added
     */
    protected int
    addEndpointConsumer(EndpointConsumer endpointConsumer)
    {
        int consumerID = createConsumerID();
        synchronized (epConsumers) {
            epConsumers.put(new Integer(consumerID), endpointConsumer);
        }
        return consumerID;
    }

    /** Removes EndpointConsumer from the hashmap given consumerID
     *  
     * @param consumerID for the EndpointConsumer to remove
     */
    private void
    removeEndpointConsumer(int consumerID)
    {
        synchronized (epConsumers) {
            epConsumers.remove(new Integer(consumerID));
	}
    }

    /** Returns the EndpointConsumer for a given consumerID
     *
     * @param consumerID The ID for which the EndpointConsumer is desired
     * 
     * @return EndpointConsumer for consumerID
     */
    private EndpointConsumer
    _getEndpointConsumer(int consumerID)
    {
        EndpointConsumer endpointConsumer = 
            (EndpointConsumer)epConsumers.get(new Integer(consumerID));
        return endpointConsumer;
    }

    /** Removes references MessageEndpointFactory and linked EndpointConsumer
     * from hashmaps
     */
    private void
    removeFromMaps(int factoryID)
    {
        int consumerID = _getConsumerIDbyFactoryID(factoryID);
        removeEndpointConsumer(consumerID);
        removeMessageFactory(factoryID);
        removeFactorytoConsumerLink(factoryID);
    }

    /** Removes all the consumers from epConsumers after
     * first closing them.
     */
    private void
    removeAllConsumers()
    {
        synchronized (epFactories) {

            Collection factories = epFactories.entrySet();
            
            if(factories != null) {

                Iterator iter = factories.iterator();
                while (iter.hasNext()) {
                    Map.Entry entry = (Map.Entry)iter.next();
                    int factoryID = ((Integer)entry.getKey()).intValue();
                    int consumerID = _getConsumerIDbyFactoryID(factoryID);
                    EndpointConsumer ec = _getEndpointConsumer(consumerID);

                    try{
                        ec.stopMessageConsumer();
                    } catch(Exception ex) {
                        //No exception for this method, print
                        //XXX:log:tharakan
                        System.err.println("MQJMSRA:RA::Error:stopMessageConsumer exception:ignoring");
                        //ex.printStackTrace();
                    }
                    //removeFromMaps(factoryID); 
                }                       
                clearMaps();
            }
        }
    }

    private void clearMaps() {

        /* clear hash maps */
        epFactories.clear();
        epConsumers.clear();
        epFactoryToConsumer.clear();
    }
    
    /** Sets the Method that is called in the MessageListener
     *
     */
    private void
    _setOnMessageMethod()
    {
        Method onMessageMethod = null;
        try {
            Class msgListenerClass = javax.jms.MessageListener.class;
            Class[] paramTypes = { javax.jms.Message.class };
            onMessageMethod = msgListenerClass.getMethod("onMessage", paramTypes);
            
        } catch (NoSuchMethodException ex) {

            ex.printStackTrace();
        }
        onMessage = onMessageMethod;
    }  

    private void
    _adjustInAppClientContainer()
    {
        //System.out.println("MQJMSRA:RA:AIACC()");
        String inACC_SysProp = System.getProperty("com.sun.messaging.jms.ra.inACC");
        if (inACC_SysProp != null) {
            System.err.println("MQJMSRA:RA:AIACC:SystemProp com.sun.messaging.jms.ra.inACC is NOT null!!");
            if ("true".equals(inACC_SysProp)) {
                System.err.println("MQJMSRA:RA:AIACC:setting inACC true");
                this.inAppClientContainer = true;
            } else {
                System.err.println("MQJMSRA:RA:AIACC:setting inACC false");
                this.inAppClientContainer = false;
            }
        } else {
            //System.out.println("MQJMSRA:RA:AIACC:SystemProp com.sun.messaging.jms.ra.inACC IS NULL!! Try WorkMgr");
            //Try to set inACC correctly depending on workMgr availability
            if (workMgr != null) {
                //System.out.println("MQJMSRA:RA:AIACC:WorkMgr is NOT null!!");
                //Try to do work
                try {
                    workMgr.doWork(
                        new Work()
                        {
                            public void
                            run()
                            {
                                //System.out.println("MQJMSRA:RA:AIACC:Working...!");
                                return;
                            }

                            public void
                            release()
                            {
                                //System.out.println("MQJMSRA:RA:AIACC:Released...!");
                                return;
                            }
                         }
                    );
                    //System.out.println("MQJMSRA:RA:AIACC:leaving inACC as set since WorkMgr available and successful. inACC="+this.inAppClientContainer);
                    //
                    //Leave inAppClientContainer setting as set -- i.e. don't change it if WorkMgr is successful
                    //e.g. If Test wants it to be true - don't set it to false if the workMgr is successful
                    //
                } catch (Exception we) {
                    //System.out.println("MQJMSRA:RA:AIACC:setting inACC true as WorkMgr available but unsuccessful");
                    //Force it to be in the AppClient Container if unable to use WorkMgr
                    this.inAppClientContainer = true;
                }
            } else {
                //System.out.println("MQJMSRA:RA:AIACC:setting inACC true as WorkMgr unavailable");
                //Force it to be in the AppClient Container if WorkMgr is NULL
                this.inAppClientContainer = true;
            }
        }
        //System.out.println("MQJMSRA:RA:AIACC:exiting with inACC set to " + this.inAppClientContainer);
    }

    // Info Methods
    public String
    toString()
    {
        return ("SJSMQ JMS ResourceAdaapter configuration=\n"+
            "\traUID                               ="+raUID+"\n"+
            "\tbrokerType                          ="+brokerType+"\n"+
            "\tbrokerInstanceName                  ="+brokerInstanceName+"\n"+
            "\tbrokerBindAddress                   ="+brokerBindAddress+"\n"+
            "\tbrokerPort                          ="+brokerPort+"\n"+
            "\tbrokerHomeDir                       ="+brokerHomeDir+"\n"+
            "\tbrokerVarDir                        ="+brokerVarDir+"\n"+
            "\tbrokerJavaDir                       ="+brokerJavaDir+"\n"+
            "\tbrokerArgs                          ="+brokerArgs+"\n"+
            "\tbrokerStartTimeout                  ="+brokerStartTimeout+"\n"+
            "\tadminUsername                       ="+adminUsername+"\n"+
            "\tadminPassFile                       ="+adminPassFile+"\n"+
            "\tuseJNDIRmiServiceURL                ="+useJNDIRmiServiceURL+"\n"+
            "\trmiRegistryPort                     ="+rmiRegistryPort+"\n"+
            "\tstartRmiRegistry                    ="+startRmiRegistry+"\n"+
            "\tuseSSLJMXConnector                  ="+useSSLJMXConnector+"\n"+
            "\tbrokerEnableHA                      ="+brokerEnableHA+"\n"+
            "\tclusterId                           ="+clusterId+"\n"+
            "\tbrokerId                            ="+brokerId+"\n"+
            "\tjmxServiceURL                       ="+jmxServiceURL+"\n"+
            "\tdbType                              ="+dbType+"\n"+
            "\tdbProps                             ="+(dbProps != null ? dbProps.toString() : "null") +"\n"+
            "\tdsProps                             ="+(dsProps != null ? dsProps.toString() : "null") +"\n"+
            "\tConnectionURL                       ="+connectionURL+"\n"+
            "\tUserName                            ="+userName+"\n"+
            "\tReconnectEnabled                    ="+reconnectEnabled+"\n"+
            "\tReconnectInterval                   ="+reconnectInterval+"\n"+
            "\tReconnectAttempts                   ="+reconnectAttempts+"\n"+
            "\tAddressListBehavior                 ="+addressListBehavior+"\n"+
            "\tAddressListIterations               ="+addressListIterations+"\n"+
            "\tInAppClientContainer                ="+inAppClientContainer+"\n"+
            "\tInClusteredContainer                ="+inClusteredContainer+"\n"+
            //"\thaRequired                          ="+haRequired+"\n"+
            "\tGroupName                           ="+groupName+"\n");
    }

    public static com.sun.messaging.jms.ra.ConnectionManager
    _getConnectionManager()
    {
        return _cm;
    }

    /**
    * Validates a string to be a proper Java Identifier
    *
    * @param name The string to be validated.
    *   
    * @return <code>true</code> if the name is valid;
    *         <code>false</code> if the name is invalid.
    */  
    private static final boolean isNameValidIdentifier(String name) {
        //Invalid if name is null or empty.
        if (name == null || "".equals(name)) {
            return false;
        }
        //Verify identifier start character and part
        char[] namechars = name.toCharArray();
        if (Character.isJavaIdentifierStart(namechars[0])) {
            for (int i = 1; i<namechars.length; i++) {
                //isValid = Character.isLetterOrDigit(namechars[i]); //str.charAt(i));
                if (!Character.isJavaIdentifierPart(namechars[i])) {
                    //Invalid if body characters are not valid using isJavaIdentifierPart().
                    return false;
                }
            }  
        } else {
            //Invalid if first character is not valid using isJavaIdentifierStart().
            return false;
        }
        return true;
    }

    /**
    * Validates a string to be a valid name (alphanumeric + '_' only)
    *
    * @param name The string to be validated.
    *   
    * @return <code>true</code> if the name is valid;
    *         <code>false</code> if the name is invalid.
    */  
    private static final boolean isNameValidAlphaNumeric_(String name) {
        //Invalid if name is null or empty.
        if (name == null || "".equals(name)) {
            return false;
        }
        char[] namechars = name.toCharArray();
        for (int i = 0; i<namechars.length; i++) {
            if (!(Character.isLetterOrDigit(namechars[i])) && (namechars[i] != '_')) {
                return false;
            }
        }  
        return true;
    }

    private Properties _getEffectiveBrokerProps()
    {
        Properties props = new Properties();

        if (brokerEnableHA) {
            props.setProperty("imq.cluster.ha", "true");
            if (clusterId != null) props.setProperty("imq.cluster.clusterid", clusterId);
        } else {
            if (connectionURL != null && !("".equals(connectionURL)) ) {
                props.setProperty("imq.cluster.brokerlist", connectionURL);
            }
            if (masterBroker != null) {
                props.setProperty("imq.cluster.masterbroker", masterBroker);
            }
        }

        if (dbType != null) {
            props.setProperty("imq.persist.store", "jdbc");

            if (DB_TYPE_HADB.equals(dbType)) {
                props.setProperty(IMQ_JDBC_VENDOR, dbType);
                props.setProperty(IMQ_BROKERID, brokerId);
                if (dbProps.containsKey(HADB_USER)) {
                    props.setProperty(IMQ_HADB_USER, dbProps.getProperty(HADB_USER));
                }
                if (dbProps.containsKey(HADB_PASSWORD)) {
                    props.setProperty(IMQ_HADB_PASSWORD, dbProps.getProperty(HADB_PASSWORD));
                }

                if (dsProps.containsKey(HADB_SERVERLIST)) {
                    props.setProperty(IMQ_HADB_DSPROP_SERVERLIST, dsProps.getProperty(HADB_SERVERLIST));
                }
            }

        }
        return props;
    }
}

