/*
 * The contents of this file are subject to the JOnAS Public License Version
 * 1.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License on the JOnAS web site.
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied.
 * See the License for the specific terms governing rights and limitations under
 * the License.
 *
 * The Original Code is JOnAS application server code released July 1999.
 *
 * The Initial Developer of the Original Code is Bull S.A.
 * The Original Code and portions created by Bull S.A. are
 *    Copyright (C) 1999 Bull S.A. All Rights Reserved.
 *
 * Contributor(s): ______________________________________.
 *
 * --------------------------------------------------------------------------
 * $Id: TimerManager.java,v 1.3 2002/01/11 08:19:44 jonas Exp $
 * --------------------------------------------------------------------------
 */


package org.objectweb.jonas_timer;

import java.util.Vector;
import java.util.Enumeration;

/**
 * Clock thread for a TimerManager
 * Every second, decrement timers and launch action if expired
 */
class Clock extends Thread {

    private TimerManager tmgr;

    public Clock(TimerManager tmgr) {
	super("JonasClock");
	TraceTimer.debug("Clock constructor");
	this.tmgr = tmgr;
    }

    public void run() {
	tmgr.clock();
    }
}

/**
 * Batch thread for a TimerManager
 * pocess all expired timers
 */
class Batch extends Thread {

    private TimerManager tmgr;

    public Batch(TimerManager tmgr) {
	super("JonasBatch");
	TraceTimer.debug("Batch constructor");
	this.tmgr = tmgr;
    }

    public void run() {
	tmgr.batch();
    }
}

/**
 * A timer manager manages 2 lists of timers with 2 threads
 * One thread is a clock which decrements timers every second
 * and passes them when expired in a list of expired timers.
 * The other thread looks in the list of expired timers to process
 * them.
 */
public class TimerManager {

    // threads managing the service.
    private static Batch batchThread;
    private static Clock clockThread;

    // lists
    private Vector timerList = new Vector();
    private Vector expiredList = new Vector();

    private static TimerManager unique = null;
    private static boolean shuttingdown = false;

    /**
     * Constructor
     */
    private TimerManager() {
	// launch threads for timers
	batchThread = new Batch(this);
        batchThread.setDaemon(true);
	batchThread.start();
	clockThread = new Clock(this);
        clockThread.setDaemon(true);
	clockThread.start();
    }

    /**
     * Get an instance of the TimerManager
     */
    public static TimerManager getInstance() {
	if (unique == null)
	    unique = new TimerManager();
	return unique;
    }

    /**
     * stop the service
     * @param force tell the manager NOT to wait for the timers to be completed
     */
    public static void stop(boolean force) {
	TraceTimer.debug("Stop TimerManager");
	TimerManager tmgr = getInstance();
	shuttingdown = true;
	while (clockThread.isAlive() || batchThread.isAlive()) {
	    try {
		Thread.sleep(100);
	    } catch (InterruptedException e) {
		break;
	    }
	}
	TraceTimer.debug("TimerManager has stopped");
    }

    public static void stop() {
        stop(true);
    }

    /**
     * cney speed up the clock x1000 when shutting down
     * update all timers in the list
     * each timer expired is put in a special list of expired timers
     * they will be processed then by the Batch Thread.
     */
    public void clock() {
	while (true) {
	    try {
		Thread.currentThread().sleep(shuttingdown?1:1000);	// 1 second or 1ms shen shuttingdown
		synchronized(timerList) {
		    int found = 0;
		    boolean empty = true;
		    for (int i = 0; i < timerList.size(); i++) {
			TimerEvent t = (TimerEvent) timerList.elementAt(i);
			if (!t.isStopped()) {
			    empty = false;
			}
			if (t.update() <= 0) {
			    timerList.removeElementAt(i--);
			    if (t.valid()) {
				expiredList.addElement(t);
				found++;
				if (t.ispermanent() && !shuttingdown) {
				    t.restart();
				    timerList.addElement(t);
				}
			    }
			}
			// Be sure there is no more ref on bean in this local variable.
			t = null;
		    }
		    if (found > 0) {
			timerList.notify();
		    } else {
			if (empty && shuttingdown) {
			    break;
			}
		    }
		}
	    } catch (InterruptedException e) {
		TraceTimer.error("Timer interrupted");
	    }
	}
	synchronized(timerList) { // notify batch so that function can return.
	    timerList.notify();
	}
    }

    /**
     * process all expired timers
     */
    public void batch() {

	while (!(shuttingdown && timerList.isEmpty() && expiredList.isEmpty())) {
	    TimerEvent t;
	    synchronized(timerList) {
		while (expiredList.isEmpty()) {
		    if (shuttingdown) return;
		    try {
			timerList.wait();
		    } catch (Exception e) {
			TraceTimer.error("Exception in Batch: ",e);
		    }
		}
		t = (TimerEvent) expiredList.elementAt(0);
		expiredList.removeElementAt(0);
	    }
	    // Do not keep the lock during the processing of the timer
	    t.process();
	}
    }

    /**
     * add a new timer in the list
     */
    public TimerEvent addTimer(TimerEventListener tel, long timeout, Object arg, boolean permanent) {
	TimerEvent te = new TimerEvent(tel, timeout, arg, permanent);
	synchronized(timerList) {
	    timerList.addElement(te);
	}
	return te;
    }

    /**
     * remove a timer from the list. this is not very efficient.
     * A better way to do this is TimerEvent.unset()
     * @deprecated
     */
    public void removeTimer(TimerEvent te) {
	synchronized(timerList) {
	    timerList.removeElement(te);
	}
    }
}
