#ifndef METHOD_H__
#define METHOD_H__

#include <jni.h>
#include <jvmpi.h>
#include <jmp.h>
#include <methodtime.h>

/** The structure used to represent a method. */
struct method {
    char* method_name;        /** The name of the method "toLowerCase" */
    char* method_signature;   /** The method signature "()V" */
    char* jmpname;            /** The human readable method name "int size ()" */
    jint start_lineno;        /** The start line of the method. */
    jint end_lineno;          /** The end line of the method. */
    jmethodID method_id;      /** The id of the method. */

    methodtime time_used;     /** The time this method has used. */
    methodtime restore_time;  /** The time this method has used (before the reset). */
    long calls;               /** The number of calls to this method. */
    long restore_calls;       /** The number of calls to this method before the reset. */
    cls* owner;               /** The class that is owner of this method. */
    mvector* called_methods;  /** The methods that this method has called. */

    int allocated_objects;    /** How many objects were created in this method */
    int restore_objects;      /** The number of alloced objects before the reset. */ 
    jlong allocated_memory;   /** How many bytes were consumed for allocating objects */
    jlong restore_memory;     /** The number of alloced bytes before the reset. */

    int modified;              /** Hack to avoid updating ui to much. */
};

/** Create a new method structure with the given attributes. */
method* method_new (const char* method_name, char* method_signature, 
		    jint start_lineno, jint end_lineno,
		    jmethodID method_id, cls* owner);

/** Delete the given method structure. */
void method_free (method* m);

/** This is the hash function for method structures. */
size_t method_jmphash_func (void* c, size_t len);

/** Check if two methods are equal. 
 *  return 0 if c1 == c2.
 *  return -1 if c1 < c2.
 *  return 1 if c1 > c2
 */
int method_cmp_func (void* c1, void* c2);

/** Print some statistics on the given class to stdout. 
 */
void method_print (method* c);

/** Set the method id of the given method. */
inline void method_set_method_id (method* m, jmethodID method_id);

/** Get the method id from the given method. */
inline jmethodID method_get_method_id (const method* m);

/** Get the method name from the given method. */
inline char* method_get_method_name (method* m);

/** Get the method signature from the given method. */
inline char* method_get_method_signature (method* m);

/** Get the human readable name and signature. */
inline char* method_get_method_jmpname (method* m);

/** Get the line number this method starts on. */
inline jint method_get_method_start_line_number (method* m);

/** Get the line number this method ends on. */
inline jint method_get_method_end_line_number (method* m);

/** Get the class id from the given method. */
inline jobjectID method_get_class_id (method* m);

/** Get the number of calls to the method. */
inline long method_get_calls (method* m);

/** Get the number of calls to the method. */
inline long method_get_restore_calls (method* m);

/** Get the time used by the given method. */
inline methodtime* method_get_time_used (method* m);

/** Get the time used by the given method. */
inline methodtime* method_get_restore_time (method* m);

/** Get the owner of this method. */
inline cls* method_get_owner (method* m);

/** Reset counters to zero. */
inline void method_reset_count (method* m);

/** Restore counters to real values. */
inline void method_restore_count (method* m);

/** Check if this method has changed since last call. */
inline int method_check_modified (method* m);

/** Set updated status of this method. */
inline void method_set_modified (method* m, int flag);

/** Returns the number of objects allocated by this method */
inline int method_get_allocated_objects (method* m);

/** Returns the number of bytes used for allocating objects by this method */
inline jlong method_get_allocated_memory (method* m);

#endif /* METHOD_H__ */

/* Emacs Local Variables: */
/* Emacs mode:C */
/* Emacs c-indentation-style:"gnu" */
/* Emacs c-hanging-braces-alist:((brace-list-open)(brace-entry-open)(defun-open after)(substatement-open after)(block-close . c-snug-do-while)(extern-lang-open after)) */
/* Emacs c-cleanup-list:(brace-else-brace brace-elseif-brace space-before-funcall) */
/* Emacs c-basic-offset:4 */
/* Emacs End: */
