#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <jmp-config.h>
#include <jmp.h>
#include <ui.h>
#include <comparators.h>
#include <dumper.h>

/** This file implements the ui interface. 
 *  The ui provided by this file is: none, dumping to file works.
 *
 *  Some methods are marked with "This method does nothing." 
 *  Thoose methods are only here to enable the no-ui to link correctly.
 */

static int quit = 0;
/** id of last dump file... */
static int dump_id = -1;

/** Are we dumping on a timer basis? */
static long  dumptimer = 0;   /* The countdown value (10)... */
static long  dumptime = 0;    /* Current value (1, 2, 3, 4,....) */

/** our current class comparator */
static int (*cls_compr) (const void* v1, const void* v2) = cls_compr_size;
/** The current thread comparator */
static int (*jthread_compr) (const void* v1, const void* v2) = jthread_compr_name;
/** The current method comparator */
static int (*method_compr) (const void* v1, const void* v2) = method_compr_time;

static cls** classlist = NULL;
static jthread** threadslist = NULL;
static method** methodlist = NULL;
static int classes_count = 0;
static int threads_count = 0;
static int methods_count = 0;

int run_updates (void* data) {
    if (dumptimer > 0) {
	dumptime++;
	if (dumptime == dumptimer) {
	    run_data_dump ();
	    dumptime = 0;
	}
    }
    return 1;
}

void init_ui (void) {
    /* do nothing. */
}

void gtkthread (void* data) {
    while (!quit) {
	sleep (1);
	run_updates (NULL);
    }
    ui_shutdown_complete ();
}

/** Tell the ui system to shut itself down. */
void quit_ui (void) {
    quit = 1;
}

/** Update the statistics. */
void updateUI (hashtab* cls, hashtab* methods) {
}

/** Count the number of threads that are currently active */
static void count_threads (void* data) {
    threads_count++;
}

/* Add a thread to the array of threads. */
static void add_threads_row (void* data) {
    jthread* t = (jthread*)data;
    threadslist[threads_count++] = t;
}

/** Count the number of classes that have more than zero instances. */
static void count_classes (void* data, cls_usage* cu) {
    cls* c = (cls*)data;
    if (cls_get_instances (c) != 0)
	classes_count++;
}

/** Add a row to the array of data. */
static void add_class_row (void* data) {
    cls* c = (cls*)data;    
    if (cls_get_instances (c) != 0)
	classlist[classes_count++] = c;
}

/** Count the number of methodods that have used any time. */
static void count_methods (void* data) {
    method* m = (method*)data;
    if (!methodtime_is_zero (&m->time_used))
	methods_count++;
}

/** Add a row to the array of data. */
static void add_methods_row (void* data) {
    method* m = (method*)data;
    if (!methodtime_is_zero (&m->time_used))
	methodlist[methods_count++] = m;
}

/** Return the id of the dump file. */
int dump_data (char* dumpfile, hashtab* classes, hashtab* methods, hashtab* threads) {
    FILE* f;
    int i;
    char filename[128];
    
    snprintf (filename, 128, "%s-%d.txt", 
	      (dumpfile == NULL ? "jmp_dump" : dumpfile), ++dump_id);    
    f = fopen (filename, "w");
    if (f == NULL)
	return -1;

    fprintf (f, "Threads\n");
    fprintf (f, "thread\tgroup\tparent\tclass\tmethod\n");
    fprintf (f, "--------------------------------------------------------------\n");
    threads_count = 0;
    jmphash_for_each ((jmphash_iter_f)count_threads, threads);
    if (threads_count) {
	threadslist = malloc (threads_count * sizeof(jthread*));
	if (threadslist) {
	    threads_count = 0;
	    jmphash_for_each ((jmphash_iter_f)add_threads_row, threads);
	    qsort(threadslist, threads_count, sizeof(jthread*), jthread_compr);
	    for (i = 0; i < threads_count; i++)
		dump_jthread_row (threadslist[i], f);
	    free (threadslist);
	}
    }

    fprintf (f, "\n\n\nClassdump\n");
    fprintf (f, "class_name\t#instaces\tmax #instances\tsize\t#GC\ttenure\n");
    fprintf (f, "--------------------------------------------------------------\n");

    classes_count = 0;
    jmphash_for_each ((jmphash_iter_f)count_classes, classes); 
    classlist = malloc (classes_count * sizeof (cls*));
    if (classlist) {
	classes_count = 0;
	jmphash_for_each ((jmphash_iter_f)add_class_row, classes); 
	qsort (classlist, classes_count, sizeof (cls*), cls_compr);
	for (i = 0; i < classes_count; i++) 
	    dump_class_row (classlist[i], f);
	free (classlist);
    }

    fprintf (f, "\n\n\nMethods\n");
    fprintf (f, "class name\tmethod\tsec\tcalls\tsubsec\t#alloced instances\t#alloced bytes\n");
    fprintf (f, "--------------------------------------------------------------\n");
    methods_count = 0;
    jmphash_for_each ((jmphash_iter_f)count_methods, methods);
    methodlist = malloc (methods_count * sizeof (method*));
    if (methodlist) {
	methods_count = 0;
	jmphash_for_each ((jmphash_iter_f)add_methods_row, methods);
	qsort (methodlist, methods_count, sizeof (method*), method_compr);
	for (i = 0; i < methods_count; i++) 
	    dump_method_row (methodlist[i], f);
    }
    free (methodlist);

    fflush (f);
    fclose (f);
    return dump_id;
}

/** Set the dump timer... */
void set_dump_timer (int dt) {
    dumptimer = dt;
}

/** Get the current dump timer. */
int get_dump_timer () {
    return dumptimer;
}

/** Set the status text... */
void set_status (char* text) {
}

/** Set the status text... */
void set_status_lock (char* text) {
}

/** methods used to show a window with owners to objects of the given type.
 *  This method does nothing.
 * @param objects the hash table of objects to look in.
 * @param c the class of objects to show instance owners for.
 */
void show_instance_owners (hashtab* objects, cls* c) {
}

/** method used to show a window with the owners to the given object. 
 *  This method does nothing.
 * @param objects the hash table of objects to look in.
 * @param obj the object to show instance owners for.
 */
void show_owner_for_object (hashtab* objects, obj* o) {
}

/** method used to show owner statistics.
 *  This method does nothing.
 * @param objects the hash table of objects to look in.
 * @param c the class of objects to show instance owners for.
 */
void show_owners_statistics (hashtab* objects, cls* c) {
}

/** Show the objects that are alloced by the given method.
 *  This method does nothing.
 */
void show_objects_alloced_by_method (hashtab* objects, method* m) {
}

/** Show the objects that are alloced by the given method. 
 *  This method does nothing.
*/
void show_objects_alloced_by_class (hashtab* objects, cls* c) {
}

/** Check if we have any ui-handling to do. */
int events_pending (void) {
    return 0;
}

/** Update the ui-toolkit (gtk_main_iteration ()) */
int ui_iteration (void) {
    return 0;
}

void show_deadlock (visited_threads* vt) {
    while (vt != NULL) {
	fprintf (stdout, "%s holding %s (%p)", 
		 jthread_get_thread_name (vt->mi->owner), 
		 vt->mi->name, vt->mi->id); 
	if (vt->next) 
	    fprintf (stdout, ", trying to enter %s (%p)\n", 
		     vt->next->mi->name, vt->next->mi->id);
	else 
	    fprintf (stdout, "\n");
	vt = vt->next;
    }
}

/* Emacs Local Variables: */
/* Emacs mode:C */
/* Emacs c-indentation-style:"gnu" */
/* Emacs c-hanging-braces-alist:((brace-list-open)(brace-entry-open)(defun-open after)(substatement-open after)(block-close . c-snug-do-while)(extern-lang-open after)) */
/* Emacs c-cleanup-list:(brace-else-brace brace-elseif-brace space-before-funcall) */
/* Emacs c-basic-offset:4 */
/* Emacs End: */
