/*********************************************************************
 *
 *      Copyright (C) 2002-2004 Nathan Fiedler
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 * $Id: EvaluateAction.java 1113 2004-01-25 22:25:32Z nfiedler $
 *
 ********************************************************************/

package com.bluemarsh.jswat.action;

import com.bluemarsh.jswat.ContextManager;
import com.bluemarsh.jswat.Session;
import com.bluemarsh.jswat.expr.EvaluationException;
import com.bluemarsh.jswat.expr.Evaluator;
import com.bluemarsh.jswat.util.Strings;
import com.bluemarsh.jswat.util.Variables;
import com.sun.jdi.ThreadReference;
import com.sun.jdi.Value;
import java.awt.Container;
import java.awt.Frame;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.Insets;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import javax.swing.JButton;
import javax.swing.JDialog;
import javax.swing.JLabel;
import javax.swing.JScrollPane;
import javax.swing.JTextArea;
import javax.swing.JTextField;
import javax.swing.WindowConstants;

/**
 * Implements the evaluate action used to evaluate Java expressions.
 *
 * @author  Nathan Fiedler
 */
public class EvaluateAction extends JSwatAction {

    /**
     * Creates a new EvaluateAction object with the default action
     * command string of "evaluate".
     */
    public EvaluateAction() {
        super("evaluate");
    } // EvaluateAction

    /**
     * Performs the evaluate action.
     *
     * @param  event  action event
     */
    public void actionPerformed(ActionEvent event) {
        Frame topFrame = getFrame(event);
        JDialog dialog = new EvaluateDialog(topFrame);
        dialog.setDefaultCloseOperation(WindowConstants.DISPOSE_ON_CLOSE);
        dialog.pack();
        dialog.setLocationRelativeTo(topFrame);
        dialog.setVisible(true);
    } // actionPerformed
} // EvaluateAction

/**
 * Class EvaluateDialog is a dialog for evaluating expressions.
 *
 * @author  Nathan Fiedler
 */
class EvaluateDialog extends JDialog implements ActionListener {
    /** Text field to receive expression. */
    private JTextField textField;
    /** Text area to show evaluation results. */
    private JTextArea textArea;
    /** Button to close the dialog. */
    private JButton closeButton;

    /**
     * Constructs a EvaluateDialog with the given parent frame.
     *
     * @param  topFrame  parent frame.
     */
    public EvaluateDialog(Frame topFrame) {
        super(topFrame, Bundle.getString("Evaluate.title"));
        Container pane = getContentPane();
        GridBagLayout gbl = new GridBagLayout();
        pane.setLayout(gbl);
        GridBagConstraints gbc = new GridBagConstraints();
        gbc.insets = new Insets(3, 3, 3, 3);

        JLabel label = new JLabel(Bundle.getString("Evaluate.enterExpr"));
        gbc.anchor = GridBagConstraints.WEST;
        gbc.gridwidth = GridBagConstraints.REMAINDER;
        gbl.setConstraints(label, gbc);
        pane.add(label);

        textField = new JTextField(30);
        textField.addActionListener(this);
        gbc.gridwidth = GridBagConstraints.RELATIVE;
        gbl.setConstraints(textField, gbc);
        pane.add(textField);

        JButton button = new JButton(Bundle.getString("Evaluate.evalLabel"));
        button.addActionListener(this);
        gbc.gridwidth = GridBagConstraints.REMAINDER;
        gbl.setConstraints(button, gbc);
        pane.add(button);

        label = new JLabel(Bundle.getString("Evaluate.result"));
        gbl.setConstraints(label, gbc);
        pane.add(label);

        textArea = new JTextArea(10, 30);
        textArea.setEditable(false);
        JScrollPane scroller = new JScrollPane(textArea);
        gbc.fill = GridBagConstraints.HORIZONTAL;
        gbl.setConstraints(scroller, gbc);
        pane.add(scroller);

        closeButton = new JButton(Bundle.getString("closeButton"));
        closeButton.addActionListener(this);
        gbc.anchor = GridBagConstraints.CENTER;
        gbc.fill = GridBagConstraints.NONE;
        gbc.gridheight = GridBagConstraints.REMAINDER;
        gbl.setConstraints(closeButton, gbc);
        pane.add(closeButton);
    } // EvaluateDialog

    /**
     * Performs the evaluate action.
     *
     * @param  event  action event
     */
    public void actionPerformed(ActionEvent event) {
        Object source = event.getSource();
        if (source == closeButton) {
            dispose();
            return;
        }

        // Get the current thread.
        Session session = JSwatAction.getSession(event);
        ContextManager ctxtman = (ContextManager)
            session.getManager(ContextManager.class);
        ThreadReference thread = ctxtman.getCurrentThread();
        int frame = ctxtman.getCurrentFrame();

        String expr = textField.getText();
        if (expr.length() == 0) {
            textArea.setText("");
            return;
        }

        Evaluator eval = new Evaluator(expr);
        String s;
        try {
            Object o = eval.evaluate(thread, frame);
            if (o instanceof Value) {
                Value v = (Value) o;
                s = Variables.printValue(v, thread, "\n");
            } else {
                if (o == null) {
                    s = "null";
                } else {
                    s = o.toString();
                }
            }
        } catch (EvaluationException ee) {
            s = Bundle.getString("Evaluate.error") + ' '
                + Strings.exceptionToString(ee);
        } catch (Exception e) {
            s = Bundle.getString("Evaluate.error") + ' '
                + Strings.exceptionToString(e);
        }

        textArea.setText(s);
    } // actionPerformed
} // EvaluateDialog
