// avcodec_sample.0.4.9.cpp

// A small sample program that shows how to use libavformat and libavcodec to
// read video from a file.
//
// This version is for the 0.4.9-pre1 release of ffmpeg. This release adds the
// av_read_frame() API call, which simplifies the reading of video frames
// considerably.
//
// Use
//
// g++ -o avcodec_sample.0.4.9 avcodec_sample.0.4.9.cpp -lavformat
//     -lavcodec -lz
//
// to build (assuming libavformat and libavcodec are correctly installed on
// your system).
//
// Run using
//
// avcodec_sample.0.4.9 myvideofile.mpg
//
// to write the first five frames from "myvideofile.mpg" to disk in PPM
// format.

#include "qffmpeg.h"
#include "qffmpegconverter.h"
#include "qffmpeglogger.h"
#include "libavformat/avformat.h"
#include "libavcodec/avcodec.h"
#include <qfileinfo.h>
#include <qregexp.h>
#include <kdebug.h>

QAVImage::QAVImage(AVFrame* frame, int width, int height)
  : QImage(width, height, 32)
{
  if(!frame)
    return;
  unsigned char* data = frame->data[0];
  long i = 0;

  setAlphaBuffer(true);
  for (int y = 0; y < height; y++)
  {
    for (int x = 0; x < width; x++)
    {
      int r = (int)data[i++];
      int g = (int)data[i++];
      int b = (int)data[i++];
      setPixel(x, y, qRgba(r, g, b, 255));
    }
  }
}

int64_t QAVTime::toAVTime() const
{
  return (uint64_t)((
      (double)hour() *  (60.0*60.0) +
      (double)minute() *  60.0 +
      (double)second() +
      (double)msec() / 1000.0 ) * AV_TIME_BASE);
}

void QAVTime::set(int64_t avTime)
{
  int64_t seconds = avTime / AV_TIME_BASE;
  int64_t ms = (((avTime % AV_TIME_BASE) * 1000) / AV_TIME_BASE);
  setHMS(seconds / (60*60), (seconds / 60) % 60, seconds % 60, ms);
}

QFFMpegFile::QFFMpegFile(const QString& filename)
{
  reset();
  if(!filename.isEmpty())
    open(filename);
}

QFFMpegFile::~QFFMpegFile()
{
  freeAVData();
}

void QFFMpegFile::reset()
{
  m_filename = "";
  m_pFrameRGB = 0;
  m_pFrame = 0;
  m_buffer = 0;
  m_pFormatCtx = 0;
  m_pCodecCtx = 0;
  m_videoStream = -1;
}

void QFFMpegFile::freeAVData()
{
  // Free the RGB image
  if(m_buffer)
    delete [] m_buffer;
  if(m_pFrameRGB)
    av_free(m_pFrameRGB);

  // Free the YUV frame
  if(m_pFrame)
    av_free(m_pFrame);

  // Close the video file
  if(m_pFormatCtx)
    av_close_input_file(m_pFormatCtx);

  reset();
}

const AVInputFormat* QFFMpegFile::fileFormat() const
{
  return (m_pFormatCtx)?m_pFormatCtx->iformat:0;
}

uint QFFMpegFile::audioSampleRate() const
{
  if(hasAudio())
  {
    AVCodecContext* pCodecCtx =
        m_pFormatCtx->streams[m_audioStreams[0]]->codec;
    return (pCodecCtx)?pCodecCtx->sample_rate:0;
  }
  else
    return 0;
}

int64_t QFFMpegFile::avDuration() const
{
  if(m_pFormatCtx)
  {
    if(m_pFormatCtx->duration < 0)
      return 0;
    return m_pFormatCtx->duration;
  }
  return 0;
}

QTime QFFMpegFile::duration() const
{
  QAVTime result;
  result.set(avDuration());
  return result;
}

bool QFFMpegFile::open(const QString& filename)
{
  // Free previous file
  freeAVData();

  m_filename = filename;
  if(filename.isEmpty())
    return false;

  int i;
  AVCodec* pCodec;
  int numBytes;
  //AVFormatParameters params;
  //AVInputFormat* file_iformat;

  //memset(&params, 0, sizeof(params));

  // Open video file
  //file_iformat = av_find_input_format(filename);
  //if(av_open_input_file(&m_pFormatCtx, filename, file_iformat, 0, &params)!=0)
  if(av_open_input_file(&m_pFormatCtx, (const char*)filename.local8Bit(), 
     NULL, 0, NULL) != 0)
  {
    kdDebug() << "Couldn't open file." << endl;
    return false;
  }

  // Retrieve stream information
  if(av_find_stream_info(m_pFormatCtx) < 0)
  {
    kdDebug() << "Couldn't find stream information." << endl;
    return false;
  }

  // Dump information about file onto standard error
  //dump_format(m_pFormatCtx, 0, filename, false);

  // Find the first video stream
  m_videoStream = -1;
  m_audioStreams.clear();
  for(i=0; i<m_pFormatCtx->nb_streams; i++)
  {
    if(m_pFormatCtx->streams[i]->codec->codec_type == CODEC_TYPE_VIDEO)
      m_videoStream = i;
    else if(m_pFormatCtx->streams[i]->codec->codec_type == CODEC_TYPE_AUDIO)
      m_audioStreams.append(i);
  }
  if(m_videoStream == -1 && m_audioStreams.count() == 0)
  {
    kdDebug() << "Didn't find a stream." << endl;
    return false;
  }

  if(m_videoStream != -1)
  {
    // Get a pointer to the codec context for the video stream
    m_pCodecCtx = m_pFormatCtx->streams[m_videoStream]->codec;

    // Find the decoder for the video stream
    pCodec=avcodec_find_decoder(m_pCodecCtx->codec_id);
    if(pCodec==NULL)
    {
      kdDebug() << "Codec not found." << endl;
      return false;
    }

    // Open codec
    if(avcodec_open(m_pCodecCtx, pCodec) < 0)
    {
      kdDebug() << "Could not open codec." << endl;
      return false;
    }

    // Hack to correct wrong frame rates that seem to be generated by some
    // codecs
    if(m_pCodecCtx->time_base.den>1000 && m_pCodecCtx->time_base.num==1)
      m_pCodecCtx->time_base.den=1000;

    // Allocate video frame
    m_pFrame=avcodec_alloc_frame();

    // Allocate an AVFrame structure
    m_pFrameRGB=avcodec_alloc_frame();
    if(m_pFrameRGB==NULL)
    {
      kdDebug() << "RGB frame allocation failed." << endl;
      return false;
    }

    // Determine required buffer size and allocate buffer
    numBytes=avpicture_get_size(PIX_FMT_RGB24, m_pCodecCtx->width,
                                m_pCodecCtx->height);
    m_buffer=new uint8_t[numBytes];

    // Assign appropriate parts of buffer to image planes in pFrameRGB
    avpicture_fill((AVPicture *)m_pFrameRGB, m_buffer, PIX_FMT_RGB24,
                    m_pCodecCtx->width, m_pCodecCtx->height);
  }
  return true;
}

AVFrame* QFFMpegFile::frame() const
{
  AVPacket packet;
  int frameFinished;

  if(!m_pFormatCtx || !m_pFrame || !hasVideo())
    return 0;

  while(av_read_frame(m_pFormatCtx, &packet) >= 0)
  {
    // Is this a packet from the video stream?
    if(packet.stream_index == m_videoStream)
    {
      // Decode video frame
      avcodec_decode_video(m_pCodecCtx, m_pFrame, &frameFinished,
                           packet.data, packet.size);

      // Did we get a video frame?
      if(frameFinished)
      {
        // Convert the image from its native format to RGB
        img_convert((AVPicture*)m_pFrameRGB, PIX_FMT_RGB24,
                    (AVPicture*)m_pFrame, m_pCodecCtx->pix_fmt,
                    m_pCodecCtx->width, m_pCodecCtx->height);
        return m_pFrameRGB;
      }
    }
    // Free the packet that was allocated by av_read_frame
    av_free_packet(&packet);
  }
  return 0;
}

bool QFFMpegFile::seek(int64_t timestamp, bool backwards)
{
  if(!m_pFormatCtx || !m_pFrame)
    return false;

  int64_t fl = frameLength();
  if(timestamp > m_pos - fl && timestamp < m_pos + fl)
  {
    m_pos = timestamp + fl;
    return true;
  }

  int seek_flags = backwards ? AVSEEK_FLAG_BACKWARD : 0;
  int rv = av_seek_frame(m_pFormatCtx, -1,
                         m_pFormatCtx->start_time + timestamp,
                         seek_flags);
  if (rv >= 0)
  {
    m_pos = timestamp + fl;
    if (m_videoStream >= 0)
    {
      //packet_queue_flush(&is->videoq);
      avcodec_flush_buffers(m_pFormatCtx->streams[m_videoStream]->codec);
    }
  }
  return (rv >= 0);
}

int QFFMpegFile::width() const
{
  if(m_videoStream >= 0)
    return m_pFormatCtx->streams[m_videoStream]->codec->width;
  return 0;
}

int QFFMpegFile::height() const
{
  if(m_videoStream >= 0)
    return m_pFormatCtx->streams[m_videoStream]->codec->height;
  return 0;
}

int QFFMpegFile::aspectRatio() const
{
  if(m_videoStream >= 0)
  {
    AVCodecContext* enc = m_pFormatCtx->streams[m_videoStream]->codec;
    double ratio = av_q2d(enc->sample_aspect_ratio) * enc->width / enc->height;
    if(ratio > 1.5)
      return 1; // 16:9
  }
  return 0; // 4:3
}

void QFFMpegFile::SaveFrame(AVFrame *pFrame, int width, int height) const
{
  FILE *pFile;
  char szFilename[32];
  int  y;
  static int iFrame = 0;

    // Open file
  snprintf(szFilename, 32, "/tmp/frame%d.ppm", iFrame);
  pFile=fopen(szFilename, "wb");
  if(pFile==NULL)
    return;

    // Write header
  fprintf(pFile, "P6 %d %d 255\n", width, height);

    // Write pixel data
  for(y=0; y<height; y++)
    fwrite(pFrame->data[0]+y*pFrame->linesize[0], 1, width*3, pFile);

    // Close file
  fclose(pFile);
  ++iFrame;
}

double QFFMpegFile::frameRate() const
{
  if(m_pFormatCtx && m_videoStream != -1)
    return (double)m_pFormatCtx->streams[m_videoStream]->codec->time_base.den /
        (double)m_pFormatCtx->streams[m_videoStream]->codec->time_base.num;
  else
    return 0.0;
}

int64_t QFFMpegFile::frameLength() const
{
  double fr = 1.0 / frameRate();
  return (int64_t)(fr * AV_TIME_BASE);
}

bool QFFMpegFile::isDVDCompatible() const
{
  const AVInputFormat* format = fileFormat();
  if(format)
  {
    if(hasVideo())
    {
      //kdDebug() << k_funcinfo << format->long_name << endl;
      if(strcmp(format->long_name, "MPEG PS format") != 0)
        return false;
      // TODO: Check resolution
    }
    else if(hasAudio())
    {
      if(!format || strcmp(format->long_name, "MPEG audio") != 0)
        return false;
      if(audioSampleRate() != 48000)
        return false;
    }
  }
  return true;
}

QFFMpeg::QFFMpeg() : m_ff(0)
{
  // Register all formats and codecs
  av_register_all();
  connect(QFFMpegLogger::self(), SIGNAL(message(const QString&)),
          this, SLOT(messageSlot(const QString&)));
}

QFFMpeg::QFFMpeg(QString file) : m_ff(0)
{
  // Register all formats and codecs
  av_register_all();
  connect(QFFMpegLogger::self(), SIGNAL(message(const QString&)),
          this, SLOT(messageSlot(const QString&)));
  addFile(file);
}

QFFMpeg::~QFFMpeg()
{
  clear();
}

void QFFMpeg::messageSlot(const QString& s)
{
  if(s.startsWith("frame="))
    return; // Dont log progress messages
  if(s.startsWith("size="))
    return; // Dont log size messages
  emit message(s);
}

void QFFMpeg::progressSlot(int progress)
{
  emit convertProgress(progress);
}

void QFFMpeg::clear()
{
  m_files.clear();
}

uint QFFMpeg::audioStreamCount() const
{
  return m_files.first().audioStreams().count();
}

QStringList QFFMpeg::fileNames() const
{
  QStringList result;

  for(QFFMpegFileList::const_iterator it = m_files.begin();
      it != m_files.end(); ++it)
    result.append((*it).fileName());
  return result;
}

double QFFMpeg::frameRate() const
{
  return m_files.first().frameRate();
}

AVFrame* QFFMpeg::rawFrame(QTime qtime)
{
  bool backwards = (m_position > qtime);
  m_position = qtime;
  int64_t pos = m_position.toAVTime();

  for(QFFMpegFileList::iterator it = m_files.begin();
      it != m_files.end(); ++it)
  {
    if(pos < (*it).avDuration())
    {
      (*it).seek(pos, backwards);
      return (*it).frame();
    }
    else
      pos -= (*it).avDuration();
  }
  return 0;
}

QTime QFFMpeg::duration() const
{
  QAVTime result;
  int64_t d = 0;

  for(QFFMpegFileList::const_iterator it = m_files.begin();
      it != m_files.end(); ++it)
  {
    d += (*it).avDuration();
  }
  result.set(d);
  //kdDebug() << k_funcinfo << result << endl;
  return result;
}

bool QFFMpeg::addFile(const QString& file)
{
  m_files.append(QFFMpegFile());
  return m_files.last().open(file);
}

bool QFFMpeg::isDVDCompatible() const
{
  for(QFFMpegFileList::const_iterator it = m_files.begin();
      it != m_files.end(); ++it)
  {
    if(!(*it).isDVDCompatible())
      return false;
  }
  return true;
}

bool QFFMpeg::convertTo(QFFMpegConvertTo type, int input, const QString& output)
{
  int max = 0;
  m_ff = new QFFMpegConverter(max);
  bool result = false;

  if(m_files[input].hasVideo())
  {
    QAVTime time = duration();
    double d = (double)time.toAVTime() / (double)AV_TIME_BASE;
    max = (int)(d * frameRate());

    // No audio progress
    connect(m_ff, SIGNAL(progress(int)),
            this, SLOT(progressSlot(int)));
  }

  QString files = m_files[input].fileName(); //fileNames().join(" ");

  m_ff->set("i", (const char*)files.local8Bit());
  for(QFFMpegConvertTo::iterator it = type.begin(); it != type.end(); ++it)
  {
    kdDebug() << k_funcinfo << (*it).key << " = " << (*it).value << endl;
    m_ff->set((*it).key, (*it).value);
  }

  m_ff->set_output((const char*)output.local8Bit());
  result = (m_ff->transcode() == 0);
  delete m_ff;
  m_ff = 0;
  return result;
}

void QFFMpeg::stop()
{
  if(m_ff)
    m_ff->stop(1);
}

QDateTime QFFMpeg::lastModified()
{
  QDateTime result;

  result.setTime_t(0);
  for(QFFMpegFileList::const_iterator it = m_files.begin();
      it != m_files.end(); ++it)
  {
    QFileInfo fi((*it).fileName());
    if(fi.lastModified() > result)
      result = fi.lastModified();
  }
  return result;
}

#include "qffmpeg.moc"

/*
int QFFMpeg::test(const char *filename)
{
    AVFormatContext *pFormatCtx;
    int             i, videoStream;
    AVCodecContext  *pCodecCtx;
    AVCodec         *pCodec;
    AVFrame         *pFrame;
    AVFrame         *pFrameRGB;
    AVPacket        packet;
    int             frameFinished;
    int             numBytes;
    uint8_t         *buffer;

    // Register all formats and codecs
    av_register_all();

    // Open video file
    if(av_open_input_file(&pFormatCtx, filename, NULL, 0, NULL)!=0)
        return false; // Couldn't open file

    // Retrieve stream information
    if(av_find_stream_info(m_pFormatCtx)<0)
        return false; // Couldn't find stream information

    // Dump information about file onto standard error
    dump_format(m_pFormatCtx, 0, filename, false);

    // Find the first video stream
    videoStream=-1;
    for(i=0; i<pFormatCtx->nb_streams; i++)
        if(m_pFormatCtx->streams[i]->codec.codec_type==CODEC_TYPE_VIDEO)
        {
            videoStream=i;
            break;
        }
    if(videoStream==-1)
        return false; // Didn't find a video stream

    // Get a pointer to the codec context for the video stream
    pCodecCtx=&pFormatCtx->streams[videoStream]->codec;

    // Find the decoder for the video stream
    pCodec=avcodec_find_decoder(m_pCodecCtx->codec_id);
    if(pCodec==NULL)
        return false; // Codec not found

    // Open codec
    if(avcodec_open(m_pCodecCtx, pCodec)<0)
        return false; // Could not open codec

    // Hack to correct wrong frame rates that seem to be generated by some
    // codecs
    if(m_pCodecCtx->time_base.den>1000 && pCodecCtx->time_base.num==1)
      pCodecCtx->time_base.den=1000;

    // Allocate video frame
    pFrame=avcodec_alloc_frame();

    // Allocate an AVFrame structure
    pFrameRGB=avcodec_alloc_frame();
    if(pFrameRGB==NULL)
        return false;

    // Determine required buffer size and allocate buffer
    numBytes=avpicture_get_size(PIX_FMT_RGB24, pCodecCtx->width,
        pCodecCtx->height);
    buffer=new uint8_t[numBytes];

    // Assign appropriate parts of buffer to image planes in pFrameRGB
    avpicture_fill((AVPicture *)pFrameRGB, buffer, PIX_FMT_RGB24,
        pCodecCtx->width, pCodecCtx->height);

    // Read frames and save first five frames to disk
    i=0;
    while(av_read_frame(m_pFormatCtx, &packet)>=0)
    {
        // Is this a packet from the video stream?
        if(packet.stream_index==videoStream)
        {
            // Decode video frame
            avcodec_decode_video(m_pCodecCtx, pFrame, &frameFinished,
                packet.data, packet.size);

            // Did we get a video frame?
            if(frameFinished)
            {
                // Convert the image from its native format to RGB
                img_convert((AVPicture *)pFrameRGB, PIX_FMT_RGB24,
                    (AVPicture*)pFrame, pCodecCtx->pix_fmt, pCodecCtx->width,
                    pCodecCtx->height);

                // Save the frame to disk
                if(++i<=5)
                    SaveFrame(pFrameRGB, pCodecCtx->width, pCodecCtx->height,
                        i);
            }
        }

        // Free the packet that was allocated by av_read_frame
        av_free_packet(&packet);
    }

    // Free the RGB image
    delete [] buffer;
    av_free(pFrameRGB);

    // Free the YUV frame
    av_free(pFrame);

    // Close the codec
    avcodec_close(m_pCodecCtx);

    // Close the video file
    av_close_input_file(m_pFormatCtx);

    return 0;
}
*/
