/***************************************************************************
                          setiparse.h  -  description
                             -------------------
    begin                : Tue Jan 1 2002
    copyright            : (C) 2002 by Sebastian Schildt
    email                : sebastian@frozenlight.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#ifndef SETIPARSE_H
#define SETIPARSE_H

#include <string>
#include <time.h>
#include <stdio.h>

using namespace std;

/**
  *@author Sebastian Schildt
  */

/** This structure stores all relevant data of a gaussian signal.
  * Gaussians are signals whose power should increase and then decrease with a
  * bell-shaped curve in time. The idea behind this is that Signals from a
  * distant transmitter should get stronger and then weaker as the telescope's
  * focal point moves across that area of the sky. The gaussian test is only applied
  * for frequency resolutions greater than or equal to 0.59 Hz. Since noise can
  * sometimes randomly simulate a gaussian, the SETI@home client only returns
  * gaussians stronger than 3.2 times the average noise level with a fit (chisq)
  *less than 8.8. */
struct Gaussian {
  double score; ///<  This value is determined by calculating (power/chisq).
	double sigma; ///<  Sigma represents the width of the gaussian.
	double power; ///< This value tells us how strong the signal is relative to noise.
	double trueMean; ///< The fitted mean value.
/** A fit value indicating how well a rising and falling signal fits an ideal
  * gaussian (bell curve) profile.
  *
  * A lower fit value means a better fit. */
	double chiSquare;
	double chirpRate; ///< The chirp rate at which this gaussian was found
	
  int bin; ///< An integer value representing the gaussian.
  int fftInd; ///< The fft index value
	int fftLen; ///< The fft length (number of data points)

	double data[64]; ///<  This array stores the gaussian shape

	bool valid; ///< True if the struct contains a valid gaussian
};

/** This structure stores all relevant data of a pulse.
  *
  * Pulses are a series of evenly spaced spikes (in time). To find such a
  * series, the SETI@home client applies a special test called a
  * "fast folding algorithm". Pulses are searched for all frequency resolutions
  * greater than or equal to 0.59 Hz. Any pulse with a score of greater than 1
  * will be reported when the client returns a result to Berkeley.
  */
struct Pulse {
	double power; ///< The power of the pulse relative to the noise level
  /** The score of the pulse.
    *
    * Because both interference signals (RFI) and random noise can simulate a
    * pulsed signal, a minimum threshold for acceptance is set. This threshold
    * is calculated dynamically and depends upon the period and the number of
    * times the data has been folded. The score is the ratio of the pulse
    * amplitude to this threshold value.
    */
	double score;
	double mean;  ///< The mean value of the data
	double chirpRate; ///< The chirp rate at which this pulse was found
	double period; ///< The period of the pulse in seconds

	int fftLen; ///<  The fft length (number of data points)
	int dataLen; ///< The amount of data values in the array (variable for pulses)

	short data[512]; ///< This array stores the pulse profile

	bool valid; ///< True if the structure contains a valid pulse
};

/**This structure stores all relevant data of a triplet.
 *
 * Triplets are sets of three evenly spaced spikes. The SETI@home client tests
 * for triplets by looking at every pair of spikes above a certain threshold
 * power. It then looks for another spike precisely between the two spikes. If
 * one is found, a triplet is logged and sent back to the SETI@home server.
 */
struct Triplet {
  double power; ///< The power of the triplet relative to the noise level
	double score; ///< The score of the triplet. (Currently the score is identical to the power)
	double mean; ///<  The mean value of the data
	double chirpRate; ///< The chirp rate at which this pulse was found.
  /** The period of the triplet.
    *
    * This value tells us how far apart the three spikes are (in seconds).
    */
	double period;

	int fftLen; ///< The fft length (number of data points).
	int dataLen; ///< The amount of data values in the array (should be always 512 for triplets).
	int peak1; ///< Marks the position of the first pulse of the triplet in the data set.
	int peak2; ///< Marks the position of the second pulse of the triplet in the data set.
	int peak3; ///< Marks the position of the third pulse of the triplet in the data set.

	short data[512]; ///< This array stores the triplet profile.

	bool valid; ///< True if the structure contains a valid triplet
};


/** The setiparse class is a C++ class that allows convenient access to all the
  * information in the state files of a SETI@home client.
  *
  * Using the SetiContainer is very simple. Here is a short code snippet:
  *
  * \code
  * setiparse sp;
	* sp.setSetiPath("/home/sebastian/setidir/");
  *
	* //Start parsing process
	* sp.parseSAH();
  *
  * string username = sp.getUserName();
  * int    progressInPercent = sp.getPercent();
  * \endcode
  * This short snippet creates a setiparse object. With the parseSAH() function
  * the *.sah files get parsed and the data is filled into the setiparse class.
  * You may now use setiparse access methods to extract all information you
  * need.
  *
  * All information from the client's output files is organized in a logical
  * manner using dedicated structures. For example, there are the Pulse structure,
  * the Triplet structure, or the Gaussian structure.
  * All inforamtion can be accessed via getXX methods. This get methods have
  * speaking names like getTimeLastResult(), getUserName() which should make the
  * resulting code very readable. Additionaly all get methods are const, so you
  * may work with constant reference of setiparse in you program.
  *
  *
  * @author Sebastian Schildt (sebastian@frozenlight.de)
  */
class setiparse {
public: 
	setiparse(); ///< The constructor
	~setiparse(); ///< The destructor

/** This sets the path to the set directory */
	void setSetiPath(string path);
	string getSetiPath() const; ///< @return the path to seti

	/* get state.sah vars */
	float getProc() const;
	int getPercent() const;
	int getNcfft() const;
	double getChirp() const;
	double getWuCpuTime() const;
	Gaussian getGaussian() const;
	Pulse getPulse() const;
	Triplet getTriplet() const;

	/* get user_info.sah vars */
	int getUserId() const;
	string getUserName() const;
	string getTimeRegistered() const;
	string getTimeLastResult() const;
	int getNrCompleted() const;
	double getCpuTime() const;	

	/* get work_unit.sah vars */
	string getWu_name() const;
	double getWu_StartRa() const;
	double getWu_EndRa() const;
	double getWu_StartDec() const;
	double getWu_EndDec() const;
	double getWu_AngleRange() const;
	double getWu_SubbandCenter() const;
	double getWu_SubbandBase() const;
	double getWu_SubbandSampleRate() const;
	int    getWu_SubbandNumber() const;

	int parseSAH();

private: // Private attributes
  /* state.sah vars */
  int percent; //how many % of a WU is done
	float proc; //the procnumber
	int ncfft; //Number off Fast Fourier Transformations done so far
  double chirp; //Chirp rate (Doppler)
	double wuCpuTime; //How long is this WU being processed (s)

	Gaussian gau; // The gaussian, if found
	Pulse 	 pls;
	Triplet  trp;

	/* user_info.sah*/
	int     userId; //User id
  string  userName; //User name
	string  timeRegistered; //Date when user registered
  string  timeLastResult;  // Date when last result was returned
	int     nrCompleted;  //nr WU completed
	double  cpuTime; //Used CPU time / sec
	string  SetiPath;

	/* work_unit.sah */
	string wu_name;
	double wu_StartRa;
	double wu_EndRa;
	double wu_StartDec;
	double wu_EndDec;
	double wu_AngleRange;
	double wu_SubbandCenter;
	double wu_SubbandBase;
	double wu_SubbandSampleRate;
	int    wu_SubbandNumber;

	void analyseStateLine(string cl);
	void analyseUserLine(string cl);
	void analyseWULine(string cl);

	void extractProc(string cl);
	void extractInt(string cl,int startpos, int* dest);
	void extractDouble(string cl,int startpos, double* dest);
	void extractString(string cl, int startpos, string* dest);
	void extractTime(string cl, int startpos, tm* dest);
	void extractGPot(string cl); // Extract gaussian potential energy
	void extractPPot(string cl); //Extract pulse potential energy
	void extractTPot(string cl); //Extract triplet potential energy
};

#endif
