package com.limegroup.gnutella.gui;

import javax.swing.*;
import java.awt.*;

/**
 * Window that displays the splash screen.  This loads the splash screen
 * image, places it on the center of the screen, and allows dynamic
 * updating of the status text for loading the application.
 */
//2345678|012345678|012345678|012345678|012345678|012345678|012345678|012345678|
final class SplashWindow extends JWindow {

    /**
     * Constant handle to the glass pane that handles drawing text
     * on top of the splash screen.
     */
    private static final SplashGlassPane GLASS_PANE = 
        new SplashGlassPane();

    /**
     * Creates a new SplashWindow, setting its location, size, etc.
     */
    SplashWindow() {
        final ImageIcon splashIcon = 
            ResourceManager.instance().getThemeImage("splash.gif");
        final Image image = splashIcon.getImage();
        final Dimension size = 
            new Dimension(image.getWidth(null) + 2, image.getHeight(null) + 2);
        this.setSize(size);

        final Dimension screenSize =
            Toolkit.getDefaultToolkit().getScreenSize();
        setLocation((screenSize.width - size.width) / 2,
                    (screenSize.height - size.height) / 2);

        final JLabel splashLabel = new JLabel(splashIcon);
        splashLabel.setBorder(BorderFactory.createLineBorder(Color.black, 1));
        this.getContentPane().add(splashLabel, BorderLayout.CENTER);

        this.setGlassPane(GLASS_PANE);
        this.pack();
        this.setVisible(true);
        GLASS_PANE.setVisible(true);
    }

    /**
     * Sets the loading status text to display in the splash 
     * screen window.
     *
     * @param text the text to display
     */
    static void setStatusText(String text) {
        GLASS_PANE.setText(text);
    }

    /**
     * A private glass for the glass pane that handles drawing
     * status text above the background image.
     */
    private static final class SplashGlassPane extends JPanel {

        /**
         * Handle for the panel that contains the text.
         */
        private static JPanel TEXT_PANEL = new JPanel();

        /**
         * The label for the text.
         */
        private static JLabel TEXT_LABEL = new JLabel();

        /**
         * Constructor lays out the panels and label, sets them to
         * be transparent, etc.
         */
        private SplashGlassPane() {
            this.setOpaque(false);
            this.setLayout(new BoxLayout(this, BoxLayout.Y_AXIS));
            this.add(Box.createVerticalGlue());
            TEXT_PANEL.setOpaque(false);
            TEXT_PANEL.setLayout(new BoxLayout(TEXT_PANEL, 
                                               BoxLayout.X_AXIS));
            TEXT_PANEL.add(Box.createHorizontalGlue());
            // make sure label doesn't clip....
            TEXT_LABEL.setMinimumSize(new Dimension(40, 3));
            TEXT_PANEL.add(TEXT_LABEL);
            TEXT_PANEL.add(GUIMediator.getHorizontalSeparator());
            this.add(TEXT_PANEL);
            this.add(GUIMediator.getVerticalSeparator());
        }

        /**
         * Sets the text to display for the status area.
         *
         * @param text the text to display
         */
        private static void setText(String text) {
            FontMetrics fm = TEXT_LABEL.getFontMetrics(TEXT_LABEL.getFont());
            TEXT_LABEL.setPreferredSize(new Dimension(fm.stringWidth(text),
                                                      fm.getHeight()));
            TEXT_LABEL.setText(text);
        }

    }

}

