package com.limegroup.gnutella.gui;

import java.io.*;
import java.awt.*;
import java.awt.event.*;

import javax.swing.*;
import javax.swing.border.*;
import javax.swing.event.*;
import com.limegroup.gnutella.util.CommonUtils;
import com.limegroup.gnutella.gui.mp3.*;
import com.limegroup.gnutella.settings.ThemeFileHandler;

/**
 * The component for the space at the bottom of the main application
 * window, including the connected status and the media player.
 */
public final class StatusLine implements ThemeObserver {
    
    /**
     * The left most panel containing the connect[ed]/disconnect[ed]
     * icons and buttons. Switcher lets you toggle between the two.
     */
    private JPanel switchedPanel;


    /**
     * <tt>CardLayout</tt> for switching between connected and 
     * disconnected views.
     */
    private CardLayout switcher;
    
    /**
     * The label with number of files being shared statistics.
     */
    private JLabel sharingLabel;
    
    
    /**
     * Takes care of all the MediaPlayer stuff.
     */
    private MediaPlayerComponent _mediaPlayer;

    /**
     * Constant for the <tt>JPanel</tt> that displays the status line.
     */
    private final JPanel PANEL = new BoxPanel(BoxPanel.X_AXIS);

	/**
	 * Constant for the <tt>JLabel</tt> instance to use when connected.
	 */
	private final JLabel CONNECTED_LABEL = new JLabel();

	/**
	 * Constant for the <tt>JLabel</tt> instance to use when disconnected.
	 */
	private final JLabel DISCONNECTED_LABEL = new JLabel();
    
    /**
     * Creates a new status line in the disconnected state.
     */
    public StatusLine() {        
        GUIMediator.setSplashScreenString(
            GUIMediator.getStringResource("SPLASH_STATUS_STATUS_WINDOW"));  
        
        //Make components
        createConnectPanel();
        sharingLabel = new JLabel("           ");
        sharingLabel.setHorizontalAlignment(SwingConstants.CENTER);
        // I don't want the label to clip...
        sharingLabel.setMinimumSize(new Dimension(20, 3));
        String toolTip = 
            GUIMediator.getStringResource("STATISTICS_SHARING_TOOLTIP");
        sharingLabel.setToolTipText(toolTip);
        setConnected(false);
        
        // Tweak appearance
        PANEL.setBorder(BorderFactory.createLoweredBevelBorder());
        
        // Put them together.
        JPanel leftPanel = new JPanel(new FlowLayout(FlowLayout.LEFT));
        
        // add connect icon and sharing label...        
        leftPanel.add(switchedPanel);
        leftPanel.add(sharingLabel);
        PANEL.add(leftPanel);;

        JPanel centerPanel = new BoxPanel(BoxPanel.X_AXIS);
        if(!GUIMediator.hasDonated()) {
            centerPanel.add(Box.createHorizontalGlue());
            centerPanel.add(new StatusLinkHandler().getComponent());
            centerPanel.add(Box.createHorizontalGlue());
        }        

        PANEL.add(centerPanel);
        
        // unfortunately, we can't have the media player on the mac...
        if (!CommonUtils.isMacClassic()) {
            // now show the mp3 player...        
            _mediaPlayer = new MediaPlayerComponent();
            JPanel mediaPanel = _mediaPlayer.getMediaPanel();
            PANEL.add(mediaPanel);
        }
		GUIMediator.addThemeObserver(this);
    }
    
    /**
     * Sets up switchedPanel and switcher so you have
     * connect/disconnect icons.
     *
     * @modifies switchedPanel, switcher
     */
    private void createConnectPanel() {
		updateTheme();
        switchedPanel = new JPanel();
        switcher = new CardLayout();
        switchedPanel.setLayout(switcher);
        
        //1. Create connected mode stuff
        JPanel connectedPanel = new JPanel();
        connectedPanel.setLayout(new FlowLayout(FlowLayout.LEFT));
        //Icon connectedIcon = GUIMediator.getThemeImage("connected.gif");
        connectedPanel.add(CONNECTED_LABEL);
        switchedPanel.add("connect", connectedPanel);
        
        //2. Create disconnected mode stuff
        JPanel disconnectedPanel = new JPanel();
        disconnectedPanel.setLayout(new FlowLayout(FlowLayout.LEFT));
        //Icon disconnectedIcon = GUIMediator.getThemeImage("disconnected.gif");
        disconnectedPanel.add(DISCONNECTED_LABEL);
        switchedPanel.add("disconnect", disconnectedPanel);
        
        switchedPanel.setMaximumSize(new Dimension(60, 30));
    }
    
	// inherit doc commment
	public void updateTheme() {
		Icon connectedIcon = GUIMediator.getThemeImage("connected.gif");
        Icon disconnectedIcon = GUIMediator.getThemeImage("disconnected.gif");
		CONNECTED_LABEL.setIcon(connectedIcon);
		DISCONNECTED_LABEL.setIcon(disconnectedIcon);
		if(_mediaPlayer != null) 
			_mediaPlayer.updateTheme();
	}

    /**
     * Notifies this that we're connected.
     *
     * @modifies this
     */
    public void setConnected(final boolean connected) {
        if (connected) {
            synchronized(switchedPanel) {
                switcher.first(switchedPanel);
            }
        } else {
            synchronized(switchedPanel) {
                switcher.last(switchedPanel);
            }
        }
    }
    
    /**
     * Sets the horizon statistics for this.
     * @modifies this
     * @return A displayable Horizon string.
     */
    public String setStatistics(long hosts, long files, long kbytes, int share, int pending) {
        //Shorten with KB/MB/GB/TB as needed.
        String txt;
        if (hosts == 0)
            txt = STATS_DISCONNECTED_STRING;
        else
            txt = GUIUtils.toUnitnumber(files, false) +
                  " " + STATS_FILE_STRING + " / " +
                  GUIUtils.toUnitbytes(kbytes * 1024) +
                  " " + STATS_AVAILABLE_STRING;
        
        // if nothing is pending shared, display as normal
        if( pending == 0 ) 
            sharingLabel.setText(STATS_SHARING_STRING + " " + 
                                 String.valueOf(share) + 
                                 " " + STATS_FILE_STRING);
        //otherwise display as  'shared / total'
        else
            sharingLabel.setText(STATS_SHARING_STRING + " " +
                                String.valueOf(share) + " / " +
                                String.valueOf(pending + share) +
                                " " + STATS_FILE_STRING);
                                
        if (share == 0) {
			Color notSharingColor = ThemeFileHandler.NOT_SHARING_LABEL_COLOR.getValue();
			sharingLabel.setForeground(notSharingColor);
		} else
            sharingLabel.setForeground((Color)UIManager.get("Label.foreground"));
        return txt;
    }
    
    
    /**
     * The String for part of the stats string, ie "available".
     */
    private static final String STATS_AVAILABLE_STRING = 
        GUIMediator.getStringResource("STATISTICS_AVAILABLE");;
    
    /**
     * The String for part of the stats string, ie "files".
     */
    private static final String STATS_FILE_STRING = 
        GUIMediator.getStringResource("STATISTICS_FILES");
    
    /**
     * The String for part of the stats string, ie "no files...".
     */
    private static final String STATS_DISCONNECTED_STRING =             
        GUIMediator.getStringResource("STATISTICS_DISCONNECTED");
    
    /**
     * The String for part of the stats string, ie "no files...".
     */
    private static final String STATS_SHARING_STRING =             
        GUIMediator.getStringResource("STATISTICS_SHARING");
    
    
    public void audioFileDoubleClicked() {
        _mediaPlayer.audioFileDoubleClicked();
    }
    
    public void launchAudio(File toPlay) {
        _mediaPlayer.launchAudio(toPlay);
    }

    /**
      * Accessor for the <tt>JComponent</tt> instance that contains all 
      * of the panels for the status line.
      *
      * @return the <tt>JComponent</tt> instance that contains all 
      *  of the panels for the status line
      */
    public JComponent getComponent() {
        return PANEL;
    }
}
