package com.limegroup.gnutella.gui.menu;

import com.limegroup.gnutella.util.CommonUtils;
import com.limegroup.gnutella.gui.GUIMediator;
import java.io.*;
import java.awt.event.*;


/**
 * Handles all of the contents of the help menu in the menu bar.  This 
 * includes such items as the link to the "Using LimeWire" page of the
 * web site as well as links to the forum, faq, "tell a friend", etc.
 */
//2345678|012345678|012345678|012345678|012345678|012345678|012345678|012345678|
final class HelpMenu extends AbstractMenu {
	
	/**
	 * Constant handle to the listener class for opening up the main
	 * help page.
	 */
	private final HelpListener HELP_LISTENER = new HelpListener();

	/**
	 * Constant handle to the listener class for opening up the faq page.
	 */
	private final FAQListener FAQ_LISTENER = new FAQListener();

	/**
	 * Constant handle to the listener class for opening up the forum page.
	 */
	private final ForumListener FORUM_LISTENER = new ForumListener();

	/**
	 * Constant handle to the listener class for opening up the "tell a
	 * friend" page.
	 */
	private final FriendListener FRIEND_LISTENER = new FriendListener();

	/**
	 * Constant handle to the listener class for opening up the page
	 * to submit a review of the application.
	 */
	private final ReviewListener REVIEW_LISTENER = new ReviewListener();

	/**
	 * Constant handle to the listener class for opening up the "about"
	 * window.
	 */
	private final AboutListener ABOUT_LISTENER = new AboutListener();
	
	/**
	 * Constant handle to the listener class for displaying an error.
	 */
	private final ErrorListener ERROR_LISTENER = new ErrorListener();

	/**
	 * Creates a new <tt>HelpMenu</tt>, using the <tt>key</tt> 
	 * argument for setting the locale-specific title and 
	 * accessibility text.
	 *
	 * @param key the key for locale-specific string resources unique
	 *            to the menu
	 */
	HelpMenu(final String key) {
		super(key);
		addMenuItem("HELP_USING_LIMEWIRE", HELP_LISTENER);
		addMenuItem("HELP_FAQ", FAQ_LISTENER);
		addMenuItem("HELP_FORUM", FORUM_LISTENER);
		addMenuItem("HELP_FRIEND", FRIEND_LISTENER);
		addMenuItem("HELP_REVIEW", REVIEW_LISTENER);
		if(CommonUtils.isMacOSX()) return;
		addSeparator();
		addMenuItem("HELP_ABOUT", ABOUT_LISTENER);
		if(CommonUtils.isTestingVersion()) {
		    addSeparator();
		    addMenuItem("HELP_ERROR", ERROR_LISTENER);
		}
	}

	/**
	 * Opens an error report, for testing.
	 */
	private class ErrorListener implements ActionListener {
	    public void actionPerformed(ActionEvent e) {
	        GUIMediator.showInternalError(new Exception("Generated Error"));
	    }
	}
	
	/**
	 * Opens the main help page in the default browser, displaying
	 * an error message if the browser could not be launched 
	 * successfully.
	 */
	private class HelpListener implements ActionListener {
		public void actionPerformed(ActionEvent e) {
			try {
				String url = "http://www.limewire.com/support.htm";
				GUIMediator.openURL(url);
			}
			// hhmmn... could not locate the browser.
			// lets let them know
			catch(IOException ex){
				String msgKey = "ERROR_LOCATING_BROWSER_SUPPORT";
				GUIMediator.showError(msgKey);
			}
		}
	}

	/**
	 * Opens the faq page in the default browser, displaying
	 * an error message if the browser could not be launched 
	 * successfully.
	 */
	private class FAQListener implements ActionListener {
		public void actionPerformed(ActionEvent e) {
			try {
				String url = "http://www.limewire.com/support/faq.htm";
				GUIMediator.openURL(url);
			}
			// hhmmn... could not locate the browser.
			// lets let them know
			catch(IOException ex) {
				String msgKey = "ERROR_LOCATING_BROWSER_FAQ";
				GUIMediator.showError(msgKey);
			}
		}
	}
	
	/**
	 * Opens the forum page in the default browser, displaying
	 * an error message if the browser could not be launched 
	 * successfully.
	 */
	private class ForumListener implements ActionListener {
		public void actionPerformed(ActionEvent e) {
			try {
				String url = "http://www.limewire.com/forum.htm";
				GUIMediator.openURL(url);
			}
			// hhmmn... could not locate the browser.
			// lets let them know
			catch(IOException ex) {
				String msgKey = "ERROR_LOCATING_BROWSER_FORUM";
				GUIMediator.showError(msgKey);
			}
		}
	}

	/**
	 * Opens the "tell a friend" page in the default browser, displaying
	 * an error message if the browser could not be launched 
	 * successfully.
	 */
	private class FriendListener implements ActionListener {
		public void actionPerformed(ActionEvent e) {
			try {
				String url = "http://www.limewire.com/index.jsp/tell_friend";
				GUIMediator.openURL(url);
			}
			// hhmmn... could not locate the browser.
			// lets let them know
			catch(IOException ex) {
				String msgKey = "ERROR_LOCATING_BROWSER_FRIEND";
				GUIMediator.showError(msgKey);
			}
		}
	}

	/**
	 * Opens the page to review the application in the default browser, 
	 * displaying an error message if the browser could not be launched 
	 * successfully.<p>
	 *
	 * Opens up different links depending on the user's operating system.
	 */
	private class ReviewListener implements ActionListener {
		public void actionPerformed(ActionEvent e) {
			String url;
			if(CommonUtils.isWindows()) {
				url = "http://www.limewire.com/review/pc.htm";
			} else if(CommonUtils.isMacClassic()) {
				url = "http://www.limewire.com/review/mac_classic.htm";
			} else if(CommonUtils.isMacOSX()) {
				url = "http://www.limewire.com/review/mac_osx.htm";
			} else if(CommonUtils.isLinux()) {
				url = "http://www.limewire.com/review/linux.htm";
			} else if(CommonUtils.isSolaris()) {
				url = "http://www.limewire.com/review/linux.htm";
			} else {
				url = "http://www.limewire.com/review/pc.htm";
			}

			try {
				GUIMediator.openURL(url);
			}
			// hhmmn... could not locate the browser.
			// lets let them know
			catch(IOException ex) {
				String msgKey = "ERROR_LOCATING_BROWSER_GENERAL";
				GUIMediator.showError(msgKey);
			}
		}
	}
	
	/**
	 * Shows the about window with more information about the application.
	 */
	private class AboutListener implements ActionListener {				
		/**
		 * Implements the <tt>ActionListener</tt> interface, showing the
		 * about window.
		 */
		public void actionPerformed(ActionEvent e) {
			GUIMediator.showAboutWindow();
		}	   
	}
}
