package com.limegroup.gnutella.gui.mp3;

import java.io.*;
import javazoom.jl.converter.*;
import javazoom.jl.decoder.*;
import javax.sound.sampled.*;

/**
 * Class BasicPlayer
 *
 * - Facilitates control of the JavaLayer Decoder
 * - Provides an interface to play, stop, pause, seek an MP3
 * - Uses callbacks to send messages to the client 
 *   (which implements BasicPlayerListener)
 *
 * - This class can be substituted by another player which may be able to play
 *   more media types.  Right now, this class does not support seeking very
 *    well, so that is something to work on.
 * 
 */
final class BasicPlayer extends AbstractAudioPlayer {
	
	/** BasicPlayer status */
	private int status = STATUS_STOPPED;
	private static Thread playthread;	// plays mp3 in seperate thread
	
	private WaveStreamObuffer output;	// decoded frame output buffer
	private Decoder decoder;		    // the decoder
	private SourceDataLine line = null;	// line enables access to sound hardware
	private FloatControl Jvolume;
	private FloatControl Jpan;

    // frame length in bytes (default size 413)
	private int currentFrameLength = 413;		
	
    private float volumeLevel = 0F;
	private float panLevel = 0F;
	private int frameSeek = 0;			    // seek bar value
	
    public int getFrameSeek() {
        return frameSeek;
    }
	private String currentMP3;		// mp3 to be played
	
	/**
	 * Variable for the number of frames read for the current file.  This
	 * does not need to be volatile because only one thread changes it.
	 */
	private int framesRead = 0;
	
	public int getFramesRead() {
        return framesRead;
    }

    // lock on this object when paused - no need to idle loop.  see doPlay().
    private Object pauseLock = new Object();

	//private BasicPlayerListener parent;
	
	public BasicPlayer() {
		output = null;
		//this.parent = parent;
	}

	/**
	 * class PlayThread
	 *
	 * - starts decoding in a seperate thread
	 *
	 */
	private class PlayThread extends Thread {
		String playFile;
		public PlayThread(String mp3File) {
		    super("PlayThread");
			playFile = mp3File;
		}
		public void run() {
			try {
				  // play the mp3 file
				doplay(playFile);
			} 
            catch (Exception ignored) {}
		}
	}
    
	/**
	 *
	 * - called by client to start a new playthread
	 *   and begin mp3 playback
	 *
	 */
	private void play(String mp3File) {
		currentMP3 = mp3File;
		playthread = new PlayThread(mp3File);
		playthread.start();
	}


	/**
	 *
	 * - called by client to start a new playthread
	 *   and begin mp3 playback
	 *
	 */
    public void play(File mp3File) {
        try {
            play(mp3File.getCanonicalPath());
        }
        catch (Exception ignored) {};
    }


	/**
	 * play
	 *
	 * - loops playback until no more frames can be decoded
	 * - plays the mp3 file
	 *
	 */
	private void doplay(String sourceName) throws JavaLayerException {
		framesRead = 0;
		  // initialize the decoder
		decoder = new Decoder(null);

		if (line != null) {
			  // delete any current line that is open
			line = null;
		}

        InputStream in = null;
        boolean seekSetup = false;
		try {
			  // get the file input stream
			in = openInput(sourceName);
			  // create a bit stream
			Bitstream stream = new Bitstream(in);
			
			status = STATUS_PLAYING;
			while(status != STATUS_STOPPED) {
				if (status == STATUS_PLAYING) {

                    /******* 
                     * stop playing ASAP!!
                     */
                    if (status == STATUS_STOPPED)
                        throw new Exception();

                    // read a header from the bitstream
					Header header = stream.readFrame();
					if (header == null) 
                        break;

                    // update frame counter
					framesRead++;

                    // get the frame size
					currentFrameLength = header.framesize;

                    /******* 
                     * stop playing ASAP!!
                     */
                    if (status == STATUS_STOPPED)
                        throw new Exception();

                    //skip forward frameSeek frames
					while (frameSeek > 0) {
						stream.closeFrame();
						frameSeek--;
						header = stream.readFrame();
						framesRead++;
					}
					
                    // callback to AudioPlayerListener to update song position
                    if (!seekSetup) {
                        //parent.setUpSeek(countFrames(sourceName));
                        this.fireSeekSetupRequired(countFrames(sourceName));
                        seekSetup = true;
                    }
                    //this.fireAudioPositionUpdated(framesRead);
					//parent.updateSongPosition(framesRead);
					
                    /******* 
                     * stop playing ASAP!!
                     */
                    if (status == STATUS_STOPPED)
                        throw new Exception();
                    
                    if (line == null) {
                        int channels = 
                        (header.mode()==Header.SINGLE_CHANNEL) ? 1 : 2 ;

                        int freq = header.frequency();
                        // initialize a wavestream output buffer
                        output = new WaveStreamObuffer(channels,freq);
                        decoder.setOutputBuffer(output);
                        // initialize the line
                        line = getALine(channels, freq);
                        setUpControls();
                        line.start();
                    }

                    // decode the frame
                    decoder.decodeFrame(header,stream);

					
                    /******* 
                     * stop playing ASAP!!
                     */
                    if (status == STATUS_STOPPED)
                        throw new Exception();

                    // get the decoded frame from the output buffer
					byte [] b = output.get_data();

                    // write the decoded frame to the output line
					if (line != null && line.isOpen())
						line.write(b,0,b.length);
					stream.closeFrame();

				} /* if */

                if (status == STATUS_PAUSED) {
                    synchronized (pauseLock) {
                        // i'll be woken up when unpaused, or thrown out when
                        // i'm ready to go....
                        pauseLock.wait();
                    }
                } /* if */
			} /* while */
		} 
        catch (Exception ignored) {
            // do we really want to ignore this???
        } 
        finally {
			/*
			 * clean up objects
			 * - close output buffer
			 * - drain, stop and close the line
			 * - delete the line
             * - close access to the mp3 file
			 */
			if (output != null) output.close();
			if (line != null) {
				try {
					line.drain();
					line.stop();
					line.close();
				} 
                catch (Exception e) {} 
                finally {
					line = null;
				}
			}
            if (in != null) {
                try {
                    in.close();
                }
                catch (Exception e) {}
                finally {
                    in = null;
                }
            }
		}
        // callback to AudioPlayerListener to inform the current 
        // song has finished
        if (status != STATUS_STOPPED)
            status = STATUS_STOPPED;
		//parent.playComplete();
		this.firePlayComplete();
	} /* play() */

	
	/**
	 * getALine
	 *
	 * - returns a SourceDataLine object if there is a line available for 
     *   playback
	 *
	 */
	private SourceDataLine getALine(int channels, int freq) throws Exception {
		AudioFormat audioFormat = new AudioFormat((float)freq,16,channels,true,false);
        // create the new line info
		DataLine.Info info = new DataLine.Info(SourceDataLine.class,audioFormat);
        // get the line from the audio system
		SourceDataLine aline = (SourceDataLine) AudioSystem.getLine(info);
        // open the line with the specified format
		aline.open(audioFormat, aline.getBufferSize());
		return aline;
	}
	
	/**
	 * fileLength
	 *
	 * - returns the length in bytes of the specified file
	 *
	 */
	public int fileLength(String sourceName) {	
		File phile = new File(sourceName);
		if (phile.exists()) return (int) phile.length();
		return -1;
	}
	
	/**
	 * countFrames
	 *
	 * - returns the number of frames in the specified file
	 *
	 */
	public int countFrames(String sourceName) {
		int fLength = fileLength(sourceName);
		if (fLength != -1) {
			return fLength/currentFrameLength;
        }
        return -1;
    }
	
	/**
	 * seek
	 *
	 * - updates the current song position
	 * - doplay() checks frameSeek each iteration and
	 *   fast forwards or rewinds depending where
	 *   we are in the file
	 *
	 */
    public void seek(int seekPosition) {
        frameSeek = seekPosition;
    }
	
	/**
	 * unpause
	 *
	 * - unpauses playback
	 *
	 */
	public void unpause() {
		status = STATUS_PLAYING;
        synchronized (pauseLock) {
            pauseLock.notify();
        }
		if (line != null) {
			line.start();
		}	
	}
	
	/**
	 * pause
	 *
	 * - causes playback to pause
	 *
	 */
	public void pause() {
		if (line != null) {
			line.stop();
		}
        if (status != STATUS_STOPPED)
            status = STATUS_PAUSED;
	}
	
	/**
	 * stop
	 *
	 * - causes playback to stop and reset
	 *
	 */
    private static final int MAX_JOIN_TIME = 3000;  // 3 seconds...
	public void stop() {
		if (line != null) {
			line.flush();
			line.stop();
            // need to add a close here.  if a stop is done and immediately
            // followed by a play, the play was failing.  the close seems to fix
            // it.
            line.close(); 
		}
        if (status == STATUS_PAUSED) 
            playthread.interrupt();
        status = STATUS_STOPPED;
        try { // unfortunately, we should wait of the playthread to stop
            if (playthread != null)
                playthread.join(MAX_JOIN_TIME);
        }
        catch (InterruptedException ie) {};
	}

	/**
	 * setUpControls
	 *
	 * - asks the line for some controls, and if the
	 *   line has those controls it  retrieves them from
	 *   the line and then sets the state of that control
	 *   to the level in volumeLevel or panLevel.
	 *   Volumelevel and panlevel are set by the Jplayer (client)
	 *   when the user moves a GUI control
	 *
	 */
	public void setUpControls() {
		if (line.isControlSupported(FloatControl.Type.MASTER_GAIN)) {
			Jvolume = (FloatControl)line.getControl(FloatControl.Type.MASTER_GAIN);
			setVolume(volumeLevel);
		}
		if (line.isControlSupported(FloatControl.Type.PAN)) {
			Jpan = (FloatControl)line.getControl(FloatControl.Type.PAN);
			setPan(panLevel);	
		}
	}
	
	public void setPan(float newPanLevel) {
		panLevel = newPanLevel;
		if (status != STATUS_STOPPED) {
			Jpan.setValue(newPanLevel);
		}				
	}
	
	public void setVolume(float newVolumeLevel) {
		volumeLevel = newVolumeLevel;
		if (status != STATUS_STOPPED) {
			Jvolume.setValue(volumeLevel);
		}
	}
	
	/**
	 * getStatus
	 *
	 * - called by client (JPlayer) to determine what
	 *   the BasicPlayer is doing at that momemt
	 * - returns STATUS_PLAYING, STATUS_STOPPED, STATUS_PAUSED
	 *
	 */
	public int getStatus() {
		return status;
	}
	
	/**
	 * openInput
	 *
	 * - creates a BufferedInputStream from the specified file
	 *   and returns it
	 *
	 */
	protected InputStream openInput(String fileName) throws IOException {
		File file = new File(fileName);
		InputStream fileIn = new FileInputStream(file);
		BufferedInputStream buffIn = new BufferedInputStream(fileIn);
		return buffIn;
	}
	
    // inherit doc comment
	public void refresh() {
	    if(status == STATUS_PLAYING) {
	        this.fireAudioPositionUpdated(framesRead);
	    }
	}
}
