package com.limegroup.gnutella.gui.options;

import com.limegroup.gnutella.gui.*;
import com.limegroup.gnutella.settings.SettingsHandler;

import javax.swing.*;
import java.awt.*;
import java.io.*;

/**
 * This class acts as a mediator for the different components of the options
 * window.  This class maintains references to the 
 * <tt>OptionsTreeManager</tt> and <tt>OptionsPaneManager</tt>, the two
 * primary classes that it delegates to.
 */
//2345678|012345678|012345678|012345678|012345678|012345678|012345678|012345678|
public final class OptionsMediator implements ThemeObserver {

	/**
	 * Constant for the key for the root node in the tree.
	 */
	public final static String ROOT_NODE_KEY = "OPTIONS_ROOT_NODE";

	/**
	 * Singleton constant for easy access to the options mediator.
	 */
	private final static OptionsMediator INSTANCE =
		new OptionsMediator();

	/**
	 * Constant for the class that manages the current options pane 
	 * displayed to the user.  It is fine to construct these here since 
	 * they do not reference this class.
	 */
	private static OptionsPaneManager _paneManager =
		new OptionsPaneManager();

	/**
	 * Class that handles constructing all of the elements of the options 
     * windows.
	 */
	private static OptionsConstructor _constructor = 
		new OptionsConstructor(new OptionsTreeManager(), _paneManager);

	/**
	 * Boolean value for whether or not the save directory has changed since
	 * the last time settings were applied.
	 */
	private boolean _saveDirectoryChanged = false;


	/** 
	 * Singleton accessor for this class. 
	 *
	 * @return the <tt>OptionsMediator</tt> instance
	 */
	public static synchronized OptionsMediator instance() {
		return INSTANCE;
	}

	/** 
	 * Private constructor to ensure that this class cannot be constructed 
	 * from another class.  The constructor does very little to alleviate
	 * construction conflicts with classes that may use the mediator.
	 */
	private OptionsMediator() {
		GUIMediator.setSplashScreenString(
		    GUIMediator.getStringResource("SPLASH_STATUS_OPTIONS_WINDOW"));
        GUIMediator.addThemeObserver(this);
	}

	/**
	 * Makes the options window either visible or not visible depending 
	 * on the boolean argument.
	 *
	 * @param visible <tt>boolean</tt> value specifying whether the 
	 *  options window should be made visible or not visible
	 */
	public final void setOptionsVisible(boolean visible) {
		_saveDirectoryChanged = false;
		_paneManager.initOptions();
		_constructor.setOptionsVisible(visible);        
	}


    /** Returns if the Options Box is visible.
     *  @return true if the Options Box is visible.
     */
    public final boolean isOptionsVisible() {
        return _constructor.isOptionsVisible();
    }

	/**
	 * Returns a locale-specific <tt>String</tt> from the 
	 * <tt>GUIMediator</tt> class.
	 *
	 * @param key the key of the desired locale-specific <tt>String</tt>
	 *
	 * @return the locale-specific <tt>String</tt> corresponding to the 
	 *         <tt>key</tt> argument.
	 */
	public final String getString(final String key) {
		return GUIMediator.getStringResource(key);
	}

	/**
	 * Returns the specified image.
	 *
	 * @param imageName the name of the icon to return
	 * @return the <tt>ImageIcon</tt> object specified in the param string
	 */
	public final ImageIcon getImage(final String imageName) {
		return GUIMediator.getThemeImage(imageName);
	}

	/**
	 * Handles the selection of a new panel as the currently visible panel.
	 *
	 * @param key the unique identifying key of the panel to show
	 */
	public final void handleSelection(final String key) {
		_paneManager.show(key);
	}

	/**
	 * Applies the current settings in the options windows, storing them
	 * to disk.  This method delegates to the <tt>OptionsPaneManager</tt>.
	 *
	 * @throws IOException if the options could not be fully applied
	 */
	public final void applyOptions() throws IOException {
		_paneManager.applyOptions();
        SettingsHandler.save();
	}


	/**
	 * Returns whether or not the save directory has changed since the last
	 * time settings were stored.
	 *
	 * @return <tt>true</tt> if the save directory has changed since the last
	 *  time the settings were applied, <tt>false</tt> otherwise
	 */
	public boolean getSaveDirectoryChanged() {
		return _saveDirectoryChanged;
	}

	/**
	 * Sets whether or not the save directory has changed since the last time that 
	 * the settings were applied.
	 *
	 * @param directoryChanged specifies whether or not the save directory has
	 *  changed since the last time settings were applied
	 */
	public void setSaveDirectoryChanged(boolean directoryChanged) {
		_saveDirectoryChanged = directoryChanged;
	}
	
	/**
	 * Returns the main <tt>JDialog</tt> instance for the options window,
	 * allowing other components to position themselves accordingly.
	 *
	 * @return the main options <tt>JDialog</tt> window
	 */
	public JDialog getMainOptionsComponent() {
	    return _constructor.getMainOptionsComponent();
	}

    // Implements ThemeObserver interface
    public void updateTheme() {
        _paneManager = new OptionsPaneManager();
        _constructor = new OptionsConstructor(new OptionsTreeManager(), 
                                              _paneManager);
    }

//  	public static void main(String args[]) {
//  		OptionsMediator mediator = OptionsMediator.instance();
//  		mediator.setOptionsVisible(true);
//  	}
}


