package com.limegroup.gnutella.gui.search;

import com.limegroup.gnutella.gui.*;
import com.sun.java.util.collections.*;
import java.awt.Color;
import java.awt.Insets;
import java.awt.CardLayout;
import java.awt.GridLayout;
import java.awt.event.*;
import javax.swing.*;
import java.util.Enumeration;

import com.limegroup.gnutella.MediaType;
import com.limegroup.gnutella.xml.*;
import com.limegroup.gnutella.xml.gui.*;
import com.limegroup.gnutella.settings.ThemeFileHandler;

/**
 * This class manages the search input area of the search window,
 * including the various text fields for each category for the 
 * current search type and the radio buttons for selecting the search 
 * type.
 */
final class SearchInputManager implements ThemeObserver {
    
    
    /** The color to use for the bottom of the fade.. 
     */
    static Color _fadeColor = ThemeFileHandler.SEARCH_FADE_COLOR.getValue();
    
    /** The number of steps to use in the fade interpolation.
     */
    static final int FADE_STEPS = 62;
    
    /** Constant for <tt>AutoCompleteTextField</tt> for the search field.
     */
    private final AutoCompleteTextField SEARCH_FIELD =
        new SearchField(15);
    //new AutoCompleteTextField(15);
    
    /** Constant <tt>JButton</tt> for the search button.
     */
    private final JButton SEARCH_BUTTON = 
        new JButton(GUIMediator.getStringResource("SEARCH_SEARCH_BUTTON_LABEL"));
    
    /** Constant <tt>JButton</tt> for the stop button.
     */
    private final JButton STOP_BUTTON = 
        new JButton(GUIMediator.getStringResource("SEARCH_STOP_BUTTON_LABEL"));
    
    /** The button group of type specifying radio buttons.
     */
    private final ButtonGroup RADIO_GROUP = new ButtonGroup();
    
    /** The panel that contains the search box and options.
     */
    private final JPanel SEARCH_BOX_PANEL = new BoxPanel(BoxPanel.Y_AXIS);
    
    /** The Label for the default search field.
     */
    private final JLabel DEFAULT_SEARCH_LABEL = 
        new JLabel("", SwingConstants.LEADING);
    
    /** All these Data Structures lying around is disconcerting.  Going to
     *  centralize everything in here.
     */
    private final SchemaMapper SCHEMA_MAPPER = new SchemaMapper();
        
    /** Constant for the search panel <tt>CardLayout</tt>.
     */	 
    private static final CardLayout SEARCH_CARDS = new CardLayout();
    
    /** Used for switching between different search types.
     */
    private static final JPanel MAIN_PANEL = new JPanel(SEARCH_CARDS);

    /** The color to user for the top of the fade, equal to the 
     *  background.
     */
	static Color _backColor = MAIN_PANEL.getBackground();
    
    /** The array of NamedMediaTypes.
     */
    private NamedMediaType[] namedMediaTypes;
    
    /** Set by the radio buttons when they are chosen.  Use this to set the
     *  appropriate filter on the search.
     */
    private int mediaIndexToUse;
    
    private static final String DEFAULT_SEARCH_KEY = "default";

	private static final JPanel RADIO_PANEL = new JPanel();


	private final Ditherer DITHERER = new Ditherer();
    
    /**
     * constructs a new search input manager class, including all displayed
     * elements for search input.
     *
     * @param ssv the <tt>StandardSearchView</tt> mediator class
     */
    SearchInputManager() {        
        // create radio buttons for specific searches
        // *--------------------------------------------------------------------
        // currently we hard code the number of fields.  this may be a practical
        // limitation in the future....
        JPanel[] radioPanels;
        List cumulativeList = 
            NamedMediaType.convertMediaTypes(MediaType.getDefaultMediaTypes());
        //add the schemas
        LimeXMLSchemaRepository rep = LimeXMLSchemaRepository.instance();
        String[] schemas = rep.getAvailableSchemaURIs();
        for (int i = 0; i < schemas.length; i++) {
            String schemaURI = schemas[i];
            // the method name below is not accurate as, in fact, it returns
            // a non-localized identifier name for the XSD schema
            String schemaStr = LimeXMLSchema.getDisplayString(schemaURI);
            if (!NamedMediaType.isDefaultType(schemaStr)){ //not default type
                // Note that there is no resource key defined, so the
                // NamedMediaType constructor will capitalize the media type
                // name to form a default name
                NamedMediaType n =
                    new NamedMediaType(new MediaType(schemaStr, null, null));
                cumulativeList.add(n); // this may not be really added
            }
        }
        namedMediaTypes = new NamedMediaType[cumulativeList.size()];
        cumulativeList.toArray(namedMediaTypes);
        radioPanels = new JPanel[namedMediaTypes.length];
        for (int i = 0; i < namedMediaTypes.length; i++) {
            radioPanels[i] = new JPanel(new GridLayout(1, 0, 0, 0));
            radioPanels[i].setOpaque(false);
        }
		Color buttonColor = ThemeFileHandler.SEARCH_BUTTON_COLOR.getValue();
        for (int i = 0; i < namedMediaTypes.length; i++) {
            final String typeString = namedMediaTypes[i].toString();
            final String type = namedMediaTypes[i].getUnderLyingMediaType().toString();
            JRadioButton button = new JRadioButton(typeString);
			
            button.setForeground(buttonColor);
            button.setOpaque(false);
            button.setActionCommand(type);
            button.addActionListener(new RadioButtonListener(i));
            RADIO_GROUP.add(button);
            radioPanels[i / 2].add(button);
            if (i == 0)
                button.setSelected(true);
        }
        Box radioBox = new Box(BoxLayout.Y_AXIS);
        for (int i = 0; i < radioPanels.length; i++)
            radioBox.add(radioPanels[i]);
        RADIO_PANEL.setBackground(_fadeColor);
        RADIO_PANEL.add(radioBox);
        
        SEARCH_BUTTON.setToolTipText
            (GUIMediator.getStringResource("SEARCH_SEARCH_BUTTON_TIP"));
        STOP_BUTTON.setToolTipText
            (GUIMediator.getStringResource("SEARCH_STOP_BUTTON_TIP"));   
        
        // add the default view...
        JPanel defaultSearch = new DitherPanel(DITHERER);
        JPanel defaultSearchInner = new BoxPanel(BoxPanel.Y_AXIS);
        defaultSearchInner.setOpaque(false);
        
        String fileNameString =
			GUIMediator.getStringResource("SEARCH_DEFAULT_FIELD_TITLE");
        
        DEFAULT_SEARCH_LABEL.setText(""+namedMediaTypes[0]);
        DEFAULT_SEARCH_LABEL.setOpaque(false);
        
		JPanel labelPanel = new BoxPanel(BoxPanel.X_AXIS);
		labelPanel.setOpaque(false);
		labelPanel.add(new JLabel(fileNameString));
		labelPanel.add(Box.createHorizontalGlue());		

		defaultSearchInner.add(labelPanel);
		defaultSearchInner.add(Box.createVerticalStrut(3));
        defaultSearchInner.add(SEARCH_FIELD);
        defaultSearch.add(DEFAULT_SEARCH_LABEL);
        defaultSearch.add(Box.createHorizontalStrut(40));
        defaultSearch.add(defaultSearchInner);
        defaultSearch.setOpaque(false);
        MAIN_PANEL.add(defaultSearch, DEFAULT_SEARCH_KEY);
        
        //add schema items
        ActionListener xmlSearchListener = new SearchListener();
        for (int i = 0; i < schemas.length; i++) {
            // get the URI and string and associate them in a map...
            String schemaURI = schemas[i];
            String schemaStr = LimeXMLSchema.getDisplayString(schemaURI);
            SCHEMA_MAPPER.associateActionWithSchemaURI(schemaStr, schemaURI);
            
            // put them in the stack of cards....
            LimeXMLSchema schema = rep.getSchema(schemaURI);
            InputPanel inputPanel = new InputPanel(schema, xmlSearchListener);
            inputPanel.setOpaque(false);
            JScrollPane scroller = new DitherScrollPane(inputPanel, DITHERER);
            scroller.getViewport().setOpaque(false);
            MAIN_PANEL.add(scroller, schemaStr);
            SCHEMA_MAPPER.associateActionWithInputPanel(schemaStr, inputPanel);
        }
        // set the default view....
        SEARCH_CARDS.show(MAIN_PANEL, DEFAULT_SEARCH_KEY);
        
        Box lowerPanel = new Box(BoxLayout.X_AXIS);
        SEARCH_BUTTON.setMargin(new Insets(0,0,0,0));
        STOP_BUTTON.setMargin(new Insets(0,0,0,0));
        lowerPanel.add(SEARCH_BUTTON);
        lowerPanel.add(STOP_BUTTON);
        
        JPanel innerRadio = new BoxPanel(BoxLayout.Y_AXIS);
        innerRadio.setOpaque(false);
        innerRadio.setBorder(BorderFactory.createEtchedBorder());
        innerRadio.add(RADIO_PANEL);
        
        JPanel outer = new BoxPanel(BoxLayout.Y_AXIS);
        outer.add(MAIN_PANEL);
        outer.add(innerRadio);
        
        SEARCH_BOX_PANEL.add(outer);
        SEARCH_BOX_PANEL.add(lowerPanel); 
        
        SEARCH_BUTTON.addActionListener(new SearchListener());
        SEARCH_FIELD.addActionListener(new SearchListener());
        STOP_BUTTON.addActionListener(new StopListener());
    }
     
    /**
     * Returns the <tt>JComponent</tt> instance containing the UI elements
     * for the search input section of the search tab.
     *
     * @return the <tt>JComponent</tt> instance containing the UI elements
     *  for the search input section of the search tab
     */
    JComponent getSearchBoxPanel() {
        return SEARCH_BOX_PANEL;
    }
    
    /**
     * Returns the string currently in the search field.
     *
     * @return the string currently in the search field
     */
    String getSearchString() {
        return SEARCH_FIELD.getText().trim();
    }
    
    /**
     * Sets the search string in the search field.
     *
     * @param searchString the string to load into the search field
     */
    void setSearchString(String searchString) {
        SEARCH_FIELD.setText(searchString);
    }
    
    /**
     * Notifies the search window of the current connection status,
     * making any necessary changes as a result, such as the 
     * enabled/disabled status of the search button.
     *
     * @param connected the connected status of the client
     */
    void setConnected(final boolean connected) {
        SEARCH_BUTTON.setEnabled(connected);
    }
    
    /**
     * Returns the currently selected <tt>MediaType</tt> for the search 
     * input panel.
     *
     * @return the currently selected <tt>MediaType</tt> for the search 
     *  input panel
     */
    MediaType getSelectedMediaType() {
        return namedMediaTypes[mediaIndexToUse].getUnderLyingMediaType();
    }
    
    /**
     * Returns the currently selected <tt>InputPanel</tt> instance.
     * 
     * @return  the currently selected <tt>InputPanel</tt> instance
     */
    InputPanel getCurrentInputPanel() {
        String action = RADIO_GROUP.getSelection().getActionCommand();
        return SCHEMA_MAPPER.getActionInputPanelAssociation(action);		
    }

	// inherit doc comment
	public void updateTheme() {
		_backColor = MAIN_PANEL.getBackground();
		_fadeColor = ThemeFileHandler.SEARCH_FADE_COLOR.getValue();			
		RADIO_PANEL.setBackground(_fadeColor);
		
		Color buttonColor = ThemeFileHandler.SEARCH_BUTTON_COLOR.getValue();
		
		Enumeration buttons = RADIO_GROUP.getElements();
		while(buttons.hasMoreElements()) {
			AbstractButton curButton = (AbstractButton)buttons.nextElement();
			curButton.setForeground(buttonColor);
			curButton.setOpaque(false);
		}
		DITHERER.updateTheme();
	}
	
	/**
	 * Stores a basic search (not XML) for AutoCompleteness
	 */
	public void storeSearch() {
	    SEARCH_FIELD.addToDictionary();
	}
    
    /**
     * The listener for the radio buttons.
     */
    private class RadioButtonListener implements ActionListener {
        
        private final int MEDIA_INDEX;
        
        /** @param  mediaTypeIndex fill this in so you can associate it with 
         *  MediaType.
         */
        public RadioButtonListener(int mediaTypeIndex) {
            MEDIA_INDEX = mediaTypeIndex;
        }
        
        public void actionPerformed(ActionEvent e) {
            mediaIndexToUse = MEDIA_INDEX;
            String schemaURI = 
                SCHEMA_MAPPER.getActionSchemaAssociation(e.getActionCommand());
            if (schemaURI != null)
                SEARCH_CARDS.show(MAIN_PANEL, e.getActionCommand());
            else {
                DEFAULT_SEARCH_LABEL.setText(""+namedMediaTypes[MEDIA_INDEX]);
                SEARCH_CARDS.show(MAIN_PANEL, DEFAULT_SEARCH_KEY);
            }
        }
    }
    
    /**
     * Helper class for mapping to schemas.
     */
    private class SchemaMapper {
        
        /** Maps Buttons to schemaURIs & InputPanels*/
        private Map itemsToSchemaURI = new HashMap();
        
        private void associateActionWithSchemaURI(String action,
                String schemaURI) {
            Object[] contents = (Object[]) itemsToSchemaURI.get(action);
            if (contents != null)
                ;
            else
                contents = new Object[2];
            contents[0] = schemaURI;
            itemsToSchemaURI.put(action, contents);
        }
        
        /** @return a SchemaURI String.  May return null.
         */
        private String getActionSchemaAssociation(String action) {
            String retString = null;
            Object[] contents =  (Object[]) itemsToSchemaURI.get(action);
            if (contents != null)
                retString = (String) contents[0];
            return retString;
        }
        
        private void associateActionWithInputPanel(String action,
                InputPanel input) {
            Object[] contents =  (Object[]) itemsToSchemaURI.get(action);
            if (contents != null)
                ;
            else
                contents = new Object[2];
            contents[1] = input;
            itemsToSchemaURI.put(action, contents);
        }

        /** @return a SchemaURI String.  May return null.
         */
        private InputPanel getActionInputPanelAssociation(String action) {
            InputPanel retIP = null;
            Object[] contents =  (Object[]) itemsToSchemaURI.get(action);
            if (contents != null)
                retIP = (InputPanel) contents[1];
            return retIP;
        }
    }
}
