package com.limegroup.gnutella.gui.search;

import com.sun.java.util.collections.*;
import com.limegroup.gnutella.*;
import com.limegroup.gnutella.util.*;
import com.limegroup.gnutella.gui.*;
import com.limegroup.gnutella.xml.*;
import java.util.Properties;
import javax.swing.*;
import java.awt.event.*;


/**
 * @author Sumeet Thadani
 * The ResultPanel has a handle to this class. This class will be responsible
 * for selecting lines based on some filters that can be set.
 * <p>
 * Other decisions like which rows to project are delegated to another
 * TreeTableModel called the TableLineProjector
 */
abstract class TableColumnFilter extends ResultPanelModel {

    private ArrayList /** of LimeXMLSchema */ schemas;
    /**
     * maps the column numbers to the displayed column numbers (the 
     * index of item in the map is the index displayed)
     */
    private ArrayList /** of String */ columnNames;

    private ArrayList /** of Integer */ selectedColIndices;
    
    /**
     * number of columns is default view
     */
    private final int normalCols = 9;

    private DisplayManager dispManager ;

    private JPopupMenu menu;
    
    /**
     * Used for creating the column selection menu correctly
     */
    private HashMap /** JCheckBoxMenuItem -> Pair( offet, LimeXMLSchema) */
        itemToPair;
    
    private ResultPanel rp;
    
    //constructor
    public TableColumnFilter(TableLine root, boolean grouped, 
                              ResultPanel rp) {
        super(root,grouped);
        this.rp = rp;
        columnNames = new ArrayList();
        columnNames.add(GUIMediator.getStringResource("RESULT_PANEL_QUALITY"));
        columnNames.add(GUIMediator.getStringResource("RESULT_PANEL_COUNT"));
        columnNames.add(GUIMediator.getStringResource("RESULT_PANEL_NAME"));
        columnNames.add(GUIMediator.getStringResource("RESULT_PANEL_TYPE"));
        columnNames.add(GUIMediator.getStringResource("RESULT_PANEL_SIZE"));
        columnNames.add(GUIMediator.getStringResource("RESULT_PANEL_SPEED"));
        columnNames.add(GUIMediator.getStringResource("RESULT_PANEL_CHAT"));
        columnNames.add(GUIMediator.getStringResource("RESULT_PANEL_LOCATION"));
        columnNames.add(GUIMediator.getStringResource("RESULT_PANEL_VENDOR"));
        schemas = new ArrayList();
        dispManager = DisplayManager.instance();
        selectedColIndices = new ArrayList();
        for (int i=0; i<normalCols; i++){
            String val = dispManager.getValue(DisplayManager.getKeyString(i));
            if(Integer.parseInt(val) > -1)
                selectedColIndices.add(new Integer(i));
        }
    }
    
    ///////////////////////////TreeModel interface//////////////////// 
    /**
     * column is the index of the column 
     */
    public Object getValueAt(Object node, int column) {
        TableLine line = (TableLine)node;        
        Pair p = getIndexSchemaPairForColumn(column);
        if (p==null)
            return "";
        column = p.getKey();
        LimeXMLSchema sch = (LimeXMLSchema)p.getElement();
        return line.getValue(sch,column);
    }
    
    ///////////////////////TableModel interface//////////////////
    
    //Note: All these methods will delegate to the projector
    public int getColumnCount() {
        return selectedColIndices.size();
    }
    
    public String getColumnName(int column) {
        int realIndex = translateColumnValue(column);
        return (String)columnNames.get(realIndex);
    }
    
    public Class getColumnClass(int column) {
        if(column >= columnNames.size()){
            Assert.that(false,"Bad column in table");
            return null;
        }
        //translate the column
        column = translateColumnValue(column);
        switch(column){
            //The following line is critical for JTreeTable to work.
        case DisplayManager.NAME_COL: return TreeTableModel.class;    
        case DisplayManager.EXTENSION_COL: return String.class;
        case DisplayManager.SIZE_COL: return String.class;
        case DisplayManager.SPEED_COL: return ResultSpeed.class;
        case DisplayManager.LOCATION_COL: return CachingEndpoint.class;
        case DisplayManager.QUALITY_COL: return QualityHolder.class;
        case DisplayManager.CHAT_COL: return ChatHolder.class;	
        case DisplayManager.VENDOR_COL:return String.class;
        default:
            return String.class;
        }
    }

    ////////////////other methods////////////////////////////////        
    
   /**
     * add the fields of this schema so that we can display them later
     * <p>
     * @param schema is not allowed to be null
     */
    public void addSchema(LimeXMLSchema schema){
        schemas.add(schema);//add the schema
        Iterator fields = schema.getCanonicalizedFields().iterator();
        while(fields.hasNext()){
            String rawName = ((SchemaFieldInfo)fields.next()).
                                                   getCanonicalizedFieldName();
            String dispName=dispManager.getDisplayName
                                             (rawName,schema.getSchemaURI());
            //always add the column name to the array.
            columnNames.add(dispName);
            //we are going to add the this column to selectedColIndices only
            //if we get a value from displayManager.
            String val = dispManager.getValue(rawName);
            if(Integer.parseInt(val)>-1)
                selectedColIndices.add(new Integer(columnNames.size()-1));
        }
    }
    
    /**
     * returns the identifier of the column
     */
    public Object getColumnId(int column) {
        Object id;
        Pair p = getIndexSchemaPairForColumn(column);
        LimeXMLSchema sch = (LimeXMLSchema)p.getElement();
        column = p.getKey();
        if(sch == null)
            id = DisplayManager.getKeyString(column);
        else {
            List l = sch.getCanonicalizedFields();
            SchemaFieldInfo sfi=(SchemaFieldInfo)l.get(column);
            id = sfi.getCanonicalizedFieldName();
        }
        return id;
    }
    
    public int getPreferredWidth(int column){
        String keyString;
        Pair p = getIndexSchemaPairForColumn(column);
        LimeXMLSchema sch = (LimeXMLSchema)p.getElement();
        column = p.getKey();
        if(sch == null)
            keyString = DisplayManager.getKeyString(column);
        else{
            List l = sch.getCanonicalizedFields();
            SchemaFieldInfo sfi=(SchemaFieldInfo)l.get(column);
            keyString = sfi.getCanonicalizedFieldName();
        }
        return Integer.parseInt(dispManager.getValue(keyString));
    }

    public JPopupMenu getColumnSelectionMenu() {
        menu = new JPopupMenu();
        itemToPair = new HashMap(); /*JCheckBoxItem -> Pair(Schema,index) */
        JMenu subMenu=null;
        //create the menu....and add action listers to each of them
        int selIndex = 0;//index of the next selected column
        int selValue=((Integer)selectedColIndices.get(selIndex)).intValue();
        
        //schema book keeping
        int currSchemaIndex=-1;//index of the current schema we will deal with
        LimeXMLSchema currSchema = null;//
        int currSchemaStartIndex = 0;//intialize
        int currSchemaEndIndex = normalCols;//initialize to 1 past current sch
        int fieldCount=0;
        
        Iterator iter = columnNames.iterator();        

        for(int i=0; iter.hasNext() ; i++){
            if(i==currSchemaEndIndex){//ok next schema
                currSchemaIndex++;//even last iteration should work
                currSchema =(LimeXMLSchema)schemas.get(currSchemaIndex);
                subMenu = new JMenu
                (LimeXMLSchema.getDisplayString(currSchema.getSchemaURI()));
                menu.add(subMenu);
                currSchemaStartIndex=currSchemaEndIndex;
                int z = currSchema.getCanonicalizedFields().size();
                currSchemaEndIndex += z;
                fieldCount = 0;
            }
            String label = (String)iter.next();
            boolean selected = false;
            if(i == selValue){
                selected = true;
                if(selIndex < selectedColIndices.size()-1){//increment till end
                    selIndex++;
                    selValue=
                    ((Integer)selectedColIndices.get(selIndex)).intValue();
                }
            }
            JCheckBoxMenuItem item = new JCheckBoxMenuItem(label,selected);
            Pair p = new Pair(fieldCount, currSchema);
            itemToPair.put(item,p);
            fieldCount++;
            if(subMenu == null)
                menu.add(item);
            else
                subMenu.add(item);
            item.addActionListener(new SelectionActionListener());
        }
        return menu;
    }

    /**
     * translates the column to a pair, which is the format the RPM needs,
     * and then asks super to deal with it
     */
    void sort(int column, boolean ascending) {
        //1. translate the column into a (schema,offsetPair)
        Pair p = getIndexSchemaPairForColumn(column);
        //2. ask underlying model to sort itself
        super.sort(p,ascending);
    }

    /////////////////private helper methods ///////////////
    private  int getDefaultWidth(String rawName){
        int i = rawName.indexOf("^");
        if(i < 0)
            return 30;//defualt size
        return Integer.parseInt(rawName.substring(i+1));
    }
 
    private int translateColumnValue(int column){
        return ((Integer)selectedColIndices.get(column)).intValue();     
    }
        
    Pair getIndexSchemaPairForColumn(int column){        
        //translate the column
        column = translateColumnValue(column);
        if(column < normalCols)//we need the value from a normal column
            return new Pair(column, null);//no schema and same index
        //find the schema
        Iterator iter = schemas.iterator();
        LimeXMLSchema sch;
        int cumulativeColumnCount=normalCols;//initialize
        int tmp;
        while(iter.hasNext()){
            sch = (LimeXMLSchema)iter.next();
            int size = sch.getCanonicalizedFields().size();
            tmp = cumulativeColumnCount;
            cumulativeColumnCount += size;//the total number of columns seen 
            if(column < cumulativeColumnCount)//this is NOT the correct schema
                return new Pair(column-tmp,sch);
        }
        return null;
    }
    
    /**
     * private inncer class to do some remodelling based on the selected value
     */
    private class SelectionActionListener implements ActionListener {
        public void actionPerformed(ActionEvent e) {
            JCheckBoxMenuItem item = (JCheckBoxMenuItem)e.getSource();
            Pair p = (Pair)itemToPair.get(item);
            int offset = p.getKey();
            LimeXMLSchema sch = (LimeXMLSchema)p.getElement();
            //find the index of the field in columnNames
            int index =-1;//intialize
            if(sch==null)
                index = offset;
            else{
                index = normalCols;
                Iterator iter = schemas.iterator();
                while(iter.hasNext()){
                    LimeXMLSchema cSch = (LimeXMLSchema)iter.next();
                    if(sch != cSch)
                        index += cSch.getCanonicalizedFields().size();
                    else
                        break;                        
                }
                index += offset;
                //System.out.println("Sumeet the index = "+index);
            }                        
            //add that index at the correct location in selectedColIndices
            if(item.getState()) {//if checked insert
                Iterator iter = selectedColIndices.iterator();
                boolean found = false;
                boolean exists = false;
                int lower = -1;//initialize
                int higher= ((Integer)iter.next()).intValue();
                int i = 0;//i is index of higher
                while(iter.hasNext()){
                    lower = higher;
                    higher = ((Integer)iter.next()).intValue();
                    i++;
                    if(index==lower || index == higher){
                        exists = true;
                        break; //it's already in there
                    }
                    else if(index > lower && index < higher){
                        found = true;
                        break;
                    }                    
                }
                if(exists){//its already in there return
                    menu = null;
                    itemToPair = null;
                    return;
                }
                if(found)//put it in the middle
                    selectedColIndices.add(i,new Integer(index));
                else//not found append to end
                    selectedColIndices.add(new Integer(index));                
                //make changes to the displayManager
                String rawName = "";
                int prefWidth =-1;
                if(sch == null){//normal column being added
                    rawName = DisplayManager.getKeyString(offset);
                    prefWidth = DisplayManager.getDefaultWidth(offset);
                }
                else{
                    Properties props=
                                 dispManager.getProperties(sch.getSchemaURI());
                    List fields = sch.getCanonicalizedFields();
                    SchemaFieldInfo sfi = (SchemaFieldInfo) fields.get(offset);
                    rawName = sfi.getCanonicalizedFieldName();
                    String dispName = props.getProperty(rawName,"null");
                    prefWidth = getDefaultWidth(dispName);
                }
                //System.out.println("Sumeet "+rawName+", "+prefWidth);
                dispManager.addColumn(rawName,prefWidth);
                rp.columnsChanged();                
            }
            else{ //item was unchecked, remove from selectedColIndices
                if(selectedColIndices.size()==1){//only 1 col, can't remove
                    rp.showColumnWarning();
                    return;
                }
                Iterator iter = selectedColIndices.iterator();
                boolean found = false;
                int i = -1;
                while(iter.hasNext()){
                    int val = ((Integer)iter.next()).intValue();
                    i++;//i points to current index
                    if(val==index){
                        //System.out.println("Sumeet: found index "+ i);
                        found = true;
                        break;
                    }
                }
                if(found){
                    //System.out.println("Sumeet: removing index "+ i);
                    selectedColIndices.remove(i);//remove it
                    String rawName = ""; //initialize
                    if(sch == null)//normal column
                        rawName = DisplayManager.getKeyString(offset);
                    else{
                        List fields = sch.getCanonicalizedFields();
                        SchemaFieldInfo sf=(SchemaFieldInfo)fields.get(offset);
                        rawName = sf.getCanonicalizedFieldName();
                    }
                    dispManager.removeColumn(rawName);
                    rp.columnsChanged();
                }
            }
            menu = null;
            itemToPair = null;
        }
    }
        
}//end of class

