package com.limegroup.gnutella.gui.tables;

import javax.swing.table.TableColumn;
import javax.swing.table.TableCellRenderer;

import com.limegroup.gnutella.gui.GUIMediator;

/**
 * A personalized TableColumn for storing extended information.
 * This class provides support for storing:
 * <ul>
 * <li> The model number of the column
 *      (this doubles as the default order number) </li>
 * <li> The messageBundle ID of the column </li>
 * <li> The default width of the column
 *      (as opposed to the current preferredSize) </li>
 * <li> The default visibility of the column
 *      (as opposed to the current visibility) </li>
 * <li> The class of this column </li>
 * </ul>
 */
public class LimeTableColumn extends TableColumn {

    /**
     * Variable for the HeaderRenderer for all components.
     */
    public static final TableCellRenderer HEADER_RENDERER =
        new SortHeaderRenderer();

    private final boolean defaultVisibility;
    private final int defaultWidth;
    private final String messageId;
    private final String name;
    private final Class clazz;
    
    private boolean initialized = false;

    public String toString() {
        return messageId;
    }

    /**
     * Creates a new TableColumn.  If you do not know a default width,
     * use -1.
     */
    public LimeTableColumn(int model, final String id,
                    int width, boolean vis, Class clazz) {
        super(model);
        initialized = true;

        this.defaultVisibility = vis;

        this.defaultWidth = width;
        if( defaultWidth != -1 )
            super.setPreferredWidth(width);

        this.messageId = id;
        super.setIdentifier(id);

        this.name = GUIMediator.getStringResource(id);
        super.setHeaderValue( name );

        this.clazz = clazz;

        super.setHeaderRenderer( HEADER_RENDERER );
    }

    /**
     * Gets the default visibility for this column.
     */
    public boolean getDefaultVisibility() {
        return defaultVisibility;
    }

    /**
     * Gets the default width for this column.
     */
    public int getDefaultWidth() {
        return defaultWidth;
    }

    /**
     * Gets the default order for this column.
     */
    public int getDefaultOrder() {
        return getModelIndex();
    }

    /**
     * Get the name, as a string.
     */
    public String getName() {
        return name;
    }

    /**
     * Gets the class of this column.
     */
    public Class getColumnClass() {
        return clazz;
    }

    /**
     * Gets the Id as a string.
     */
    public String getId() {
        return messageId;
    }
    
    /**
     * The following methods are overridden to ensure that we never
     * accidentally change the default values.  This is absolutely
     * necessary so that the DefaultColumnPreferenceHandler can correctly
     * write the default values to the settings.
     * Unfortunately, Java 1.1.8 calls some of these methods during
     * construction.  As a workaround, we check a boolean variable
     * and silently return if it is during initialization.
     */

    /**
     * Disallows changing of model number
     */
    public void setModelIndex(int idx) {
        if(!initialized) return;
        throw new IllegalStateException("cannot change model index");
    }

    /**
     * Disallows changing of header value
     */
    public void setHeaderValue(Object val) {
        if(!initialized) return;
        throw new IllegalStateException("cannot change header value");
    }

    /**
     * Disallows changing of identifier
     */
    public void setIdentifier(Object id) {
        if(!initialized) return;
        throw new IllegalStateException("cannot change id");
    }
}

        
