/****************************************************************************
**
** $Id: ImportVendorDialogImpl.cpp,v 1.2 2003/10/23 00:22:39 hemer Exp $
**
** Copyright (C) 2001-2003 Frank Hemer <frank@hemer.org>
**
**----------------------------------------------------------------------------
**
** This class implements the cvs pserver auth protocol as nonblocking
**
**----------------------------------------------------------------------------
**
** LinCVS is available under two different licenses:
**
** If LinCVS is linked against the GPLed version of Qt 
** LinCVS is released under the terms of GPL also.
**
** If LinCVS is linked against a nonGPLed version of Qt 
** LinCVS is released under the terms of the 
** LinCVS License for non-Unix platforms (LLNU)
**
**
** LinCVS License for non-Unix platforms (LLNU):
**
** Redistribution and use in binary form, without modification, 
** are permitted provided that the following conditions are met:
**
** 1. Redistributions in binary form must reproduce the above copyright
**    notice, this list of conditions and the following disclaimer in the
**    documentation and/or other materials provided with the distribution.
** 2. It is not permitted to distribute the binary package under a name
**    different than LinCVS.
** 3. The name of the authors may not be used to endorse or promote
**    products derived from this software without specific prior written
**    permission.
** 4. The source code is the creative property of the authors.
**    Extensions and development under the terms of the Gnu Public License
**    are limited to the Unix platform. Any distribution or compilation of 
**    the source code against libraries licensed other than gpl requires 
**    the written permission of the authors.
**
**
** THIS SOFTWARE IS PROVIDED BY THE AUTHOR "AS IS" AND ANY EXPRESS OR 
** IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED 
** WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE 
** ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY 
** DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL 
** DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE 
** GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS 
** INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, 
** WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING 
** NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS 
** SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
**
**
**
** LinCVS License for Unix platforms:
**
** This program is free software; you can redistribute it and/or modify 
** it under the terms of the GNU General Public License as published by 
** the Free Software Foundation; either version 2 of the License, or 
** (at your  option) any later version.  This program is distributed in 
** the hope that it will be useful, but WITHOUT ANY WARRANTY; without 
** even the implied warranty of MERCHANTABILITY or FITNESS FOR A 
** PARTICULAR PURPOSE.
**
** See the GNU General Public License for more details.
**
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software Foundation,
** Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
**
*****************************************************************************/

#include "ImportVendorDialogImpl.h"

#include <qapplication.h>
#include <qfiledialog.h>
#include <qpushbutton.h>
#include <qmessagebox.h>
#include <qcombobox.h>
#include <qcheckbox.h>
#include <qwhatsthis.h>

#include "WrappersIgnoreDialogImpl.h"
#include "Validators.h"
#include "pixmapcache.h"

ImportVendorDialogImpl::ImportVendorDialogImpl(QString caption,
					       const QIconSet& whatsThisIconSet,
					       QString topDir,
					       QString workDir,
					       QString commitInfoTxt,
					       QWidget * parent,
					       const char* name,
					       bool modal, WFlags f)
  : ImportVendorDialog(parent, name, modal, f),
    m_topCvsDir(topDir),
    m_workDir(workDir),
    m_vendorTagListOnEdit(FALSE),
    m_releaseTagListOnEdit(FALSE),
    m_pOldVendorTagPixmap(NULL),
    m_pOldReleaseTagPixmap(NULL)
{
  m_pWhatsThis->setIconSet(whatsThisIconSet);
  m_pWhatsThis->setMaximumWidth(m_pWhatsThis->height());
  setCaption(caption);

  m_VendorTag->setAutoCompletion(FALSE);
  m_VendorTag->setValidator(new TagValidator(m_VendorTag));
  connect( m_VendorTag, SIGNAL(textChanged(const QString &)), this, SLOT(textChangedVendorTag(const QString &)));
  connect( m_VendorTag, SIGNAL(activated(const QString &)), this, SLOT(activatedVendorTag(const QString &)));

  m_ReleaseTag->setAutoCompletion(FALSE);
  m_ReleaseTag->setValidator(new TagValidator(m_ReleaseTag));
  connect( m_ReleaseTag, SIGNAL(textChanged(const QString &)), this, SLOT(textChangedReleaseTag(const QString &)));
  connect( m_ReleaseTag, SIGNAL(activated(const QString &)), this, SLOT(activatedReleaseTag(const QString &)));

  m_VendorBranch->setValidator(new RevisionValidator(m_VendorBranch));

  tagListUpdated();

  CommitInfo->setText(commitInfoTxt);

  //build ignorelist from cvs default + homedir/.cvsignore settings
  //per dir is ignored because these are appended to the -I option anyway
  m_ignoreList = QStringList::split(" ",CVSIGNOREFILES);
  QFile cvsignore(QDir::homeDirPath() + "/.cvsignore");
  if(cvsignore.open(IO_ReadOnly)) {
    QTextStream ignoreStream(&cvsignore); 
    while(!ignoreStream.eof()) {
      QString line = ignoreStream.readLine().simplifyWhiteSpace();
      if (!line.isEmpty()) {
	m_ignoreList.append(line);
      }
    }
    cvsignore.close();
  }
  m_ignoreString = m_ignoreList.join(" ");
  if (m_ignoreString.find("!") > -1) m_ignoreString = m_ignoreString.mid(m_ignoreString.findRev("!", -1) + 1);

  m_binaryList = BINARYFILESPECLIST;

  m_pTemplateField = 0;
  connect( CommitInfoChoice, SIGNAL( activated( int ) ),
	   this, SLOT( selected( int ) ) );
  updateTemplate();

}

ImportVendorDialogImpl::~ImportVendorDialogImpl() {

}

void ImportVendorDialogImpl::browseDirClicked() {
  QString importDir = QFileDialog::getExistingDirectory (m_workDir, this, NULL,
						  tr("Choose import directory"),
						  true);

  if (!importDir.isEmpty ()) {
    if (importDir.endsWith("/") && (importDir.length() > 1)) importDir.truncate(importDir.length()-1);
    ImportDir->setCurrentText(importDir);
  }
}

void ImportVendorDialogImpl::editIgnoreClicked() {
  WrappersIgnoreDialogImpl* dlg = new WrappersIgnoreDialogImpl(&m_ignoreList,
                                                               *(m_pWhatsThis->iconSet()),
                                                               tr("Files to ignore"),
                                                               this, 0, true);
  dlg->exec();
  m_ignoreString = m_ignoreList.join(" ");
  if (m_ignoreString.find("!") > -1) m_ignoreString = m_ignoreString.mid(m_ignoreString.findRev("!", -1) + 1);
  delete dlg;
}

void ImportVendorDialogImpl::editBinaryClicked() {
  WrappersIgnoreDialogImpl* dlg = new WrappersIgnoreDialogImpl(&m_binaryList,
                                                               *(m_pWhatsThis->iconSet()),
                                                               tr("Import files as binary"),
                                                               this, 0, true);
  dlg->exec();
  delete dlg;
}

void ImportVendorDialogImpl::previewClicked() {

  m_bScanInterrupted = false;

  //previewDlg = new ImportPreview();
  QApplication::setOverrideCursor(Qt::waitCursor);

  progressDlg = new ProgressDialog(this,NULL,true, Qt::WStyle_Customize | Qt::WStyle_NoBorder);
  connect(progressDlg->CancelButton,SIGNAL(clicked()),this,SLOT(cancelPreviewScan()));
  progressDlg->m_InfoMessage->setText("Scanning dir:");
  progressDlg->show();
  progressDlg->update();

  previewDlg = new ImportPreview(this, NULL, true,
                                 Qt::WStyle_SysMenu | Qt::WStyle_MinMax | 
                                 Qt::WStyle_DialogBorder | Qt::WType_Dialog |
				 Qt::WStyle_ContextHelp);
  previewDlg->m_pWhatsThis->setIconSet(*(m_pWhatsThis->iconSet()));
  previewDlg->m_pWhatsThis->setMaximumWidth(previewDlg->m_pWhatsThis->height());

  addToPreview(ImportDir->currentText().replace(QRegExp ("/+$"),""));

  progressDlg->hide();
  delete(progressDlg);

  QApplication::restoreOverrideCursor();
  qApp->processEvents();
  
  if (!m_bScanInterrupted) {
    previewDlg->exec();
  }
  
  delete previewDlg;
}

void ImportVendorDialogImpl::addToPreview( QString& path) {

  progressDlg->m_InfoText->setText(path);
  qApp->processEvents();
  if (m_bScanInterrupted) {
    return;
  }

  QDir D(path);
  D.setFilter( QDir::Files | QDir::Dirs | QDir::Hidden);
  QStringList allDirEntrieslist = D.entryList();
  allDirEntrieslist.remove( "." );
  allDirEntrieslist.remove( ".." );
  allDirEntrieslist.remove( "CVS" );

  //collect ignore info
  QString filesToIgnore = m_ignoreString;
  QFile cvsignore(path + "/.cvsignore");
  if(cvsignore.open(IO_ReadOnly)) {
    QTextStream ignoreStream(&cvsignore); 
    while(!ignoreStream.eof()) {
      filesToIgnore += " " + ignoreStream.readLine().simplifyWhiteSpace();
    }
    cvsignore.close();
  }
  filesToIgnore.replace(QRegExp("\\."),"\\.");
  filesToIgnore.replace(QRegExp("\\?"),".");
  filesToIgnore.replace(QRegExp("\\*"),".*");
  filesToIgnore.replace(QRegExp("\\$"), "\\$");
  if (filesToIgnore.find("!") > -1) filesToIgnore = filesToIgnore.mid(filesToIgnore.findRev("!", -1) + 1);

  filesToIgnore.replace(QRegExp(" "), "|");
  filesToIgnore = "^(" + filesToIgnore + ")$";

  //collect binary info
  QString binaryFiles = getBinaryWildcards().join(" ");
  
  binaryFiles.replace(QRegExp("\\."),"\\.");
  binaryFiles.replace(QRegExp("\\?"),".");
  binaryFiles.replace(QRegExp("\\*"),".*");
  binaryFiles.replace(QRegExp("\\$"), "\\$");
  
  binaryFiles.replace(QRegExp(" "), "|");
  binaryFiles = "^(" + binaryFiles + ")$";

  //now insert entries to appropriate view
  QStringList::Iterator it;
  for ( it = allDirEntrieslist.begin(); it != allDirEntrieslist.end(); it++ ) {

    QString file = path + "/" + (*it);
    QFileInfo fileInfo( file);

    if(fileInfo.isDir() && fileInfo.isReadable() && fileInfo.isExecutable() && ((*it).find(QRegExp(filesToIgnore)) == -1) ) {// dir

      addToPreview(file.replace(QRegExp ("/+$"),""));

    } else if(fileInfo.isFile() && fileInfo.isReadable()) {// file

      if ((*it).find(QRegExp(filesToIgnore)) != -1) {// ignore

	previewDlg->m_IgnoreList->insertItem(new QListViewItem(previewDlg->m_IgnoreList,file));

      } else {// don't ignore

	if ((*it).find(QRegExp(binaryFiles)) != -1) {// binary
	  previewDlg->m_ImportList->insertItem(new QListViewItem(previewDlg->m_ImportList,file,"-kb"));
	} else {// not binary
	  previewDlg->m_ImportList->insertItem(new QListViewItem(previewDlg->m_ImportList,file));
	}
      }
    } else {

      previewDlg->m_RestList->insertItem(new QListViewItem(previewDlg->m_RestList,file));
      
    }
  }

}

void ImportVendorDialogImpl::fetchVendorButtonClicked() {

  m_rTagList.clear();
  m_vTagList.clear();
  m_tagMap.clear();
  QString strCmd = CvsOptions::cmprStr();
  strCmd += " log -b -h";
  QString files = "";
  callInteractive( m_topCvsDir, m_workDir, strCmd,
		   files, LOG_CMD,
		   ExtApps::g_cvsRsh.path,  //additional options of cvsRsh not supported yet
		   TRUE);

} 

QStringList ImportVendorDialogImpl::getIgnoreWildcards() {
  return QStringList::split(" ",m_ignoreString);
}

QStringList ImportVendorDialogImpl::getBinaryWildcards() {
  return m_binaryList;
}

void ImportVendorDialogImpl::cancelPreviewScan() {
  m_bScanInterrupted = true;
}

void ImportVendorDialogImpl::setEnabled(bool state) {
  ImportVendorDialog::setEnabled(state);
}

void ImportVendorDialogImpl::tagListUpdated() {

  m_vendorTagListOnUpdate = TRUE;
  m_releaseTagListOnUpdate = TRUE;
  m_vendorTagListOnEdit = FALSE;
  m_releaseTagListOnEdit = FALSE;
  m_VendorTag->clear();
  m_ReleaseTag->clear();

  m_VendorTag->insertStringList(m_vTagList);
  m_ReleaseTag->insertStringList(m_rTagList);

  setPixmaps(m_VendorTag);
  m_oldVendorTagText = m_VendorTag->currentText();
  m_vendorTagListOnUpdate = FALSE;
  setPixmaps(m_ReleaseTag);
  m_oldReleaseTagText = m_VendorTag->currentText();
  m_releaseTagListOnUpdate = FALSE;

  setVendorBranch(m_VendorTag->currentText());
}

void ImportVendorDialogImpl::setPixmaps(QComboBox * box) {
  int i;
  for (i = 0; i < box->count(); ++i) {
    QString txt = box->text(i);
    if (txt.startsWith("T: ")) box->changeItem(findEmbeddedPixmap("TagTag30x16"),txt.mid(3),i);
    else if (txt.startsWith("V: ")) box->changeItem(findEmbeddedPixmap("BranchTag30x16"),txt.mid(3),i);
  }
  if (box == m_VendorTag) m_oldVendorTagIdx = 0;
  else if (box == m_ReleaseTag) m_oldReleaseTagIdx = 0;

  if (i == 0) {
    box->insertItem(findEmbeddedPixmap("Tag30x16"),"",box == m_VendorTag ? m_oldVendorTagIdx : m_oldReleaseTagIdx);
  }
  box->setCurrentItem(box == m_VendorTag ? m_oldVendorTagIdx : m_oldReleaseTagIdx);//removes T: or B: for the lineEdit
}

void ImportVendorDialogImpl::activatedVendorTag(const QString & s) {

  if (m_pOldVendorTagPixmap) {
    m_VendorTag->changeItem(*m_pOldVendorTagPixmap,m_oldVendorTagText,m_oldVendorTagIdx);
    m_pOldVendorTagPixmap = NULL;
  }
  m_oldVendorTagText = s;
  m_oldVendorTagIdx = m_VendorTag->currentItem();
  m_vendorTagListOnEdit = FALSE;
  m_VendorTag->lineEdit()->selectAll();
}

void ImportVendorDialogImpl::activatedReleaseTag(const QString & s) {

  if (m_pOldReleaseTagPixmap) {
    m_ReleaseTag->changeItem(*m_pOldReleaseTagPixmap,m_oldReleaseTagText,m_oldReleaseTagIdx);
    m_pOldReleaseTagPixmap = NULL;
  }
  m_oldReleaseTagText = s;
  m_oldReleaseTagIdx = m_ReleaseTag->currentItem();
  m_releaseTagListOnEdit = FALSE;
  m_ReleaseTag->lineEdit()->selectAll();
}

void ImportVendorDialogImpl::textChangedVendorTag(const QString &txt) {

  if (m_vendorTagListOnUpdate) return;

  static int lastLen = 0;
  int idx = -1;
  for (int i = 0; i<m_VendorTag->count(); ++i) {
    if (m_VendorTag->text(i).startsWith(txt)) {
      idx = i;
      break;
    }
  }

  if (idx > -1 && (lastLen < (int)txt.length()) || (m_VendorTag->text(idx).length() == txt.length()) ) {

    m_vendorTagListOnUpdate = TRUE;
    int cursorPos = m_VendorTag->lineEdit()->cursorPosition();
    m_VendorTag->setCurrentItem(idx);
    QString current = m_VendorTag->currentText();
    activatedVendorTag(current);
    if (current.length() != txt.length()) m_VendorTag->lineEdit()->setSelection(current.length(), txt.length()-current.length());
    else m_VendorTag->lineEdit()->setCursorPosition(cursorPos);
    m_vendorTagListOnUpdate = FALSE;
    setVendorBranch(current);

  } else if (m_oldVendorTagIdx == m_VendorTag->currentItem()) {

    if (!m_vendorTagListOnEdit) {
      m_vendorTagListOnEdit = TRUE;
      QPixmap const * p = m_VendorTag->pixmap(m_oldVendorTagIdx);
      if (p) {
	m_pOldVendorTagPixmap = new QPixmap(*p);
      } else {
	m_pOldVendorTagPixmap = NULL;
      }
      m_VendorTag->changeItem(findEmbeddedPixmap("Tag30x16"),m_oldVendorTagText,m_oldVendorTagIdx);
    }

    setVendorBranch(txt);

  } else {
    setVendorBranch(txt);
  }

  lastLen = txt.length();
}

void ImportVendorDialogImpl::textChangedReleaseTag(const QString &txt) {

  if (m_releaseTagListOnUpdate) return;

  static int lastLen = 0;
  int idx = -1;
  for (int i = 0; i<m_ReleaseTag->count(); ++i) {
    if (m_ReleaseTag->text(i).startsWith(txt)) {
      idx = i;
      break;
    }
  }

  if (idx > -1 && (lastLen < (int)txt.length()) || (m_ReleaseTag->text(idx).length() == txt.length()) ) {

    m_releaseTagListOnUpdate = TRUE;
    int cursorPos = m_ReleaseTag->lineEdit()->cursorPosition();
    m_ReleaseTag->setCurrentItem(idx);
    QString current = m_ReleaseTag->currentText();
    activatedReleaseTag(current);
    if (current.length() != txt.length()) m_ReleaseTag->lineEdit()->setSelection(current.length(), txt.length()-current.length());
    else m_ReleaseTag->lineEdit()->setCursorPosition(cursorPos);
    m_releaseTagListOnUpdate = FALSE;

  } else if (m_oldReleaseTagIdx == m_ReleaseTag->currentItem()) {

    if (!m_releaseTagListOnEdit) {
      m_releaseTagListOnEdit = TRUE;
      QPixmap const * p = m_ReleaseTag->pixmap(m_oldReleaseTagIdx);
      if (p) {
	m_pOldReleaseTagPixmap = new QPixmap(*p);
      } else {
	m_pOldReleaseTagPixmap = NULL;
      }
      m_ReleaseTag->changeItem(findEmbeddedPixmap("Tag30x16"),m_oldReleaseTagText,m_oldReleaseTagIdx);
    }
  }

  lastLen = txt.length();
}

void ImportVendorDialogImpl::setVendorBranch(const QString &txt) {
  QMap<QString,QString>::ConstIterator it = m_tagMap.find(txt);
  if (it != m_tagMap.end()) {
    DefaultBranchCheckBox->setEnabled(FALSE);
    DefaultBranchCheckBox->setChecked(FALSE);
    m_VendorBranch->setReadOnly(TRUE);
    m_VendorBranch->setText(it.data());
  } else {
    m_VendorBranch->setText("");
    m_VendorBranch->setReadOnly(FALSE);
    DefaultBranchCheckBox->setEnabled(TRUE);
  }
}

void ImportVendorDialogImpl::updateTemplate()
{
  int i;
  for (i = CommitInfoChoice->count(); i>0; --i)
    CommitInfoChoice->removeItem(i-1);

  delete [] m_pTemplateField;
  m_pTemplateField = 0;    //needed if count <= 0...

  QString hstDir = tmpDir + "/Template";
  QDir d(hstDir);

  const QStringList list = d.entryList(QDir::Files | QDir::NoSymLinks, QDir::Time);
  QString str;
  int curItem = -1;
  int curPos = 0;
  int count = (int)list.count();
  if (count>0) {
    m_pTemplateField = new QString[count];
  }

  for (i = 0; i < count; i++) {
    QFile tmpFile(hstDir + "/" + list[i]);
    if (tmpFile.open(IO_ReadOnly)) {
      QTextStream hstFile(&tmpFile); 
      str = "";
      m_pTemplateField[curPos] = "";
      while(!hstFile.eof()) {
	QString line;
        line = hstFile.readLine();
	str += line + " ";
	m_pTemplateField[curPos] += line + "\n";
      }
      str += " ";
      tmpFile.close();

      QString abbrev = str.replace( QRegExp("[\\a\\f\\n\\r\\t\\v]"),"").stripWhiteSpace();

      if (list[i] == "rcsinfo") {
          if (!abbrev.isEmpty()) {
              CommitInfo->setText(m_pTemplateField[curPos]);
              curItem = curPos;
          }
      }

      if (!abbrev.isEmpty()) {
	CommitInfoChoice->insertItem(abbrev);
	if (CommitInfo->text().stripWhiteSpace() == str.stripWhiteSpace()) curItem = curPos;
	curPos++;
      }
    }

    //remove all obsolete history files (i >= historySize)
    if ( (i >= HistorySize::g_commitinfo-1) && (list[i] != "rcsinfo") ) tmpFile.remove();
  }

  if (curItem > -1) CommitInfoChoice->setCurrentItem(curItem);
  if (CommitInfo->text().isEmpty()) {
      CommitInfoChoice->setCurrentItem(0);
      if (count>0) {
	CommitInfo->setText(m_pTemplateField[0]);
      }
  }
}

void ImportVendorDialogImpl::selected(int item)
{
  CommitInfo->setText(m_pTemplateField[item]);
}

void ImportVendorDialogImpl::cvsCallStarted() {
  setEnabled(FALSE);
  QApplication::setOverrideCursor(Qt::waitCursor);
}

void ImportVendorDialogImpl::cvsCallFinished() {
  QApplication::restoreOverrideCursor();
  setEnabled(TRUE);
}

void ImportVendorDialogImpl::afterCall( int cmd, CvsBuffer* output, bool failed) {
  cvsCallFinished();
  if (failed) return;
  int pos = 0;

  QString line;
  unsigned int len = (*output).numLines();
  for (unsigned int i = 0; i<len;i++) {
    line = (*output).textLine(i);
    if (line.isEmpty()) continue;

    switch (cmd) {
      case LOG_CMD: {
	if (pos == 0) {
	  if (line.startsWith("symbolic names:") ) pos = 1;
	}
	else if ( line.startsWith("keyword substitution:")
		  || (line.startsWith("total revisions:"))
		  || (line.startsWith("===================================================================")) ) {
	  pos = 0;
	} else if (line.startsWith("\t")) {

	  int pos1 = line.find(':');
	  QString tag;
	  QString rev;
	  if (pos1 > -1) {
	    tag = line.mid(1,pos1-1);
	    rev = line.mid(pos1+2);
	    if (rev.contains('.') == 2) {
	      QString entry = "V: "+tag;
	      if (m_vTagList.find(entry) == m_vTagList.end()) {
		m_vTagList.append(entry);
	      }
	    } else if (rev.contains('.') % 2) {
	      rev.replace(QRegExp("\\.0\\."),".");
	      if ( !(rev.contains('.') % 2) ) {
		continue;//ignore branch with more than two dots
	      } else {
		QString entry = "T: "+tag;
		if (m_rTagList.find(entry) == m_rTagList.end()) {
		  m_rTagList.append(entry);
		}
	      }
	    } else {
	      qDebug("ignoring unknown entry: ->"+tag+"<-, rev: ->"+rev+"<-");
	      continue;
	    }
	    if (!m_tagMap.contains(tag)) {
	      m_tagMap.insert(tag,rev);
	    }
	  }
	} else {
	  qDebug("unknown line on GET_TAG_INFO: "+line);
	  continue;
	}
	break;
      }
    }
  }

  switch (cmd) {
    case LOG_CMD: {
      m_vTagList.sort();
      m_rTagList.sort();
      tagListUpdated();
      break;
    }
    default: qDebug("ImportVendorDialogImpl: unknown cmd: "+QString::number(cmd));
  }
}

QString ImportVendorDialogImpl::getImportDir() {
  QString importDir = ImportDir->currentText();
  if (importDir.endsWith("/") && (importDir.length() > 1)) importDir.truncate(importDir.length()-1);
  return importDir;
}

QString ImportVendorDialogImpl::getVendorTag() {
  return m_VendorTag->currentText();
}

QString ImportVendorDialogImpl::getVendorBranch() {
  return m_VendorBranch->text();
}

QString ImportVendorDialogImpl::getReleaseTag() {
  return m_ReleaseTag->currentText();
}

QString ImportVendorDialogImpl::getCommitInfo() {
  return CommitInfo->text();
}

void ImportVendorDialogImpl::accept() {

  if (ImportDir->currentText().isEmpty()) {
    QMessageBox::warning(0, tr("Warning"), tr("You have to enter an import directory"), QMessageBox::Ok, QMessageBox::NoButton);
  } else if (CommitInfo->text().isEmpty()) {
    QMessageBox::warning(0, tr("Warning"), tr("You have to enter an import message"), QMessageBox::Ok, QMessageBox::NoButton);
  } else if (m_VendorTag->currentText().isEmpty()) {
    QMessageBox::warning(0, tr("Warning"), tr("You have to enter a vendor tag"), QMessageBox::Ok, QMessageBox::NoButton);
  } else if (m_ReleaseTag->currentText().isEmpty()) {
    QMessageBox::warning(0, tr("Warning"), tr("You have to enter a release tag"), QMessageBox::Ok, QMessageBox::NoButton);
  } else if ( (!DefaultBranchCheckBox->isChecked()) && (m_VendorBranch->text().isEmpty()) ) {
    QMessageBox::warning(0, tr("Warning"), tr("You have to enter a vendor branch"), QMessageBox::Ok, QMessageBox::NoButton);
  } else if ( (!DefaultBranchCheckBox->isChecked()) && ( !(m_VendorBranch->text().contains('.') == 2)) ) {
    QMessageBox::warning(0, tr("Warning"), tr("A vendor branch requires exactly two dots: x.y.z"), QMessageBox::Ok, QMessageBox::NoButton);
  } else {
    ImportVendorDialog::accept();
  }

}

void ImportVendorDialogImpl::enterWhatsThisMode()
{
   QWhatsThis::enterWhatsThisMode();
}
