/****************************************************************************
**
** $Id: ResolvDialogImpl.cpp,v 1.25 2004/01/11 22:28:40 riemer Exp $
**
** Copyright (C) 2001-2004 Frank Hemer <frank@hemer.org>
**
**
**----------------------------------------------------------------------------
**
**----------------------------------------------------------------------------
**
** LinCVS is available under two different licenses:
**
** If LinCVS is linked against the GPLed version of Qt 
** LinCVS is released under the terms of GPL also.
**
** If LinCVS is linked against a nonGPLed version of Qt 
** LinCVS is released under the terms of the 
** LinCVS License for non-Unix platforms (LLNU)
**
**
** LinCVS License for non-Unix platforms (LLNU):
**
** Redistribution and use in binary form, without modification, 
** are permitted provided that the following conditions are met:
**
** 1. Redistributions in binary form must reproduce the above copyright
**    notice, this list of conditions and the following disclaimer in the
**    documentation and/or other materials provided with the distribution.
** 2. It is not permitted to distribute the binary package under a name
**    different than LinCVS.
** 3. The name of the authors may not be used to endorse or promote
**    products derived from this software without specific prior written
**    permission.
** 4. The source code is the creative property of the authors.
**    Extensions and development under the terms of the Gnu Public License
**    are limited to the Unix platform. Any distribution or compilation of 
**    the source code against libraries licensed other than gpl requires 
**    the written permission of the authors.
**
**
** THIS SOFTWARE IS PROVIDED BY THE AUTHOR "AS IS" AND ANY EXPRESS OR 
** IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED 
** WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE 
** ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY 
** DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL 
** DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE 
** GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS 
** INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, 
** WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING 
** NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS 
** SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
**
**
**
** LinCVS License for Unix platforms:
**
** This program is free software; you can redistribute it and/or modify 
** it under the terms of the GNU General Public License as published by 
** the Free Software Foundation; either version 2 of the License, or 
** (at your  option) any later version.  This program is distributed in 
** the hope that it will be useful, but WITHOUT ANY WARRANTY; without 
** even the implied warranty of MERCHANTABILITY or FITNESS FOR A 
** PARTICULAR PURPOSE.
**
** See the GNU General Public License for more details.
**
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software Foundation,
** Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
**
*****************************************************************************/

#include "config.h"

#include <qapplication.h>
#include <qtimer.h>
#include <qfile.h>
#include <qpainter.h>
#include <qmessagebox.h>
#include <qfont.h>
#include <qwhatsthis.h>
#include <qpushbutton.h>

#include "ResolvDialogImpl.h"
#include "globals.h"
#include "FileTableImpl.h"

//FileTable ID's
const int LEFT = 0;
const int RIGHT = 1;
const int MERGE = 2;

//colors for the tables
const QColor sourceBg(255,255,255);//white
const QColor mergeBg(255,165,0);//orange
const QColor numBg(190,190,190);//gray
const QColor numSelect(0,255,0);//green
const QColor conflict(255,80,60);//light red
const QColor conflictSelect(120,160,215);//light blue
const QColor merge(60,225,255);//light red
const QColor marker(190,190,190);//gray
const QColor fillColor(125,255,125);//green


ResolvDialogImpl::ResolvDialogImpl( QString fileName,
				    const QIconSet &whatsThisIconSet,
				    QWidget* parent,
				    const char* name,
				    bool modal,
				    WFlags fl )
  : ResolvDialog( parent, name, modal, fl)
{
  m_pWhatsThis->setIconSet(whatsThisIconSet);
  m_pWhatsThis->setMaximumWidth(m_pWhatsThis->height());

  m_stop = false;
  m_fileName = fileName;
  installEventFilter( this);//get rid of the anoying whatsthis popup menu on a rightclick

  TextLabel1->setText(tr("Sandbox (unmerged)"));
  TextLabel3->setText(tr("Result (preview)"));

  if (!Font::g_diff.isEmpty()) {
    QFont f;
    if (f.fromString(Font::g_diff)) {
      m_FileLeft->setFont(f);
      m_FileRight->setFont(f);
      m_FileMerged->setFont(f);
    } else Font::g_diff = QString::null;
  }

  m_FileLeft->setBackground(sourceBg);
  m_FileRight->setBackground(sourceBg);
  m_FileMerged->setBackground(mergeBg);

  m_FileLeft->setID(LEFT);
  m_FileRight->setID(RIGHT);
  m_FileMerged->setID(MERGE);

  connect(m_FileLeft,SIGNAL(xScrollValueChanged(int)),m_FileRight,SLOT(setXScrollPos(int)));
  connect(m_FileRight,SIGNAL(xScrollValueChanged(int)),m_FileMerged,SLOT(setXScrollPos(int)));
  connect(m_FileMerged,SIGNAL(xScrollValueChanged(int)),m_FileLeft,SLOT(setXScrollPos(int)));
  connect(m_FileLeft,SIGNAL(yScrollValueChanged(int)),m_FileRight,SLOT(setYScrollPos(int)));
  connect(m_FileRight,SIGNAL(yScrollValueChanged(int)),m_FileMerged,SLOT(setYScrollPos(int)));
  connect(m_FileMerged,SIGNAL(yScrollValueChanged(int)),m_FileLeft,SLOT(setYScrollPos(int)));

  connect(m_FileLeft,SIGNAL(focused(int,int)),m_FileRight,SLOT(setFocus(int,int)));
  connect(m_FileLeft,SIGNAL(focused(int,int)),m_FileMerged,SLOT(setFocus(int,int)));
  connect(m_FileRight,SIGNAL(focused(int,int)),m_FileLeft,SLOT(setFocus(int,int)));
  connect(m_FileRight,SIGNAL(focused(int,int)),m_FileMerged,SLOT(setFocus(int,int)));
  connect(m_FileMerged,SIGNAL(focused(int,int)),m_FileLeft,SLOT(setFocus(int,int)));
  connect(m_FileMerged,SIGNAL(focused(int,int)),m_FileRight,SLOT(setFocus(int,int)));

  connect(m_FileLeft,SIGNAL(unselected(int,int)),this,SLOT(unselected(int,int)));
  connect(m_FileLeft,SIGNAL(unSelectedBlock(int,int,int)),this,SLOT(unSelectedBlock(int,int,int)));
  connect(m_FileLeft,SIGNAL(selectionChanged(int,int,QString)),this,SLOT(selected(int,int,QString)));
  connect(m_FileLeft,SIGNAL(selectedBlock(int,int,int)),this,SLOT(selectedBlock(int,int,int)));

  connect(m_FileRight,SIGNAL(unselected(int,int)),this,SLOT(unselected(int,int)));
  connect(m_FileRight,SIGNAL(unSelectedBlock(int,int,int)),this,SLOT(unSelectedBlock(int,int,int)));
  connect(m_FileRight,SIGNAL(selectionChanged(int,int,QString)),this,SLOT(selected(int,int,QString)));
  connect(m_FileRight,SIGNAL(selectedBlock(int,int,int)),this,SLOT(selectedBlock(int,int,int)));

  connect(m_FileMerged,SIGNAL(clear(int,int)),this,SLOT(clear(int,int)));
  connect(m_FileMerged,SIGNAL(clicked(int,int)),this,SLOT(clear(int,int)));

  QTimer::singleShot(0, this, SLOT(setFile()));//move eventhandling to dialog eventloop
}

ResolvDialogImpl::~ResolvDialogImpl() {
}

bool ResolvDialogImpl::eventFilter( QObject *o, QEvent *e ) {
  if ( e->type() == QEvent::ContextMenu ) {
    return TRUE; // eat event
  } else {
    // standard event processing
    return QWidget::eventFilter( o, e );
  }
}

void ResolvDialogImpl::cancelRead() {
  m_stop = true;
  m_FileLeft->cancelRead();
  m_FileRight->cancelRead();
  m_FileMerged->cancelRead();
}

void ResolvDialogImpl::setFile() {

  QApplication::setOverrideCursor(waitCursor);
  QString fName = m_fileName.mid(m_fileName.findRev("/")+1);
  ProgressDialog* progressDlg = new ProgressDialog(this, "ProgressDialog", true,
						   Qt::WStyle_Customize | Qt::WStyle_NoBorder | WDestructiveClose);
  connect(progressDlg->CancelButton,SIGNAL(clicked()),this,SLOT(cancelRead()));
  progressDlg->m_InfoMessage->setText(tr("Reading file: "));
  progressDlg->m_InfoText->setText(fName);
  progressDlg->show();
  progressDlg->update();

  FileTableImpl* fL = m_FileLeft;
  FileTableImpl* fR = m_FileRight;
  FileTableImpl* fM = m_FileMerged;

  m_blockVector = new QPtrVector<block>(100);
  m_blockVector->setAutoDelete(true);
  m_currentBlock = NULL;

  QString parse = "";
  int mode = 0;
  int maxWidth = 0;
  QFontMetrics fm(font());

  QFile f;
  QString line;

  f.setName(m_fileName);
  if(f.open(IO_ReadOnly)) {
    QTextStream textStream(&f); 

    int lineCount = 0;
    int blockStart = 0;
    
    int leftLineCount = 0;
    int rightLineCount = 0;
    int mergeLineCount = 0;
    QString leftBlock = "";
    QString rightBlock = "";
    QString mergeBlock = "";
    QString conflictSeparator;

    while(!textStream.atEnd()) {
      line = textStream.readLine();
      lineCount++;
    }
    f.reset();
    fL->reserveLines(lineCount);
    fR->reserveLines(lineCount);
    fM->reserveLines(lineCount);

    lineCount = 0;
    while(!textStream.atEnd()) {

      qApp->processEvents();
      if (m_stop) break;

      line = textStream.readLine();
      int width = fm.width(line);
      if (width>maxWidth) maxWidth = width;
      switch (mode) {
        case 0: {
	  if (line.startsWith("<<<<<<<")) {
	    if (!parse.isEmpty()) {
	      fL->addLine(parse,true);
	      fR->addLine(parse,true);
	      fM->addLine(parse,true);
	    }

	    fL->addLine(line+"\n",true,numSelect,numBg,marker,marker);
	    fM->addLine(line+"\n",true,numSelect,numBg,marker,marker);

	    blockStart = lineCount+1;
	    addLine(lineCount);
	    initBlock();
	    mode = 1;
	    parse = "";
	  } else {
	    parse += line+"\n";
	    addLine(lineCount);
	  }
	  break;
	}
        case 1: {
	  if (line.startsWith("=======")) {
	    conflictSeparator = line+"\n";
	    fR->addLine(conflictSeparator,true,numSelect,numBg,marker,marker);
	    leftBlock += parse;
	    mergeBlock += parse+line+"\n";
	    mode = 2;
	    addLine(lineCount);
	    parse = "";
	  } else {
	    parse += line+"\n";
	    addLine(lineCount);
	    leftLineCount++;
	    mergeLineCount++;
	  }
	  break;
	}
        case 2: {
	  if (line.startsWith(">>>>>>>")) {
	    TextLabel2->setText(tr("Revision: ")+line.mid(8));
	    rightBlock += parse;

	    if (leftBlock.isEmpty()) {
	      leftBlock += "\n";
	      leftLineCount++;
	    }
	    if (rightBlock.isEmpty()) {
	      rightBlock += "\n";
	      rightLineCount++;
	    }

	    fL->addLine(leftBlock,false,numSelect,numBg,conflictSelect,conflict);
	    fR->addLine(rightBlock,false,numSelect,numBg,conflictSelect,conflict);
	    fM->addLine(mergeBlock,true,numSelect,numBg,merge,merge);

	    leftBlock = "";
	    rightBlock = "";

	    int i = 0;
	    for (i=leftLineCount;i<=mergeLineCount;i++) {
	      leftBlock += "\n";
	    }
	    for (i=rightLineCount;i<=mergeLineCount;i++) {
	      rightBlock += "\n";
	    }

	    fL->addLine(leftBlock,true,numSelect,numBg,fillColor,fillColor);
	    fR->addLine(rightBlock,true,numSelect,numBg,fillColor,fillColor);

	    fL->addLine(conflictSeparator,true,numSelect,numBg,marker,marker);
	    fR->addLine(line+"\n",true,numSelect,numBg,marker,marker);
	    fM->addLine(line+"\n",true,numSelect,numBg,marker,marker);

	    setCurrentBlock(blockStart,lineCount-1);
	    m_currentBlock = NULL;
	    addLine(lineCount);

	    leftBlock = "";
	    rightBlock = "";
	    mergeBlock = "";
	    leftLineCount = 0;
	    rightLineCount = 0;
	    mergeLineCount = 0;
	    mode = 0;
	    parse = "";
	  } else {
	    parse += line+"\n";
	    mergeBlock += line+"\n";
	    addLine(lineCount);
	    rightLineCount++;
	    mergeLineCount++;
	  }
	  break;
	}
      }
      lineCount++;
    }

    if (!parse.isEmpty() ) {
      fL->addLine(parse,true);
      fR->addLine(parse,true);
      fM->addLine(parse,true);
    }
    fL->adjustSize(maxWidth);
    fR->adjustSize(maxWidth);
    fM->adjustSize(maxWidth);
    fL->setXScrollPos(0);
    fL->setYScrollPos(0);

    f.close();
    m_linesTotal = lineCount;
  }
  progressDlg->hide();
  delete(progressDlg);

  update();
  QApplication::restoreOverrideCursor();

  if (m_stop) reject();
}

void ResolvDialogImpl::addLine(unsigned int line) {
  if (line >= m_blockVector->size()) m_blockVector->resize(m_blockVector->size()+100);
  m_blockVector->insert(line,m_currentBlock);
}

void ResolvDialogImpl::initBlock() {
  m_currentBlock = new block;
  m_currentBlock->from = -1;
  m_currentBlock->to = -1;
  m_currentBlock->fileMap[LEFT] = NULL;
  m_currentBlock->fileMap[RIGHT] = NULL;
  m_currentBlock->mergeMap = NULL;
}

void ResolvDialogImpl::setCurrentBlock(int from, int to) {
  m_currentBlock->from = from;
  m_currentBlock->to = to;
  
  m_currentBlock->fileMap[LEFT] = new int[to-from];
  m_currentBlock->fileMap[RIGHT] = new int[to-from];
  m_currentBlock->mergeMap = new reference[to-from];

  m_currentBlock->pos = 0;

  cBlock* cbl = new cBlock;
  cbl->from = from;
  cbl->to = to;
  if (m_conflictVector.count() >= m_conflictVector.size()) m_conflictVector.resize(m_conflictVector.size()+10);
  m_conflictVector.insert(m_conflictVector.count(),cbl);
}

int ResolvDialogImpl::appendLine(int fileTable, int line) {

  block* hlpBlock = m_blockVector->at((unsigned int)line);
  int mergePos = hlpBlock->pos;

  hlpBlock->mergeMap[mergePos].fileTable = fileTable;
  hlpBlock->mergeMap[mergePos].line = line;

  hlpBlock->fileMap[fileTable][line-hlpBlock->from] = mergePos;
  hlpBlock->pos++;

  return mergePos;
}

void ResolvDialogImpl::clear(int, int row) {
  block* hlp = m_blockVector->at((unsigned int)row);
  if (hlp) {
    if (row >= hlp->from+hlp->pos) return;
    int pos = row-hlp->from;
    if (hlp->mergeMap[pos].fileTable == LEFT) {
      m_FileLeft->setSelected(hlp->mergeMap[pos].line,false);
    } else {
      m_FileRight->setSelected(hlp->mergeMap[pos].line,false);
    }
    unselected(hlp->mergeMap[pos].fileTable,hlp->mergeMap[pos].line);
  }
}

void ResolvDialogImpl::selected(int id,int row,QString txt) {

  block* hlp = m_blockVector->at((unsigned int)row);
  if (hlp) {
    int pos = appendLine(id,row);
    if (!pos) {//clear block
      QString* tmp = new QString();
      int i = 0;
      for (i=hlp->from;i<=hlp->to;i++) {
	(*tmp) += m_FileMerged->getText(i)+"\n";
	m_FileMerged->setText(i,"");
      }
      hlp->oriTxt = tmp;
    }
    m_FileMerged->setText(hlp->from+pos,txt);
  }
}

void ResolvDialogImpl::unselected(int id,int row) {

  block* hlp = m_blockVector->at((unsigned int)row);
  if (hlp) {
    int pos = hlp->fileMap[id][row-hlp->from];
    int i = 0;
    int j = 0;
    m_FileMerged->move(pos+hlp->from+1,-1,hlp->pos-pos-1);
    for (i=pos+hlp->from,j=pos;j<hlp->pos-1;i++,j++) {
      hlp->mergeMap[j] = hlp->mergeMap[j+1];
      hlp->fileMap[hlp->mergeMap[j].fileTable][hlp->mergeMap[j].line-hlp->from]--;
    }
    hlp->pos--;
    if (hlp->pos == 0) {
      QString* tmp = hlp->oriTxt;
      i = hlp->from;
      j = 0;
      while ( (j = tmp->find("\n"))>-1) {
	m_FileMerged->setText(i++,tmp->mid(0,j));
	*tmp = tmp->mid(j+1);
      }
      *tmp = "";
    } else {
      m_FileMerged->setText(i,"");
    }
  }
}

void ResolvDialogImpl::selectedBlock(int id, int from, int to) {

  QString txt;
  int range = to-from+1;

  block* hlp = m_blockVector->at((unsigned int)from);
  if (!hlp->pos) {//clear
    QString* tmp = new QString();
    int i = 0;
    for (i=hlp->from;i<hlp->from+range;i++) {
      (*tmp) += m_FileMerged->getText(i)+"\n";
    }
    for (i=hlp->from+range;i<=hlp->to;i++) {
      (*tmp) += m_FileMerged->getText(i)+"\n";
      m_FileMerged->setText(i,"");
    }
    hlp->oriTxt = tmp;
  }

  int hFrom = hlp->from;

  switch( id) {
    case LEFT: {
      for (int i = from;i<=to; i++) {
	txt = m_FileLeft->getText(i);
	int pos = appendLine(id,i);
	m_FileMerged->setText(hFrom+pos,txt);
      }
      break;
    }
    case RIGHT: {
      for (int i = from;i<=to; i++) {
	txt = m_FileRight->getText(i);
	int pos = appendLine(id,i);
	m_FileMerged->setText(hFrom+pos,txt);
      }
      break;
    }
  }
}

void ResolvDialogImpl::unSelectedBlock(int id, int from, int to) {

  block* hlp = m_blockVector->at((unsigned int)from);
  if (hlp) {

    int j;
    int blockPos;
    int lower = hlp->pos;

    for ( blockPos=from; blockPos <= to; blockPos++) {//reorganize mergeFile lines

      int mappedMergePos = hlp->fileMap[id][blockPos-hlp->from];
      if (mappedMergePos < lower) lower = mappedMergePos;//find startpoint for rewrite

      for (j=mappedMergePos;j<hlp->pos-1;j++) {//update pointers
	hlp->mergeMap[j] = hlp->mergeMap[j+1];
	hlp->fileMap[hlp->mergeMap[j].fileTable][hlp->mergeMap[j].line-hlp->from]--;
      }
      hlp->pos--;
    }

    if (hlp->pos == 0) {//reset to unchanged
      QString* tmp = hlp->oriTxt;
      int i = hlp->from;
      j = 0;
      while ( (j = tmp->find("\n"))>-1) {
	m_FileMerged->setText(i++,tmp->mid(0,j));
	*tmp = tmp->mid(j+1);
      }
      *tmp = "";
    } else {//modify m_FileMerged

      for (j=lower;j<hlp->pos;j++) {

	switch(hlp->mergeMap[j].fileTable) {
	  case LEFT: {
	    m_FileMerged->setText(hlp->from+j,m_FileLeft->getText(hlp->mergeMap[j].line));
	    break;
	  }
	  case RIGHT: {
	    m_FileMerged->setText(hlp->from+j,m_FileRight->getText(hlp->mergeMap[j].line));
	    break;
	  }
	}
      }
      for (j=hlp->from+hlp->pos;j<=hlp->from+hlp->pos+to-from;j++) {
	m_FileMerged->setText(j,"");
      }
    }
  }
}

void ResolvDialogImpl::writeFile() {

  QFile f(m_fileName);
  QString line;
  QString lf = getSystemLF();
  block* hlp = NULL;

  if(f.open(IO_WriteOnly)) {
    QTextStream textStream(&f); 
    int i = 0;
    int to = -1;
    int skipLines = 0;
    while (true) {

      line = m_FileMerged->getText(i++);
      if (line.isNull()) break;
      if (i == to) {
	continue;
      }
      if (i>to && (skipLines)) {
	skipLines--;
	continue;
      }
      if ( (hlp = m_blockVector->at(i))) {//one in advance
	if (hlp->pos) {
	  if (i == hlp->from) {
	    to = hlp->from+hlp->pos+1;
	    skipLines = hlp->to+2-to;//skip empty lines
	    continue;
	  }
	}
      }
      textStream << line;
      textStream << lf;
    }
    f.flush();
    f.close();

    accept();

  } else {

    QString fName = m_fileName.mid(m_fileName.findRev("/")+1);
    QMessageBox::warning(this, tr("Warning"), tr("Cannot save:\n") +
			 fName +
			 tr(",\nthe file is write-protected.\nPlease run edit first."),
			 QMessageBox::Ok,QMessageBox::NoButton,QMessageBox::NoButton);
  }
}

void ResolvDialogImpl::paintEvent(QPaintEvent *) {
  QPainter painter(this);

  int x1 = m_FileLeft->pos().x() + m_FileLeft->size().width();
  int x2 = m_FileRight->pos().x() + m_FileRight->size().width();
  int w1 = m_FileRight->pos().x() - x1;
  int w2 = m_FileMerged->pos().x() - x2;
  int top = m_FileLeft->pos().y();
  int h = m_FileLeft->size().height();

  m_FileLeft->adjustScrollBarOffsets(top,h);

  double ppl = (double)h/(double)m_linesTotal;

  unsigned int i;
  for (i=0;i<m_conflictVector.count();i++) {
    int y;
    int height;

    cBlock* bl = m_conflictVector.at(i);
    y = (int)((bl->from+1)*ppl);
    height = (int)((bl->to - bl->from)*ppl);
    if (height<1) {
      height = 1;
    }

    painter.setPen(conflict);
    painter.setBrush(conflict);
    painter.drawRect(x1,top+y,w1,height);
    painter.drawRect(x2,top+y,w2,height);
  }
}

void ResolvDialogImpl::nextClicked() {
  int line = m_FileLeft->getFocusedLine();
  if (line == -1) line = m_FileLeft->getTopLine();
  int i;
  for (i=0;i<(int)m_conflictVector.count();i++) {
    cBlock* bl = m_conflictVector.at(i);
    if (bl->from <= line) continue;
    else {
      m_FileLeft->setFocusRow(bl->from);
      break;
    }
  }
}

void ResolvDialogImpl::previousClicked() {
  int line = m_FileLeft->getFocusedLine();
  if (line == -1) line = m_FileLeft->getTopLine();
  int i;
  for (i=(int)m_conflictVector.count()-1;i>=0;i--) {
    cBlock* bl = m_conflictVector.at(i);
    if (bl->from >= line) continue;
    else {
      m_FileLeft->setFocusRow(bl->from);
      break;
    }
  }
}

void ResolvDialogImpl::enterWhatsThisMode()
{
   QWhatsThis::enterWhatsThisMode();
}
