/****************************************************************************
 **
 ** Copyright (C) 2001-2005 Tilo Riemer <riemer@lincvs.org> and
 **                         Frank Hemer <frank@hemer.org>
 **
 **
 **----------------------------------------------------------------------------
 **
 **----------------------------------------------------------------------------
 **
 ** LinCVS is available under two different licenses:
 **
 ** If LinCVS is linked against the GPLed version of Qt 
 ** LinCVS is released under the terms of GPL also.
 **
 ** If LinCVS is linked against a nonGPLed version of Qt 
 ** LinCVS is released under the terms of the 
 ** LinCVS License for non-Unix platforms (LLNU)
 **
 **
 ** LinCVS License for non-Unix platforms (LLNU):
 **
 ** Redistribution and use in binary form, without modification, 
 ** are permitted provided that the following conditions are met:
 **
 ** 1. Redistributions in binary form must reproduce the above copyright
 **    notice, this list of conditions and the following disclaimer in the
 **    documentation and/or other materials provided with the distribution.
 ** 2. It is not permitted to distribute the binary package under a name
 **    different than LinCVS.
 ** 3. The name of the authors may not be used to endorse or promote
 **    products derived from this software without specific prior written
 **    permission.
 ** 4. The source code is the creative property of the authors.
 **    Extensions and development under the terms of the Gnu Public License
 **    are limited to the Unix platform. Any distribution or compilation of 
 **    the source code against libraries licensed other than gpl requires 
 **    the written permission of the authors.
 **
 **
 ** THIS SOFTWARE IS PROVIDED BY THE AUTHOR "AS IS" AND ANY EXPRESS OR 
 ** IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED 
 ** WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE 
 ** ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY 
 ** DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL 
 ** DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE 
 ** GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS 
 ** INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, 
 ** WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING 
 ** NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS 
 ** SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 **
 **
 **
 ** LinCVS License for Unix platforms:
 **
 ** This program is free software; you can redistribute it and/or modify 
 ** it under the terms of the GNU General Public License as published by 
 ** the Free Software Foundation; either version 2 of the License, or 
 ** (at your  option) any later version.  This program is distributed in 
 ** the hope that it will be useful, but WITHOUT ANY WARRANTY; without 
 ** even the implied warranty of MERCHANTABILITY or FITNESS FOR A 
 ** PARTICULAR PURPOSE.
 **
 ** See the GNU General Public License for more details.
 **
 ** You should have received a copy of the GNU General Public License
 ** along with this program; if not, write to the Free Software Foundation,
 ** Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 **
 *****************************************************************************/

#include "Annotate.h"

#include <qapplication.h>
#include <assert.h>

#include "globals.h"
#include "LinCVSLog.h"
#include "AnnotateDialogImpl.h"
#include "AnnotateGrepLineDialogImpl.h"
#include "AnnotateGrepLinesDialogImpl.h"

Annotate * Annotate::annotateFile(const QIconSet &whatsThisIconSet, QWidget* parent, CvsDirListView * workBench)
{
   return setup(whatsThisIconSet, parent, workBench, false, FILE);
}

Annotate * Annotate::annotateGrepLineFile(const QIconSet &whatsThisIconSet, QWidget* parent, CvsDirListView * workBench)
{
   return setup(whatsThisIconSet, parent, workBench, false, LINEFILE);
}

Annotate * Annotate::annotateGrepLinesFile(const QIconSet &whatsThisIconSet, QWidget* parent, CvsDirListView * workBench)
{
   return setup(whatsThisIconSet, parent, workBench, false, MULTILINEFILE);
}

Annotate * Annotate::annotateDir(const QIconSet &whatsThisIconSet, QWidget* parent, CvsDirListView * workBench)
{
   return setup(whatsThisIconSet, parent, workBench, true, DIR);
}

Annotate * Annotate::annotateGrepLineDir(const QIconSet &whatsThisIconSet, QWidget* parent, CvsDirListView * workBench)
{
   return setup(whatsThisIconSet, parent, workBench, true, LINEDIR);
}

Annotate * Annotate::annotateGrepLinesDir(const QIconSet &whatsThisIconSet, QWidget* parent, CvsDirListView * workBench)
{
   return setup(whatsThisIconSet, parent, workBench, true, MULTILINEDIR);
}

Annotate * Annotate::setup(const QIconSet &whatsThisIconSet, QWidget* parent, CvsDirListView * workBench, bool isDir, int mode) {

   DirBase * dir = workBench->selectedItem();
   if (!dir) return NULL;

   if (dir->getType() == DirBase::Cvs) {

      QDialog * dlg = NULL;
      switch(mode) {
	 case FILE:
	 case DIR: {
	    
	    break;
	 }
	 case LINEFILE:
	 case LINEDIR: {
	    dlg = new AnnotateGrepLineDialogImpl(whatsThisIconSet,parent,"AnnotateGrepLine",true);
	    break;
	 }
	 case MULTILINEFILE:
	 case MULTILINEDIR: {
	    dlg = new AnnotateGrepLinesDialogImpl(whatsThisIconSet,parent,"AnnotateGrepLine",true);
	    break;
	 }
	 default: {
	    assert(false);
	 }
      }

      if ( dlg && !dlg->exec() ) {
	 delete dlg;
	 dlg = NULL;
      } else {
	 Annotate * p = new Annotate(whatsThisIconSet, parent, workBench, dir, isDir, mode);
	 if (dlg) p->m_dlg = dlg;
	 p->acceptCvs();
	 return p;
      }
   } else {
      qDebug("Annotate: Method not implemented");
   }
   return NULL;
}

Annotate::Annotate(const QIconSet &whatsThisIconSet, QWidget* parent, CvsDirListView * workBench, DirBase * dir, bool isDir, int mode)
   :  m_whatsThisIconSet(whatsThisIconSet),
      m_parent(parent),
      m_workBench(workBench),
      m_dir(dir),
      m_dlg(NULL),
      m_isDir(isDir),
      m_mode(mode),
      m_file(""),
      m_rev("")
{
   connect(this,SIGNAL(deleteObject(QObject *)),parent,SLOT(slot_deleteObject(QObject *)));
}

Annotate::~Annotate() {
   if (m_dlg) delete m_dlg;
   m_dlg = NULL;
}

void Annotate::acceptCvs() {

   assert(m_dir);

   m_searchLines.clear();
   m_caseSensitive = false;
   QString command = "annotate";
   QString file = "";
   int cmd = -1;

   switch(m_mode) {
      case FILE:
      case LINEFILE:
      case MULTILINEFILE: {
         m_dir->getNameAndRevOfFirstSelectedFile(&file, &m_rev);
         if (file.isEmpty() || m_rev.isEmpty()) {
            reject();
            return;
         }
         m_file = file;
         file = "-r " + m_rev + " " + masqWs(file.replace("\"", PLACEHOLDER_FOR_QUOTATION_MARKS));
         break;
      }
      case LINEDIR:
      case DIR:
      case MULTILINEDIR: {
	 QString dirBranch = m_dir->getDirTag();
	 QString dirType = "";
	 if (!dirBranch.isEmpty()) {
	    dirType = dirBranch.left(1);
	    dirBranch = dirBranch.mid(1);
	    if (dirType=="T" || dirType=="N") {
	       command += " -r " + dirBranch;
	    } else if (dirType=="D") {
	       command += " -D " + dirBranch;
	    } else {
	       Debug::g_pLog->log(Debug::LL_THE_OLD_MAN_AND_THE_SEA,"unknown dirBranchType: ->"+dirType+"<-");
	    }
	 }
	 m_file = m_dir->repository();
	 break;
      }
      default: {
	 assert(false);
      }
   }

   switch(m_mode) {
      case FILE:
      case DIR: {
	 cmd = CVS_ANNOTATE_CMD;
	 break;
      }
      case LINEFILE:
      case LINEDIR: {
	 AnnotateGrepLineDialogImpl * dlg = static_cast<AnnotateGrepLineDialogImpl *>(m_dlg);
	 assert(dlg);
	 m_searchLines.append( dlg->getSearchLine() );
	 m_caseSensitive = dlg->isCaseSensitive();
	 cmd = CVS_ANNOTATE_GREP_LINE_CMD;
	 break;
      }
      case MULTILINEFILE:
      case MULTILINEDIR: {
	 AnnotateGrepLinesDialogImpl * dlg = static_cast<AnnotateGrepLinesDialogImpl *>(m_dlg);
	 assert(dlg);
	 m_searchLines = dlg->getSearchLines();
	 m_caseSensitive = dlg->isCaseSensitive();
	 cmd = CVS_ANNOTATE_GREP_LINES_CMD;
	 break;
      }
      default: {
	 assert(false);
      }
   }

   delete m_dlg;
   m_dlg = NULL;

   QString dir = m_dir->fullName();
   QString topModule = m_dir->topControlledDir()->relativeName();
   
   callInteractive( topModule, dir, command,
	 file, cmd,
	 ExtApps::g_cvsRsh.path,  //additional options of cvsRsh not supported yet
	 NOROOT);
}

void Annotate::reject() {
   emit deleteObject(this);
}

void Annotate::cvsCallStarted() {
   QApplication::setOverrideCursor(Qt::waitCursor);
}

void Annotate::cvsCallFinished() {
   QApplication::restoreOverrideCursor();
}

void Annotate::afterCall(int cmd,CvsBuffer * buf,bool failed) {
   cvsCallFinished();
   if (failed) {
      reject();
      return;
   }

   AnnotateDialogImpl * dlg = new AnnotateDialogImpl ( m_whatsThisIconSet,
	 m_parent, "AnnotateDlg", LookAndFeel::g_nonModalF | WDestructiveClose);

   switch( cmd) {
      case CVS_ANNOTATE_CMD: {
	 dlg->parseCvsAnnotate (m_file, buf, m_rev);
	 break;
      }
      case CVS_ANNOTATE_GREP_LINE_CMD: {
	 dlg->parseCvsAnnotateGrepLine (m_file, buf, m_rev, m_searchLines[0], m_caseSensitive);
	 break;
      }
      case CVS_ANNOTATE_GREP_LINES_CMD: {
	 dlg->parseCvsAnnotateGrepLines (m_file, buf, m_rev, &m_searchLines, m_caseSensitive);
	 break;
      }
      default: {
	 qDebug("Annotate::afterCall: Unknown cmd");
      }
   }
   dlg->show();
   reject();
}
