/* **********************************************************
 * Copyright 1998 VMware, Inc.  All rights reserved. -- VMware Confidential
 * **********************************************************/


/*
 * hgfsProto.h -- 
 *
 * Header file for data types and message formats used in the
 * Host/Guest File System (hgfs) protocol. [bac]
 */


#ifndef _HGFS_PROTO_H_
# define _HGFS_PROTO_H_

#define INCLUDE_ALLOW_USERLEVEL
#define INCLUDE_ALLOW_MODULE
#define INCLUDE_ALLOW_DISTRIBUTE
#include "includeCheck.h"

#include "vm_basic_types.h"
#include "hgfs.h"

/*
 * Handle used by the server to identify files and searches. Used
 * by the driver to match server replies with pending requests.
 */
typedef uint32 HgfsHandle;


/*
 * Opcodes for server operations.
 *
 * Changing the ordering of this enum will break the protocol; new ops
 * should be added at the end (but before HGFS_OP_MAX).
 */
typedef enum {
   HGFS_OP_OPEN,          /* Open file */
   HGFS_OP_READ,          /* Read from file */
   HGFS_OP_WRITE,         /* Write to file */
   HGFS_OP_CLOSE,         /* Close file */
   HGFS_OP_SEARCH_OPEN,   /* Start new search */
   HGFS_OP_SEARCH_READ,   /* Get next search response */
   HGFS_OP_SEARCH_CLOSE,  /* End a search */
   HGFS_OP_GETATTR,       /* Get file attributes */
   HGFS_OP_SETATTR,       /* Set file attributes */
   HGFS_OP_CREATE_DIR,    /* Create new directory */
   HGFS_OP_DELETE_FILE,   /* Delete a file */
   HGFS_OP_DELETE_DIR,    /* Delete a directory */
   HGFS_OP_RENAME,        /* Rename a file or directory */
   HGFS_OP_QUERY_VOLUME_INFO, /* Query volume information */
   HGFS_OP_MAX,           /* Dummy op, must be last in enum */
} HgfsOp;


/*
 * File types, used in HgfsAttr. We only support regular files and
 * directories.
 *
 * Changing the order of this enum will break the protocol; new types
 * should be added at the end.
 */
typedef enum {
   HGFS_FILE_TYPE_REGULAR,
   HGFS_FILE_TYPE_DIRECTORY,
} HgfsFileType;


/*
 * Open flags
 *
 * Changing the order of this enum will break stuff.
 */
typedef enum {             //  File doesn't exist   File exists
   HGFS_OPEN,              //  error
   HGFS_OPEN_EMPTY,        //  error               size = 0
   HGFS_OPEN_CREATE,       //  create
   HGFS_OPEN_CREATE_SAFE,  //  create              error
   HGFS_OPEN_CREATE_EMPTY, //  create              size = 0
} HgfsOpenFlags;

/*
 * Write flags
 */
typedef uint8 HgfsWriteFlags;

#define HGFS_WRITE_APPEND 1

/*
 * Permissions bits.
 *
 * These are intentionally similar to Unix permissions bits, and we
 * convert to/from Unix permissions using simple shift operations, so
 * don't change these or you will break things.
 */
typedef uint8 HgfsPermissions;

#define HGFS_PERM_READ  4
#define HGFS_PERM_WRITE 2
#define HGFS_PERM_EXEC  1

/*
 * Flags to indicate in a setattr request which fields should be
 * updated.
 */
typedef uint8 HgfsAttrChanges;

#define HGFS_ATTR_SIZE         1
#define HGFS_ATTR_CREATE_TIME  2
#define HGFS_ATTR_ACCESS_TIME  4
#define HGFS_ATTR_WRITE_TIME   8
#define HGFS_ATTR_CHANGE_TIME  16
#define HGFS_ATTR_PERMISSIONS  32

/*
 * File attributes
 *
 * The four time fields below are in Windows NT format, which is in
 * units of 100ns since Jan 1, 1601, UTC.
 */
typedef
#include "vmware_pack_begin.h"
struct HgfsAttr {
   HgfsFileType type;            /* File type */
   uint64 size;                  /* File size (in bytes) */
   uint64 creationTime;          /* Creation time. Ignored by linux. */
   uint64 accessTime;            /* Time of last access */
   uint64 writeTime;             /* Time of last write */
   uint64 attrChangeTime;        /* Time file attributess were last
                                  * changed. Ignored by Windows. */
   HgfsPermissions permissions;  /* Permissions bits */
}
#include "vmware_pack_end.h"
HgfsAttr;

/*
 * Cross-platform filename representation
 *
 * Cross-platform (CP) names are represented by a string with each
 * path component separated by NULs, and terminated with a final NUL,
 * but with no leading path separator.
 *
 * For example, the representations of a linux and windows name
 * are as follows, with "0" meaning NUL.
 *
 * Original name             Cross-platform name
 * -----------------------------------------------------
 * "/home/bac/temp"    ->    "home0bac0temp0"
 * "C:\temp\file.txt"  ->    "C0temp0file.txt0"
 *
 * Note that as in the example above, windows should strip the colon
 * off of drive letters as part of the conversion. Aside from that,
 * all characters in each path component should be left unescaped and
 * unmodified. Each OS is responsible for escaping any characters that
 * are not legal in its filenames when converting FROM the CP name
 * format, and unescaping them when converting TO the CP name format.
 *
 * In some requests (OPEN, GETATTR, SETATTR, DELETE, CREATE_DIR) the
 * CP name is used to represent a particular file, but it is also used
 * to represent a search pattern for looking up files using
 * SEARCH_OPEN.
 */ 
typedef
#include "vmware_pack_begin.h"
struct HgfsFileName {
   uint32 length; /* does NOT include terminating NUL */
   char name[1];
}
#include "vmware_pack_end.h"
HgfsFileName;


/*
 * Request/reply structs. These are the first members of all
 * operation request and reply messages, respectively.
 */

typedef
#include "vmware_pack_begin.h"
struct HgfsRequest {
   HgfsHandle id;        /* Opaque request ID used by the driver */
   HgfsOp op;
}
#include "vmware_pack_end.h"
HgfsRequest;

typedef
#include "vmware_pack_begin.h"
struct HgfsReply {
   HgfsHandle id;        /* Opaque request ID used by the driver */
   HgfsStatus status;
}
#include "vmware_pack_end.h"
HgfsReply;



/*
 * Messages for our file operations
 */ 


typedef
#include "vmware_pack_begin.h"
struct HgfsRequestOpen {
   HgfsRequest header;
   HgfsOpenMode mode;            /* which type of access is requested */
   HgfsOpenFlags flags;          /* which flags to open the file with */
   HgfsPermissions permissions;  /* which permissions to *create* a new file with */
   HgfsFileName fileName;
}
#include "vmware_pack_end.h"
HgfsRequestOpen;


typedef
#include "vmware_pack_begin.h"
struct HgfsReplyOpen {
   HgfsReply header;
   HgfsHandle file;      /* Opaque file ID used by the server */
}
#include "vmware_pack_end.h"
HgfsReplyOpen;


typedef
#include "vmware_pack_begin.h"
struct HgfsRequestRead {
   HgfsRequest header;
   HgfsHandle file;      /* Opaque file ID used by the server */
   uint64 offset;
   uint32 requiredSize;
}
#include "vmware_pack_end.h"
HgfsRequestRead;


typedef
#include "vmware_pack_begin.h"
struct HgfsReplyRead {
   HgfsReply header;
   uint32 actualSize;
   char payload[1];
}
#include "vmware_pack_end.h"
HgfsReplyRead;


typedef
#include "vmware_pack_begin.h"
struct HgfsRequestWrite {
   HgfsRequest header;
   HgfsHandle file;      /* Opaque file ID used by the server */
   HgfsWriteFlags flags;
   uint64 offset;
   uint32 requiredSize;
   char payload[1];
}
#include "vmware_pack_end.h"
HgfsRequestWrite;


typedef
#include "vmware_pack_begin.h"
struct HgfsReplyWrite {
   HgfsReply header;
   uint32 actualSize;
}
#include "vmware_pack_end.h"
HgfsReplyWrite;


typedef
#include "vmware_pack_begin.h"
struct HgfsRequestClose {
   HgfsRequest header;
   HgfsHandle file;      /* Opaque file ID used by the server */
}
#include "vmware_pack_end.h"
HgfsRequestClose;


typedef
#include "vmware_pack_begin.h"
struct HgfsReplyClose {
   HgfsReply header;
}
#include "vmware_pack_end.h"
HgfsReplyClose;


typedef
#include "vmware_pack_begin.h"
struct HgfsRequestSearchOpen {
   HgfsRequest header;
   HgfsFileName dirName;
}
#include "vmware_pack_end.h"
HgfsRequestSearchOpen;


typedef
#include "vmware_pack_begin.h"
struct HgfsReplySearchOpen {
   HgfsReply header;
   HgfsHandle search;    /* Opaque search ID used by the server */
}
#include "vmware_pack_end.h"
HgfsReplySearchOpen;


typedef
#include "vmware_pack_begin.h"
struct HgfsRequestSearchRead {
   HgfsRequest header;
   HgfsHandle search;    /* Opaque search ID used by the server */
   uint32 offset;        /* The first result is offset 0 */
}
#include "vmware_pack_end.h"
HgfsRequestSearchRead;


typedef
#include "vmware_pack_begin.h"
struct HgfsReplySearchRead {
   HgfsReply header;
   HgfsAttr attr;
   HgfsFileName fileName;
   /* fileName.length = 0 means "no entry at this offset" */
}
#include "vmware_pack_end.h"
HgfsReplySearchRead;


typedef
#include "vmware_pack_begin.h"
struct HgfsRequestSearchClose {
   HgfsRequest header;
   HgfsHandle search;    /* Opaque search ID used by the server */
}
#include "vmware_pack_end.h"
HgfsRequestSearchClose;


typedef
#include "vmware_pack_begin.h"
struct HgfsReplySearchClose {
   HgfsReply header;
}
#include "vmware_pack_end.h"
HgfsReplySearchClose;


typedef
#include "vmware_pack_begin.h"
struct HgfsRequestGetattr {
   HgfsRequest header;
   HgfsFileName fileName;
}
#include "vmware_pack_end.h"
HgfsRequestGetattr;


typedef
#include "vmware_pack_begin.h"
struct HgfsReplyGetattr {
   HgfsReply header;
   HgfsAttr attr;
}
#include "vmware_pack_end.h"
HgfsReplyGetattr;


typedef
#include "vmware_pack_begin.h"
struct HgfsRequestSetattr {
   HgfsRequest header;
   HgfsAttrChanges update;  /* Which fields need to be updated */
   HgfsAttr attr;
   HgfsFileName fileName;
}
#include "vmware_pack_end.h"
HgfsRequestSetattr;


typedef
#include "vmware_pack_begin.h"
struct HgfsReplySetattr {
   HgfsReply header;
}
#include "vmware_pack_end.h"
HgfsReplySetattr;


typedef
#include "vmware_pack_begin.h"
struct HgfsRequestCreateDir {
   HgfsRequest header;
   HgfsPermissions permissions;
   HgfsFileName fileName;
}
#include "vmware_pack_end.h"
HgfsRequestCreateDir;


typedef
#include "vmware_pack_begin.h"
struct HgfsReplyCreateDir {
   HgfsReply header;
}
#include "vmware_pack_end.h"
HgfsReplyCreateDir;


typedef
#include "vmware_pack_begin.h"
struct HgfsRequestDelete {
   HgfsRequest header;
   HgfsFileName fileName;
}
#include "vmware_pack_end.h"
HgfsRequestDelete;


typedef
#include "vmware_pack_begin.h"
struct HgfsReplyDelete {
   HgfsReply header;
}
#include "vmware_pack_end.h"
HgfsReplyDelete;


/*
 * The size of the HgfsFileName struct is variable depending on the
 * length of the name, so you can't use request->newName to get the
 * actual address of the new name, because where it starts is
 * dependant on how long the oldName is. To get the address of
 * newName, use this:
 *
 *          &oldName + sizeof(HgfsFileName) + oldName.length
 */
typedef
#include "vmware_pack_begin.h"
struct HgfsRequestRename {
   HgfsRequest header;
   HgfsFileName oldName;
   HgfsFileName newName; 
}
#include "vmware_pack_end.h"
HgfsRequestRename;


typedef
#include "vmware_pack_begin.h"
struct HgfsReplyRename {
   HgfsReply header;
}
#include "vmware_pack_end.h"
HgfsReplyRename;

typedef
#include "vmware_pack_begin.h"
struct HgfsRequestQueryVolume {
   HgfsRequest header;
   HgfsFileName fileName;
}
#include "vmware_pack_end.h"
HgfsRequestQueryVolume;

typedef
#include "vmware_pack_begin.h"
struct HgfsReplyQueryVolume {
   HgfsReply header;
   uint64 freeBytes;
   uint64 totalBytes;
}
#include "vmware_pack_end.h"
HgfsReplyQueryVolume;


#endif /* _HGFS_PROTO_H_ */
