/* GraphModelSupport.java
 * =========================================================================
 * This file is part of the GrInvIn project - http://www.grinvin.org
 * 
 * Copyright (C) 2005-2007 Universiteit Gent
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * A copy of the GNU General Public License can be found in the file
 * LICENSE.txt provided with the source distribution of this program (see
 * the META-INF directory in the source jar). This license can also be
 * found on the GNU website at http://www.gnu.org/licenses/gpl.html.
 * 
 * If you did not receive a copy of the GNU General Public License along
 * with this program, contact the lead developer, or write to the Free
 * Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 */

package org.grinvin;

import javax.swing.event.EventListenerList;

/**
 * Helper class for implementations of {@link GraphModel}. 
 * Models can delegate listener management and notification to
 * an object of this class.
 */
public class GraphModelSupport {
    
    //
    private EventListenerList listenerList = new EventListenerList ();
    
    // [kc] for performance reasons
    private static final Class<GraphListener>
            GRAPH_LISTENER_CLASS = GraphListener.class;
    
    // implements GraphModel
    public void addGraphListener (GraphListener l) {
        listenerList.add (GRAPH_LISTENER_CLASS, l);
    }
    
    // implements GraphModel
    public void removeGraphListener (GraphListener l) {
        listenerList.remove (GRAPH_LISTENER_CLASS, l);
    }
    
    /**
     * Notify all listeners of the fact that a vertex was added
     * to this graph.
     */
    protected void fireVertexAdded (Vertex vertex) {
        Object[] listeners = listenerList.getListenerList ();
        for (int i = listeners.length-2; i>=0; i-=2)
            if (listeners[i]==GRAPH_LISTENER_CLASS)
                ((GraphListener)listeners[i+1]).vertexAdded (vertex);
    }
    
    /**
     * Notify all listeners of the fact that an edge was added
     * to this graph.
     */
    protected void fireEdgeAdded (Edge edge) {
        Object[] listeners = listenerList.getListenerList ();
        for (int i = listeners.length-2; i>=0; i-=2)
            if (listeners[i]==GRAPH_LISTENER_CLASS)
                ((GraphListener)listeners[i+1]).edgeAdded (edge);
    }
    
    /**
     * Notify all listeners of the fact that a vertex was removed
     * from this graph.
     */
    protected void fireVertexRemoved (Vertex vertex) {
        Object[] listeners = listenerList.getListenerList ();
        for (int i = listeners.length-2; i>=0; i-=2)
            if (listeners[i]==GRAPH_LISTENER_CLASS)
                ((GraphListener)listeners[i+1]).vertexRemoved (vertex);
    }
    
    /**
     * Notify all listeners of the fact that a vertex was restored
     * to this graph.
     */
    protected void fireVertexRestored (Vertex vertex) {
        Object[] listeners = listenerList.getListenerList ();
        for (int i = listeners.length-2; i>=0; i-=2)
            if (listeners[i]==GRAPH_LISTENER_CLASS)
                ((GraphListener)listeners[i+1]).vertexRestored (vertex);
    }
    
    /**
     * Notify all listeners of the fact that an edge was removed
     * from this graph.
     */
    protected void fireEdgeRemoved (Edge edge) {
        Object[] listeners = listenerList.getListenerList ();
        for (int i = listeners.length-2; i>=0; i-=2)
            if (listeners[i]==GRAPH_LISTENER_CLASS)
                ((GraphListener)listeners[i+1]).edgeRemoved (edge);
    }
    
    /**
     * Notify all listeners of the fact that an edge was restored
     * to this graph.
     */
    protected void fireEdgeRestored (Edge edge) {
        Object[] listeners = listenerList.getListenerList ();
        for (int i = listeners.length-2; i>=0; i-=2)
            if (listeners[i]==GRAPH_LISTENER_CLASS)
                ((GraphListener)listeners[i+1]).edgeRestored (edge);
    }
    
    /**
     * Notify all listeners of the fact that vertex information
     * was changed.
     */
    protected void fireVertexChanged (Vertex vertex) {
        Object[] listeners = listenerList.getListenerList ();
        for (int i = listeners.length-2; i>=0; i-=2)
            if (listeners[i]==GRAPH_LISTENER_CLASS)
                ((GraphListener)listeners[i+1]).vertexChanged (vertex);
    }
    
    /**
     * Notify all listeners of the fact that edge information
     * was changed.
     */
    protected void fireEdgeChanged (Edge edge) {
        Object[] listeners = listenerList.getListenerList ();
        for (int i = listeners.length-2; i>=0; i-=2)
            if (listeners[i]==GRAPH_LISTENER_CLASS)
                ((GraphListener)listeners[i+1]).edgeChanged (edge);
    }
    
    /**
     * Notify all listeners of a bulk change to this graph.
     */
    protected void fireGraphChanged () {
        Object[] listeners = listenerList.getListenerList ();
        for (int i = listeners.length-2; i>=0; i-=2)
            if (listeners[i]==GRAPH_LISTENER_CLASS)
                ((GraphListener)listeners[i+1]).graphChanged ();
    }
}
