/* Operators.java
 * =========================================================================
 * This file is part of the GrInvIn project - http://www.grinvin.org
 * 
 * Copyright (C) 2005-2007 Universiteit Gent
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * A copy of the GNU General Public License can be found in the file
 * LICENSE.txt provided with the source distribution of this program (see
 * the META-INF directory in the source jar). This license can also be
 * found on the GNU website at http://www.gnu.org/licenses/gpl.html.
 * 
 * If you did not receive a copy of the GNU General Public License along
 * with this program, contact the lead developer, or write to the Free
 * Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 */

package org.grinvin.engine.apengine;


public class Operators {
    
    //
    public interface Operator {
        public String renderString(String[] args);
    }
    
    //
    public enum UnaryOperator implements Operator {
        MINUSONE("-1") {
            public double eval(double x) { return x - 1; }
            public String renderString(String[] args) { return args[0] + " - 1"; }
        },
        PLUSONE("+1") {
            public double eval(double x) { return x + 1; }
            public String renderString(String[] args) { return args[0] + " + 1"; }
        },
        TIMESTWO("*2") {
            public double eval(double x) { return x * 2; }
            public String renderString(String[] args) { return args[0] + " * 2"; }
        },
        DIVTWO("/2") {
            public double eval(double x) { return x / 2; }
            public String renderString(String[] args) { return args[0] + " / 2"; }
        },
        SQUARED("^2") {
            public double eval(double x) { return x * x; }
            public String renderString(String[] args) { return args[0] + " ^ 2"; }
        },
        NEGATE("-") {
            public double eval(double x) { return -x; }
            public String renderString(String[] args) { return "-" + args[0]; }
        },
        ONEOVER("1/") {
            public double eval(double x) {
                if(almostZero(x))
                    return Double.NaN;
                else
                    return 1 / x;
            }
            public String renderString(String[] args) { return "1 / " +  args[0]; }
        };
        
        private String name;
        
        UnaryOperator(String name) {
            this.name = name;
        }
        
        public String toString() {
            return name;
        }
        
        public abstract double eval(double x);
    }
    
    //
    public enum BinaryOperator implements Operator {
        PLUS("+", true) { 
            public double eval(double x, double y) { return x + y; }
            public String renderString(String[] args) { return args[0] + " + " + args[1]; }
        },
        MINUS("-", false) { 
            public double eval(double x, double y) { return x - y; }
            public String renderString(String[] args) { return args[0] + " - " + args[1]; }
        },
        TIMES("*", true) { 
            public double eval(double x, double y) { return x * y; }
            public String renderString(String[] args) { return args[0] + " * " + args[1]; }
        },
        DIVIDE("/", false) {
            public double eval(double x, double y) {
                if(almostZero(y))
                    return Double.NaN;
                else
                    return x / y;
            }
            public String renderString(String[] args) { return args[0] + " / " + args[1]; }
        },
        POWER("^", false) { 
            public double eval(double x, double y) { return Math.pow(x, y); }
            public String renderString(String[] args) { return args[0] + " ^ " + args[1]; }
        };
        
        private String name;
        
        private boolean commutative;
        
        BinaryOperator(String name, boolean commutative) {
            this.name = name;
            this.commutative = commutative;
        }
        
        public String toString() {
            return name;
        }
        
        public abstract double eval(double x, double y);
        
        public boolean isCommutative() {
            return commutative;
        }
    }
    
    //
    public static class Invariant implements Operator {
        
        public int number;
        
        public Invariant(int number) {
            this.number = number;
        }
        
        public String toString() {
            return "I" + number;
        }
        
        public String renderString(String[] args) {
            return toString();
        }
    }
    
    //
    private static boolean almostZero(double value) {
        return Math.abs(value) < (1e-6);
    }
    
}
