/* GraphBrowserWindow.java
 * =========================================================================
 * This file is part of the GrInvIn project - http://www.grinvin.org
 * 
 * Copyright (C) 2005-2007 Universiteit Gent
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * A copy of the GNU General Public License can be found in the file
 * LICENSE.txt provided with the source distribution of this program (see
 * the META-INF directory in the source jar). This license can also be
 * found on the GNU website at http://www.gnu.org/licenses/gpl.html.
 * 
 * If you did not receive a copy of the GNU General Public License along
 * with this program, contact the lead developer, or write to the Free
 * Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 */

package org.grinvin.graphbrowser;

import be.ugent.caagt.swirl.commands.AttributedCommandManager;

import java.awt.BorderLayout;
import java.io.InputStream;
import java.util.ResourceBundle;
import javax.swing.JFrame;
import javax.swing.event.ListDataEvent;

import org.grinvin.help.HelpManager;
import org.grinvin.list.GraphInvariantListModel;
import org.grinvin.list.GraphList;
import org.grinvin.list.GraphListElement;
import org.grinvin.list.GraphListModel;
import org.grinvin.list.GraphListModelListener;
import org.grinvin.list.InvariantList;

import org.pietschy.command.LoadException;

/**
 * A window that displays the GraphList/GraphTableList
 */
public class GraphBrowserWindow extends JFrame 
        implements HasGraphList, HasInvariantList {
    
    //
    private static final ResourceBundle BUNDLE
            = ResourceBundle.getBundle("org.grinvin.graphbrowser.graphbrowserwindow");
    
    //
    private GraphBrowserMultiPanel multipanel;
    
    //
    private GraphInvariantListModel model;
    
    //
    private AttributedCommandManager commandManager;

    public String getName() {
        return model.getName ();
    }
    
    /**
     * Creates a new instance of GraphBrowserWindow
     */
    public GraphBrowserWindow(GraphInvariantListModel model) {
        super (model.getName());
       
        this.multipanel = new GraphBrowserMultiPanel(model);
        this.model = model;
        model.getGraphListModel().addGraphListModelListener(new GraphBrowserListenerHelper(this));
        add(multipanel);
        
        commandManager = new AttributedCommandManager();
        commandManager.setAttribute(GraphBrowserMultiPanel.class.getName(), multipanel);
        commandManager.setAttribute(GraphBrowserWindow.class.getName(), this);
        commandManager.setResourceBundle(BUNDLE);
        
        try {
            InputStream ins = GraphBrowserWindow.class.getResourceAsStream("graphbrowserwindow.cmds");
            commandManager.load(ins);
        } catch (LoadException lex) {
            throw new RuntimeException("Could not initialize menus", lex);
        }
        
        add(commandManager.getGroup("toolbar").createToolBar(), BorderLayout.NORTH);
        
        //make sure the GraphCellList view is the default view
        setCurrentComponent("graphCellList");
        
        setJMenuBar(commandManager.getGroup("main.menu").createMenuBar());
        
        HelpManager.enableHelpKey(this.getRootPane(), "intro");
        
        pack();
    }
    
    public String getCurrentComponent() {
        return multipanel.getCurrentComponent();
    }

    public GraphList getGraphList() {
        return model.getGraphList();
    }

    public InvariantList getInvariantList() {
        return model.getInvariantListModel();
    }

    public void setCurrentComponent(String component) throws RuntimeException {
        // TODO: the above should be solved in a different way
        if ("graphInvariantTable".equals(component)) {
            commandManager.getCommand("view.invarianttable").execute();
        } else if ("graphCellList".equals(component)) {
            commandManager.getCommand("view.graphcellist").execute();
        } else if ("graphList".equals(component)) {
            commandManager.getCommand("view.graphlist").execute();
        } else {
            throw new RuntimeException("invalid component");
        }
        
    }
    
    public void fireGraphListNameChanged() {
        this.setTitle(model.getGraphListModel().getName());
    }
    
    public void setInvariantTableDividerLocation(int location){
        multipanel.setInvariantTableDividerLocation(location);
    }
    
    public int getInvariantTableDividerLocation(){
        return multipanel.getInvariantTableDividerLocation();
    }
    
    public void setGraphListDividerLocation(int location){
        multipanel.setGraphListDividerLocation(location);
    }
    
    public int getGraphListDividerLocation(){
        return multipanel.getGraphListDividerLocation();
    }

    public void setGraphPropertiesComponentDividerLocation(int location){
        multipanel.setGraphPropertiesComponentDividerLocation(location);
    }
    
    public int getGraphPropertiesComponentDividerLocation(){
        return multipanel.getGraphPropertiesComponentDividerLocation();
    }

   //
    private class GraphBrowserListenerHelper implements GraphListModelListener {
        
        //
        private GraphBrowserWindow window;
        
        public GraphBrowserListenerHelper(GraphBrowserWindow window) {
            this.window = window;
        }
        
        public void intervalAdded(ListDataEvent e) {
            //ignore
        }
        
        public void intervalRemoved(ListDataEvent e) {
            //ignore
        }
        
        public void contentsChanged(ListDataEvent e) {
            //ignore
        }
        
        public void graphListModelNameChanged(GraphListModel graphListModel) {
            window.fireGraphListNameChanged();
        }
        
    }
    
}
